﻿using OpenCvSharp;
using OpenCvSharp.Extensions;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using cvPoint = OpenCvSharp.Point;
using cvRect = OpenCvSharp.Rect;
using cvSize = OpenCvSharp.Size;
using Point = System.Drawing.Point;

namespace TexMaster
{
    public enum CanvasType { BITMAP, VECTOR };          //# Enum Draw Type
    
    public enum UnitType { DOT, INCH, CM };

    public partial class Canvas : Form
    {
        //# 

        public Mat src, dst;
        public List<Mat> matList;
        public cvPoint start, end;

        //# curve
        public int curCnt;
        public cvPoint[] curPoints = new cvPoint[3];

        //# connect
        public List<cvPoint> connPoints;

        public Scalar penColor;
        public int penSize;
        public bool drawing;

        public MainMenu menu;
        public MoveCopy mMove;
        public Draw mDraw;
        public WorkArea mWorkarea;
        public Vector mVector;
        public Fill mFill;
        public ColorWay mColorWay; 
        public Palette mPalette;
        public ColorChange mColorChange;
        public PenManager mPen;
        public Clear mClear;
        public Kmean mGrouping;
        public Stitch mStitch;
        public PatternRepeat mPattern;
        public Stipple mStipple;
        public AutoRepeat mAutoRepeat;
        public Reflection mReflection;
        public Shadow mShadow;
        public Reproportion mReproportion;
        public Separate mSeparate;
        public Gradient mGradient;
        public Assemble mAssem;
        public Smooth mSmooth;
        public TextDraw mText;
        public Stripe mStripe;

        public Bitmap penCursor;
        public Color gBackColor;

        public UnitType unit;
        public CanvasType mDrawObject;

        Graphics gr;

        public Mat mask;       //# mask 영역
        public Mat protect;    //# 보호된 이미지
        public Mat pattern;
        public Mat restore;    //# 마지막 이미지 

        //# 패턴 반복
        public Mat dregular;        //# 반복시킬 패턴 이미지
        public Mat regular;         //# 전개된 이미지
        public Mat patternMask;     //# Masking 되어 뿌려질 이미지
        public Mat cropMat;

        public GraphicsPath direction;
        public List<PointF> curveList;              //# Bezier Points ( Point Count = 4 )

        public Canvas(Vector vec)
        {
            InitializeComponent();

            mVector = vec;
            connPoints = new List<cvPoint>();
            matList = new List<Mat>();
            gBackColor = Color.White;
            SetCursor();
            curveList = new List<PointF>();
        }

        private void Canvas_Load(object sender, EventArgs e)
        {
            dst = new Mat(new cvSize(PictureBox1.Width, PictureBox1.Height), MatType.CV_8UC4, new Scalar(255, 255, 255, 255));
            restore = dst.Clone();
            PictureBox1.ImageIpl = dst;
            matList.Add(dst);

            this.AllowDrop = true;

            mask = new Mat(new cvSize(PictureBox1.Width, PictureBox1.Height), MatType.CV_8UC1, 0);
            protect = new Mat();
        }

        private void pictureBox1_MouseDown(object sender, MouseEventArgs e)
        {
            PictureBox1.Focus();
            switch (mDrawObject)
            {
                case CanvasType.BITMAP:
                    bitmap_MouseDown(sender, e);
                    break;
                case CanvasType.VECTOR:
                    mVector.pictureBox1_MouseDown(sender, e);
                    break;
            }
        }

        private void pictureBox1_MouseMove(object sender, MouseEventArgs e)
        {
            switch (mDrawObject)
            {
                case CanvasType.BITMAP:
                    //PictureBox1.Cursor = new Cursor(penCursor.GetHicon());
                    bitmap_MouseMove(sender, e);
                    break;
                case CanvasType.VECTOR:
                    PictureBox1.Cursor = Cursors.Default;
                    mVector.pictureBox1_MouseMove(sender, e);
                    break;
            }
        }

        private void pictureBox1_MouseUp(object sender, MouseEventArgs e)
        {
            switch (mDrawObject)
            {
                case CanvasType.BITMAP:
                    bitmap_MouseUp(sender, e);
                    break;
                case CanvasType.VECTOR:
                    mVector.pictureBox1_MouseUp(sender, e);
                    break;
            }
        }

        private void pictureBox1_Paint(object sender, PaintEventArgs e)
        {
            mWorkarea.WorkArea_Paint(e.Graphics);

            if (!drawing) return;

            if (mVector.tsb_antialias.Checked)
            {
                e.Graphics.SmoothingMode = SmoothingMode.AntiAlias;
            }
            else
            {
                e.Graphics.SmoothingMode = SmoothingMode.None;
            }

            switch (mDrawObject)
            {
                case CanvasType.BITMAP:
                    switch (MainMenu.mode)
                    {
                        case MainMode.FILL:
                            mFill.fill_Paint(e);
                            break;
                        case MainMode.TEXT:
                            mText.text_Paint(e);
                            break;
                        case MainMode.STIPPLE:
                            mStipple.stipple_Paint(e);
                            break;
                        case MainMode.ASSEM:
                            mAssem.assem_Paint(e);
                            break;
                        case MainMode.GRADIENT:
                            mGradient.gradient_Paint(e);
                            break;
                        case MainMode.SEPARATE:
                            mSeparate.separate_Paint(e);
                            break;
                        case MainMode.MOVE:
                            mMove.move_Paint(e);
                            break;
                        case MainMode.REPEAT:
                            mAutoRepeat.repeat_Paint(e);
                            break;
                        case MainMode.MIRROR:
                            mReflection.reflection_Paint(e);
                            break;
                        case MainMode.ZOOM:
                            mReproportion.zoom_Paint(e);
                            break;
                        case MainMode.DRAW:
                            if (mPen.chk_select.Checked)
                            {
                                mPen.pen_Paint(e);
                            }
                            break;
                        case MainMode.COLOR:
                            mColorChange.color_Paint(e);
                            break;
                    }

                    mVector.DrawDataList(e.Graphics);
                    //# 방향 표시
                    if (direction != null)
                    {
                        e.Graphics.DrawPath(new Pen(Color.Gray, 1), direction);
                    }

                    if (mPen.penMode == PenManager.PenShape.KASURI)
                    {
                        if (mPen.currKasuriPath != null) e.Graphics.DrawPath(Pens.Black, mPen.currKasuriPath);
                        if (mPen.kasuriPaths.Count == (int)mPen.nud_branch.Value)
                        {
                            for (int i = 0; i < mPen.allBranchCount; i++)
                            {
                                GraphicsPath tempPath = new GraphicsPath();
                                List<PointF> tempPts = new List<PointF>();
                                for (int j = 0; j < mPen.pointCount; j++)
                                {
                                    PointF temp = mPen.AllPoints[i, j];
                                    int size = mVector.handleSize;

                                    if (i % (mPen.twigCount + 1) == 0) //가지만 네모 그려주기 잔가지는 안그려줌
                                    {
                                        if (j == 0) e.Graphics.DrawRectangle(Pens.Red, Rectangle.Round(new RectangleF(temp.X - size, temp.Y - size, size * 2, size * 2)));
                                        else e.Graphics.DrawRectangle(Pens.DeepSkyBlue, Rectangle.Round(new RectangleF(temp.X - size, temp.Y - size, size * 2, size * 2)));
                                    }
                                    tempPts.Add(temp);
                                }
                                tempPath.AddCurve(tempPts.ToArray());
                                e.Graphics.DrawPath(Pens.Black, tempPath);
                            }
                        }
                        else
                        {
                            for (int i = 0; i < mPen.kasuriPaths.Count; i++)
                            {
                                e.Graphics.DrawPath(Pens.Black, mPen.kasuriPaths[i]);
                                for (int j = 0; j < mPen.curvePts[i].Count; j++)
                                {
                                    PointF temp = mPen.curvePts[i][j];
                                    int size = mVector.handleSize;
                                    if (j == 0) e.Graphics.DrawRectangle(Pens.Red, Rectangle.Round(new RectangleF(temp.X - size, temp.Y - size, size * 2, size * 2)));
                                    else e.Graphics.DrawRectangle(Pens.DeepSkyBlue, Rectangle.Round(new RectangleF(temp.X - size, temp.Y - size, size * 2, size * 2)));
                                }
                            }
                        }
                    }
                    break;
                case CanvasType.VECTOR:
                    mVector.pictureBox1_Paint(e.Graphics);
                    break;
            }
        }

        //# total canvas event
        private void bitmap_MouseDown(object sender, MouseEventArgs e)
        {
            Console.WriteLine("Mode : {0}", MainMenu.mode);
            Point startPaste;
            start = end = new cvPoint(e.X, e.Y);
            //Pen Size, Color 초기화
            Color c = mPen.penColor;
            penColor = new Scalar(c.B, c.G, c.R, c.A);
            penSize = mPen.penSize;

            switch (MainMenu.mode)
            {
                case MainMode.TEXT:
                    mText.text_MouseDown(e);
                    break;
                case MainMode.SMOOTH:
                    mSmooth.smooth_MouseDown(e);
                    break;
                case MainMode.FILL:
                    mFill.fill_MouseDown(e);
                    break;
                case MainMode.ASSEM:
                    mAssem.assem_MouseDown(e);
                    break;
                case MainMode.SEPARATE:
                    mSeparate.separate_MouseDown(e);
                    break;
                case MainMode.WORKAREA:
                    mWorkarea.work_MouseDown(sender, e);
                    break;
                case MainMode.MOVE:
                    mMove.move_MouseDown(e);
                    break;
                case MainMode.COLOR:
                    mColorChange.color_MouseDown(e);
                    break;
                case MainMode.STIPPLE:
                    mStipple.stipple_MouseDown(e);
                    break;
                case MainMode.GRADIENT:
                    mGradient.gradient_MouseDown(e);
                    drawing = true;
                    break;
                case MainMode.ZOOM:
                    mReproportion.reproportion_MouseDown(e);
                    break;
                case MainMode.DRAW:
                case MainMode.REPEAT:
                case MainMode.MIRROR:
                    //# mouse 이벤트가 아닌 입력된 크기의 도형을 paste 할때의 동작
                    if (mPen.chk_select.Checked)
                    {
                        mPen.pen_MouseDown(e);
                        return;
                    }

                    if (!IsMouseInCanvas(e.Location, dst, true)) return;

                    if (mPen.penMode == PenManager.PenShape.KASURI)
                    {
                        mPen.Kasuri_MouseDown(e);
                        return;
                    }

                    if (mDraw.shapeMat != null)
                    {
                        dst = LastList(matList).Clone();
                        startPaste = new Point(PointToClient(MousePosition).X - mDraw.shapeMat.Width / 2, PointToClient(MousePosition).Y - mDraw.shapeMat.Height / 2);
                        dst[new cvRect(startPaste.X, startPaste.Y, mDraw.shapeMat.Width, mDraw.shapeMat.Height)].SetTo(penColor, mDraw.shapeMat);
                        mDraw.shapeMat = null;
                        restore = dst.Clone();
                        matList.Add(dst);
                        PictureBox1.ImageIpl = dst;
                        drawing = false;
                        return;
                    }
                    //# 마우스 down 동작
                    else
                    {
                        if (!drawing || mDraw.tsb_conn.Checked || mDraw.shape == TShape.CURVE)
                        {
                            drawing = true;  //연결된 선을 그릴 때는 계속 Drawing 상태 유지
                        }
                        else
                        {
                            matList.Add(dst);
                            drawing = false;
                            return;
                        }

                        if (mDraw.tsb_conn.Checked && mDraw.shape == TShape.LINE)
                        {
                            connPoints.Add(start);
                        }

                        if (mReflection.isReflect) mReflection.SetReflectionMat(LastList(matList));
                        else PictureBox1.ImageIpl = LastList(matList);

                        if (curCnt == 0) //curCount가 1이 아닐 때만 image를 받아와야한다
                        {
                            src = LastList(matList);
                            dst = LastList(matList);
                            restore = dst.Clone();
                        }

                        switch (mDraw.shape)
                        {
                            case TShape.PATH:
                                //# mask
                                if (mPen.penMode == PenManager.PenShape.MASK)
                                {
                                    if (mPen.rdo_fill.Checked)
                                    {
                                        Cv2.FloodFill(mask, new cvPoint(e.X, e.Y), new Scalar(255, 255, 255, 255));
                                        DrawMask();
                                    }
                                }
                                break;
                            case TShape.LINE:
                                if (mDraw.tsb_conn.Checked)
                                {
                                    matList.Add(dst);
                                    PictureBox1.ImageIpl = dst;
                                }
                                break;
                            case TShape.CURVE:
                                curCnt++;
                                if (curCnt == 1)
                                {
                                    if (connPoints.Count == 0)
                                    {
                                        matList.Add(dst);
                                        if (mReflection.isReflect) mReflection.SetReflectionMat(dst);
                                        else if (mAutoRepeat.isRepeat) mAutoRepeat.SetRepeatImage();
                                        else PictureBox1.ImageIpl = dst;
                                    }
                                    curPoints[0] = start;
                                }
                                else if (curCnt == 2)
                                {
                                    curPoints[1] = start;
                                    if (mReflection.isReflect) mReflection.SetReflectionMat(dst);
                                    else if (mAutoRepeat.isRepeat) mAutoRepeat.SetRepeatImage();
                                    else PictureBox1.ImageIpl = dst;
                                }
                                else if (curCnt == 3)
                                {
                                    curCnt = 0;                         //# 곡선 그리기를 완료하면 Count를 초기화 해야함
                                    drawing = false;                    //# 곡선 그리기를 완료했으므로 isDrawing = false
                                    if (mDraw.tsb_conn.Checked)         //# 이어진 곡선 그리기
                                    {
                                        if (connPoints.Count == 0) connPoints.Add(curPoints[0]);
                                        matList.Add(dst);
                                        if (mReflection.isReflect) mReflection.SetReflectionMat(dst);
                                        else if (mAutoRepeat.isRepeat) mAutoRepeat.SetRepeatImage();
                                        else PictureBox1.ImageIpl = dst;
                                        drawing = true;
                                        curCnt++;
                                        switch (mDraw.gCurveMode)
                                        {
                                            case 1:
                                                curPoints[0] = curPoints[1];
                                                connPoints.Add(curPoints[1]);
                                                break;
                                            case 2:
                                                curPoints[0] = curPoints[2];
                                                connPoints.Add(curPoints[2]);
                                                break;
                                        }
                                    }
                                    else
                                    {
                                        matList.Add(dst);                 //# 그림 그리기 완료하면 List에 Mat 저장하기
                                        if (mReflection.isReflect) mReflection.SetReflectionMat(dst);
                                        else if (mAutoRepeat.isRepeat) mAutoRepeat.SetRepeatImage();
                                        else PictureBox1.Image = BitmapConverter.ToBitmap(dst);
                                    }
                                }
                                return;

                            case TShape.OUTLINE:
                                cvPoint[][] contours;
                                HierarchyIndex[] hierarchy;

                                Mat hsv = new Mat();
                                Color selColor = Color.FromArgb(dst.At<int>(e.Y, e.X));
                                Scalar s = new Scalar(selColor.B, selColor.G, selColor.R, selColor.A);
                                Mat ori = dst.Clone();
                                Cv2.CvtColor(ori, hsv, ColorConversionCodes.BGR2HSV);
                                Cv2.InRange(ori, s, s, hsv);
                                Cv2.FindContours(hsv, out contours, out hierarchy, RetrievalModes.Tree, ContourApproximationModes.ApproxTC89KCOS);
                                double min = double.MaxValue;
                                int index = 0;

                                for (int i = 0; i < contours.Length; i++)
                                {
                                    for (int j = 0; j < contours[i].Length; j++)
                                    {
                                        double distance = Math.Sqrt(Math.Pow(e.X - contours[i][j].X, 2) + Math.Pow(e.Y - contours[i][j].Y, 2));

                                        if (min > distance)
                                        {
                                            min = distance;
                                            index = i;
                                        }
                                    }
                                }

                                Cv2.DrawContours(dst, contours, index, penColor, penSize);
                                PictureBox1.ImageIpl = dst;
                                matList.Add(dst);
                                drawing = false;
                                break;
                        }
                    }
                    break;
                case MainMode.STRIPE:
                    if (mStripe.stripeMat != null)
                    {
                        dst = LastList(matList).Clone();
                        startPaste = new Point(PointToClient(MousePosition).X - mStripe.stripeMat.Width / 2, PointToClient(MousePosition).Y - mStripe.stripeMat.Height / 2);
                        dst[new cvRect(startPaste.X, startPaste.Y, mStripe.stripeMat.Width, mStripe.stripeMat.Height)] = mStripe.stripeMat;
                        mStripe.stripeMat = null;
                        restore = dst.Clone();
                        matList.Add(dst);
                        PictureBox1.ImageIpl = dst;
                        drawing = false;
                        return;
                    }
                    break;
                case MainMode.PATTERN:
                    drawing = true;
                    if (mPattern.method == PatternRepeat.PatternMethod.REGULAR)
                    {
                        //# 이미지 크롭
                        Rectangle r = Rectangle.Round(mWorkarea.totalPath.GetBounds());
                        cvRect rect = new cvRect((int)r.X, (int)r.Y, (int)r.Width, (int)r.Height);
                        cropMat = new Mat();
                        dst.CopyTo(cropMat, mWorkarea.holeMat);
                        cropMat[rect].CopyTo(cropMat);

                        if (mPattern.isSelPos)
                        {
                            if (mWorkarea.totalPath.IsVisible(e.Location))
                            {
                                PictureBox1.Cursor = new Cursor(penCursor.GetHicon());
                                mPattern.btn_position.Enabled = true;
                                mPattern.isSelPos = false;

                                //# 이미지 분할 및 재배열
                                cvPoint center = new cvPoint(e.X - r.X, e.Y - r.Y);
                                dregular = ReArrange(center, cropMat);
                                return;
                            }
                        }
                        else
                        {
                            if (dregular == null)
                            {
                                dregular = cropMat;
                            }
                            //# 이미지 반복
                            int row = dst.Height / dregular.Height;
                            int col = dst.Width / dregular.Width;

                            Mat vMat = new Mat();
                            for (int i = 0; i < row; i++)
                            {
                                Mat hMat = new Mat();
                                for (int j = 0; j < col; j++)
                                {
                                    if (j == 0)
                                    {
                                        Cv2.HConcat(dregular, dregular, hMat);
                                    }
                                    else
                                    {
                                        Cv2.HConcat(hMat, dregular, hMat);
                                    }
                                }

                                if (i == 0)
                                {
                                    Cv2.VConcat(hMat, hMat, vMat);
                                }
                                else
                                {
                                    Cv2.VConcat(vMat, hMat, vMat);
                                }
                            }

                            //# 이미지 분할 및 재배열
                            cvPoint center = new cvPoint(e.X % dregular.Width, e.Y % dregular.Height);
                            regular = ReArrange(center, vMat);
                            regular = regular[new cvRect(0, 0, dst.Width, dst.Height)];
                            patternMask = new Mat(new cvSize(dst.Width, dst.Height), MatType.CV_8UC1, 0);
                        }
                    }
                    else if (mPattern.method == PatternRepeat.PatternMethod.SPRAY)
                    {
                        if (mPattern.cmb_method.Text == "Curve")
                        {
                            if (direction == null)
                            {
                                direction = new GraphicsPath();
                                curveList.Add(e.Location);
                                drawing = true;
                            }
                            //# connected drawing
                            else
                            {
                                curveList.Add(e.Location);
                                if (curveList.Count == 2)
                                {
                                    curveList.Add(e.Location);
                                    curveList.Add(e.Location);
                                    curveList[curveList.Count - 1] = curveList[curveList.Count - 3];
                                }
                                else if (curveList.Count == 5)
                                {
                                    curveList.RemoveAt(curveList.Count - 1);
                                    direction = new GraphicsPath();
                                    direction.AddBeziers(curveList.ToArray());
                                    direction.Flatten();
                                    int max = cropMat.Width >= cropMat.Height ? cropMat.Width : cropMat.Height;
                                    int length = (int)(max * Math.Sqrt(2));
                                    Mat resizeMat = new Mat(new cvSize(length, length), MatType.CV_8UC4, 0);
                                    int x = resizeMat.Width / 2 - cropMat.Width / 2;
                                    int y = resizeMat.Height / 2 - cropMat.Height / 2;
                                    resizeMat[new cvRect(x, y, cropMat.Width, cropMat.Height)] = cropMat;
                                    SprayData(direction, resizeMat);
                                    direction = null;
                                    curveList.Clear();
                                    PictureBox1.Refresh();
                                    drawing = false;
                                }
                            }
                        }
                        else
                        {
                            direction = new GraphicsPath();
                            drawing = true;
                        }
                    }
                    break;
            }
        }

        private void bitmap_MouseMove(object sender, MouseEventArgs e)
        {
            if (!drawing) return;
            switch (MainMenu.mode)
            {
                case MainMode.STRIPE:
                    mStripe.stripe_MouseMove(e);
                    break;
                case MainMode.TEXT:
                    mText.text_MouseMove(e);
                    break;
                case MainMode.FILL:
                    mFill.fill_MouseMove(e);
                    break;
                case MainMode.ASSEM:
                    mAssem.assem_MouseMove(e);
                    break;
                case MainMode.SEPARATE:
                    mSeparate.separate_MouseMove(e);
                    break;
                case MainMode.GRADIENT:
                    mGradient.gradient_MouseMove(e);
                    return;
                case MainMode.STIPPLE:
                    mStipple.stipple_MouseMove(e);
                    break;
                case MainMode.WORKAREA:
                    mWorkarea.work_MouseMove(sender, e);
                    PictureBox1.Refresh();
                    break;
                case MainMode.MOVE:
                    mMove.move_MouseMove(e);
                    PictureBox1.Refresh();
                    return;
                case MainMode.COLOR:
                    mColorChange.color_MouseMove(e);
                    return;
                case MainMode.ZOOM:
                    mReproportion.reproportion_MouseMove(e);
                    break;
                case MainMode.DRAW:
                case MainMode.REPEAT:
                case MainMode.MIRROR:
                    if (mPen.penMode == PenManager.PenShape.AIRBRUSH || mPen.chk_select.Checked)
                    {
                        mPen.pen_MouseMove(e);
                        return;
                    }

                    if (mDraw.shapeMat != null)
                    {
                        mDraw.draw_MouseMove(e);
                    }
                    else
                    {
                        end = new cvPoint(e.X, e.Y);
                        DrawShape(e);
                    }

                    if (mPen.penMode == PenManager.PenShape.KASURI)
                    {
                        mPen.Kasuri_MouseMove(e);
                        return;
                    }
                    break;
                case MainMode.PATTERN:
                    if (mPattern.isSelPos)
                    {
                        if (mWorkarea.totalPath.IsVisible(e.Location))
                        {
                            PictureBox1.Cursor = Cursors.Cross;
                        }
                        else
                        {
                            PictureBox1.Cursor = new Cursor(penCursor.GetHicon());
                        }
                    }
                    if (mPattern.method == PatternRepeat.PatternMethod.SPRAY)
                    {
                        Point startPt = new Point(start.X, start.Y);
                        Point endPt = new Point(end.X, end.Y);

                        if (mPattern.cmb_method.Text == "Free")
                        {
                            if (direction == null) return;
                            direction.AddLine(startPt, endPt);
                            start = new cvPoint(e.X, e.Y);
                        }
                        else if (mPattern.cmb_method.Text == "Line")
                        {
                            direction = new GraphicsPath();
                            if ((System.Windows.Input.Keyboard.GetKeyStates(System.Windows.Input.Key.LeftShift) & System.Windows.Input.KeyStates.Down) > 0)
                            {
                                //# 수평선
                                GraphicsPath line = new GraphicsPath();
                                double angle = 360 - GetAngle(start, end);
                                if ((angle >= 45 && angle < 135) || (angle >= 225 && angle < 315))
                                {
                                    line.AddLine(startPt, new Point(startPt.X, endPt.Y));
                                }
                                else
                                {
                                    line.AddLine(startPt, new Point(endPt.X, startPt.Y));
                                }
                                direction = line;
                            }
                            else
                            {
                                direction = new GraphicsPath();
                                direction.AddLine(startPt, endPt);
                            }
                        }
                        else if (mPattern.cmb_method.Text == "Curve")
                        {
                            if (direction == null) return;
                            direction = new GraphicsPath();
                            if (curveList.Count > 1)
                            {
                                curveList[1] = GetSymmetryPoint(curveList[3], e.Location);
                                curveList[2] = GetSymmetryPoint(curveList[3], e.Location);

                                int bX = (int)(curveList[0].X + (2.0f / 3.0f) * (curveList[1].X - curveList[0].X));
                                int bY = (int)(curveList[0].Y + (2.0f / 3.0f) * (curveList[1].Y - curveList[0].Y));
                                int cX = (int)(curveList[3].X + (2.0f / 3.0f) * (curveList[1].X - curveList[3].X));
                                int cY = (int)(curveList[3].Y + (2.0f / 3.0f) * (curveList[1].Y - curveList[3].Y));

                                curveList[1] = new Point(bX, bY);
                                curveList[2] = new Point(cX, cY);

                                direction.AddBeziers(curveList.ToArray());
                            }
                            else
                            {
                                direction.AddLine(startPt, endPt);
                            }
                        }
                        else if (mPattern.cmb_method.Text == "Rectangle")
                        {
                            direction = new GraphicsPath();
                            if ((System.Windows.Input.Keyboard.GetKeyStates(System.Windows.Input.Key.LeftShift) & System.Windows.Input.KeyStates.Down) > 0)
                                direction.AddRectangle(mVector.GetSquare(startPt.X, startPt.Y, endPt.X, endPt.Y));

                            else
                                direction.AddRectangle(mVector.GetRect(startPt.X, startPt.Y, endPt.X, endPt.Y));
                        }
                        else if (mPattern.cmb_method.Text == "Ellipse")
                        {
                            direction = new GraphicsPath();
                            if ((System.Windows.Input.Keyboard.GetKeyStates(System.Windows.Input.Key.LeftShift) & System.Windows.Input.KeyStates.Down) > 0)
                                direction.AddEllipse(mVector.GetSquare(startPt.X, startPt.Y, endPt.X, endPt.Y));
                            else
                                direction.AddEllipse(mVector.GetRect(startPt.X, startPt.Y, endPt.X, endPt.Y));
                        }

                        PictureBox1.Refresh();
                    }
                    end = new cvPoint(e.X, e.Y);
                    break;
            }
        }

        private void bitmap_MouseUp(object sender, MouseEventArgs e)
        {
            switch (MainMenu.mode)
            {
                case MainMode.FILL:
                    mFill.fill_MouseUp(e);
                    break;
                case MainMode.TEXT:
                    mText.text_MouseUp(e);
                    break;
                case MainMode.STIPPLE:
                    mStipple.stipple_MouseUp(e);
                    break;
                case MainMode.ASSEM:
                    mAssem.assem_MouseUp(e);
                    break;
                case MainMode.SEPARATE:
                    mSeparate.separate_MouseUp(e);
                    break;
                case MainMode.COLOR:
                    mColorChange.color_MouseUp(e);
                    break;
                case MainMode.WORKAREA:
                    mWorkarea.work_MouseUp(sender, e);
                    return;
                case MainMode.ZOOM:
                    mReproportion.reproportion_MouseUp(e);
                    break;
                case MainMode.DRAW:
                case MainMode.REPEAT:
                case MainMode.MIRROR:
                    if (mPen.chk_select.Checked)
                    {
                        mPen.pen_MouseUp(e);
                        return;
                    }

                    if (mPen.penMode == PenManager.PenShape.KASURI)
                    {
                        mPen.Kasuri_MouseUp(e);
                        return;
                    }

                    if (mDraw.shape == TShape.PATH)
                    {
                        drawing = false;
                        matList.Add(dst);

                        if (mPen.penMode == PenManager.PenShape.MASK && mask != null)
                        {
                            Mat temp = new Mat();
                            Cv2.BitwiseNot(mask, temp);
                            protect = mPen.oriMat.Clone();
                            protect.SetTo(0, temp);
                        }
                    }
                    if (mAutoRepeat.isRepeat)
                    {
                        mAutoRepeat.SetRepeatImage();
                    }
                    else if (mReflection.isReflect)
                    {
                        mReflection.SetReflectionMat(LastList(matList));
                    }
                    else//DRAW 모드 일 때만 restore 갱신
                    {
                        restore = dst.Clone();
                    }
                    break;
                case MainMode.PATTERN:
                    if (mPattern.method == PatternRepeat.PatternMethod.SPRAY)
                    {
                        if (mPattern.cmb_method.Text != "Curve")
                        {
                            if (direction != null)
                            {
                                direction.Flatten();
                                int max = cropMat.Width >= cropMat.Height ? cropMat.Width : cropMat.Height;
                                int length = (int)(max * Math.Sqrt(2));
                                Mat resizeMat = new Mat(new cvSize(length, length), MatType.CV_8UC4, 0);
                                int x = resizeMat.Width / 2 - cropMat.Width / 2;
                                int y = resizeMat.Height / 2 - cropMat.Height / 2;
                                resizeMat[new cvRect(x, y, cropMat.Width, cropMat.Height)] = cropMat;
                                SprayData(direction, resizeMat);
                                direction = null;
                                drawing = false;
                                PictureBox1.Refresh();
                            }
                        }
                    }
                    return;
                case MainMode.GRADIENT:
                    mGradient.gradient_MouseUp(e);
                    return;
                default:
                    matList.Add(dst);
                    drawing = false;
                    break;
            }
        }

        //# pattern
        public Mat ReArrange(cvPoint center, Mat mat)
        {
            Mat lt = mat[new cvRect(0, 0, center.X - 1, center.Y - 1)];                                 //# lt - 1
            Mat rt = mat[new cvRect(center.X, 0, mat.Width - center.X, center.Y - 1)];                  //# rt - 2
            Mat lb = mat[new cvRect(0, center.Y, center.X - 1, mat.Height - center.Y)];                 //# lb - 3
            Mat rb = mat[new cvRect(center.X, center.Y, mat.Width - center.X, mat.Height - center.Y)];  //# rb - 4

            //# 이미지 재배열
            Mat hMat1 = new Mat(), hMat2 = new Mat();
            Cv2.HConcat(rb, lb, hMat1);
            Cv2.HConcat(rt, lt, hMat2);
            Mat vMat = new Mat();
            Cv2.VConcat(hMat1, hMat2, vMat);
            return vMat;
        }

        //# spray
        public void SprayData(GraphicsPath path, Mat cropMat)
        {
            int width = cropMat.Width;
            int height = cropMat.Height;

            if (mPattern.cmb_method.Text == "Line" || mPattern.cmb_method.Text == "Rectangle")
            {
                List<float> angleList = new List<float>();
                List<PointF> pointList = new List<PointF>();
                GetSprayPoint(path, ref angleList, ref pointList);
                for (int i = 0; i < pointList.Count; i++)
                {
                    Mat matrix = Cv2.GetRotationMatrix2D(new Point2f(width / 2, height / 2), 180 / Math.PI * angleList[i], 1);
                    Mat mat = new Mat();
                    Cv2.WarpAffine(cropMat, mat, matrix, new cvSize(width, height));
                    int x = (int)(pointList[i].X - cropMat.Width / 2);
                    int y = (int)(pointList[i].Y - cropMat.Height / 2);

                    for (int row = 1; row < mat.Height - 1; row++)
                    {
                        for (int col = 1; col < mat.Width - 1; col++)
                        {
                            Color c = Color.FromArgb(mat.At<int>(row, col));
                            if (c.A == 255)
                                dst.Set<int>(y + row, x + col, mat.At<int>(row, col));
                        }
                    }
                }
            }
            else
            {
                List<int> indexList = GetSprayIndex(path);

                float dx, dy, dist;
                int idx = 0;

                for (int i = 0; i < path.PointCount - 1; i++)
                {
                    dx = path.PathPoints[i + 1].X - path.PathPoints[i].X;
                    dy = path.PathPoints[i + 1].Y - path.PathPoints[i].Y;
                    dist = (float)Math.Sqrt(dx * dx + dy * dy);
                    dx /= dist;
                    dy /= dist;

                    float angle = (float)(180 / Math.PI * Math.Atan2(dy, dx));
                    if (idx < indexList.Count && i == indexList[idx])
                    {
                        Mat matrix = Cv2.GetRotationMatrix2D(new Point2f(width / 2, height / 2), angle, 1);
                        Mat mat = new Mat();
                        Cv2.WarpAffine(cropMat, mat, matrix, new cvSize(width, height));
                        int x = (int)(path.PathPoints[i].X - cropMat.Width / 2);
                        int y = (int)(path.PathPoints[i].Y - cropMat.Height / 2);

                        for (int row = 0; row < mat.Height; row++)
                        {
                            for (int col = 0; col < mat.Width; col++)
                            {
                                Color c = Color.FromArgb(mat.At<int>(row, col));
                                if (c.A == 255)
                                    dst.Set<int>(y + row, x + col, mat.At<int>(row, col));
                            }
                        }
                        idx++;
                    }
                }
            }

            PictureBox1.ImageIpl = dst;
        }

        private PointF GetSymmetryPoint(PointF p1, PointF p2, bool reverse = false)
        {
            float x, y;
            if (!reverse)
            {
                x = 2 * p1.X - p2.X;
                y = 2 * p1.Y - p2.Y;
            }
            else
            {
                x = 2 * p2.X - p1.X;
                y = 2 * p2.Y - p1.Y;
            }

            PointF result = new PointF(x, y);
            return result;
        }

        private List<int> GetSprayIndex(GraphicsPath path)
        {
            List<int> indexList = new List<int>();
            float totLength = 0;

            //# 전체 길이 구하기
            for (int i = 0; i < path.PointCount - 1; i++)
            {
                float dx = path.PathPoints[i + 1].X - path.PathPoints[i].X;
                float dy = path.PathPoints[i + 1].Y - path.PathPoints[i].Y;
                totLength += (float)Math.Sqrt(dx * dx + dy * dy);
            }

            //# 간격 구하기
            float unitLength = totLength / Convert.ToSingle(mPattern.nud_cnt.Value);

            totLength = 0;

            //# 간격마다의 점 구하기
            for (int i = 0; i < path.PointCount - 1; i++)
            {
                float dx = path.PathPoints[i + 1].X - path.PathPoints[i].X;
                float dy = path.PathPoints[i + 1].Y - path.PathPoints[i].Y;
                totLength += (float)Math.Sqrt(dx * dx + dy * dy);

                while (totLength > unitLength || totLength > 10F)
                {
                    totLength = totLength - unitLength;
                    indexList.Add(i);
                }
            }
            return indexList;
        }

        private void GetSprayPoint(GraphicsPath path, ref List<float> angleList, ref List<PointF> pointList)
        {
            if (!(path.PointCount > 0)) return; //# path가 없으면 return

            //# create local variable
            pointList = new List<PointF>();     //# 좌표 list
            angleList = new List<float>();      //# 각도 list
            float totLength = 0;                //# 길이
            PointF sPoint;                      //# 계산된 좌표
            float dx;                           //# x 변화량
            float dy;                           //# y 변화량

            //# 전체 길이 구하기
            for (int i = 0; i < path.PointCount; i++)
            {
                //# 마지막 점이 닫힌 패스인 경우 첫점과의 길이도 더해준다.
                if (i == path.PointCount - 1)
                {
                    if (path.PathTypes[i] == 129)
                    {
                        dx = path.PathPoints[0].X - path.PathPoints[i].X;
                        dy = path.PathPoints[0].Y - path.PathPoints[i].Y;
                    }
                    else
                    {
                        //# 열린 패스이면 더이상 반복하지 않고 빠져나간다.
                        break;
                    }
                }
                else
                {
                    dx = path.PathPoints[i + 1].X - path.PathPoints[i].X;
                    dy = path.PathPoints[i + 1].Y - path.PathPoints[i].Y;
                }

                totLength += (float)Math.Sqrt(dx * dx + dy * dy);

            }

            //# 간격 구하기
            float unitLength = totLength / Convert.ToSingle(mPattern.nud_cnt.Value);

            totLength = 0;
            float extraLength = 0;

            for (int i = 0; i < path.PointCount; i++)
            {
                if (i == path.PointCount - 1)
                {
                    if (path.PathTypes[i] == 129)
                    {
                        dx = path.PathPoints[0].X - path.PathPoints[i].X;
                        dy = path.PathPoints[0].Y - path.PathPoints[i].Y;
                    }
                    else
                    {
                        //# 열린 패스이면 더이상 반복하지 않고 빠져나간다.
                        break;
                    }
                }
                else
                {
                    dx = path.PathPoints[i + 1].X - path.PathPoints[i].X;
                    dy = path.PathPoints[i + 1].Y - path.PathPoints[i].Y;
                }

                totLength += (float)Math.Sqrt(dx * dx + dy * dy);
                float angle = (float)(2 * Math.PI - Math.Atan2(dy, dx));

                //# 전체 길이가 단위길이보다 짧으면 전체길이를 반복문을 돌며 추가해준다.
                if (totLength <= unitLength) continue;

                //# 변화량 초기화
                dx = 0;
                dy = 0;

                //# 한 변을 단위 길이만큼 증가시키며 반복
                while (totLength >= unitLength)
                {
                    //# 직전 변에서 남은 길이만큼 빼주어 거리 계산
                    if (extraLength > 0)
                    {
                        dx += (int)((unitLength - extraLength) * (float)Math.Cos(angle));
                        dy += (int)((unitLength - extraLength) * -(float)Math.Sin(angle));
                        extraLength = 0;
                    }
                    //# 단위길이만큼의 변화량을 계산
                    else
                    {
                        dx += (int)(unitLength * (float)Math.Cos(angle));
                        dy += (int)(unitLength * -(float)Math.Sin(angle));
                    }

                    //# 포인트 계산 / 추가
                    sPoint = new PointF(path.PathPoints[i].X + dx, path.PathPoints[i].Y + dy);
                    pointList.Add(sPoint);
                    angleList.Add(angle);

                    //# 변화한 만큼 전체 길이 조정
                    totLength -= unitLength;
                }

                //# 남은 길이 계산
                extraLength = totLength % unitLength;
            }

            //# 추가된 포인트의 개수와 개수가 일치하지 않을 경우의 처리
            while (pointList.Count != Convert.ToInt32(mPattern.nud_cnt.Value))
            {
                //# 개수가 부족할 경우 첫점을 추가
                if (pointList.Count < Convert.ToInt32(mPattern.nud_cnt.Value))
                {
                    pointList.Add(path.PathPoints[0]);
                    angleList.Add(angleList[0]);
                }
                //# 개수가 클 경우 마지막 점을 제거
                else if (pointList.Count > Convert.ToInt32(mPattern.nud_cnt.Value))
                {
                    pointList.RemoveAt(pointList.Count - 1);
                    angleList.RemoveAt(angleList.Count - 1);
                }
            }
        }

        public Mat LastList(List<Mat> matList)
        {
            return matList[matList.Count - 1];
        }

        public cvPoint LastList(List<cvPoint> points)
        {
            return points[points.Count - 1];
        }

        /// <summary>
        /// 전체 Mat을 읽으면서 입력한 Color와 같은 Color가 있으면 선택한 색으로 칠한다.
        /// 전체 Mat을 읽으면 속도가 느려지기 때문에 개선 필요
        /// </summary>
        /// <param name="src"></param>
        /// <param name="c_point"></param>
        /// <param name="p_color"></param>
        private void WholeFill(Mat src, cvPoint c_point, Scalar p_color)
        {
            Vec3b color = p_color.ToVec3b();
            var c_pixel = src.At<Vec3b>(c_point.Y, c_point.X);

            for (int i = 0; i < src.Height; i++)
            {
                for (int j = 0; j < src.Width; j++)
                {
                    var srcPixel = src.At<Vec3b>(i, j);
                    if (srcPixel == c_pixel)
                    {
                        src.Set<Vec3b>(i, j, color);
                    }
                }
            }
        }

        /// <summary>
        /// Bezier 곡선을 구성하는 Point들로 Line을 연결해서 Bezier 곡선을 그려주는 함수
        /// </summary>
        /// <param name="dst"></param>
        /// <param name="point"></param>
        public void DrawCurve(Mat dst, cvPoint point)
        {
            List<cvPoint> bList = new List<cvPoint>();
            curPoints[2] = point; //# 곡선을 만들 세 번째 점 추가
            for (float i = 0; i < 1.0; i += 0.01f)
            {
                int x = 0; int y = 0;
                if (mDraw.gCurveMode == 1)
                {
                    x = QuadBezier(curPoints[0].X, curPoints[2].X, curPoints[1].X, i);
                    y = QuadBezier(curPoints[0].Y, curPoints[2].Y, curPoints[1].Y, i);
                }
                else if (mDraw.gCurveMode == 2)
                {
                    x = QuadBezier(curPoints[0].X, curPoints[1].X, curPoints[2].X, i);
                    y = QuadBezier(curPoints[0].Y, curPoints[1].Y, curPoints[2].Y, i);
                }

                bList.Add(new cvPoint(x, y));
            }
            for (int i = 0; i < bList.Count - 1; i++)
            {
                Cv2.Line(dst, bList[i], bList[i + 1], penColor, Convert.ToInt32(penSize));
            }
        }

        public void DrawShape(MouseEventArgs e)
        {
            if (!IsMouseInCanvas(e.Location, dst, false)) return;
            switch (mDraw.shape)
            {
                case TShape.PATH:
                    Mat tempMat = new Mat(new cvSize(penSize, penSize), MatType.CV_8UC4, 0);
                    Cv2.Circle(tempMat, penSize / 2, penSize / 2, penSize / 2, Scalar.White, -1, LineTypes.Link8);
                    Random rand = new Random();

                    if (MainMenu.mode == MainMode.PATTERN)
                    {
                        if (mPattern.method == PatternRepeat.PatternMethod.IRREGULAR) //# 불규칙
                        {
                            for (int i = 0; i < penSize - 1; i++)
                            {
                                for (int j = 0; j < penSize - 1; j++)
                                {
                                    Color c = Color.FromArgb(tempMat.At<int>(i, j));
                                    if (c.R == 255 && c.G == 255 && c.B == 255) //# 원 내부
                                    {
                                        cvPoint pt = new cvPoint(e.Y - penSize / 2 + j, e.X - penSize / 2 + i);
                                        int randomCnt = rand.Next(0, 100);

                                        for (int x = 0; x < mPattern.colCnt.Count; x++)
                                        {
                                            Color col = Color.FromArgb(mPattern.colList[x]);
                                            Scalar s = new Scalar(col.B, col.G, col.R, col.A);
                                            if (mPattern.colCnt[x] >= randomCnt)
                                            {
                                                dst.Set<Vec3b>(pt.X, pt.Y, s.ToVec3b());
                                                break;
                                            }
                                        }
                                    }
                                }
                            }
                        }
                        else if (mPattern.method == PatternRepeat.PatternMethod.REGULAR)    //# 규칙
                        {
                            if (patternMask != null)
                            {
                                Cv2.Line(patternMask, start, end, new Scalar(255, 255, 255, 255), penSize);
                                Mat temp = new Mat();
                                regular.CopyTo(temp, patternMask);

                                dst.SetTo(0, patternMask);
                                dst = dst + temp;
                            }
                        }

                        start = end;
                        PictureBox1.ImageIpl = dst;
                        restore = dst.Clone();
                        return;
                    }

                    switch (mPen.penMode)
                    {
                        case PenManager.PenShape.ERASER:
                            mClear.Eraser(e);
                            break;
                        case PenManager.PenShape.NORMAL:
                            Cv2.Line(dst, start, end, penColor, penSize);
                            break;
                        case PenManager.PenShape.CRAYON:
                            for (int i = 0; i < penSize - 1; i++)
                            {
                                for (int j = 0; j < penSize - 1; j++)
                                {
                                    Color c = Color.FromArgb(tempMat.At<int>(i, j));
                                    if (c.R == 255 && c.G == 255 && c.B == 255)
                                    {
                                        cvPoint pt = new cvPoint(e.Y - penSize / 2 + j, e.X - penSize / 2 + i);
                                        if (rand.Next(0, (int)(1 / (0.05 * (float)(mPen.nud_density2.Value)))) == 0)
                                        {
                                            dst.Set<Vec3b>(pt.X, pt.Y, penColor.ToVec3b());
                                        }
                                        else
                                        {
                                            dst.Set<Vec3b>(pt.X, pt.Y, dst.At<Vec3b>(pt.X, pt.Y));
                                        }
                                    }
                                }
                            }
                            break;
                        case PenManager.PenShape.SCRATCH:
                            double yIvt = (penSize * Convert.ToInt32(mPen.txt_scale.Text) / 100 * Math.Cos((180 - (int)GetAngle(start, end)) * Math.PI / 180));
                            double xIvt = (penSize * Convert.ToInt32(mPen.txt_scale.Text) / 100 * Math.Sin((180 - (int)GetAngle(start, end)) * Math.PI / 180));
                            int level = 0;
                            for (int i = 0; i < (int)Math.Round(mPen.nud_bristles.Value); i++)
                            {
                                if (i < (int)(mPen.nud_bristles.Value / 2))
                                {
                                    ++level;
                                    Cv2.Line(dst, new cvPoint(start.X - level * xIvt, start.Y - level * yIvt), new cvPoint(end.X - level * xIvt, end.Y - level * yIvt), penColor, penSize);
                                }
                                else if (i == (int)(mPen.nud_bristles.Value / 2))
                                {
                                    level = 0;
                                    Cv2.Line(dst, start, end, penColor, penSize);
                                }
                                else if (i > (int)(mPen.nud_bristles.Value / 2))
                                {
                                    ++level;
                                    Cv2.Line(dst, new cvPoint(start.X + level * xIvt, start.Y + level * yIvt), new cvPoint(end.X + level * xIvt, end.Y + level * yIvt), penColor, penSize);
                                }
                            }
                            break;
                        case PenManager.PenShape.MASK:
                            if (mPen.chk_delete.Checked)
                            {
                                Cv2.Line(mask, start, end, 0, penSize);
                            }
                            else
                            {
                                Cv2.Line(mask, start, end, new Scalar(255, 255, 255, 255), penSize);
                            }
                            DrawMask();
                            break;
                    }

                    if (!mReflection.isReflect && !mAutoRepeat.isRepeat && mPen.penMode != PenManager.PenShape.MASK && protect != null)    //# 보호된 영역 처리
                    {
                        dst.SetTo(0, mask);
                        dst = dst + protect;
                    }

                    PictureBox1.ImageIpl = dst;
                    restore = dst.Clone();
                    start = end;
                    break;
                case TShape.LINE:
                    if (MainMenu.mode == MainMode.STITCH)
                    {
                        dst = src.Clone();

                        int yInv1 = (int)(penSize * (int)mStitch.nud_space1.Value * Math.Cos((180 - (int)GetAngle(start, end)) * Math.PI / 180));
                        int xInv1 = (int)(penSize * (int)mStitch.nud_space1.Value * Math.Sin((180 - (int)GetAngle(start, end)) * Math.PI / 180));
                        int yInv2 = (int)(penSize * (int)mStitch.nud_space2.Value * Math.Cos((180 - (int)GetAngle(start, end)) * Math.PI / 180));
                        int xInv2 = (int)(penSize * (int)mStitch.nud_space2.Value * Math.Sin((180 - (int)GetAngle(start, end)) * Math.PI / 180));

                        mStitch.DrawStitch(new cvPoint(start.X + xInv1, start.Y + yInv1), new cvPoint(end.X + xInv1, end.Y + yInv1), mStitch.cmb_line1.SelectedIndex, (int)mStitch.nud_dot1.Value, (int)mStitch.nud_gap1.Value);
                        mStitch.DrawStitch(start, end, mStitch.cmb_line2.SelectedIndex, (int)mStitch.nud_dot2.Value, (int)mStitch.nud_gap2.Value);
                        mStitch.DrawStitch(new cvPoint(start.X - xInv2, start.Y - yInv2), new cvPoint(end.X - xInv2, end.Y - yInv2), mStitch.cmb_line3.SelectedIndex, (int)mStitch.nud_dot3.Value, (int)mStitch.nud_gap3.Value);
                    }
                    else
                    {
                        if (!mDraw.tsb_conn.Checked)
                        {
                            dst = src.Clone();
                            Cv2.Line(dst, start, end, penColor, penSize);
                        }
                        else
                        {
                            curCnt++;
                            dst = LastList(matList).Clone();
                            Cv2.Line(dst, LastList(connPoints), end, penColor, penSize);
                        }
                    }
                    PictureBox1.ImageIpl = dst;
                    restore = dst.Clone();
                    break;
                case TShape.CURVE:
                    if (!mDraw.tsb_conn.Checked)
                    {
                        dst = src.Clone();
                    }
                    else
                    {
                        dst = LastList(matList).Clone();
                    }

                    if (curCnt == 1)
                    {
                        Cv2.Line(dst, new cvPoint(curPoints[0].X, curPoints[0].Y), end, penColor, penSize);
                    }
                    else if (curCnt == 2)
                    {
                        DrawCurve(dst, end);
                    }
                    PictureBox1.ImageIpl = dst;
                    restore = dst.Clone();
                    break;
                case TShape.RECT:
                    dst = src.Clone();
                    if (!mDraw.tsb_fill.Checked)
                    {
                        Cv2.Rectangle(dst, CreateCvRect(start, end), penColor, penSize);
                    }
                    else
                    {
                        Cv2.Rectangle(dst, CreateCvRect(start, end), penColor, -1, LineTypes.Link8);
                    }
                    PictureBox1.ImageIpl = dst;
                    restore = dst.Clone();
                    break;
                case TShape.SQUARE:
                    dst = src.Clone();
                    if (!mDraw.tsb_fill.Checked)
                        Cv2.Rectangle(dst, CreateCvSquare(start, end), penColor, penSize);
                    else
                        Cv2.Rectangle(dst, CreateCvSquare(start, end), penColor, -1, LineTypes.Link8);
                    PictureBox1.ImageIpl = dst;
                    restore = dst.Clone();
                    break;
                case TShape.CIRCLE:
                    dst = src.Clone();
                    if (!mDraw.tsb_fill.Checked)
                        Cv2.Circle(dst, start.X, start.Y, Math.Abs(end.X - start.X), penColor, penSize);
                    else
                        Cv2.Circle(dst, start.X, start.Y, Math.Abs(end.X - start.X), penColor, -1, LineTypes.Link8);
                    PictureBox1.ImageIpl = dst;
                    restore = dst.Clone();
                    break;
                case TShape.ELLIPSE:
                    dst = src.Clone();
                    if (!mDraw.tsb_fill.Checked)
                        Cv2.Ellipse(dst, start, new cvSize(Math.Abs(end.X - start.X), Math.Abs(e.Location.Y - start.Y)), 0, 0, 360, penColor, penSize);
                    else
                        Cv2.Ellipse(dst, start, new cvSize(Math.Abs(end.X - start.X), Math.Abs(e.Location.Y - start.Y)), 0, 0, 360, penColor, -1, LineTypes.Link8);
                    PictureBox1.ImageIpl = dst;
                    restore = dst.Clone();
                    break;
                default:
                    break;
            }
        }

        public void DrawMask()
        {
            //# 영역 밖과 안의 데이터 저장을 위해 toggle mask를 구한다.
            Mat toggle = new Mat();
            Cv2.BitwiseNot(mask, toggle);

            //# 영역 내부 역상 이미지와 영역 외부 기존 이미지를 구한다.
            Mat reverse = new Mat();
            Mat other = new Mat();
            mPen.revMat.CopyTo(reverse, mask);
            mPen.oriMat.CopyTo(other, toggle);

            //# 두 이미지를 Or 연산으로 더한다.
            Cv2.BitwiseOr(reverse, other, dst);
        }

        public Bitmap ClipToRegion(Bitmap img, Region rg)
        {
            Bitmap clipReigon = new Bitmap(img.Width, img.Height);
            using (Graphics gr = Graphics.FromImage(clipReigon))
            {
                gr.SmoothingMode = SmoothingMode.AntiAlias;
                gr.SetClip(rg, CombineMode.Replace);
                gr.Clear(Color.White);
                gr.DrawImage(img, new Point(0, 0));
            }
            return clipReigon;
        }

        public bool IsMouseInCanvas(Point mouse, Mat dst, bool isDown)
        {
            if (mouse.X < 0 || mouse.X > dst.Width || mouse.Y < 0 || mouse.Y > dst.Height) return false;
            return true;
        }

        /// <summary>
        /// 식에 값을 대입하여 Bezier Point를 구하는 함수
        /// </summary>
        /// <param name="A"></param>
        /// <param name="B"></param>
        /// <param name="C"></param>
        /// <param name="t"></param>
        /// <returns></returns>
        private int QuadBezier(int A, int B, int C, float t)
        {
            if (t == 0) return A;
            if (t == 1) return C;

            float s = 1 - t;

            return (int)(Math.Pow(s, 2) * A + 2 * (s * t) * B + Math.Pow(t, 2) * C);
        }

        /// <summary>
        /// 점 2개 사이의 거리를 구함
        /// </summary>
        /// <param name="start"></param>
        /// <param name="end"></param>
        /// <returns></returns>
        public double GetLength(cvPoint start, cvPoint end)
        {
            //return Math.Round(Math.Sqrt(Math.Pow(Math.Abs(start.X - end.X), 2) + Math.Pow(Math.Abs(start.Y - end.Y), 2)) * 0.026458333, 2);
            return Math.Sqrt(Math.Pow(Math.Abs(start.X - end.X), 2) + Math.Pow(Math.Abs(start.Y - end.Y), 2));
        }

        /// <summary>
        /// pixel 단위의 길이를 입력받아 Cm 단위로 변경
        /// </summary>
        /// <param name="length"></param>
        /// <returns></returns>
        public double GetLength(float length)
        {
            return Math.Round(length * 0.026458333, 2);
        }

        /// <summary>
        /// 두 점을 이은 직선의 각도를 계산하는 함수
        /// </summary>
        /// <param name="start"></param>
        /// <param name="end"></param>
        /// <returns></returns>
        public double GetAngle(cvPoint start, cvPoint end)
        {
            double result = Math.Round(Math.Atan2(end.Y - start.Y, end.X - start.X) * (180.0 / Math.PI), 2);
            if (result < 0)
            {
                result = 360 + result;

            }
            return result;
        }

        /// <summary>
        /// 사각형 Draw할 때 모든 방향에서 그릴 수 있도록 startX, endX, startY, endY를 계산하는 함수(Bitmap Drawing)
        /// </summary>
        /// <param name="startX"></param>
        /// <param name="startY"></param>
        /// <param name="endX"></param>
        /// <param name="endY"></param>
        /// <returns></returns>
        public cvRect CreateCvRect(cvPoint start, cvPoint end)
        {
            cvRect rect;
            if (end.X - start.X > 0 && end.Y - start.Y > 0)
            {
                rect = new cvRect(start.X, start.Y, Math.Abs(end.X - start.X), Math.Abs(end.Y - start.Y));
            }
            else if (end.X - start.X > 0 && end.Y - start.Y < 0)
            {
                rect = new cvRect(start.X, end.Y, Math.Abs(end.X - start.X), Math.Abs(end.Y - start.Y));
            }
            else if (end.X - start.X < 0 && end.Y - start.Y > 0)
            {
                rect = new cvRect(end.X, start.Y, Math.Abs(end.X - start.X), Math.Abs(end.Y - start.Y));
            }
            else
            {
                rect = new cvRect(end.X, end.Y, Math.Abs(end.X - start.X), Math.Abs(end.Y - start.Y));
            }
            return rect;
        }

        /// <summary>
        /// 정사각형을 Draw할 때 모든 방향에서 그릴 수 있도록 startX, endX, startY, endY를 계산하는 함수(Bitmap Drawing)
        /// </summary>
        /// <param name="startX"></param>
        /// <param name="startY"></param>
        /// <param name="endX"></param>
        /// <param name="endY"></param>
        /// <returns></returns>
        public cvRect CreateCvSquare(cvPoint start, cvPoint end)
        {
            cvRect square;
            if (end.X - start.X > 0 && end.Y - start.Y > 0)
            {
                square = new cvRect(start.X, start.Y, Math.Abs(end.Y - start.Y), Math.Abs(end.Y - start.Y));
            }
            else if (end.X - start.X > 0 && end.Y - start.Y < 0)
            {
                square = new cvRect(start.X, end.Y, Math.Abs(end.Y - start.Y), Math.Abs(end.Y - start.Y));
            }

            else if (end.X - start.X < 0 && end.Y - start.Y > 0)
            {
                if (end.X >= end.Y) square = new cvRect(end.X, start.Y, Math.Abs(end.Y - start.Y), Math.Abs(end.Y - start.Y));
                else square = new cvRect(end.X, start.Y, Math.Abs(end.X - start.X), Math.Abs(end.X - start.X));
            }
            else
            {
                if (end.X >= end.Y) square = new cvRect(end.X, start.Y - (start.X - end.X), Math.Abs(end.X - start.X), Math.Abs(end.X - start.X));
                else square = new cvRect(start.X - (start.Y - end.Y), end.Y, Math.Abs(end.Y - start.Y), Math.Abs(end.Y - start.Y));
            }
            return square;
        }

        public Rectangle CreateRect(int startX, int startY, int endX, int endY)
        {
            Rectangle rect;
            if (endX - startX > 0 && endY - startY > 0)
            {
                rect = new Rectangle(startX, startY, Math.Abs(endX - startX), Math.Abs(endY - startY));
            }
            else if (endX - startX > 0 && endY - startY < 0)
            {
                rect = new Rectangle(startX, endY, Math.Abs(endX - startX), Math.Abs(endY - startY));
            }
            else if (endX - startX < 0 && endY - startY > 0)
            {
                rect = new Rectangle(endX, startY, Math.Abs(endX - startX), Math.Abs(endY - startY));
            }
            else
            {
                rect = new Rectangle(endX, endY, Math.Abs(endX - startX), Math.Abs(endY - startY));
            }
            return rect;
        }

        public Rectangle CreateSquare(int startX, int startY, int endX, int endY)
        {
            Rectangle square;
            if (endX - startX > 0 && endY - startY > 0)
            {
                square = new Rectangle(startX, startY, Math.Abs(endY - startY), Math.Abs(endY - startY));
            }
            else if (endX - startX > 0 && endY - startY < 0)
            {
                square = new Rectangle(startX, endY, Math.Abs(endY - startY), Math.Abs(endY - startY));
            }

            else if (endX - startX < 0 && endY - startY > 0)
            {
                if (endX >= endY) square = new Rectangle(endX, startY, Math.Abs(endY - startY), Math.Abs(endY - startY));
                else square = new Rectangle(endX, startY, Math.Abs(endX - startX), Math.Abs(endX - startX));
            }
            else
            {
                if (endX >= endY) square = new Rectangle(endX, startY - (startX - endX), Math.Abs(endX - startX), Math.Abs(endX - startX));
                else square = new Rectangle(startX - (startY - endY), endY, Math.Abs(endY - startY), Math.Abs(endY - startY));
            }
            square.Height = square.Width;

            return square;
        }

        private void pictureBox1_MouseEnter(object sender, EventArgs e)
        {
            if (MainMenu.mode == MainMode.DRAW && mDraw.shape == TShape.PATH)
            {
                PictureBox1.Cursor = new Cursor(penCursor.GetHicon());
            }
            else
            {
                PictureBox1.Cursor = Cursors.Arrow;
            }
        }

        /// <summary>
        /// Canvas에 Bitmap Cursor를 보여줌
        /// </summary>
        public void SetCursor()
        {
            int cursorSize = 30; //# 커서 십자가 사이즈
            int penSize;
            if (mPen != null)
            {
                penSize = mPen.penSize;
            }
            else
            {
                penSize = 5;
            }

            if (penSize <= cursorSize)
            {
                penCursor = new Bitmap(cursorSize, cursorSize);
                Graphics g = Graphics.FromImage(penCursor);
                g.Clear(Color.Transparent);
                g.DrawLine(Pens.Black, cursorSize / 2, 0, cursorSize / 2, cursorSize - 1);
                g.DrawLine(Pens.Black, 0, cursorSize / 2, cursorSize - 1, cursorSize / 2);
                g.DrawEllipse(Pens.Black, (cursorSize - penSize) / 2, (cursorSize - penSize) / 2, penSize, penSize);
            }
            else
            {
                penCursor = new Bitmap(penSize, penSize);
                Graphics g = Graphics.FromImage(penCursor);
                g.Clear(Color.Transparent);
                g.DrawLine(Pens.Black, penSize / 2, penSize / 2 - cursorSize / 2, penSize / 2, penSize / 2 + cursorSize / 2);
                g.DrawLine(Pens.Black, penSize / 2 - cursorSize / 2, penSize / 2, penSize / 2 + cursorSize / 2, penSize / 2);
                g.DrawEllipse(Pens.Black, 0, 0, penSize - 1, penSize - 1);
            }
        }

        private void panel1_Scroll(object sender, ScrollEventArgs e)
        {
            mVector.drawing = true;
            PictureBox1.Refresh();
            mVector.drawing = false;
        }

        private void Canvas_DragDrop(object sender, DragEventArgs e)
        {
            string[] filePathArray = e.Data.GetData(DataFormats.FileDrop) as string[];
            if (filePathArray != null)
            {
                foreach (string filePath in filePathArray)
                {
                    Console.WriteLine(filePath);
                    Image img = Image.FromFile(filePath);

                    Vector vector = new Vector();
                    vector.MdiParent = this.MdiParent;

                    Canvas canvas = new Canvas(vector);
                    canvas.MdiParent = this.MdiParent;
                    canvas.Text = filePath;
                    canvas.Height = img.Height + statusStrip1.Height + 38;
                    canvas.Width = img.Width + 16;

                    PenManager specialPen = new PenManager(canvas);
                    specialPen.MdiParent = this.MdiParent;

                    Palette palette = new Palette(canvas);
                    palette.MdiParent = this.MdiParent;

                    MainMenu mainmenu = new MainMenu(canvas, vector);
                    mainmenu.MdiParent = this.MdiParent;

                    canvas.StartPosition = FormStartPosition.Manual;
                    canvas.Location = new Point(0, 0);
                    canvas.Show();
                    canvas.dst = BitmapConverter.ToMat((Bitmap)img);
                    canvas.matList.Add(canvas.dst);
                    canvas.PictureBox1.ImageIpl = canvas.dst;
                }
            }
        }

        private void Canvas_DragEnter(object sender, DragEventArgs e)
        {
            Console.WriteLine("DragEnter");
            
        }

        private void Canvas_DragLeave(object sender, EventArgs e)
        {
            Console.WriteLine("DragLeave");
        }

        private void Canvas_DragOver(object sender, DragEventArgs e)
        {
            if (e.Data.GetDataPresent(DataFormats.FileDrop))
            {
                e.Effect = DragDropEffects.Copy;
            }
            else
            {
                e.Effect = DragDropEffects.None;
            }
        }

        private void Canvas_KeyDown(object sender, KeyEventArgs e)
        {
            switch (mDrawObject)
            {
                case CanvasType.BITMAP:
                    this.HotKey(e.KeyCode, e.Modifiers);
                    break;
                case CanvasType.VECTOR:
                    mVector.HotKey(e.KeyCode, e.Modifiers);
                    break;
            }
        }

        //# 단축키
        public void HotKey(Keys key, Keys modifier)
        {
            if (modifier == Keys.Control)
            {
                switch (key)
                {
                    case Keys.A:
                        mWorkarea.SelAll();
                        break;
                }
            }
            else
            {
                switch (key)
                {
                    case Keys.X:
                        dst = LastList(matList);
                        PictureBox1.ImageIpl = dst;
                        restore = dst.Clone();

                        if (mWorkarea.connPath != null)
                        {
                            mWorkarea.connPath.CloseAllFigures();
                            mWorkarea.AddArea(mWorkarea.connPath);

                            mWorkarea.pathPoints = null;
                            mWorkarea.pathByte = null;
                            mWorkarea.connPath = null;
                            mWorkarea.connPathList.Clear();

                            PictureBox1.Refresh();
                        }
                        drawing = false;
                        break;
                    case Keys.OemOpenBrackets:
                        if (mPen.nud_size.Value <= 1) return;
                        mPen.nud_size.Value--;
                        SetCursor();
                        PictureBox1.Cursor = new Cursor(penCursor.GetHicon());
                        break;
                    case Keys.OemCloseBrackets:
                        if (mPen.nud_size.Value >= 50) return;
                        mPen.nud_size.Value++;
                        PictureBox1.Cursor = new Cursor(penCursor.GetHicon());
                        break;
                }
            }
        }
    }
}
