﻿using OpenCvSharp;
using OpenCvSharp.Extensions;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using Point = System.Drawing.Point;

namespace TexMaster
{
    public enum TPCanvasSizeType
    {
        cstA4,
        cstA3,
        cstA2,
        cstA1,
        cstA0,
        cstFree,
        cstLetter,
        cstLegal,
        cstTabloid
    }

    public enum TPCanvasOrientation
    {
        coPortrait,
        coLandscape
    }

    public enum TUnit
    {
        uDot,
        uInch,
        uCm
    }

    public struct TPCanvasInfo
    {
        public TPCanvasSizeType SizeType;
        public TPCanvasOrientation Orientation;
        public int DotsPerInch;
        public int Width;
        public int Height;

        public void SetSize(TPCanvasSizeType st, int w = 0, int h = 0)
        {
            SizeType = st;
            if (st.Equals(TPCanvasSizeType.cstFree))
            {
                Width = w;
                Height = h;
            }
            else
            {
                Width = GetWidth();
                Height = GetHeight();

            }
        }

        public int GetWidth()
        {
            if (SizeType.Equals(TPCanvasSizeType.cstFree))
            {
                return Width;
            }

            if (Orientation.Equals(TPCanvasOrientation.coPortrait))
            {
                switch (SizeType)
                {
                    case TPCanvasSizeType.cstA0:
                        return (int)(31.2 * DotsPerInch);
                    case TPCanvasSizeType.cstA1:
                        return (int)(22.0 * DotsPerInch);
                    case TPCanvasSizeType.cstA2:
                        return (int)(15.6 * DotsPerInch);
                    case TPCanvasSizeType.cstA3:
                        return (int)(11.0 * DotsPerInch);
                    case TPCanvasSizeType.cstA4:
                        return (int)(7.8 * DotsPerInch);
                    case TPCanvasSizeType.cstLetter:
                        return (int)(8.0 * DotsPerInch);
                    case TPCanvasSizeType.cstLegal:
                        return (int)(8.0 * DotsPerInch);
                    case TPCanvasSizeType.cstTabloid:
                        return (int)(10.3 * DotsPerInch);
                    default:
                        return (int)(7.8 * DotsPerInch);
                }
            }
            else
            {
                switch (SizeType)
                {
                    case TPCanvasSizeType.cstA0:
                        return (int)(44.0 * DotsPerInch);
                    case TPCanvasSizeType.cstA1:
                        return (int)(31.2 * DotsPerInch);
                    case TPCanvasSizeType.cstA2:
                        return (int)(22.0 * DotsPerInch);
                    case TPCanvasSizeType.cstA3:
                        return (int)(15.6 * DotsPerInch);
                    case TPCanvasSizeType.cstA4:
                        return (int)(11.0 * DotsPerInch);
                    case TPCanvasSizeType.cstLetter:
                        return (int)(10.3 * DotsPerInch);
                    case TPCanvasSizeType.cstLegal:
                        return (int)(13.2 * DotsPerInch);
                    case TPCanvasSizeType.cstTabloid:
                        return (int)(16.0 * DotsPerInch);
                    default:
                        return (int)(11.0 * DotsPerInch);
                }
            }
        }

        public int GetHeight()
        {
            if (SizeType.Equals(TPCanvasSizeType.cstFree))
            {
                return Height;
            }

            if (Orientation.Equals(TPCanvasOrientation.coPortrait))
            {
                switch (SizeType)
                {
                    case TPCanvasSizeType.cstA0:
                        return (int)(44.0 * DotsPerInch);
                    case TPCanvasSizeType.cstA1:
                        return (int)(31.2 * DotsPerInch);
                    case TPCanvasSizeType.cstA2:
                        return (int)(22.0 * DotsPerInch);
                    case TPCanvasSizeType.cstA3:
                        return (int)(15.6 * DotsPerInch);
                    case TPCanvasSizeType.cstA4:
                        return (int)(11.0 * DotsPerInch);
                    case TPCanvasSizeType.cstLetter:
                        return (int)(10.3 * DotsPerInch);
                    case TPCanvasSizeType.cstLegal:
                        return (int)(13.2 * DotsPerInch);
                    case TPCanvasSizeType.cstTabloid:
                        return (int)(16.0 * DotsPerInch);
                    default:
                        return (int)(11.0 * DotsPerInch);
                }
            }
            else
            {
                switch (SizeType)
                {
                    case TPCanvasSizeType.cstA0:
                        return (int)(31.2 * DotsPerInch);
                    case TPCanvasSizeType.cstA1:
                        return (int)(22.0 * DotsPerInch);
                    case TPCanvasSizeType.cstA2:
                        return (int)(15.6 * DotsPerInch);
                    case TPCanvasSizeType.cstA3:
                        return (int)(11.0 * DotsPerInch);
                    case TPCanvasSizeType.cstA4:
                        return (int)(7.8 * DotsPerInch);
                    case TPCanvasSizeType.cstLetter:
                        return (int)(8.0 * DotsPerInch);
                    case TPCanvasSizeType.cstLegal:
                        return (int)(8.0 * DotsPerInch);
                    case TPCanvasSizeType.cstTabloid:
                        return (int)(10.3 * DotsPerInch);
                    default:
                        return (int)(7.8 * DotsPerInch);
                }
            }
        }

        public string Name(TUnit unit, int w, int h)
        {
            switch (SizeType)
            {
                case TPCanvasSizeType.cstA0:
                    return "A0";
                case TPCanvasSizeType.cstA1:
                    return "A1";
                case TPCanvasSizeType.cstA2:
                    return "A2";
                case TPCanvasSizeType.cstA3:
                    return "A3";
                case TPCanvasSizeType.cstA4:
                    return "A4";
                case TPCanvasSizeType.cstLetter:
                    return "Letter";
                case TPCanvasSizeType.cstLegal:
                    return "Legal";
                case TPCanvasSizeType.cstTabloid:
                    return "Tabloid";
                default:
                    if (unit.Equals(TUnit.uDot))
                        return String.Format("%d x %d", w, h);
                    else if (unit.Equals(TUnit.uInch))
                        return String.Format("%f x %f", (double)w / DotsPerInch, (double)h / DotsPerInch);
                    else if (unit.Equals(TUnit.uCm))
                        return String.Format("%f x %f", 2.54 * w / DotsPerInch, 2.54 * h / DotsPerInch);
                    break;
            }
            return "";
        }
    }

    public class MultiColor
    {
        public bool isSelect;
        public bool isDelete;

        public List<int> colList;
        public List<int> lastList;

        public Point start, end;
        public bool drawing;

        public MultiColor()
        {
            isSelect = false;
            isDelete = false;
            colList = new List<int>();
            lastList = new List<int>();
        }

        public Mat GetHoleMat(GraphicsPath workPath, int width, int height)
        {
            Mat hsv = new Mat();
            if (workPath != null)
            {
                //# 이미지 생성해서 WorkPath 파란색으로 칠함(마스킹 용이하게 하기위함)
                Bitmap maskImg = new Bitmap(width, height);
                Graphics maskGr = Graphics.FromImage(maskImg);
                Color cMask = Color.FromArgb(0, 0, 255); //# blue
                Scalar sMask = new Scalar(cMask.B, cMask.G, cMask.R, cMask.A);
                maskGr.FillPath(new SolidBrush(cMask), workPath);

                //# make mask mat
                Mat ori = BitmapConverter.ToMat(maskImg);
                Cv2.CvtColor(ori, hsv, ColorConversionCodes.BGR2HSV);
                Cv2.InRange(ori, sMask, sMask, hsv);
            }
            return hsv;
        }

        public Mat GetWorkMat(Mat totalMat, GraphicsPath workPath)
        {
            Mat cropMat = new Mat();
            //# 작업 구역이 설정 되어있을 때
            if (workPath != null)
            {
                Rectangle bound = Rectangle.Round(workPath.GetBounds());
                Rect rect = new Rect(bound.X, bound.Y, bound.Width, bound.Height);
                totalMat.CopyTo(cropMat, GetHoleMat(workPath, totalMat.Width, totalMat.Height));
                cropMat[rect].CopyTo(cropMat);
            }
            //# 작업 구역이 설정 되어있지 않을 때 - 전체 영역
            else
            {
                cropMat = totalMat;
            }
            return cropMat;
        }

        public void PanelClick(object sender, EventArgs e)
        {
            lastList = colList;
            colList.Remove(((Panel)sender).BackColor.ToArgb());
            ((Panel)sender).Parent.Controls.Remove((Panel)sender);
        }

        public void SelectAll(Panel parentPanel, Mat totalMat, GraphicsPath workPath)
        {
            if (GetWorkMat(totalMat, workPath) == null) return;

            Mat tempMat = GetWorkMat(totalMat, workPath).Clone();
            colList.Clear();

            for (int row = 0; row < tempMat.Height - 1; row++)
            {
                for (int col = 0; col < tempMat.Width - 1; col++)
                {
                    if (!colList.Contains(tempMat.At<int>(row, col)))
                    {
                        colList.Add(tempMat.At<int>(row, col));
                    }
                }
            }

            lastList = colList;
            parentPanel.Controls.Clear();
            for (int i = 0; i < colList.Count; i++)
            {
                Panel panel = new Panel();
                panel.Click += PanelClick;
                panel.Width = 17;
                panel.Height = 17;
                panel.BackColor = Color.FromArgb(colList[i]);
                panel.BorderStyle = BorderStyle.FixedSingle;
                parentPanel.Controls.Add(panel);
            }
        }

        public void SelectClear(Panel parentPanel)
        {
            colList.Clear();
            lastList = colList;
            parentPanel.Controls.Clear();
        }

        public void SelectRestore(Panel parentPanel)
        {
            colList = lastList;
            parentPanel.Controls.Clear();
            for (int i = 0; i < colList.Count; i++)
            {
                Panel panel = new Panel();
                panel.Click += PanelClick;
                panel.Width = 17;
                panel.Height = 17;
                panel.BackColor = Color.FromArgb(colList[i]);
                panel.BorderStyle = BorderStyle.FixedSingle;
                parentPanel.Controls.Add(panel);
            }
        }

        public void SetSelectState(Button button)
        {
            ToolTip toolTip = new ToolTip();
            toolTip.ShowAlways = true;

            if (!isSelect)
            {
                isSelect = true;
                button.Image = new Bitmap(Application.StartupPath + "\\Image\\select.bmp");
                toolTip.SetToolTip(button, "단색");
            }
            else
            {
                isSelect = false;
                button.Image = new Bitmap(Application.StartupPath + "\\Image\\area.bmp");
                toolTip.SetToolTip(button, "구역색");
            }
        }

        public void SetInsertState(Button button)
        {
            ToolTip toolTip = new ToolTip();
            toolTip.ShowAlways = true;

            if (isDelete)
            {
                isDelete = false;
                button.Image = new Bitmap(Application.StartupPath + "\\Image\\insert.bmp");
                toolTip.SetToolTip(button, "삽입");
            }
            else
            {
                isDelete = true;
                button.Image = new Bitmap(Application.StartupPath + "\\Image\\delete.bmp");
                toolTip.SetToolTip(button, "삭제");
            }
        }

        public void MouseDown(Mat totalMat, GraphicsPath workPath, Panel parentPanel, MouseEventArgs e)
        {
            Mat cropMat = GetWorkMat(totalMat, workPath);
            if (e.Button == MouseButtons.Middle)
            {
                if (isSelect)    //# 단색 추출
                {
                    if (!isDelete)  //# 색상 추가
                    {
                        if (colList.Contains(cropMat.At<int>(e.Y, e.X))) return;
                        Panel panel = new Panel();
                        panel.Click += PanelClick;
                        panel.Width = 17;
                        panel.Height = 17;
                        panel.BackColor = Color.FromArgb(cropMat.At<int>(e.Y, e.X));
                        colList.Add(cropMat.At<int>(e.Y, e.X));
                        panel.BorderStyle = BorderStyle.FixedSingle;
                        parentPanel.Controls.Add(panel);
                    }
                    else            //# 색상 삭제
                    {
                        if (!colList.Contains(cropMat.At<int>(e.Y, e.X))) return;
                        colList.Remove(cropMat.At<int>(e.Y, e.X));

                        parentPanel.Controls.Clear();
                        for (int i = 0; i < colList.Count; i++)
                        {
                            Panel panel = new Panel();
                            panel.Click += PanelClick;
                            panel.Width = 17;
                            panel.Height = 17;
                            panel.BackColor = Color.FromArgb(colList[i]);
                            panel.BorderStyle = BorderStyle.FixedSingle;
                            parentPanel.Controls.Add(panel);
                        }
                    }
                }
                else
                {
                    start = e.Location;
                    drawing = true;
                }
            }
        }

        public void MouseMove(MouseEventArgs e)
        {
            if (!isSelect)
            {
                end = e.Location;
            }
        }

        public void MouseUp(Mat totalMat, Panel parentPanel, MouseEventArgs e)
        {
            drawing = false;

            if (!isSelect)
            {
                if (e.Button == MouseButtons.Middle)
                {
                    end = e.Location;
                    int x = Math.Min(start.X, end.X);
                    int y = Math.Min(start.Y, end.Y);
                    int width = Math.Abs(end.X - start.X);
                    int height = Math.Abs(end.Y - start.Y);
                    Rectangle rect = new Rectangle(x, y, width, height);

                    for (int row = rect.Top; row < rect.Top + rect.Height - 1; row++)
                    {
                        for (int col = rect.Left; col < rect.Left + rect.Width - 1; col++)
                        {
                            Color c = Color.FromArgb(totalMat.At<int>(row, col));
                            if (c.A == 255)
                            {
                                if (!isDelete && !colList.Contains(totalMat.At<int>(row, col)))
                                {
                                    colList.Add(totalMat.At<int>(row, col));
                                }
                                else if (isDelete && colList.Contains(totalMat.At<int>(row, col)))
                                {
                                    colList.Remove(totalMat.At<int>(row, col));
                                }
                            }
                        }
                    }

                    parentPanel.Controls.Clear();
                    for (int i = 0; i < colList.Count; i++)
                    {
                        Panel panel = new Panel();
                        panel.Click += PanelClick;
                        panel.Width = 17;
                        panel.Height = 17;
                        panel.BackColor = Color.FromArgb(colList[i]);
                        panel.BorderStyle = BorderStyle.FixedSingle;
                        parentPanel.Controls.Add(panel);
                    }
                }
            }
        }

        public void Paint(PaintEventArgs e)
        {
            if (drawing)
            {
                int x = Math.Min(start.X, end.X);
                int y = Math.Min(start.Y, end.Y);
                int width = Math.Abs(end.X - start.X);
                int height = Math.Abs(end.Y - start.Y);
                Rectangle rect = new Rectangle(x, y, width, height);
                e.Graphics.DrawRectangle(Pens.Black, rect);
            }
        }

    }
}
