﻿using OpenCvSharp;
using OpenCvSharp.Extensions;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using cvPoint = OpenCvSharp.Point;
using cvRect = OpenCvSharp.Rect;
using cvSize = OpenCvSharp.Size;
using Point = System.Drawing.Point;

namespace TexMaster
{
    public partial class Kmean : Form
    {
        Bitmap map;
        Mat source;
        Mat target;
        Mat resMat;

        public Point start, end;
        public bool drawing;
        public Pen pen;
        public List<Color> colList;
        public List<Color> selList;
        public List<Mat> matList;
        public List<Mat> redoList;
        public List<Mat> seperateList;
        public Color currentColor;

        int width;
        int height;
        Canvas mCanvas;

        public Kmean(Canvas canvas)
        {
            InitializeComponent();
            mCanvas = canvas;
            mCanvas.mGrouping = this;
            pen = new Pen(Color.Black);
            colList = new List<Color>();
            selList = new List<Color>();
            matList = new List<Mat>();
            redoList = new List<Mat>();
            seperateList = new List<Mat>();
            pen.DashStyle = System.Drawing.Drawing2D.DashStyle.Dash;
        }

        private void Kmean_Load(object sender, EventArgs e)
        {
            pnl_src.Location = new Point(0, 0);
            pnl_dst.Location = new Point(0, 0);

            if (rdo_img.Checked)
            {
                pnl_dst.Visible = true;
                pnl_src.Visible = false;
                pnl_dst.Dock = DockStyle.Fill;
            }
            else
            {
                pnl_dst.Visible = false;
                pnl_src.Visible = true;
                pnl_src.Dock = DockStyle.Fill;
            }

            InitPanel();
            if (mCanvas.dst != null)
            {
                width = mCanvas.dst.Width;
                height = mCanvas.dst.Height;
                source = mCanvas.dst.Clone();
                pic_src.ImageIpl = mCanvas.dst.Clone();
                pic_dst.ImageIpl = mCanvas.dst.Clone();
            }

            trb_zoom.Value = 5;
            trb_zoom.Maximum = cmb_zoom.Items.Count - 1;
            cmb_zoom.SelectedIndex = 5;
        }

        private void InitPanel()
        {
            pnl_blur.Visible = false;
            pnl_auto.Visible = false;
            pnl_manual.Visible = false;
            pnl_blend.Visible = false;
        }

        //# test
        private void button1_Click(object sender, EventArgs e)
        {
            OpenFileDialog fileDialog = new OpenFileDialog();
            fileDialog.Filter = "Image Files(*.BMP;*.JPG;*.GIF)|*.BMP;*.JPG;*.GIF|All files (*.*)|*.*";
            fileDialog.ShowDialog();

            try
            {
                map = (Bitmap)Image.FromFile(fileDialog.FileName);
                source = BitmapConverter.ToMat(map);
                target = source;

                pic_src.ImageIpl = source;
                pic_dst.ImageIpl = source;
                matList.Add(source);

                width = source.Width;
                height = source.Height;
                lbl_size.Text = String.Format("Size : {0} x {1}", source.Width, source.Height);
                lbl_dpi.Text = String.Format("DPI : {0}", map.VerticalResolution);

            }
            catch
            {
                MessageBox.Show("파일을 선택해주세요.");
                return;
            }

            pnl_colorChip.Controls.Clear();
            btn_manual.Enabled = false;
            pnl_color.Visible = false;
            colList.Clear();
        }

        private void kmean_MouseDown(object sender, MouseEventArgs e)
        {
            //# 색상 설정
            if (e.Button == MouseButtons.Middle)
            {
                if (pnl_manual.Visible || pnl_blend.Visible)
                {
                    start = e.Location;
                    drawing = true;
                }
            }
        }

        private void kmean_MouseMove(object sender, MouseEventArgs e)
        {
            if (pic_dst.ImageIpl != null)
            {
                lbl_location.Text = String.Format("{0} : {1}", e.X, e.Y);
                Color c = Color.FromArgb(pic_dst.ImageIpl.At<int>(e.Y, e.X));
                if (e.Button == MouseButtons.Middle)
                {
                    if (pnl_manual.Visible || pnl_blend.Visible)
                    {
                        end = e.Location;
                    }
                    pic_dst.Refresh();
                }
            }
        }

        private void kmean_MouseUp(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Middle)
            {
                end = e.Location;
                drawing = false;
                pic_dst.Refresh();

                int r = 0, g = 0, b = 0;

                Rectangle rect = mCanvas.CreateRect(start.X, start.Y, end.X, end.Y);
                if (rect.Width <= 0 || rect.Height <= 0)
                {
                    rect.Width = 1;
                    rect.Height = 1;
                }

                cvRect cvRect = new cvRect(rect.X, rect.Y, rect.Width, rect.Height);
                Mat cropMat = pic_dst.ImageIpl[cvRect];
                Scalar avg = Cv2.Mean(cropMat);
                cropMat.SetTo(avg);
                r = (int)avg.Val2;
                g = (int)avg.Val1;
                b = (int)avg.Val0;

                Color color = Color.FromArgb(r, g, b);

                if (pnl_manual.Visible)
                {
                    colList.Add(color);

                    Panel panel = new Panel();
                    panel.Click += PanelDelete;
                    panel.Width = 23;
                    panel.Height = 23;
                    panel.BackColor = color;
                    panel.BorderStyle = BorderStyle.FixedSingle;

                    pnl_colorChip.Controls.Add(panel);
                    pnl_colorChip.VerticalScroll.Value = pnl_colorChip.VerticalScroll.Maximum;
                }

                if (pnl_blend.Visible)
                {
                    pnl_selColor.BackColor = color;
                }

                if (pnl_colorChip.Controls.Count > 0)
                    btn_manual.Enabled = true;
            }
        }

        private void kmean_Paint(object sender, PaintEventArgs e)
        {
            if (drawing)
            {
                e.Graphics.DrawRectangle(pen, mCanvas.CreateRect(start.X, start.Y, end.X, end.Y));
            }
        }

        public void Kmeans(Mat input, Mat result, int k)
        {
            colList.Clear();
            int oldpercent = 0;

            using (Mat points = new Mat())
            {
                using (Mat labels = new Mat())
                {
                    using (Mat centers = new Mat())
                    {
                        int width = input.Cols;
                        int height = input.Rows;

                        points.Create(width * height, 1, MatType.CV_32FC3);
                        centers.Create(k, 1, points.Type());
                        result.Create(height, width, input.Type());

                        int i = 0;
                        for (int y = 0; y < height; y++)
                        {
                            for (int x = 0; x < width; x++, i++)
                            {
                                Vec3f vec3f = new Vec3f
                                {
                                    Item0 = input.At<Vec3b>(y, x).Item0,
                                    Item1 = input.At<Vec3b>(y, x).Item1,
                                    Item2 = input.At<Vec3b>(y, x).Item2
                                };

                                points.Set<Vec3f>(i, vec3f);
                            }

                            int percent = (i * 50) / (height * width);

                            if (percent % 10 == 0 && percent != oldpercent)
                            {
                                oldpercent = percent;
                                progressbar.Value = percent;
                            }
                        }

                        progressbar.Value = 50;

                        Cv2.Kmeans(points, k, labels, new TermCriteria(CriteriaType.Eps | CriteriaType.MaxIter, 10, 1.0), 3, KMeansFlags.PpCenters, centers);

                        i = 0;
                        for (int y = 0; y < height; y++)
                        {
                            for (int x = 0; x < width; x++, i++)
                            {
                                int idx = labels.Get<int>(i);

                                Vec3b vec3b = new Vec3b();


                                int tmp = Convert.ToInt32(Math.Round(centers.At<Vec3f>(idx).Item0));
                                tmp = tmp > 255 ? 255 : tmp < 0 ? 0 : tmp;
                                vec3b.Item0 = Convert.ToByte(tmp);

                                tmp = Convert.ToInt32(Math.Round(centers.At<Vec3f>(idx).Item1));
                                tmp = tmp > 255 ? 255 : tmp < 0 ? 0 : tmp;
                                vec3b.Item1 = Convert.ToByte(tmp);

                                tmp = Convert.ToInt32(Math.Round(centers.At<Vec3f>(idx).Item2));
                                tmp = tmp > 255 ? 255 : tmp < 0 ? 0 : tmp;
                                vec3b.Item2 = Convert.ToByte(tmp);

                                if (!colList.Contains(Color.FromArgb(vec3b.Item2, vec3b.Item1, vec3b.Item0)))
                                {
                                    colList.Add(Color.FromArgb(vec3b.Item2, vec3b.Item1, vec3b.Item0));
                                }

                                result.Set<Vec3b>(y, x, vec3b);

                                int percent = 50 + (i * 50) / (height * width);
                                if (percent % 10 == 0 && percent != oldpercent)
                                {
                                    oldpercent = percent;
                                    progressbar.Value = percent;
                                }

                            }
                        }

                        progressbar.Value = 100;
                    }
                }
            }

            progressbar.Value = 0;
        }

        public Mat GetBlur(int value)
        {
            //pictureBoxIpl2.Image = null;
            source = null;
            target = null;

            //source = BitmapConverter.ToMat(map);
            source = pic_dst.ImageIpl.Clone();
            target = source;

            Mat blur = new Mat();
            Cv2.Blur(source, blur, new cvSize(value, value), new cvPoint(-1, -1), BorderTypes.Default);
            //pictureBoxIpl2.ImageIpl = blur;
            return blur;
        }

        private void nud_blur_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                trb_blur.Value = (int)nud_blur.Value;
                pic_dst.ImageIpl = GetBlur((int)nud_blur.Value);
            }
        }

        private void trb_blur_Scroll(object sender, EventArgs e)
        {
            nud_blur.Value = trb_blur.Value;
        }

        private void nud_blur_ValueChanged(object sender, EventArgs e)
        {
            pic_dst.ImageIpl = GetBlur((int)nud_blur.Value);
        }

        //# auto grouping
        private void btn_kmean_Click(object sender, EventArgs e)
        {
            Mat tempMat = GetBlur((int)nud_blur.Value);

            Kmeans(tempMat, target, Convert.ToInt32(nud_freq.Value));

            pnl_result.Controls.Clear();
            selList.Clear();
            seperateList.Clear();

            for (int i = 0; i < colList.Count; i++)
            {
                Panel panel = new Panel();
                panel.Click += PanelClick;
                panel.Paint += PanelPaint;
                panel.Parent = pnl_result;
                panel.Size = new System.Drawing.Size(23, 23);
                panel.BackColor = colList[i];
                panel.BorderStyle = BorderStyle.FixedSingle;

                //# mat 분리 작업
                Mat hsv = new Mat();
                Scalar s = new Scalar(colList[i].B, colList[i].G, colList[i].R, colList[i].A);
                Cv2.CvtColor(pic_dst.ImageIpl, hsv, ColorConversionCodes.BGR2HSV);
                Cv2.InRange(pic_dst.ImageIpl, s, s, hsv);
                seperateList.Add(hsv);
            }

            resMat = target;

            pnl_result.VerticalScroll.Value = pnl_result.VerticalScroll.Maximum;
            pic_dst.ImageIpl = resMat;
            source = resMat;
            pnl_color.Visible = true;

            matList.Add(resMat);
            redoList.Clear();
            tsp_func.Refresh();

        }

        //# manual grouping
        private void btn_manual_Click(object sender, EventArgs e)
        {
            Mat tempMat = GetBlur((int)nud_blur.Value);
            int i = 0;
            int oldpercent = 0;

            for (int y = 0; y < tempMat.Height; y++)
            {
                for (int x = 0; x < tempMat.Width; x++, i++)
                {
                    Vec3b vec3b = new Vec3b();
                    vec3b.Item0 = tempMat.At<Vec3b>(y, x)[0];
                    vec3b.Item1 = tempMat.At<Vec3b>(y, x)[1];
                    vec3b.Item2 = tempMat.At<Vec3b>(y, x)[2];

                    Color color = Color.FromArgb(vec3b.Item2, vec3b.Item1, vec3b.Item0);
                    //int newColor = Near(color);

                    tempMat.Set<Vec3b>(y, x, Near(color));

                    int percent = (i * 100) / (tempMat.Height * tempMat.Width);
                    if (percent % 10 == 0 && percent != oldpercent)
                    {
                        oldpercent = percent;
                        progressbar.Value = percent;
                    }
                }
            }

            pnl_result.Controls.Clear();
            selList.Clear();
            seperateList.Clear();

            for (i = 0; i < colList.Count; i++)
            {
                Panel panel = new Panel();
                panel.Click += PanelClick;
                panel.Paint += PanelPaint;
                panel.Parent = pnl_result;
                panel.Size = new System.Drawing.Size(23, 23);
                panel.BackColor = colList[i];
                panel.BorderStyle = BorderStyle.FixedSingle;

                //# mat 분리 작업
                Mat hsv = new Mat();
                Scalar s = new Scalar(colList[i].B, colList[i].G, colList[i].R, colList[i].A);
                Cv2.CvtColor(pic_dst.ImageIpl, hsv, ColorConversionCodes.BGR2HSV);
                Cv2.InRange(pic_dst.ImageIpl, s, s, hsv);
                seperateList.Add(hsv);
            }

            progressbar.Value = 100;

            pnl_color.Visible = true;
            resMat = tempMat;
            matList.Add(resMat);
            redoList.Clear();
            tsp_func.Refresh();

            pnl_result.VerticalScroll.Value = pnl_result.VerticalScroll.Maximum;
            pic_dst.ImageIpl = resMat;
            source = resMat;

            progressbar.Value = 0;
        }

        private void PanelDelete(object sender, EventArgs e)
        {
            Console.WriteLine("Delete ColorChip");
            colList.Remove(((Panel)sender).BackColor);
            pnl_colorChip.Controls.Remove((Panel)sender);
            pnl_colorChip.VerticalScroll.Value = pnl_colorChip.VerticalScroll.Maximum;
        }

        private void PanelPaint(object sender, PaintEventArgs e)
        {
            Pen pen = new Pen(Color.Black, 1);
            if (selList.Contains(((Panel)sender).BackColor))
            {
                e.Graphics.FillRectangle(new SolidBrush(Color.Blue), new Rectangle(-1, -1, 23, 5));
            }
            else
            {
                e.Graphics.FillRectangle(new SolidBrush(Color.White), new Rectangle(-1, -1, 23, 5));
            }
            e.Graphics.DrawRectangle(pen, new Rectangle(-1, -1, 23, 5));
        }

        private void PanelClick(object sender, EventArgs e)
        {
            Console.WriteLine("Click");
            Mat ori = matList[matList.Count - 1];

            Mat tempMat = new Mat();

            currentColor = ((Panel)sender).BackColor;
            if (selList.Contains(currentColor))
            {
                selList.Remove(currentColor);
            }
            else
            {
                selList.Add(currentColor);
            }

            ((Panel)sender).Refresh();

            Console.WriteLine(selList.Count);
            for (int i = 0; i < selList.Count; i++)
            {
                Mat hsv = new Mat();
                Scalar s = new Scalar(selList[i].B, selList[i].G, selList[i].R, selList[i].A);
                Cv2.CvtColor(ori, hsv, ColorConversionCodes.BGR2HSV);
                Cv2.InRange(ori, s, s, hsv);
                if (i == 0)
                {
                    tempMat = hsv;
                }
                else
                {
                    tempMat += hsv;
                }
            }

            Mat toggleMat = new Mat();
            Cv2.BitwiseNot(tempMat, toggleMat);

            ori.SetTo(Scalar.White, toggleMat);
            pic_dst.ImageIpl = ori;
        }

        private void radioButton_CheckedChanged(object sender, EventArgs e)
        {
            if (rdo_img.Checked)
            {
                pnl_dst.Visible = true;
                pnl_src.Visible = false;
                pnl_dst.Dock = DockStyle.Fill;
                grb_func.Enabled = true;
            }
            else
            {
                pnl_dst.Visible = false;
                pnl_src.Visible = true;
                pnl_src.Dock = DockStyle.Fill;
                grb_func.Enabled = false;
            }
        }

        private void tsb_blur_Click(object sender, EventArgs e)
        {
            InitPanel();

            if (pic_src.ImageIpl == null)
            {
                MessageBox.Show("이미지를 선택해주세요");
                return;
            }

            pnl_blur.Visible = true;
            pnl_blur.Top = grb_func.Top + grb_func.Height;
        }

        private void tsb_kmean_Click(object sender, EventArgs e)
        {
            if (pic_src.ImageIpl == null)
            {
                MessageBox.Show("이미지를 선택해주세요");
                return;
            }

            cms_kmean.Show(MousePosition);
        }

        private void tsm_color_Click(object sender, EventArgs e)
        {
            InitPanel();
            pnl_manual.Visible = true;
            pnl_colorChip.Controls.Clear();
            pnl_manual.Top = grb_func.Top + grb_func.Height;
        }

        private void tsm_freq_Click(object sender, EventArgs e)
        {
            InitPanel();
            pnl_auto.Visible = true;
            pnl_auto.Top = grb_func.Top + grb_func.Height;
        }

        private void tsp_func_Paint(object sender, PaintEventArgs e)
        {
            if (matList.Count > 1)
            {
                tsb_blend.Enabled = true;
                tsb_half.Enabled = true;
                tsb_undo.Enabled = true;
                if (redoList.Count > 0)
                {
                    tsb_redo.Enabled = true;
                }
                else
                {
                    tsb_redo.Enabled = false;
                }
            }
            else
            {
                tsb_blend.Enabled = false;
                tsb_half.Enabled = false;
                tsb_undo.Enabled = false;
                if (redoList.Count > 0)
                {
                    tsb_redo.Enabled = true;
                }
                else
                {
                    tsb_redo.Enabled = false;
                }
            }
        }

        private void tsb_undo_Click(object sender, EventArgs e)
        {
            InitPanel();

            redoList.Add(matList[matList.Count - 1]);
            matList.RemoveAt(matList.Count - 1);

            pic_dst.ImageIpl = matList[matList.Count - 1];
            pic_dst.Refresh();

            tsp_func.Refresh();
        }

        private void tsb_redo_Click(object sender, EventArgs e)
        {
            InitPanel();

            matList.Add(redoList[redoList.Count - 1]);
            redoList.RemoveAt(redoList.Count - 1);

            pic_dst.ImageIpl = matList[matList.Count - 1];
            pic_dst.Refresh();

            tsp_func.Refresh();
        }

        private void btn_clear_Click(object sender, EventArgs e)
        {
            colList.Clear();
            pnl_colorChip.Controls.Clear();
            btn_manual.Enabled = false;
        }

        private void tsb_blend_Click(object sender, EventArgs e)
        {
            InitPanel();

            pnl_blend.Visible = true;
            pnl_blend.Top = grb_func.Top + grb_func.Height;
        }

        private void btn_area_Click(object sender, EventArgs e)
        {
            cvPoint[][] contours;
            HierarchyIndex[] hierarchy;

            Mat hsv = new Mat();
            Color c = pnl_selColor.BackColor;
            Scalar selColor = new Scalar(c.B, c.G, c.R, c.A);
            Mat ori = pic_dst.ImageIpl.Clone();
            Cv2.CvtColor(ori, hsv, ColorConversionCodes.BGR2HSV);
            Cv2.InRange(ori, selColor, selColor, hsv);
            Cv2.FindContours(hsv, out contours, out hierarchy, RetrievalModes.Tree, ContourApproximationModes.ApproxTC89KCOS);

            Vec3b vec = new Vec3b();
            vec.Item0 = Color.White.B;
            vec.Item1 = Color.White.G;
            vec.Item2 = Color.White.R;

            foreach (cvPoint[] p in contours)
            {
                //Console.WriteLine(Cv2.ArcLength(p, true));
                //Console.WriteLine(Cv2.ContourArea(p, false));

                if (Cv2.ContourArea(p, false) < (int)nud_size.Value)
                {
                    for (int i = 0; i < p.Length; i++)
                    {
                        ori.Set<Vec3b>(p[i].Y, p[i].X, vec);
                    }
                }
            }

            pic_dst.ImageIpl = ori;
        }

        private void btn_outline_Click(object sender, EventArgs e)
        {
            cvPoint[][] contours;
            HierarchyIndex[] hierarchy;

            Mat hsv = new Mat();
            Scalar selColor = new Scalar(currentColor.B, currentColor.G, currentColor.R, currentColor.A);
            Mat ori = pic_dst.ImageIpl.Clone();
            Cv2.CvtColor(ori, hsv, ColorConversionCodes.BGR2HSV);
            Cv2.InRange(ori, selColor, selColor, hsv);
            Cv2.FindContours(hsv, out contours, out hierarchy, RetrievalModes.Tree, ContourApproximationModes.ApproxTC89KCOS);

            Vec3b vec = new Vec3b();
            vec.Item0 = Color.White.R;
            vec.Item1 = Color.White.G;
            vec.Item2 = Color.White.B;

            foreach (cvPoint[] p in contours)
            {
                Console.WriteLine(Cv2.ArcLength(p, false));

                if (Cv2.ArcLength(p, false) < 10)
                {
                    for (int i = 0; i < p.Length; i++)
                    {
                        ori.Set<Vec3b>(p[i].Y, p[i].X, vec);
                    }
                }

            }

            pic_dst.ImageIpl = ori;
        }

        private void btn_selClear_Click(object sender, EventArgs e)
        {
            Mat tempMat = matList[matList.Count - 1];
            for (int i = 0; i < seperateList.Count; i++)
            {
                Mat hsv = new Mat();
                Color c = colList[i];
                Scalar selColor = new Scalar(c.B, c.G, c.R, c.A);
                Mat ori = matList[matList.Count - 1];
                Cv2.CvtColor(ori, hsv, ColorConversionCodes.BGR2HSV);
                Cv2.InRange(ori, selColor, selColor, hsv);
                tempMat.SetTo(selColor, hsv);
            }

            selList.Clear();
            pnl_result.Refresh();
        }

        private void cmb_zoom_SelectedIndexChanged(object sender, EventArgs e)
        {
            trb_zoom.Value = cmb_zoom.SelectedIndex;
        }

        private void trb_zoom_ValueChanged(object sender, EventArgs e)
        {
            if (pic_dst.ImageIpl != null)
            {
                cmb_zoom.SelectedIndex = (int)trb_zoom.Value;
                float ratio = Convert.ToSingle(cmb_zoom.Text.Replace("%", "")) / 100F;
                target = new Mat();

                Cv2.Resize(source, target, new cvSize(width * ratio, height * ratio), 0, 0, InterpolationFlags.Area);
                pic_dst.ImageIpl = target;
            }

            if (pic_src.ImageIpl != null)
            {
                cmb_zoom.SelectedIndex = (int)trb_zoom.Value;
                float ratio = Convert.ToSingle(cmb_zoom.Text.Replace("%", "")) / 100F;
                target = new Mat();

                Cv2.Resize(source, target, new cvSize(width * ratio, height * ratio), 0, 0, InterpolationFlags.Area);
                pic_src.ImageIpl = target;
            }
        }

        private void Kmean_FormClosing(object sender, FormClosingEventArgs e)
        {
            DialogResult res = MessageBox.Show("적용하시겠습니까?", "Question", MessageBoxButtons.YesNoCancel);
            switch (res)
            {
                case DialogResult.Yes:
                    mCanvas.PictureBox1.ImageIpl = pic_dst.ImageIpl;
                    mCanvas.matList.Add(pic_dst.ImageIpl);
                    e.Cancel = true;
                    this.Hide();
                    return;
                case DialogResult.No:
                    e.Cancel = true;
                    this.Hide();
                    return;
                case DialogResult.Cancel:
                    e.Cancel = true;
                    return;
            }
        }

        private void btn_pre_run_Click(object sender, EventArgs e)
        {
            for (int i = 0; i < colList.Count; i++)
            {
                Panel panel = new Panel();
                panel.Click += PanelDelete;
                panel.Width = 23;
                panel.Height = 23;
                panel.BackColor = colList[i];
                panel.BorderStyle = BorderStyle.FixedSingle;

                pnl_colorChip.Controls.Add(panel);
                pnl_colorChip.VerticalScroll.Value = pnl_colorChip.VerticalScroll.Maximum;
            }
        }

        private void Kmean_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.A)
            {
                if (trb_zoom.Value == trb_zoom.Maximum) return;
                trb_zoom.Value++;
            }
            if (e.KeyCode == Keys.S)
            {
                if (trb_zoom.Value == trb_zoom.Minimum) return;
                trb_zoom.Value--;
            }
        }

        private void Kmean_VisibleChanged(object sender, EventArgs e)
        {
            InitPanel();
            source = BitmapConverter.ToMat((Bitmap)mCanvas.PictureBox1.Image);
            width = source.Width;
            height = source.Height;
            source = source.Clone();
            pic_src.ImageIpl = source.Clone();
            pic_dst.ImageIpl = source.Clone();
        }

        public Vec3b Near(Color target)
        {
            Vec3b near = new Vec3b();
            int min = int.MaxValue;

            for (int i = 0; i < colList.Count; i++)
            {
                int a = (int)(Math.Sqrt(Math.Pow((colList[i].R - target.R), 2) + Math.Pow((colList[i].G - target.G), 2) + Math.Pow((colList[i].B - target.B), 2)));

                if (min > a)
                {
                    min = a;
                    near.Item0 = colList[i].B;
                    near.Item1 = colList[i].G;
                    near.Item2 = colList[i].R;
                }
            }

            return near;

        }
    }
}
