﻿using OpenCvSharp;
using OpenCvSharp.Extensions;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Windows.Forms;
using cvPoint = OpenCvSharp.Point;
using cvSize = OpenCvSharp.Size;
using DashStyle = System.Drawing.Drawing2D.DashStyle;
using Point = System.Drawing.Point;

namespace TexMaster
{
    public enum TShape
    {
        PATH, LINE, CURVE, RECT, SQUARE, ELLIPSE, CIRCLE, OUTLINE
    }

    public partial class Draw : Form
    {
        /* global variable */
        public bool gDash = false;               //# Dashed Line Checkbox Click
        public int gCurveMode = 1;               //# Curve Point Order
        public bool gContour = false;

        public TShape shape;
        public bool isFill;

        public PenManager mPen;

        //public Bitmap currentBmp;
        //public Mat currentMat;

        public Mat shapeMat;

        private Canvas mCanvas;

        public List<ToolStripButton> tsbList;

        public Draw(Canvas canvas)
        {
            InitializeComponent();
            mCanvas = canvas;
            mCanvas.mDraw = this;
            this.mPen = mCanvas.mPen;

            tsbList = new List<ToolStripButton>();
        }

        private void Draw_Load(object sender, EventArgs e)
        {
            tsbList.Add(tsb_line);
            tsbList.Add(tsb_curve);
            tsbList.Add(tsb_rect);
            tsbList.Add(tsb_square);
            tsbList.Add(tsb_ellipse);
            tsbList.Add(tsb_circle);
            tsbList.Add(tsb_outline);

            //InitForm(TShape.LINE);
            ButtonClick(tsb_line, e);
        }

        private void InitForm(TShape shape)
        {
            List<Panel> pnlList = new List<Panel>();
            pnlList.Add(pnl_figure);

            switch (shape)
            {
                case TShape.LINE:
                    pnlList.Add(pnl_draw);
                    pnlList.Add(pnl_size);
                    lbl_length.Text = "길이";
                    lbl_angle.Text = "각도";
                    nud_length.Minimum = 0;
                    nud_length.Maximum = int.MaxValue;
                    nud_angle.Minimum = 0;
                    nud_angle.Maximum = 360;
                    pnlList.Add(pnl_dash);
                    pnlList.Add(pnl_line);
                    break;
                case TShape.CURVE:
                    pnlList.Add(pnl_draw);
                    pnlList.Add(pnl_dash);
                    pnlList.Add(pnl_curve);
                    break;
                case TShape.RECT:
                case TShape.SQUARE:
                case TShape.ELLIPSE:
                case TShape.CIRCLE:
                    pnlList.Add(pnl_draw);
                    pnlList.Add(pnl_size);
                    lbl_length.Text = "넓이";
                    lbl_angle.Text = "높이";
                    nud_length.Minimum = 0;
                    nud_length.Maximum = int.MaxValue;
                    nud_angle.Minimum = 0;
                    nud_angle.Maximum = int.MaxValue;
                    pnlList.Add(pnl_dash);

                    if (shape == TShape.ELLIPSE || shape == TShape.SQUARE)
                    {
                        pnlList.Add(pnl_ellipse);
                    }
                    break;
                case TShape.OUTLINE:
                    break;
            }

            int top = 0;
            for (int i = 0; i < pnlList.Count; i++)
            {
                pnlList[i].Visible = true;
                pnlList[i].Location = new Point(0, top);
                pnlList[i].BringToFront();
                top += pnlList[i].Height;
            }

            this.Height = pnlList[pnlList.Count - 1].Top + pnlList[pnlList.Count - 1].Height + 38;  //# 38 - Topbar Height

            this.Refresh();
        }

        private void ButtonClick(object sender, EventArgs e)
        {
            mCanvas.tsl_mode.Text = MainMenu.mode.ToString();
            shape = TShape.LINE;
            mCanvas.curCnt = 0;
            //# check clicked button
            for (int i = 0; i < tsbList.Count; i++)
            {
                tsbList[i].Checked = false;
                if (sender.Equals(tsbList[i]))
                {
                    tsbList[i].Checked = true;
                    shape = (TShape)(i + 1);
                }
            }

            Console.WriteLine(shape);
            InitForm(shape);
        }

        private void tsb_conn_Click(object sender, EventArgs e)
        {
            if (tsb_conn.Checked)
            {
                tsb_conn.Checked = false;
            }
            else
            {
                tsb_conn.Checked = true;
            }
        }

        private void tsb_fill_Click(object sender, EventArgs e)
        {
            if (tsb_fill.Checked)
            {
                isFill = false;
                tsb_fill.Checked = false;
            }
            else
            {
                isFill = true;
                tsb_fill.Checked = true;
            }
        }

        private void SizeKeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                ValueChange(sender, e);
            }
        }

        private void ValueChange(object sender, EventArgs e)
        {
            if (shape == TShape.SQUARE || shape == TShape.CIRCLE)
            {
                switch (((NumericUpDown)sender).Name)
                {
                    case "nud_length":
                        nud_angle.Value = nud_length.Value;
                        nud_angle.Refresh();
                        break;
                    case "nud_angle":
                        nud_length.Value = nud_angle.Value;
                        nud_length.Refresh();
                        break;
                }
            }

            Color c = mCanvas.mPen.penColor;
            Scalar penColor = new Scalar(c.B, c.G, c.R, c.A);
            int penSize = mCanvas.mPen.penSize;
            cvPoint start = new cvPoint(), end = new cvPoint();

            float length, angle = 0;
            int width, height;

            if (shape == TShape.LINE)
            {
                length = Convert.ToSingle(nud_length.Value);
                angle = Convert.ToSingle(nud_angle.Value);
                width = (int)Math.Abs(length * Math.Cos(angle * Math.PI / 180));
                height = (int)Math.Abs(length * Math.Sin(angle * Math.PI / 180));
                Console.WriteLine("{0}, {1}", width, height);
                if (length <= 0 || angle == 0) return;
            }
            else
            {
                width = Convert.ToInt32(nud_length.Value);
                height = Convert.ToInt32(nud_angle.Value);
                start = new cvPoint(penSize, penSize);
                end = new cvPoint(width, height);
                if (width <= 0 || height <= 0) return;
            }

            cvPoint center = new cvPoint(start.X + width / 2, start.Y + height / 2);
            shapeMat = new Mat(new cvSize(width + mPen.penSize, height + mPen.penSize), MatType.CV_8UC4, 0);

            switch (shape)
            {
                case TShape.LINE:
                    if ((angle >= 0 && angle < 90) || (angle >= 180 && angle < 270))
                    {
                        start = new cvPoint(penSize, height);
                        end = new cvPoint(width, penSize);
                        Cv2.Line(shapeMat, start, end, penColor, penSize);
                    }
                    else
                    {
                        start = new cvPoint(penSize, penSize);
                        end = new cvPoint(width, height);
                        Cv2.Line(shapeMat, start, end, penColor, penSize);
                    }
                    break;
                case TShape.RECT:
                    if (!tsb_fill.Checked)
                        Cv2.Rectangle(shapeMat, mCanvas.CreateCvRect(start, end), penColor, penSize);
                    else
                        Cv2.Rectangle(shapeMat, mCanvas.CreateCvRect(start, end), penColor, -1, LineTypes.Link8);
                    break;
                case TShape.SQUARE:
                    if (!tsb_fill.Checked)
                        Cv2.Rectangle(shapeMat, mCanvas.CreateCvSquare(start, end), penColor, penSize);
                    else
                        Cv2.Rectangle(shapeMat, mCanvas.CreateCvSquare(start, end), penColor, -1, LineTypes.Link8);
                    break;
                case TShape.ELLIPSE:
                    if (!tsb_fill.Checked)
                        Cv2.Ellipse(shapeMat, center, new cvSize(width / 2, height / 2), 0, 0, 360, penColor, penSize);
                    else
                        Cv2.Ellipse(shapeMat, center, new cvSize(width / 2, height / 2), 0, 0, 360, penColor, -1, LineTypes.Link8);
                    break;
                case TShape.CIRCLE:
                    if (!tsb_fill.Checked)
                        Cv2.Circle(shapeMat, center, width / 2, penColor, penSize);
                    else
                        Cv2.Circle(shapeMat, center, width / 2, penColor, -1, LineTypes.Link8);
                    break;
            }

            mCanvas.drawing = true;
        }

        private void chk_dash_CheckedChanged(object sender, EventArgs e)
        {
            if (chk_dash.Checked)
            {
                txt_dash.Enabled = true;
                txt_blank.Enabled = true;
            }
            else
            {
                txt_dash.Enabled = false;
                txt_blank.Enabled = false;
            }
        }

        private void ChangDashPattern(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                Pen pen = new Pen(mCanvas.mPen.penColor, mCanvas.mPen.penSize);
                pen.DashStyle = DashStyle.Custom;
                pen.DashPattern = new float[2] { Convert.ToSingle(txt_dash.Text), Convert.ToSingle(txt_blank.Text) };
            }
        }

        private void btn_first_Click(object sender, EventArgs e)
        {
            if (mCanvas.connPoints.Count < 3) return;

            cvPoint firstPt = mCanvas.connPoints[0];
            cvPoint lastPt = mCanvas.LastList(mCanvas.connPoints);
            Mat lastMat = mCanvas.LastList(mCanvas.matList);

            if (tsb_fill.Checked)
            {
                List<List<cvPoint>> listOfPoints = new List<List<cvPoint>>();
                listOfPoints.Add(mCanvas.connPoints);
                Cv2.FillPoly(lastMat, listOfPoints, mCanvas.penColor);
                Cv2.Line(lastMat, lastPt, firstPt, mCanvas.penColor, mCanvas.penSize);
            }
            else
            {
                Cv2.Line(lastMat, lastPt, firstPt, mCanvas.penColor, mCanvas.penSize);
            }
            mCanvas.PictureBox1.ImageIpl = lastMat;
            mCanvas.matList.Add(lastMat);
            mCanvas.connPoints.Clear();
            mCanvas.drawing = false;
        }

        private void btn_delete_Click(object sender, EventArgs e)
        {
            if (mCanvas.connPoints.Count <= 1 || mCanvas.matList.Count == 0 || !tsb_conn.Checked) return;

            mCanvas.connPoints.Remove(mCanvas.LastList(mCanvas.connPoints));
            if (shape == TShape.CURVE) mCanvas.curPoints[0] = mCanvas.LastList(mCanvas.connPoints);
            mCanvas.matList.Remove(mCanvas.LastList(mCanvas.matList));
        }

        private void btn_last_Click(object sender, EventArgs e)
        {
            mCanvas.src = mCanvas.LastList(mCanvas.matList);
            mCanvas.dst = mCanvas.LastList(mCanvas.matList);
            mCanvas.drawing = true;
        }

        public Scalar ColorToScalar(Color color)
        {
            return new Scalar(color.B, color.G, color.R, color.A);
        }

        private void Draw_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = true;
            this.Hide();
        }

        public void draw_MouseMove(MouseEventArgs e)
        {
            mCanvas.dst = mCanvas.LastList(mCanvas.matList).Clone();
            cvPoint start = new cvPoint(e.X - shapeMat.Width / 2, e.Y - shapeMat.Height / 2);
            Color c = mCanvas.mPen.penColor;
            Scalar penColor = new Scalar(c.B, c.G, c.R, c.A);
            mCanvas.dst[new Rect(start.X, start.Y, shapeMat.Width, shapeMat.Height)].SetTo(penColor, shapeMat);
            mCanvas.PictureBox1.ImageIpl = mCanvas.dst;
        }
    }
}
