﻿using OpenCvSharp;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Windows.Forms;
using Point = System.Drawing.Point;
using cvSize = OpenCvSharp.Size;
using AForge.Imaging;

namespace TexMaster
{
    public partial class Reproportion : Form
    {
        public WZoom mZoomMode;             //# WorkArea Shape type Init
        public Canvas mCanvas;              //# Main Canvas
        public Mat zoomMat;                 //# Zoom Mat
        public Mat oriMat;                  //# Original Mat
        public Mat zMask;                   //# Zoom Mask
        public Mat roi;                     //# WorkArea in Main Canvas
        public Mat restore;
        public GraphicsPath pullPath;       //# Pull Mode Path
        public List<PointF> pullPts;        //# Pull Path Points
        public GraphicsPath rectPath;       //# Rect Mode Path
        public GraphicsPath preRectPath;    //# 사이즈 조절하기 전 Path
        public PointF start, end;           //# Mouse Down, Up Point

        private bool move;
        private bool isRatio;
        private int moveIdx;

        public enum WZoom
        {
            RECT, PULL, PERS //Perspective
        }
        public Reproportion(Canvas canvas)
        {
            InitializeComponent();
            mCanvas = canvas;
            mCanvas.mReproportion = this;
            mZoomMode = WZoom.RECT;
            cmb_unit.SelectedIndex = 0;
            pullPath = new GraphicsPath();
            rectPath = new GraphicsPath();
            zoomMat = new Mat();
            zMask = new Mat();
            restore = new Mat();
            pullPts = new List<PointF>();
            move = false;
            isRatio = false;
        }

        private void Init()
        {
            pnl_rect.Visible = false;
            pnl_pull.Visible = false;
            pnl_persp.Visible = false;
        }

        public void zoom_Paint(PaintEventArgs e)
        {
            int size = mCanvas.mVector.handleSize;
            switch (mZoomMode)
            {
                case WZoom.RECT:
                    e.Graphics.DrawPath(Pens.Black, rectPath);
                    for (int i = 0; i < rectPath.PointCount; i++)
                    {
                        e.Graphics.DrawRectangle(Pens.Red, rectPath.PathPoints[i].X - size, rectPath.PathPoints[i].Y - size, size * 2, size * 2);
                    }
                    break;
                case WZoom.PULL:
                    e.Graphics.DrawPath(Pens.Black, pullPath);
                    for (int i = 0; i < pullPath.PointCount; i++)
                    {
                        e.Graphics.DrawRectangle(Pens.Red, pullPath.PathPoints[i].X - size, pullPath.PathPoints[i].Y - size, size * 2, size * 2);
                    }
                    break;
                case WZoom.PERS:
                    break;
            }
        }

        public void reproportion_MouseDown(MouseEventArgs e)
        {
            mCanvas.drawing = true;
            int size = mCanvas.mVector.handleSize;
            switch (mZoomMode)
            {
                case WZoom.RECT:
                    for (int i = 0; i < rectPath.PointCount; i++)
                    {
                        RectangleF rect = new RectangleF(rectPath.PathPoints[i].X - size, rectPath.PathPoints[i].Y - size, size * 2, size * 2);
                        if (rect.Contains(e.Location))
                        {
                            start = new PointF(rectPath.PathPoints[(i + 2) % 4].X, rectPath.PathPoints[(i + 2) % 4].Y);
                            move = true;
                            preRectPath = (GraphicsPath)rectPath.Clone();
                            break;
                        }
                    }
                    break;
                case WZoom.PULL:
                    for (int i = 0; i < pullPath.PointCount; i++)
                    {
                        RectangleF rect = new RectangleF(pullPath.PathPoints[i].X - size, pullPath.PathPoints[i].Y - size, size * 2, size * 2);
                        if (rect.Contains(e.Location))
                        {
                            move = true;
                            moveIdx = i;
                            break;
                        }
                    }
                    break;
                case WZoom.PERS:
                    break;
            }
            mCanvas.PictureBox1.Refresh();
        }
        public void reproportion_MouseMove(MouseEventArgs e)
        {
            switch (mZoomMode)
            {
                case WZoom.RECT:
                    if (move)
                    {
                        end = e.Location;
                        rectPath = new GraphicsPath();
                        rectPath.AddRectangle(mCanvas.mVector.GetRect(start.X, start.Y, end.X, end.Y));

                        RectangleF rect = rectPath.GetBounds();
                        SetZoomMat(oriMat, zoomMat, new cvSize(rect.Width, rect.Height), (int)rect.X, (int)rect.Y);
                        SetSizeUnit();
                    }
                    break;
                case WZoom.PULL:
                    if (move)
                    {
                        pullPts[moveIdx] = e.Location;
                        pullPath = new GraphicsPath(pullPts.ToArray(), pullPath.PathTypes);
                    }
                    break;
                case WZoom.PERS:
                    break;
            }
            mCanvas.PictureBox1.Refresh();
        }
        public void reproportion_MouseUp(MouseEventArgs e)
        {
            switch (mZoomMode)
            {
                case WZoom.RECT:
                    break;
                case WZoom.PULL:
                    break;
                case WZoom.PERS:
                    break;
            }
            move = false;
            mCanvas.PictureBox1.Refresh();
            mCanvas.drawing = false;
        }


        private void Reproportion_Load(object sender, EventArgs e)
        {
            Init();
            pnl_rect.Visible = true;
            pnl_rect.Location = new Point(0, pnl_button.Height);
            this.Height = pnl_button.Height + pnl_rect.Height + 38;
        }

        private void btn_rect_Click(object sender, EventArgs e)
        {
            Init();
            pnl_rect.Visible = true;
            pnl_rect.Location = new Point(0, pnl_button.Height);
            this.Height = pnl_button.Height + pnl_rect.Height + 38;

            mZoomMode = WZoom.RECT;
            mCanvas.drawing = true;
            mCanvas.PictureBox1.Refresh();
            mCanvas.drawing = false;
        }

        private void btn_pull_Click(object sender, EventArgs e)
        {
            Init();
            pnl_pull.Visible = true;
            pnl_pull.Location = new Point(0, pnl_button.Height);
            this.Height = pnl_button.Height + pnl_pull.Height + 38;

            mZoomMode = WZoom.PULL;
            mCanvas.drawing = true;
            mCanvas.PictureBox1.Refresh();
            mCanvas.drawing = false;
        }

        private void btn_persp_Click(object sender, EventArgs e)
        {
            Init();
            pnl_persp.Visible = true;
            pnl_persp.Location = new Point(0, pnl_button.Height);
            this.Height = pnl_button.Height + pnl_persp.Height + 38;

            mZoomMode = WZoom.PERS;
            mCanvas.drawing = true;
            mCanvas.PictureBox1.Refresh();
            mCanvas.drawing = false;
        }

        private void chk_input_CheckedChanged(object sender, EventArgs e)
        {
            if (chk_input.Checked)
            {
                txt_x.Enabled = true;
                txt_y.Enabled = true;
            }
            else
            {
                txt_x.Enabled = false;
                txt_y.Enabled = false;
            }
        }

        private void btn_clear_Click(object sender, EventArgs e)
        {
            zoomMat = oriMat.Clone();
            rectPath = new GraphicsPath();
            rectPath.AddRectangle(mCanvas.mWorkarea.totalPath.GetBounds());

            mCanvas.dst = mCanvas.restore.Clone();
            mCanvas.matList.Add(mCanvas.dst.Clone());
            mCanvas.PictureBox1.ImageIpl = mCanvas.dst;
        }

        private void cmb_unit_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (MainMenu.mode != MainMode.ZOOM) return;
            SetSizeUnit();
        }

        private void SetSizeUnit()
        {
            //Todo TextBox 실시간으로 갱신하기
            // 0 = %, 1 = cm, 2 = inch, 3 = dot
            switch (cmb_unit.SelectedIndex)
            {
                case 0:
                    lbl_per_x.Text = "%";
                    lbl_per_y.Text = "%";
                    txt_x.Text = PerSwapPixel(oriMat.Width, zoomMat.Width).ToString();
                    txt_y.Text = PerSwapPixel(oriMat.Height, zoomMat.Height).ToString();
                    break;
                case 1:
                    lbl_per_x.Text = "cm";
                    lbl_per_y.Text = "cm";
                    txt_x.Text = CmSwapPixel(zoomMat.Width).ToString();
                    txt_y.Text = CmSwapPixel(zoomMat.Height).ToString();
                    break;
                case 2:
                    lbl_per_x.Text = "inch";
                    lbl_per_y.Text = "inch";
                    txt_x.Text = InchSwapPixel(zoomMat.Width).ToString();
                    txt_y.Text = InchSwapPixel(zoomMat.Height).ToString();
                    break;
                case 3:
                    lbl_per_x.Text = "dot";
                    lbl_per_y.Text = "dot";
                    //txt_x.Text = ;
                    //txt_y.Text = ;
                    break;
            }
        }

        public void InitZoomData()
        {
            RectangleF rect = mCanvas.mWorkarea.totalPath.GetBounds();
            pullPath = new GraphicsPath();
            pullPts = new List<PointF>();
            rectPath = new GraphicsPath();

            pullPath.AddRectangle(rect);
            pullPts = pullPath.PathPoints.ToList();
            rectPath.AddRectangle(rect);
            oriMat = mCanvas.restore[new Rect((int)rect.X, (int)rect.Y, (int)rect.Width, (int)rect.Height)];
            zoomMat = oriMat.Clone();
            restore = mCanvas.restore.Clone();
        }

        public void SetZoomMat(Mat ori, Mat zoom, cvSize newSize, int x, int y)
        {
            Cv2.Resize(ori, zoom, new cvSize(newSize.Width, newSize.Height), 0, 0, InterpolationFlags.Nearest);
            Rectangle rect = Rectangle.Round(preRectPath.GetBounds());
            restore[new Rect(rect.X, rect.Y, rect.Width, rect.Height)].SetTo(new Scalar(Color.FromArgb(-1).B, Color.FromArgb(-1).G, Color.FromArgb(-1).R, Color.FromArgb(-1).A));
            roi = new Mat(restore, new Rect(x, y, newSize.Width, newSize.Height));
            zoomMat.CopyTo(roi);
            mCanvas.PictureBox1.ImageIpl = restore;
            mCanvas.matList.Add(restore.Clone());

            preRectPath = (GraphicsPath)rectPath.Clone();
            //Todo 변하기 전 영역에서 변한 영역을 뺀 부분 배경색 처리 해야함
        }

        private void btn_pull_run_Click(object sender, EventArgs e)
        {
            //Perspective
            Rectangle pullRect = Rectangle.Round(pullPath.GetBounds());
            List<Point2f> srcPts = new List<Point2f>()
            {
                new Point2f(0.0f, 0.0f),
                new Point2f(0.0f, oriMat.Height),
                new Point2f(oriMat.Width, oriMat.Height),
                new Point2f(oriMat.Width, 0.0f)
            };
            List<Point2f> dstPts = new List<Point2f>()
            {
                new Point2f(pullPts[0].X - pullRect.X, pullPts[0].Y - pullRect.Y),
                new Point2f(pullPts[3].X - pullRect.X, pullPts[3].Y - pullRect.Y),
                new Point2f(pullPts[2].X - pullRect.X, pullPts[2].Y - pullRect.Y),
                new Point2f(pullPts[1].X - pullRect.X, pullPts[1].Y - pullRect.Y)
            };
            Mat pers = Cv2.GetPerspectiveTransform(srcPts, dstPts);

            Cv2.WarpPerspective(oriMat, zoomMat, pers, new cvSize(pullRect.Width, pullRect.Height), InterpolationFlags.Nearest);

            Scalar scalar = new Scalar(0, 0, 0);
            Cv2.CvtColor(zoomMat, zMask, ColorConversionCodes.BGR2HSV);
            Cv2.InRange(zoomMat, scalar, scalar, zMask);
            Cv2.BitwiseNot(zMask, zMask);

            roi = new Mat(restore, new Rect(pullRect.X, pullRect.Y, pullRect.Width, pullRect.Height));
            zoomMat.CopyTo(roi, zMask);

            mCanvas.dst = restore.Clone();
            mCanvas.restore = restore.Clone();
            mCanvas.matList.Add(mCanvas.dst.Clone());
            mCanvas.PictureBox1.ImageIpl = mCanvas.dst;

            ClearZoomData();
            this.Hide();
            MainMenu.mode = MainMode.DRAW;
        }

        private void Reproportion_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = true;
            ClearZoomData();
            this.Hide();
        }

        public float PerSwapPixel(float ori, float zoom)
        {
            return zoom / ori * 100f;
        }
        public float CmSwapPixel(float zoom)
        {
            return (float)Math.Round(zoom * 0.026458333, 2);
        }

        private void btn_rect_run_Click(object sender, EventArgs e)
        {
            mCanvas.dst = restore.Clone();
            mCanvas.restore = restore.Clone();
            mCanvas.matList.Add(mCanvas.dst.Clone());
            mCanvas.PictureBox1.ImageIpl = mCanvas.dst;

            ClearZoomData();
            this.Hide();
            MainMenu.mode = MainMode.DRAW;
        }

        private void txt_xy_KeyDown(object sender, KeyEventArgs e)
        {
            TextBox textBox = (TextBox)sender;
            if (e.KeyCode == Keys.Enter)
            {
                if (!float.TryParse(txt_x.Text, out float x) || !float.TryParse(txt_y.Text, out float y))
                {
                    MessageBox.Show("Please Input Number!");
                    return;
                }
                int newW, newH;
                float ratio = oriMat.Height / (float)oriMat.Width;
                switch (cmb_unit.SelectedIndex)
                {
                    case 0:
                        if (isRatio)
                        {

                            if (textBox.Name == "txt_x")
                            {
                                newW = (int)(x * oriMat.Width / 100f);
                                newH = (int)(x * oriMat.Height / 100f);
                                txt_y.Text = textBox.Text;
                            }
                            else
                            {
                                newH = (int)(y * oriMat.Height / 100f);
                                newW = (int)(y * oriMat.Width / 100f);
                                txt_x.Text = textBox.Text;
                            }
                        }
                        else
                        {
                            newW = (int)(x * oriMat.Width / 100f);
                            newH = (int)(y * oriMat.Height / 100f);
                        }
                        SetZoomMat(oriMat, zoomMat, new cvSize(newW, newH), (int)rectPath.GetBounds().X, (int)rectPath.GetBounds().Y);
                        break;
                    case 1:
                        if (isRatio)
                        {
                            if (textBox.Name == "txt_x")
                            {
                                newW = (int)(x / 0.026458333);
                                newH = (int)(newW * ratio);
                                txt_y.Text = (x * ratio).ToString();
                            }
                            else
                            {
                                newH = (int)(y / 0.026458333);
                                newW = (int)(newH / ratio);
                                txt_x.Text = (y / ratio).ToString();
                            }
                        }
                        else
                        {
                            newW = (int)(x / 0.026458333);
                            newH = (int)(y / 0.026458333);
                        }
                        SetZoomMat(oriMat, zoomMat, new cvSize(newW, newH), (int)rectPath.GetBounds().X, (int)rectPath.GetBounds().Y);
                        break;
                    case 2:
                        if (isRatio)
                        {
                            if (textBox.Name == "txt_x")
                            {
                                newW = (int)(x / 0.010416666);
                                newH = (int)(newW * ratio);
                                txt_y.Text = (x * ratio).ToString();
                            }
                            else
                            {
                                newH = (int)(y / 0.010416666);
                                newW = (int)(newH / ratio);
                                txt_x.Text = (y / ratio).ToString();
                            }
                        }
                        else
                        {
                            newW = (int)(x / 0.010416666);
                            newH = (int)(y / 0.010416666);
                        }
                        SetZoomMat(oriMat, zoomMat, new cvSize(newW, newH), (int)rectPath.GetBounds().X, (int)rectPath.GetBounds().Y);
                        break;
                    case 3:
                        //Todo Dot 단위 변환
                        break;
                }
            }
        }

        private void btn_link_Click(object sender, EventArgs e)
        {
            if (isRatio) isRatio = false;
            else
            {
                isRatio = true;
                zoomMat = oriMat.Clone();
                SetSizeUnit();
            }
        }

        public void ClearZoomData()
        {
            mZoomMode = WZoom.RECT;
            pullPath = null;
            pullPts = null;
            rectPath = null;
            isRatio = false;
        }
        public float InchSwapPixel(float zoom)
        {
            return (float)Math.Round(zoom * 0.010416666, 2);
        }
    }
}
