﻿using OpenCvSharp;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Windows.Forms;
using cvPoint = OpenCvSharp.Point;
using cvRect = OpenCvSharp.Rect;
using cvSize = OpenCvSharp.Size;
using Point = System.Drawing.Point;

namespace TexMaster
{
    public partial class Stipple : Form
    {
        public enum StippleType { LINE, DOT, SQUARE, PATTERN };

        StippleType type = StippleType.LINE;
        Canvas mCanvas;

        public cvPoint start, end;
        public Color backColor;

        public Mat colorMask;
        public Mat cropMat;
        public Mat cropMask;
        public List<int> applyColor;
        public List<int> lastList;
        public bool isArea;
        public bool isDelete;

        MultiColor multi;

        public Stipple(Canvas canvas)
        {
            InitializeComponent();
            mCanvas = canvas;
            mCanvas.mStipple = this;

            isArea = true;
            isDelete = false;

            applyColor = new List<int>();
            lastList = new List<int>();

            multi = new MultiColor();
        }

        public void stipple_MouseDown(MouseEventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.MouseDown(mCanvas.dst, mCanvas.mWorkarea.totalPath, pnl_color, e);
                mCanvas.drawing = true;
            }
            else
            {
                if (cmb_angle.SelectedIndex < 0)
                {
                    mCanvas.direction = new GraphicsPath();
                    mCanvas.drawing = true;
                }
            }
            start = new cvPoint(e.X, e.Y);
            backColor = Color.FromArgb(mCanvas.dst.At<int>(e.Y, e.X));
        }

        public void stipple_MouseMove(MouseEventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.MouseMove(e);
                mCanvas.PictureBox1.Refresh();
            }
            else
            {
                Point startPt = new Point(start.X, start.Y);
                Point endPt = new Point(end.X, end.Y);
                mCanvas.direction = new GraphicsPath();
                if ((System.Windows.Input.Keyboard.GetKeyStates(System.Windows.Input.Key.LeftShift) & System.Windows.Input.KeyStates.Down) > 0)
                {
                    //# 수평선
                    GraphicsPath line = new GraphicsPath();
                    double angle = 360 - mCanvas.GetAngle(start, end);
                    if ((angle >= 45 && angle < 135) || (angle >= 225 && angle < 315))
                    {
                        line.AddLine(startPt, new Point(startPt.X, endPt.Y));
                    }
                    else
                    {
                        line.AddLine(startPt, new Point(endPt.X, startPt.Y));
                    }
                    mCanvas.direction = line;
                }
                else
                {
                    mCanvas.direction = new GraphicsPath();
                    mCanvas.direction.AddLine(startPt, endPt);
                }

                mCanvas.PictureBox1.Refresh();
                end = new cvPoint(e.X, e.Y);
            }
        }

        public void stipple_MouseUp(MouseEventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.MouseUp(mCanvas.dst, pnl_color, e);
                mCanvas.drawing = false;
                mCanvas.PictureBox1.Refresh();
            }
            else
            {
                lastList = applyColor;
                float angle = 0;
                if (cmb_angle.SelectedIndex < 0)
                {
                    if (mCanvas.direction != null)
                    {
                        angle = (float)mCanvas.GetAngle(start, end);
                        mCanvas.direction = null;
                        mCanvas.drawing = false;
                        mCanvas.PictureBox1.Refresh();
                    }
                }
                else
                {
                    angle = Convert.ToSingle(cmb_angle.Text);
                    cmb_angle.SelectedIndex = -1;
                    cmb_angle.Text = String.Format("{0:0.0}", angle);
                    angle = Convert.ToSingle(cmb_angle.Text);
                }


                Mat mask = new Mat();
                if (applyColor.Count > 0)
                {
                    for (int i = 0; i < applyColor.Count; i++)
                    {
                        Mat hsv = new Mat();
                        Color c = Color.FromArgb((int)applyColor[i]);
                        Scalar selColor = new Scalar(c.B, c.G, c.R, c.A);
                        Mat ori = mCanvas.dst.Clone();
                        Cv2.CvtColor(ori, hsv, ColorConversionCodes.BGR2HSV);
                        Cv2.InRange(ori, selColor, selColor, hsv);
                        if (i == 0)
                        {
                            mask = hsv;
                        }
                        else
                        {
                            mask += hsv;
                        }
                    }
                }
                else
                {
                    mask = FloodFill(mCanvas.dst, start);
                }

                //# stipple fill 구현
                Mat fill = StippleFill(mask, angle);
                fill.CopyTo(mCanvas.dst, mask);
                mCanvas.PictureBox1.ImageIpl = mCanvas.dst;
            }
        }

        public Mat StippleFill(Mat mask, float angle)
        {
            //# 각도 계산을 위해 Mat 회전
            Mat mat1 = new Mat();
            Mat matrix = Cv2.GetRotationMatrix2D(new Point2f(mask.Width / 2, mask.Height / 2), angle, 1);
            Cv2.WarpAffine(mask, mat1, matrix, new cvSize(mask.Width, mask.Height));

            //# FloodFill mask boundary 구함
            int top = int.MaxValue, left = int.MaxValue;
            int bottom = 0, right = 0;

            for (int i = 0; i < mat1.Width; i++)
            {
                for (int j = 0; j < mat1.Height; j++)
                {
                    if (mat1.At<int>(j, i) != 0)
                    {
                        if (top > j)
                        {
                            top = j;
                        }
                        if (bottom < j)
                        {
                            bottom = j;
                        }
                    }
                }
            }

            for (int i = 0; i < mat1.Height; i++)
            {
                for (int j = 0; j < mat1.Width; j++)
                {
                    if (mat1.At<int>(i, j) != 0)
                    {
                        if (left > j)
                        {
                            left = j;
                        }
                        if (right < j)
                        {
                            right = j;
                        }
                    }
                }
            }

            Console.WriteLine("{0}, {1}, {2}, {3}", left, top, right - left, bottom - top);

            //# boundary 크기
            Mat mat2 = new Mat();

            if ((right - left) * Math.Sqrt(2) > mCanvas.dst.Width || (bottom - top) * Math.Sqrt(2) > mCanvas.dst.Height)
            {
                Console.WriteLine("범위가 큽니다.");
                int tWidth = (int)((right - left) * Math.Sqrt(2));
                int tHeight = (int)((bottom - top) * Math.Sqrt(2));
                Mat totalMat = GetStipple(tWidth, tHeight);

                matrix = Cv2.GetRotationMatrix2D(new Point2f(tWidth / 2, tHeight / 2), (-angle), 1);
                Cv2.WarpAffine(totalMat, mat2, matrix, new cvSize(totalMat.Width, totalMat.Height), InterpolationFlags.Area);

                mat2 = mat2[new Rect(tWidth / 2 - mCanvas.dst.Width / 2, tHeight / 2 - mCanvas.dst.Height / 2, mask.Width, mask.Height)];
                //Cv2.ImShow("total", mat2);
            }
            else
            {
                //# boundary에 stipple 효과 넣기
                Mat tempMat = new Mat(new cvSize(mat1.Width, mat1.Height), MatType.CV_8UC4, 0);
                tempMat[new Rect(left, top, right - left, bottom - top)] = GetStipple(right - left, bottom - top);
                //# mat 다시 회전 / mask 만큼 자르기
                matrix = Cv2.GetRotationMatrix2D(new Point2f(mask.Width / 2, mask.Height / 2), (-angle), 1);
                Cv2.WarpAffine(tempMat, mat2, matrix, new cvSize(mask.Width, mask.Height), InterpolationFlags.Area);
            }


            Mat cropMat = new Mat();
            mat2.CopyTo(cropMat, mask);
            return cropMat;
        }

        public Mat GetStipple(int width, int height)
        {
            Mat stipple = new Mat(new cvSize(width, height), MatType.CV_8UC4, 0);
            //# 배경색으로 채우기
            Scalar back = new Scalar(backColor.B, backColor.G, backColor.R, 255);
            stipple.SetTo(back);
            Color c = pnl_colorchip.BackColor;
            Scalar s = new Scalar(c.B, c.G, c.R, c.A);
            cvPoint sPt, ePt, cPt;

            if (type == StippleType.PATTERN)
            {
                cropMat.SetTo(back, cropMask);
                //Cv2.ImShow("stipple", cropMat);
            }

            for (int i = 0; i < width / nud_horz.Value; i++)
            {
                int penWidth = (int)(nud_max.Value * nud_horz.Value * i / width);
                switch (type)
                {
                    case StippleType.LINE:
                        sPt = new cvPoint((int)nud_horz.Value * i, 0);
                        ePt = new cvPoint((int)nud_horz.Value * i, height);
                        Cv2.Line(stipple, sPt, ePt, s, (int)(nud_max.Value - penWidth));
                        break;
                    case StippleType.DOT:
                        for (int j = 0; j < height / nud_vert.Value; j++)
                        {
                            cPt = new cvPoint((int)nud_horz.Value * i, (int)nud_vert.Value * j);
                            Cv2.Circle(stipple, cPt, penWidth / 2, s, -1, LineTypes.Link8);
                        }
                        break;
                    case StippleType.SQUARE:
                        for (int j = 0; j < height / nud_vert.Value; j++)
                        {
                            cPt = new cvPoint((int)nud_horz.Value * i, (int)nud_vert.Value * j);
                            sPt = new cvPoint(cPt.X - penWidth / 2, cPt.Y - penWidth / 2);
                            ePt = new cvPoint(cPt.X + penWidth / 2, cPt.Y + penWidth / 2);
                            Cv2.Rectangle(stipple, mCanvas.CreateCvSquare(sPt, ePt), s, -1, LineTypes.Link8);
                        }
                        break;
                    case StippleType.PATTERN:
                        for (int j = 0; j < height / nud_vert.Value; j++)
                        {
                            float wRatio = (float)penWidth / (float)cropMat.Width;
                            float hRatio = (float)penWidth / (float)cropMat.Height;
                            Mat resizeMat = new Mat();
                            Cv2.Resize(cropMat, resizeMat, new cvSize((int)nud_max.Value - cropMat.Width * wRatio, (int)nud_max.Value - cropMat.Height * hRatio), 0, 0, InterpolationFlags.Area);
                            cPt = new cvPoint((int)nud_horz.Value * i, (int)nud_vert.Value * j);
                            sPt = new cvPoint(cPt.X - resizeMat.Width / 2, cPt.Y - resizeMat.Height / 2);
                            ePt = new cvPoint(cPt.X + resizeMat.Width / 2, cPt.Y + resizeMat.Height / 2);
                            if (sPt.X > resizeMat.Width && sPt.X < stipple.Width - resizeMat.Width && sPt.Y > resizeMat.Height && sPt.Y < stipple.Height - resizeMat.Height)
                                stipple[new cvRect(sPt, resizeMat.Size())] = resizeMat;
                        }
                        break;

                }
            }

            //Cv2.ImShow("stipple", stipple);

            return stipple;
        }

        public void stipple_Paint(PaintEventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.Paint(e);
            }
        }

        public Mat FloodFill(Mat src, cvPoint seedPt)
        {
            Mat hsv = new Mat();

            //# 배경색 제거
            Color c = backColor;
            Scalar s = new Scalar(c.B, c.G, c.R, 255);
            Cv2.InRange(src, s, s, hsv);

            //# 채우기 전 이미지
            Mat ori = hsv.Clone();
            //# 채운 후 이미지
            Cv2.FloodFill(hsv, seedPt, 0);
            Mat diff = new Mat();
            //# 차이비교 - 채워질 영역 mask 반환
            Cv2.BitwiseXor(hsv, ori, diff);

            return diff;
        }

        private void Stipple_Load(object sender, EventArgs e)
        {
            chk_select.Checked = false;
            cmb_angle.SelectedIndex = -1;
        }

        private void chk_select_CheckedChanged(object sender, EventArgs e)
        {
            if (chk_select.Checked)
            {
                btn_new.Enabled = true;
                btn_all.Enabled = true;
                btn_area.Enabled = true;
                btn_insert.Enabled = true;
                btn_pre.Enabled = true;
            }
            else
            {
                btn_new.Enabled = false;
                btn_all.Enabled = false;
                btn_area.Enabled = false;
                btn_insert.Enabled = false;
                btn_pre.Enabled = false;
            }
        }

        private void btn_line_Click(object sender, EventArgs e)
        {
            nud_vert.Enabled = false;
            type = StippleType.LINE;
        }

        private void btn_circle_Click(object sender, EventArgs e)
        {
            nud_vert.Enabled = true;
            type = StippleType.DOT;
        }

        private void btn_rect_Click(object sender, EventArgs e)
        {
            nud_vert.Enabled = true;
            type = StippleType.SQUARE;
        }

        private void btn_pattern_Click(object sender, EventArgs e)
        {
            if (mCanvas.mWorkarea.totalPath != null)
            {
                nud_vert.Enabled = false;
                type = StippleType.PATTERN;

                Rectangle r = Rectangle.Round(mCanvas.mWorkarea.totalPath.GetBounds());
                cvRect rect = new cvRect((int)r.X, (int)r.Y, (int)r.Width, (int)r.Height);

                Mat maskedMat = new Mat();
                mCanvas.dst.CopyTo(maskedMat, mCanvas.mWorkarea.holeMat);
                maskedMat[rect].CopyTo(maskedMat);

                Color b = mCanvas.gBackColor;
                Scalar s = new Scalar(b.B, b.G, b.R, b.A);
                cropMask = new Mat();
                Cv2.CvtColor(maskedMat, cropMask, ColorConversionCodes.BGR2HSV);
                Cv2.InRange(maskedMat, s, s, cropMask);

                Mat temp = new Mat();
                Cv2.BitwiseNot(cropMask, temp);
                cropMat = new Mat();
                maskedMat.CopyTo(cropMat, temp);
            }
            else
            {
                MessageBox.Show("작업 구역을 잡으세요", "Warning", MessageBoxButtons.OK);
                nud_vert.Enabled = false;
                type = StippleType.LINE;
                btn_line.Focus();
            }
        }

        private void Stipple_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = true;
            this.Hide();
        }

        private void btn_new_Click(object sender, EventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.SelectClear(pnl_color);
            }
        }

        private void btn_all_Click(object sender, EventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.SelectAll(pnl_color, mCanvas.dst, mCanvas.mWorkarea.totalPath);
            }
        }

        private void btn_area_Click(object sender, EventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.SetSelectState(btn_area);
            }
        }

        private void btn_insert_Click(object sender, EventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.SetSelectState(btn_insert);
            }
        }

        private void btn_pre_Click(object sender, EventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.SelectRestore(pnl_color);
            }
        }
    }
}
