﻿using OpenCvSharp;
using OpenCvSharp.Extensions;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.IO;
using System.Runtime.Serialization.Formatters.Binary;
using System.Windows.Forms;
using cvPoint = OpenCvSharp.Point;

namespace TexMaster
{
    public partial class Stripe : Form
    {
        private TActive mActive;

        private List<PatternInfo> warfInfoList = new List<PatternInfo>();
        private List<PatternInfo> weftInfoList = new List<PatternInfo>();
        private PatternInfo warfInfo;
        private PatternInfo weftInfo;
        private int ratio = 100;
        private bool bDeployment = false;

        public Bitmap OutImg;
        public bool bWhole;
        public Mat stripeMat;

        [Serializable()]
        public class StripInfo
        {
            public List<PatternInfo> StripWarp = new List<PatternInfo>();
            public List<PatternInfo> StripWeft = new List<PatternInfo>();
        }

        [Serializable()]
        public class PatternInfo
        {
            public Color Color;
            public String Method;
            public float Length;
            public int Repeat;

            public PatternInfo()
            {
                Method = "Normal";
                Length = 50;
                Repeat = 1;
            }
        }

        public enum TActive
        {
            INSERT,
            DELETE
        }

        Canvas mCanvas;
        public Stripe(Canvas canvas)
        {
            InitializeComponent();

            bWhole = false;
            mCanvas = canvas;
            mCanvas.mStripe = this;
        }

        private void Stripe_Load(object sender, EventArgs e)
        {
            /* combo box initialize */
            cmb_method.SelectedIndex = 0;
            cmb_angle.SelectedIndex = 0;
            cmb_size.SelectedIndex = 0;
            txt_overlap.Text = trb_overlap.Value.ToString();
            pic_stripe.BackgroundImageLayout = ImageLayout.None;

            /* table warp */
            dgv_warf.ColumnHeadersVisible = true;
            DataGridViewCellStyle columnHeaderStyle = new DataGridViewCellStyle();

            columnHeaderStyle.BackColor = Color.Beige;
            columnHeaderStyle.Font = new Font("굴림", 9);
            dgv_warf.ColumnHeadersDefaultCellStyle = columnHeaderStyle;

            dgv_warf.Columns.Add("col1", "Color");
            dgv_warf.Columns.Add("col2", "Method");
            dgv_warf.Columns.Add("col3", "Length");
            dgv_warf.Columns.Add("col4", "Repeat");

            dgv_warf.Columns[0].ReadOnly = true;

            dgv_warf.AutoResizeColumns();
            dgv_warf.ScrollBars = ScrollBars.Both;

            /* table weft */
            dgv_weft.ColumnHeadersVisible = true;
            dgv_weft.ColumnHeadersDefaultCellStyle = columnHeaderStyle;

            dgv_weft.Columns.Add("col1", "Color");
            dgv_weft.Columns.Add("col2", "Method");
            dgv_weft.Columns.Add("col3", "Length");
            dgv_weft.Columns.Add("col4", "Repeat");

            dgv_weft.Columns[0].ReadOnly = true;

            dgv_weft.AutoResizeColumns();
            dgv_weft.ScrollBars = ScrollBars.Both;

            mActive = TActive.INSERT;
        }

        private void ImageSetPixel()
        {
            /* set image size */
            int totalWidth = 0;
            int totalHeight = 0;

            if (warfInfoList.Count == 0)
            {
                txt_thick.Enabled = true;
                totalWidth = Convert.ToInt32(txt_thick.Text);
            }
            else
            {
                txt_thick.Enabled = false;
                for (int i = 0; i < warfInfoList.Count; i++)
                {
                    totalWidth += Convert.ToInt32(warfInfoList[i].Length);
                }
            }

            if (weftInfoList.Count == 0)
            {
                txt_thick.Enabled = true;
                totalHeight = Convert.ToInt32(txt_thick.Text);
            }
            else
            {
                txt_thick.Enabled = false;
                for (int i = 0; i < weftInfoList.Count; i++)
                {
                    totalHeight += Convert.ToInt32(weftInfoList[i].Length);
                }
            }

            /* To indicate before calculation in order to display the actual */
            txt_warf.Text = totalWidth.ToString();
            txt_weft.Text = totalHeight.ToString();

            totalWidth = Convert.ToInt32((double)totalWidth * ratio / 100);
            totalHeight = Convert.ToInt32((double)totalHeight * ratio / 100);

            /* make image */
            Bitmap warfImage = new Bitmap(totalWidth, totalHeight);
            Bitmap weftImage = new Bitmap(totalWidth, totalHeight);
            Bitmap patternImage = new Bitmap(totalWidth, totalHeight);

            int startX = 0;
            int startY = 0;

            for (int i = 0; i < warfInfoList.Count; i++)
            {
                for (int x = startX; x < totalWidth; x++)
                {
                    for (int y = 0; y < totalHeight; y++)
                    {
                        warfImage.SetPixel(x, y, warfInfoList[i].Color);
                    }
                }
                startX += Convert.ToInt32(warfInfoList[i].Length * ratio / 100);
            }

            for (int i = 0; i < weftInfoList.Count; i++)
            {
                for (int x = 0; x < totalWidth; x++)
                {
                    for (int y = startY; y < totalHeight; y++)
                    {
                        weftImage.SetPixel(x, y, weftInfoList[i].Color);
                    }
                }
                startY += Convert.ToInt32(weftInfoList[i].Length * ratio / 100);
            }

            if (cmb_method.SelectedIndex == 0)              //# mesh
            {
                for (int i = 0; i < totalWidth; i++)
                {
                    for (int j = 0; j < totalHeight; j++)
                    {
                        if ((i + j) % 2 == 0)
                        {
                            patternImage.SetPixel(i, j, warfImage.GetPixel(i, j));
                        }
                        else
                        {
                            patternImage.SetPixel(i, j, weftImage.GetPixel(i, j));
                        }
                    }
                }
            }
            else if (cmb_method.SelectedIndex == 1)         //# mix
            {
                Color warfColor;
                Color weftColor;
                for (int i = 0; i < totalWidth; i++)
                {
                    for (int j = 0; j < totalHeight; j++)
                    {
                        int ratio = Convert.ToInt32(txt_overlap.Text);
                        warfColor = warfImage.GetPixel(i, j);
                        weftColor = weftImage.GetPixel(i, j);
                        int a = (int)(Convert.ToInt32(warfColor.A) * (100.0 - ratio) / 100.0) + (int)(Convert.ToInt32(weftColor.A) * ratio / 100.0);
                        int r = (int)(Convert.ToInt32(warfColor.R) * (100.0 - ratio) / 100.0) + (int)(Convert.ToInt32(weftColor.R) * ratio / 100.0);
                        int g = (int)(Convert.ToInt32(warfColor.G) * (100.0 - ratio) / 100.0) + (int)(Convert.ToInt32(weftColor.G) * ratio / 100.0);
                        int b = (int)(Convert.ToInt32(warfColor.B) * (100.0 - ratio) / 100.0) + (int)(Convert.ToInt32(weftColor.B) * ratio / 100.0);
                        patternImage.SetPixel(i, j, Color.FromArgb(a, r, g, b));
                    }
                }
            }

            if (warfInfoList.Count == 0)
            {
                pic_stripe.BackgroundImage = weftImage;
                OutImg = weftImage;
            }
            else if (weftInfoList.Count == 0)
            {
                pic_stripe.BackgroundImage = warfImage;
                OutImg = warfImage;
            }
            else
            {
                pic_stripe.BackgroundImage = patternImage;
                OutImg = patternImage;
            }
        }

        private void dgv_warp_Update()
        {
            dgv_warf.Rows.Clear();

            String length = "";
            for (int i = 0; i < warfInfoList.Count; i++)
            {
                if (button1.Text == "Dot")
                {
                    length = String.Format("{0, 0:N2}",warfInfoList[i].Length);
                }
                else if (button1.Text == "Cm")
                {
                    length = String.Format("{0, 0:N2}", warfInfoList[i].Length / 160 * 2.54F);
                }
                else if (button1.Text == "Inch")
                {
                    length = String.Format("{0, 0:N2}", warfInfoList[i].Length / 160);
                }

                dgv_warf.Rows.Add("", warfInfoList[i].Method, length, warfInfoList[i].Repeat);
                dgv_warf.Rows[i].Cells[0].Style.BackColor = warfInfoList[i].Color;
                dgv_warf.Rows[i].HeaderCell.Value = (i + 1).ToString();
                dgv_warf.CurrentCell = dgv_warf.Rows[i + 1].Cells[0];
            }
        }

        private void dgv_weft_Update()
        {
            dgv_weft.Rows.Clear();

            for (int i = 0; i < weftInfoList.Count; i++)
            {
                dgv_weft.Rows.Add("", weftInfoList[i].Method, weftInfoList[i].Length, weftInfoList[i].Repeat);
                dgv_weft.Rows[i].Cells[0].Style.BackColor = weftInfoList[i].Color;
                dgv_weft.Rows[i].HeaderCell.Value = (i + 1).ToString();
                dgv_weft.CurrentCell = dgv_weft.Rows[i + 1].Cells[0];
            }

        }


        private void dgv_warp_CellClick(object sender, DataGridViewCellEventArgs e)
        {
            switch (mActive)
            {
                case TActive.INSERT:
                    if (e.ColumnIndex == 0)
                    {
                        if (dgv_warf.Rows.Count == e.RowIndex + 1)
                        {
                            warfInfo = new PatternInfo();
                            warfInfo.Color = mCanvas.mPalette.color;
                            warfInfoList.Add(warfInfo);
                        }
                        else
                        {
                            warfInfoList[e.RowIndex].Color = mCanvas.mPalette.color;
                        }
                        dgv_warp_Update();
                    }
                    break;

                case TActive.DELETE:
                    if (warfInfoList.Count == 0) return;
                    warfInfoList.RemoveAt(e.RowIndex);
                    dgv_warp_Update();
                    break;

            }
        }

        private void dgv_weft_CellClick(object sender, DataGridViewCellEventArgs e)
        {
            switch (mActive)
            {
                case TActive.INSERT:
                    if (e.ColumnIndex == 0)
                    {
                        /* add row */
                        if (dgv_weft.Rows.Count == e.RowIndex + 1)
                        {
                            weftInfo = new PatternInfo();
                            weftInfo.Color = mCanvas.mPalette.color;
                            weftInfoList.Add(weftInfo);
                        }
                        else
                        {
                            weftInfoList[e.RowIndex].Color = mCanvas.mPalette.color;
                        }

                        dgv_weft_Update();

                    }
                    break;

                case TActive.DELETE:
                    if (weftInfoList.Count == 0) return;
                    weftInfoList.RemoveAt(e.RowIndex);
                    dgv_weft_Update();
                    break;
            }
        }

        private void btn_add_Click(object sender, EventArgs e)
        {
            mActive = TActive.INSERT;
        }

        private void btn_delete_Click(object sender, EventArgs e)
        {
            mActive = TActive.DELETE;
        }

        private void btn_copy_Click(object sender, EventArgs e)
        {
            dgv_weft.Rows.Clear();
            weftInfoList.Clear();
            weftInfoList = warfInfoList;
            dgv_weft_Update();
        }

        private void btn_del_warp_Click(object sender, EventArgs e)
        {
            dgv_warf.Rows.Clear();
            warfInfoList.Clear();
            ImageSetPixel();
        }

        private void btn_del_weft_Click(object sender, EventArgs e)
        {
            dgv_weft.Rows.Clear();
            weftInfoList.Clear();
            ImageSetPixel();
        }

        private void btn_del_all_Click(object sender, EventArgs e)
        {
            dgv_warf.Rows.Clear();
            dgv_weft.Rows.Clear();
            warfInfoList.Clear();
            weftInfoList.Clear();
            ImageSetPixel();
        }

        private void dgv_warp_CellValueChanged(object sender, DataGridViewCellEventArgs e)
        {
            if (dgv_warf.CurrentCell.ColumnIndex == 1)
            {
                warfInfoList[dgv_warf.CurrentCell.RowIndex].Method = Convert.ToString(dgv_warf.CurrentCell.Value);
            }
            else if (dgv_warf.CurrentCell.ColumnIndex == 2)
            {
                warfInfoList[dgv_warf.CurrentCell.RowIndex].Length = (float)(dgv_warf.CurrentCell.Value);
            }
            else if (dgv_warf.CurrentCell.ColumnIndex == 3)
            {
                warfInfoList[dgv_warf.CurrentCell.RowIndex].Repeat = Convert.ToInt32(dgv_warf.CurrentCell.Value);
            }
            ImageSetPixel();
        }

        private void dgv_weft_CellValueChanged(object sender, DataGridViewCellEventArgs e)
        {
            if (dgv_weft.CurrentCell.ColumnIndex == 1)
            {
                weftInfoList[dgv_weft.CurrentCell.RowIndex].Method = Convert.ToString(dgv_weft.CurrentCell.Value);
            }
            else if (dgv_weft.CurrentCell.ColumnIndex == 2)
            {
                weftInfoList[dgv_weft.CurrentCell.RowIndex].Length = (float)(dgv_warf.CurrentCell.Value);
            }
            else if (dgv_weft.CurrentCell.ColumnIndex == 3)
            {
                weftInfoList[dgv_weft.CurrentCell.RowIndex].Repeat = Convert.ToInt32(dgv_weft.CurrentCell.Value);
            }
            ImageSetPixel();
        }

        private void cmb_method_SelectedIndexChanged(object sender, EventArgs e)
        {
            ImageSetPixel();
            if (cmb_method.SelectedIndex == 0)
            {
                trb_overlap.Visible = false;
                txt_overlap.Visible = false;
            }
            else if (cmb_method.SelectedIndex == 1)
            {
                trb_overlap.Visible = true;
                txt_overlap.Visible = true;
            }
        }

        private void trb_overlap_Scroll(object sender, EventArgs e)
        {
            txt_overlap.Text = trb_overlap.Value.ToString();
            ImageSetPixel();
        }

        private void txt_thick_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                ImageSetPixel();
            }
        }

        private void btn_run_Click(object sender, EventArgs e)
        {
            stripeMat = BitmapConverter.ToMat(OutImg);
            mCanvas.drawing = true;
            this.Hide();
            //this.Close();
        }

        private void cmb_angle_SelectedIndexChanged(object sender, EventArgs e)
        {
            switch (cmb_angle.SelectedIndex)
            {
                case 0: //# 0
                    OutImg.RotateFlip(RotateFlipType.RotateNoneFlipNone);
                    break;
                case 1: //# 90
                    OutImg.RotateFlip(RotateFlipType.Rotate90FlipNone);
                    break;
                case 2: //# 180
                    OutImg.RotateFlip(RotateFlipType.Rotate180FlipNone);
                    break;
                case 3: //# 270
                    OutImg.RotateFlip(RotateFlipType.Rotate270FlipNone);
                    break;
                default:
                    break;
            }
        }

        private void cmb_size_SelectedIndexChanged(object sender, EventArgs e)
        {
            switch (cmb_size.SelectedIndex)
            {
                case 0: //# Partial
                    ratio = 100;
                    break;
                case 1: //# Total
                    int totalHeight = 0;
                    if (weftInfoList.Count == 0)
                    {
                        totalHeight = Convert.ToInt32(txt_thick.Text);
                    }
                    else
                    {
                        for (int i = 0; i < weftInfoList.Count; i++)
                        {
                            totalHeight += Convert.ToInt32(weftInfoList[i].Length);
                        }
                    }
                    ratio = Convert.ToInt32((double)pic_stripe.Height / (double)totalHeight * 100);
                    break;
                default:
                    break;
            }

            txt_ratio.Text = ratio.ToString() + " %";
            ImageSetPixel();
        }

        private void txt_overlap_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                ImageSetPixel();
            }
        }

        private void btn_spread_Click(object sender, EventArgs e)
        {
            if (bDeployment)
            {
                btn_spread.Text = "One Repeat";
                pic_stripe.BackgroundImageLayout = ImageLayout.Tile;
                bDeployment = false;
            }
            else
            {
                btn_spread.Text = "Deployment";
                pic_stripe.BackgroundImageLayout = ImageLayout.None;
                bDeployment = true;
            }
            ImageSetPixel();
        }

        private void btn_file_Click(object sender, EventArgs e)
        {
            StripInfo saveInfo = new StripInfo();

            OpenFileDialog stripOpen = new OpenFileDialog();
            stripOpen.Filter = "strip file|*.stp";
            stripOpen.ShowDialog();

            FileStream fs = new FileStream(stripOpen.FileName, FileMode.Open);
            BinaryFormatter formatter = new BinaryFormatter();

            warfInfoList.Clear();
            weftInfoList.Clear();

            try
            {
                saveInfo = (StripInfo)formatter.Deserialize(fs);
                warfInfoList = saveInfo.StripWarp;
                weftInfoList = saveInfo.StripWeft;
            }
            catch
            {
                Console.WriteLine("fail");
                throw;
            }
            finally
            {
                fs.Close();
            }

            dgv_warp_Update();
            dgv_weft_Update();
            ImageSetPixel();
        }

        private void btn_save_Click(object sender, EventArgs e)
        {
            StripInfo saveInfo = new StripInfo();
            saveInfo.StripWarp = warfInfoList;
            saveInfo.StripWeft = weftInfoList;

            SaveFileDialog stripSave = new SaveFileDialog();
            stripSave.Filter = "strip file|*.stp";
            stripSave.ShowDialog();

            FileStream fs = new FileStream(stripSave.FileName, FileMode.Create);

            BinaryFormatter formatter = new BinaryFormatter();
            try
            {
                formatter.Serialize(fs, saveInfo);
            }
            catch
            {
                Console.WriteLine("fail");
                throw;
            }
            finally
            {
                fs.Close();
            }
        }

        private void btn_one_repeat_Click(object sender, EventArgs e)
        {
            bWhole = false;
        }

        private void btn_whole_Click(object sender, EventArgs e)
        {
            bWhole = true;
        }

        float unit = 100.0F;

        private void panel1_Paint(object sender, PaintEventArgs e)
        {
            e.Graphics.DrawLine(new Pen(Color.Black), new PointF(0, 0), new PointF(panel1.Width, 0));
            e.Graphics.DrawLine(new Pen(Color.Black), new PointF(0, panel1.Height - 1), new PointF(panel1.Width, panel1.Height - 1));

            for (int i = 0; i < panel1.Width; i++)
            {
                e.Graphics.DrawLine(new Pen(Color.Black), new PointF(i * unit / 10, 22), new PointF(i * unit / 10, panel1.Height));
                e.Graphics.DrawLine(new Pen(Color.Black), new PointF(i * unit / 2, 15), new PointF(i * unit / 2, panel1.Height));
                e.Graphics.DrawLine(new Pen(Color.Black), new PointF(i * unit, 0), new PointF(i * unit, panel1.Height));
                e.Graphics.DrawString(String.Format("{0, 0:N2}", i), new Font("굴림", 8), new SolidBrush(Color.Black), new PointF(i * unit + 1, 3));
            }
        }

        private void panel2_Paint(object sender, PaintEventArgs e)
        {
            e.Graphics.DrawLine(new Pen(Color.Black), new PointF(0, 0), new PointF(0, panel2.Height));
            e.Graphics.DrawLine(new Pen(Color.Black), new PointF(panel2.Width - 1, 0), new PointF(panel2.Width - 1, panel2.Height));

            for (int i = 0; i < panel2.Height; i++)
            {
                e.Graphics.DrawLine(new Pen(Color.Black), new PointF(32, i * unit / 10), new PointF(panel1.Width, i * unit / 10));
                e.Graphics.DrawLine(new Pen(Color.Black), new PointF(25, i * unit / 2), new PointF(panel1.Width, i * unit / 2));
                e.Graphics.DrawLine(new Pen(Color.Black), new PointF(0, i * unit), new PointF(panel1.Width, i * unit));
                e.Graphics.DrawString(String.Format("{0, 0:N2}", i), new Font("굴림", 8), new SolidBrush(Color.Black), new PointF(1, i * unit + 3));
            }
        }

        private void button1_Click(object sender, EventArgs e)
        {
            if (button1.Text == "Dot")
            {
                button1.Text = "Inch";
                unit = 100.0F;
            }
            else if (button1.Text == "Inch")
            {
                button1.Text = "Cm";
                unit = 160 / 2.54F;
            }
            else if (button1.Text == "Cm")
            {
                button1.Text = "Dot";
                unit = 160;
            }

            dgv_warp_Update();
            dgv_weft.Update();
            panel1.Refresh();
            panel2.Refresh();
        }

        private void vsc_warf_Scroll(object sender, ScrollEventArgs e)
        {

        }

        private void hsc_weft_Scroll(object sender, ScrollEventArgs e)
        {

        }

        private void Stripe_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = true;
            this.Hide();
        }

        public void stripe_MouseMove(MouseEventArgs e)
        {
            mCanvas.dst = mCanvas.LastList(mCanvas.matList).Clone();
            cvPoint start = new cvPoint(e.X - stripeMat.Width / 2, e.Y - stripeMat.Height / 2);
            mCanvas.dst[new Rect(start.X, start.Y, stripeMat.Width, stripeMat.Height)] = stripeMat;
            mCanvas.PictureBox1.ImageIpl = mCanvas.dst;
        }
    }
}
