﻿using OpenCvSharp;
using OpenCvSharp.Extensions;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Windows.Forms;
using cvPoint = OpenCvSharp.Point;
using cvSize = OpenCvSharp.Size;
using Point = System.Drawing.Point;

namespace TexMaster
{
    public partial class TextDraw : Form
    {
        Canvas mCanvas;
        Point start, end;
        GraphicsPath gp;
        List<Point> curPoints;
        bool isInput;
        bool isTunnel;
        bool isMoving;

        int location;
        int index;
        Mat tunnelMat;

        List<PointF>[] restorePoints;
        List<PointF>[] tunnelPoints;

        Font font;
        Graphics gr;

        double angle;

        public TextDraw(Canvas canvas)
        {
            InitializeComponent();
            mCanvas = canvas;
            mCanvas.mText = this;
        }

        private void btn_end_Click(object sender, EventArgs e)
        {
            if (mCanvas.drawing && cmb_shape.SelectedIndex == 4)
            {
                curPoints.RemoveAt(curPoints.Count - 1);
                gp = new GraphicsPath();
                gp.AddCurve(curPoints.ToArray());
                mCanvas.PictureBox1.Refresh();

                DrawString();
                mCanvas.drawing = false;
                curPoints.Clear();
                curPoints = null;

                if (cmb_auto.SelectedIndex != 0)
                    gp = null;
            }
        }

        public void DrawTunnel()
        {
            tunnelPoints = new List<PointF>[2];
            tunnelPoints[0] = new List<PointF>();
            tunnelPoints[1] = new List<PointF>();

            restorePoints = new List<PointF>[2];
            restorePoints[0] = new List<PointF>();
            restorePoints[1] = new List<PointF>();

            float dx = (end.X - start.X) / ((int)nud_point.Value - 1);
            float dy = (end.Y - start.Y) / ((int)nud_point.Value - 1);

            float radian = (float)((90 + angle) * Math.PI / 180);

            float xOffset = (float)(font.Height * Math.Cos(radian));
            float yOffset = (float)(font.Height * -Math.Sin(radian));

            for (int i = 0; i < (int)nud_point.Value; i++)
            {
                if (i == (int)nud_point.Value - 1)
                {
                    tunnelPoints[0].Add(end);
                    tunnelPoints[1].Add(new PointF(end.X + xOffset, end.Y + yOffset));
                    restorePoints[0].Add(end);
                    restorePoints[1].Add(new PointF(end.X + xOffset, end.Y + yOffset));

                }
                else
                {
                    tunnelPoints[0].Add(new PointF(start.X + dx * i, start.Y + dy * i));
                    tunnelPoints[1].Add(new PointF(start.X + xOffset + dx * i, start.Y + yOffset + dy * i));
                    restorePoints[0].Add(new PointF(start.X + dx * i, start.Y + dy * i));
                    restorePoints[1].Add(new PointF(start.X + xOffset + dx * i, start.Y + yOffset + dy * i));
                }
            }
        }

        public void DrawString()
        {
            if (cmb_auto.SelectedIndex == 0)
            {
                txt_context.Enabled = true;
                txt_context.Focus();
            }

            Color c = mCanvas.gBackColor;
            Bitmap textImg = new Bitmap(mCanvas.dst.Width, mCanvas.dst.Height);
            gr = Graphics.FromImage(textImg);
            gr.Clear(c);

            float fontSize = (float)nud_size.Value;
            FontFamily family = FontFamily.Families[cmb_fontName.SelectedIndex];
            font = new Font(family, fontSize);

            //# 폰트 스타일
            FontStyle fontStyle = FontStyle.Regular;
            switch (cmb_style.SelectedIndex)
            {
                case 0:
                    fontStyle = FontStyle.Regular;
                    break;
                case 1:
                    fontStyle = FontStyle.Bold;
                    break;
                case 2:
                    fontStyle = FontStyle.Italic;
                    break;
                case 3:
                    fontStyle = FontStyle.Bold | FontStyle.Italic;
                    break;
            }

            if (cmb_auto.SelectedIndex == 2)    //# 자유 -> 폰트 크기 변경
            {
                float stringLength = MeasureDisplayStringWidth(gr, txt_context.Text, font);
                float pathLength = 0;
                if (cmb_shape.SelectedIndex == 4) //# 곡선
                {
                    GraphicsPath flatPath = (GraphicsPath)gp.Clone();
                    flatPath.Flatten();

                    for (int i = 0; i < flatPath.PointCount - 1; i++)
                    {
                        pathLength += (float)Math.Sqrt(Math.Pow(flatPath.PathPoints[i + 1].X - flatPath.PathPoints[i].X, 2) + Math.Pow(flatPath.PathPoints[i + 1].Y - flatPath.PathPoints[i].Y, 2));
                    }
                }
                else if (cmb_shape.SelectedIndex == 2) //# 원형
                {
                    pathLength = (float)(Math.Abs(end.Y - start.Y) * Math.PI);
                }
                else
                {
                    pathLength = (float)Math.Sqrt(Math.Pow(end.X - start.X, 2) + Math.Pow(end.Y - start.Y, 2));
                }

                float ratio = pathLength / stringLength;
                fontSize = (float)nud_size.Value * ratio;
            }

            font = new Font(family, fontSize, fontStyle);

            Color penColor = mCanvas.mPalette.color;
            DrawTextOnPath(gr, new SolidBrush(penColor), font, txt_context.Text, gp);

            Mat textMat = BitmapConverter.ToMat(textImg);
            //Cv2.ImShow("textMat", textMat);

            Mat mask = new Mat();
            Cv2.CvtColor(textMat, mask, ColorConversionCodes.BGR2HSV);
            Cv2.InRange(textMat, new Scalar(c.B, c.G, c.R, 255), new Scalar(c.B, c.G, c.R, 255), mask);

            Cv2.BitwiseNot(mask, mask);
            //Cv2.ImShow("mask", mask);

            mCanvas.dst = mCanvas.LastList(mCanvas.matList).Clone();

            tunnelMat = new Mat();
            textMat.CopyTo(mCanvas.dst, mask);
            textMat.CopyTo(tunnelMat, mask);
            mCanvas.PictureBox1.ImageIpl = mCanvas.dst;
            mCanvas.restore = mCanvas.dst.Clone();
        }

        public void DrawTextOnPath(Graphics gr, Brush brush, Font font, string txt, GraphicsPath path)
        {
            if (!(path.PointCount > 0)) return;
            if (txt == null) return;
            // Make a copy so we don't mess up the original.
            path = (GraphicsPath)path.Clone();

            bool isLine = false;
            int oldcnt = path.PointCount;
            // Flatten the path into segments.
            path.Flatten();
            int newcnt = path.PointCount;
            if (oldcnt == newcnt)
            {
                isLine = true;
            }

            // Draw characters.
            int start_ch = 0;
            PointF start_point = path.PathPoints[0], end_point = path.PathPoints[0];
            for (int i = 0; i < path.PointCount; i++)
            {
                if (i == path.PointCount - 1)
                {
                    if (path.PathTypes[i] == 129)
                    {
                        if (isLine)
                        {
                            start_point = path.PathPoints[i];
                        }
                        end_point = path.PathPoints[0];
                    }
                }
                else
                {
                    end_point = path.PathPoints[i + 1];
                    if (path.PathTypes[i] == 129)
                    {
                        if (!isLine)
                        {
                            start_point = end_point;
                        }
                        else
                        {
                            start_point = path.PathPoints[i + 1];
                        }
                    }
                    else
                    {
                        if (isLine)
                        {
                            start_point = path.PathPoints[i];
                        }
                    }

                }

                DrawTextOnSegment(gr, brush, font, txt, ref start_ch, ref start_point, end_point);
                if (start_ch >= txt.Length) break;
            }
        }

        private IEnumerable<float> GetCharacterWidths(Graphics graphics, string text, Font font)
        {
            // The length of a space. Necessary because a space measured using StringFormat.GenericTypographic has no width.
            // We can't use StringFormat.GenericDefault for the characters themselves, as it adds unwanted spacing.
            var spaceLength = graphics.MeasureString(" ", font, Point.Empty, StringFormat.GenericTypographic).Width;

            return text.Select(c => c == ' ' ? spaceLength : graphics.MeasureString(c.ToString(), font, Point.Empty, StringFormat.GenericTypographic).Width);
        }

        public void DrawTextOnSegment(Graphics gr, Brush brush, Font font, string txt, ref int first_ch, ref PointF start_point, PointF end_point)
        {
            if (txt.Length == 0) return;

            float dx = end_point.X - start_point.X;
            float dy = end_point.Y - start_point.Y;
            float dist = (float)Math.Sqrt(dx * dx + dy * dy);
            dx /= dist;
            dy /= dist;

            // See how many characters will fit.
            int last_ch = first_ch;
            while (last_ch < txt.Length)
            {
                string test_string = txt.Substring(first_ch, last_ch - first_ch + 1);
                if (GetCharacterWidths(gr, test_string, font).ToArray().Sum() > dist)
                {
                    // This is one too many characters.
                    last_ch--;
                    break;
                }
                last_ch++;
            }
            if (last_ch < first_ch) return;
            if (last_ch >= txt.Length) last_ch = txt.Length - 1;
            string chars_that_fit = txt.Substring(first_ch, last_ch - first_ch + 1);

            // Update first_ch and start_point.
            first_ch = last_ch + 1;
            var characterWidths = GetCharacterWidths(gr, chars_that_fit, font).ToArray();
            var textLength = characterWidths.Sum();

            // Rotate and translate to position the characters.
            GraphicsState state = gr.Save();
            gr.TranslateTransform(0, -font.Height, MatrixOrder.Append);
            float angle = (float)(180 * Math.Atan2(dy, dx) / Math.PI);
            gr.RotateTransform(angle, MatrixOrder.Append);
            gr.TranslateTransform(start_point.X, start_point.Y, MatrixOrder.Append);

            if (mCanvas.drawing)
            {
                gr.DrawString(chars_that_fit, font, brush, 0, 0);
            }
            // Restore the saved state.
            gr.Restore(state);

            start_point = new PointF(start_point.X + dx * textLength, start_point.Y + dy * textLength);

            //# 커서 그리기
            if (isInput)
            {
                if (cmb_shape.SelectedIndex == 2 || cmb_shape.SelectedIndex == 4) //# 곡선
                {
                    if (first_ch == txt_context.SelectionStart)
                    {
                        float line_end_X = start_point.X + font.Height * (float)Math.Sin(Math.Atan2(dy, dx));
                        float line_end_Y = start_point.Y + font.Height * -(float)Math.Cos(Math.Atan2(dy, dx));
                        gr.DrawLine(Pens.Black, start_point, new PointF(line_end_X, line_end_Y));
                    }
                }
                else
                {
                    string cursor = txt.Substring(0, txt_context.SelectionStart);
                    float cursorWidth = GetCharacterWidths(gr, cursor, font).ToArray().Sum();

                    float cursor_X = start.X + cursorWidth * (float)Math.Cos(Math.Atan2(dy, dx));
                    float cursor_Y = start.Y + cursorWidth * (float)Math.Sin(Math.Atan2(dy, dx));

                    float line_end_X = cursor_X + font.Height * (float)Math.Sin(Math.Atan2(dy, dx));
                    float line_end_Y = cursor_Y + font.Height * -(float)Math.Cos(Math.Atan2(dy, dx));
                    gr.DrawLine(Pens.Black, new PointF(cursor_X, cursor_Y), new PointF(line_end_X, line_end_Y));
                    Console.WriteLine(cursor);
                }
            }

        }

        public void text_MouseDown(MouseEventArgs e)
        {
            if (isTunnel && tunnelPoints != null)
            {
                for (int i = 0; i < tunnelPoints.Length; i++)
                {
                    for (int j = 0; j < tunnelPoints[i].Count; j++)
                    {
                        PointF pt = tunnelPoints[i][j];
                        RectangleF rect = new RectangleF(pt.X - 5, pt.Y - 5, 10, 10);
                        if (rect.Contains(e.Location))
                        {
                            isMoving = true;
                            location = i;
                            index = j;
                            mCanvas.drawing = true;
                            break;
                        }
                    }
                }

                Console.WriteLine("Mousedown {0}", isMoving);
                return;
            }

            if (cmb_auto.SelectedIndex != 0 && txt_context.Text.Trim().Length == 0)
            {
                MessageBox.Show("문자열이 입력되지 않았습니다.", "warning");
                txt_context.Focus();
                return;
            }
            else
            {
                if (cmb_shape.SelectedIndex == 3 && cmb_auto.SelectedIndex == 0)
                {
                    MessageBox.Show("터널 문자는 직접 입력할 수 없습니다.", "warning");
                    cmb_auto.SelectedIndex = 1;
                    txt_context.Focus();
                    return;
                }

                if (mCanvas.drawing && cmb_auto.SelectedIndex == 0)
                {
                    txt_context.Clear();
                    isInput = true;
                }
            }

            if (mCanvas.drawing && cmb_shape.SelectedIndex != 4)
            {
                DrawString();
                mCanvas.drawing = false;
                mCanvas.PictureBox1.Refresh();

                if (cmb_auto.SelectedIndex != 0)
                    gp = null;

                if (cmb_shape.SelectedIndex == 3) //# 터널 쓰기
                {
                    DrawTunnel();
                    btn_set.Enabled = true;
                    isTunnel = true;

                    mCanvas.drawing = true;
                    mCanvas.PictureBox1.Refresh();
                    mCanvas.drawing = false;
                }

                mCanvas.matList.Add(mCanvas.dst);
            }
            else
            {
                //# 그리기 시작
                mCanvas.drawing = true;
                start = end = e.Location;

                if (cmb_shape.SelectedIndex == 4)
                {
                    if (curPoints == null)
                    {
                        curPoints = new List<Point>();
                        curPoints.Add(start);
                        curPoints.Add(end);
                    }
                    else
                    {
                        curPoints.Add(end);
                        if (!chk_conn.Checked && curPoints.Count == 4)
                        {
                            //# 끊어진 곡선
                            gp = new GraphicsPath();
                            gp.AddCurve(curPoints.ToArray());
                            DrawString();
                            mCanvas.drawing = false;
                            curPoints.Clear();
                            curPoints = null;
                            mCanvas.PictureBox1.Refresh();
                            mCanvas.matList.Add(mCanvas.dst);

                            if (cmb_auto.SelectedIndex != 0)
                                gp = null;
                        }
                    }
                }
            }
        }

        public void text_MouseMove(MouseEventArgs e)
        {
            if (isTunnel && isMoving)
            {
                Console.WriteLine(e.Location);
                tunnelPoints[location].RemoveAt(index);
                tunnelPoints[location].Insert(index, e.Location);
                mCanvas.PictureBox1.Refresh();
            }
            else
            {
                gp = new GraphicsPath();
                switch (cmb_shape.SelectedIndex)
                {
                    case 0: //# orthogonal line
                        if ((angle >= 45 && angle < 135) || (angle >= 225 && angle < 315))
                        {
                            gp.AddLine(start, new Point(start.X, end.Y));   //# 수직
                            angle = 360 - GetAngle(start, new Point(start.X, end.Y));
                        }
                        else
                        {
                            gp.AddLine(start, new Point(end.X, start.Y));   //# 수평
                            angle = 360 - GetAngle(start, new Point(end.X, start.Y));
                        }
                        break;
                    case 1: //# diagonal line
                    case 3: //# tunnel
                        angle = 360 - GetAngle(start, end);
                        gp.AddLine(start, end);
                        break;
                    case 2: //# circle
                        gp.AddEllipse(mCanvas.mVector.GetSquare(start.X, start.Y, end.X, end.Y));
                        break;
                    case 4: //# curve
                        curPoints.RemoveAt(curPoints.Count - 1);
                        curPoints.Add(end);
                        gp.AddCurve(curPoints.ToArray());
                        break;
                }
                end = e.Location;
                mCanvas.PictureBox1.Refresh();
            }
        }

        public double GetAngle(Point start, Point end)
        {
            double result = Math.Round(Math.Atan2(end.Y - start.Y, end.X - start.X) * (180.0 / Math.PI), 2);
            if (result < 0)
            {
                result = 360 + result;

            }
            return result;
        }

        public void text_MouseUp(MouseEventArgs e)
        {
            if (isTunnel && isMoving)
            {
                isMoving = false;
                mCanvas.drawing = false;
            }
        }

        public int MeasureDisplayStringWidth(Graphics graphics, string text, Font font)
        {
            StringFormat format = new StringFormat();

            RectangleF rect = new RectangleF(0, 0, mCanvas.PictureBox1.Width, mCanvas.PictureBox1.Height);

            CharacterRange[] ranges = { new CharacterRange(0, text.Length) };

            Region[] regions = new Region[1];

            format.SetMeasurableCharacterRanges(ranges);

            regions = graphics.MeasureCharacterRanges(text, font, rect, format);
            rect = regions[0].GetBounds(graphics);

            return (int)(rect.Right + 1.0f);
        }

        public void text_Paint(PaintEventArgs e)
        {
            if (gp != null)
            {
                e.Graphics.DrawPath(Pens.Gray, gp);
            }


            if (cmb_shape.SelectedIndex == 3 && isTunnel)
            {
                if (tunnelPoints != null)
                {
                    GraphicsPath bottom = new GraphicsPath();
                    GraphicsPath top = new GraphicsPath();

                    bottom.AddCurve(tunnelPoints[0].ToArray());
                    top.AddCurve(tunnelPoints[1].ToArray());

                    e.Graphics.DrawPath(Pens.Red, bottom);
                    e.Graphics.DrawPath(Pens.Blue, top);

                    for (int i = 0; i < tunnelPoints[0].Count; i++)
                    {
                        PointF pt = tunnelPoints[0][i];
                        e.Graphics.FillRectangle(new SolidBrush(Color.Red), new RectangleF(pt.X - 5, pt.Y - 5, 10, 10));
                    }

                    for (int i = 0; i < tunnelPoints[1].Count; i++)
                    {
                        PointF pt = tunnelPoints[1][i];
                        e.Graphics.FillRectangle(new SolidBrush(Color.Blue), new RectangleF(pt.X - 5, pt.Y - 5, 10, 10));
                    }
                }
            }
        }

        private void cmb_fontName_DrawItem(object sender, DrawItemEventArgs e)
        {
            FontFamily family = FontFamily.Families[e.Index];
            Font myFont = new Font(family, 10);
            Rectangle rect = new Rectangle(2, e.Bounds.Top + 2, e.Bounds.Height, e.Bounds.Height - 4);
            e.Graphics.FillRectangle(new SolidBrush(Color.White), rect);
            e.Graphics.DrawString(FontFamily.Families[e.Index].Name, myFont, Brushes.Black,
                new RectangleF(e.Bounds.X + rect.Width, e.Bounds.Y, e.Bounds.Width, e.Bounds.Height));
            e.DrawFocusRectangle();
        }

        private void TextDraw_Load(object sender, EventArgs e)
        {
            //# Font combo box setting
            foreach (FontFamily family in FontFamily.Families)
            {
                cmb_fontName.Items.Add(family.Name);
            }
            cmb_fontName.SelectedIndex = cmb_fontName.Items.IndexOf("굴림");
            cmb_fontName.DrawMode = DrawMode.OwnerDrawVariable;
            cmb_shape.SelectedIndex = 0;
            cmb_auto.SelectedIndex = 0;
            cmb_style.SelectedIndex = 0;

            UpdateDisplay();
        }

        public void UpdateDisplay()
        {
            List<Panel> pnlList = new List<Panel>();
            pnlList.Add(pnl_font);

            switch (cmb_shape.SelectedIndex)
            {
                case 3: //# 터널
                    pnlList.Add(pnl_tunnel);
                    break;
                case 4: //# 곡선
                    pnlList.Add(pnl_option);
                    break;
            }

            pnlList.Add(pnl_context);

            if (pnlList.Count > 0)
            {
                int top = 0;
                for (int i = 0; i < pnlList.Count; i++)
                {
                    pnlList[i].Visible = true;
                    pnlList[i].Location = new Point(0, top);
                    pnlList[i].BringToFront();
                    top += pnlList[i].Height;
                }

                this.Height = pnlList[pnlList.Count - 1].Top + pnlList[pnlList.Count - 1].Height + 38;  //# 38 - Topbar Height
            }

            this.Refresh();
        }

        private void cmb_auto_SelectedIndexChanged(object sender, EventArgs e)
        {
            if (cmb_auto.SelectedIndex == 0)
            {
                txt_context.Enabled = false;
            }
            else
            {
                txt_context.Enabled = true;
            }
        }

        private void txt_context_KeyUp(object sender, KeyEventArgs e)
        {
            mCanvas.drawing = true;
            if (cmb_auto.SelectedIndex == 0)
            {
                //# line -> 가변적으로 path 변경
                if (cmb_shape.SelectedIndex == 0 || cmb_shape.SelectedIndex == 1)
                {
                    float text_len = MeasureDisplayStringWidth(gr, txt_context.Text, font);

                    float radian = (float)(angle * Math.PI / 180);

                    float dx = (float)(text_len * Math.Cos(radian));
                    float dy = (float)(text_len * -Math.Sin(radian));

                    gp = new GraphicsPath();
                    gp.AddLine(start, new PointF(start.X + dx, start.Y + dy));

                    Console.WriteLine("{0}, {1}", dx, dy);
                }

                if (e.KeyCode == Keys.Enter)
                {
                    isInput = false;
                    DrawString();
                    mCanvas.matList.Add(mCanvas.dst);
                }
                else
                {
                    DrawString();
                }
            }
            mCanvas.drawing = false;
        }

        private void btn_set_Click(object sender, EventArgs e)
        {
            btn_reset.Enabled = true;
            btn_set.Enabled = false;
            isTunnel = false;
            isMoving = false;
            mCanvas.PictureBox1.Refresh();
            mCanvas.drawing = false;

        }

        private void btn_reset_Click(object sender, EventArgs e)
        {
            btn_reset.Enabled = false;
            btn_set.Enabled = true;
            isTunnel = true;
            isMoving = true;
            mCanvas.drawing = true;
            mCanvas.PictureBox1.Refresh();
            mCanvas.drawing = false;
        }

        private void TextDraw_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = true;
            this.Hide();
        }

        private void cmb_shape_SelectedIndexChanged(object sender, EventArgs e)
        {
            UpdateDisplay();
        }
    }
}
