﻿using ClipperLib;
using OpenCvSharp;
using OpenCvSharp.Extensions;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Windows.Forms;

using cvPoint = OpenCvSharp.Point;
using Point = System.Drawing.Point;
using Polygon = System.Collections.Generic.List<ClipperLib.IntPoint>;

namespace TexMaster
{
    public enum WShape
    {
        RECT, ELLIPSE, PATH, FREE,
        SINGLE, MULTI, BORDER, UNDO
    }

    public partial class WorkArea : Form
    {
        public Canvas mCanvas;

        public WShape wShape;
        public Point start, end;
        public Pen blankPen;
        public Pen dashPen;
        public GraphicsPath totalPath;          //# total path
        public GraphicsPath connPath;           //# connect path
        public GraphicsPath currentPath;        //# current drawing path
        public List<GraphicsPath> workList;     //# path list to get the total path (use to pathfinder)
        public List<GraphicsPath> undoList;
        public int history;
        public List<Byte> pathByte;
        public List<PointF> pathPoints;
        public List<GraphicsPath> connPathList; //# connPath의 리스트 ( 한 점 뒤로 기능할 때 List를 하나씩 뺀다)
        public List<Scalar> selColors;
        public List<Scalar> lastList;
        public bool isFirst;

        public bool isClick;
        public Mat tempMat;
        public Mat holeMat;                     //# 작업구역이 설정된 관심영역 Mat
        public PictureBox backupPic;

        bool isArea;
        bool isDelete;
        GraphicsPath seltempPath;

        MultiColor multi;

        public WorkArea(Canvas canvas)
        {
            InitializeComponent();
            mCanvas = canvas;
            mCanvas.mWorkarea = this;
            blankPen = new Pen(Color.White, 1F);
            dashPen = new Pen(Color.Black, 1F);
            dashPen.DashStyle = DashStyle.Dash;
            dashPen.DashPattern = new float[2] { 4, 4 };
            workList = new List<GraphicsPath>();
            //totalPath = new GraphicsPath();
            connPathList = new List<GraphicsPath>();
            undoList = new List<GraphicsPath>();
            selColors = new List<Scalar>();

            //# 생성 시 기본 picture box 기억(main에서 생성)
            backupPic = mCanvas.PictureBox1;

            isArea = true;
            isDelete = false;

            multi = new MultiColor();
        }

        public void UpdateDisplay()
        {
            List<Panel> pnlList = new List<Panel>();
            pnlList.Add(pnl_workarea);

            //# PathFinder Error로 잠시 막아둠
            if (wShape == WShape.SINGLE || wShape == WShape.MULTI || wShape == WShape.BORDER)
            {
                rdo_add.Enabled = false;
                rdo_sub.Enabled = false;
            }
            else
            {
                rdo_add.Enabled = true;
                rdo_sub.Enabled = true;
            }

            switch (wShape)
            {
                case WShape.RECT:
                case WShape.ELLIPSE:
                    break;
                case WShape.PATH:
                case WShape.FREE:
                    pnlList.Add(pnl_path);
                    break;
                case WShape.SINGLE:
                case WShape.BORDER:
                    pnlList.Add(pnl_similar);
                    break;
                case WShape.MULTI:
                    pnlList.Add(pnl_similar);
                    pnlList.Add(pnl_surface);
                    break;
                case WShape.UNDO:
                    pnlList.Add(pnl_undo);
                    if (history > 0)
                        btn_undo.Enabled = true;
                    else
                        btn_undo.Enabled = false;

                    if (history < undoList.Count)
                        btn_redo.Enabled = true;
                    else
                        btn_redo.Enabled = false;
                    break;
                default:
                    break;

            }

            if (pnlList.Count > 0)
            {
                int top = 0;
                for (int i = 0; i < pnlList.Count; i++)
                {
                    pnlList[i].Visible = true;
                    pnlList[i].Location = new Point(0, top);
                    pnlList[i].BringToFront();
                    top += pnlList[i].Height;
                }

                this.Height = pnlList[pnlList.Count - 1].Top + pnlList[pnlList.Count - 1].Height + 38;  //# 38 - Topbar Height
            }

            this.Refresh();
        }

        public void work_MouseDown(object sender, MouseEventArgs e)
        {
            if (mCanvas.drawing)
            {
                switch (wShape)
                {
                    case WShape.PATH:
                        isFirst = true;
                        isClick = true;
                        pathByte.Add(1);
                        pathPoints.Add(e.Location);
                        connPath = new GraphicsPath(pathPoints.ToArray(), pathByte.ToArray());
                        mCanvas.PictureBox1.Refresh();
                        break;
                    default:
                        AddArea(currentPath);
                        currentPath = null;
                        lbl_size.Text = String.Format("크기 : {0}cm x {1}cm", ConvertToCm(totalPath.GetBounds().Width), ConvertToCm(totalPath.GetBounds().Height));
                        mCanvas.PictureBox1.Refresh();
                        mCanvas.drawing = false;
                        break;
                }
            }
            else
            {
                if (rdo_set.Checked && connPath == null)
                {
                    totalPath = null;
                    mCanvas.drawing = true;
                    mCanvas.PictureBox1.Refresh();
                }

                start = e.Location;
                switch (wShape)
                {
                    case WShape.PATH:
                        mCanvas.drawing = true;
                        currentPath = new GraphicsPath();
                        if (connPath == null)
                        {
                            connPath = new GraphicsPath();
                            pathByte = new List<byte>();
                            pathPoints = new List<PointF>();

                            pathByte.Add(0);
                            pathPoints.Add(e.Location);
                        }
                        break;
                    case WShape.SINGLE:
                        selColors.Clear();
                        selColors.Add(mCanvas.dst.At<int>(e.Y, e.X));
                        ContourPath();
                        break;
                    case WShape.MULTI:
                        if (!isArea)
                        {
                            AddColorChip(e.Location);
                            ContourPath();
                            return;
                        }
                        else
                        {
                            seltempPath = new GraphicsPath();
                        }
                        break;
                    case WShape.BORDER:
                        break;
                    default:
                        mCanvas.drawing = true;
                        currentPath = new GraphicsPath();
                        break;
                }
            }

            GetHoleMat();
        }

        public Mat FloodFill(Mat src, cvPoint seedPt)
        {
            Mat hsv = new Mat();

            //# 배경색 제거 - 현재 흰색 -1 임시 적용
            Color c = Color.FromArgb(-1);
            Scalar s = new Scalar(c.B, c.G, c.R, c.A);
            Cv2.InRange(src, s, s, hsv);

            //# 채우기 전 이미지
            Mat ori = hsv.Clone();
            //# 채운 후 이미지
            Cv2.FloodFill(hsv, seedPt, new Scalar(255, 255, 255, 255));

            Mat diff = new Mat();
            //# 차이비교 - 채워질 영역 mask 반환
            Cv2.BitwiseXor(hsv, ori, diff);
            return diff;
        }

        public void GetHoleMat()
        {
            if (totalPath != null)
            {
                Bitmap maskImg = new Bitmap(mCanvas.PictureBox1.Width, mCanvas.PictureBox1.Height);
                Graphics maskGr = Graphics.FromImage(maskImg);
                Color cMask = Color.FromArgb(0, 0, 255); //# blue
                maskGr.FillPath(new SolidBrush(cMask), totalPath);

                //# make mask mat
                Mat hsv = new Mat();
                Scalar sMask = new Scalar(cMask.B, cMask.G, cMask.R, cMask.A);
                Mat ori = BitmapConverter.ToMat(maskImg);
                Cv2.CvtColor(ori, hsv, ColorConversionCodes.BGR2HSV);
                Cv2.InRange(ori, sMask, sMask, hsv);

                holeMat = hsv;
                //Cv2.ImShow("holeMat", holeMat);
            }
        }

        public void work_MouseMove(object sender, MouseEventArgs e)
        {
            if (mCanvas.drawing)
            {
                end = e.Location;
                if (wShape != WShape.FREE) currentPath = new GraphicsPath();
                switch (wShape)
                {
                    case WShape.FREE:
                        currentPath.AddLine(start, end);
                        start = end;
                        break;
                    case WShape.RECT:
                        if ((System.Windows.Input.Keyboard.GetKeyStates(System.Windows.Input.Key.LeftShift) & System.Windows.Input.KeyStates.Down) > 0)
                        {
                            currentPath.AddRectangle(mCanvas.CreateSquare(start.X, start.Y, end.X, end.Y));
                            mCanvas.tsl_sub.Text = "square";
                        }
                        else
                        {
                            currentPath.AddRectangle(mCanvas.CreateRect(start.X, start.Y, end.X, end.Y));
                            mCanvas.tsl_sub.Text = "rect";
                        }
                        break;
                    case WShape.ELLIPSE:
                        if ((System.Windows.Input.Keyboard.GetKeyStates(System.Windows.Input.Key.LeftShift) & System.Windows.Input.KeyStates.Down) > 0)
                        {
                            currentPath.AddEllipse(mCanvas.CreateSquare(start.X, start.Y, end.X, end.Y));
                            mCanvas.tsl_sub.Text = "circle";
                        }
                        else
                        {
                            currentPath.AddEllipse(mCanvas.CreateRect(start.X, start.Y, end.X, end.Y));
                            mCanvas.tsl_sub.Text = "ellipse";
                        }
                        break;
                    case WShape.PATH:
                        if (!isClick) return;
                        if (pathPoints.Count == 1) return;

                        if (isFirst)//Bezier Curve Make
                        {
                            isFirst = false;
                            pathByte.RemoveAt(pathByte.Count - 1);
                            pathPoints.RemoveAt(pathPoints.Count - 1);
                            for (int i = 0; i < 3; i++)
                            {
                                pathByte.Add(3);
                                pathPoints.Add(e.Location);
                            }
                        }
                        else
                        {
                            pathPoints[pathPoints.Count - 3] = GetSymmetryPoint(pathPoints[pathPoints.Count - 1], e.Location);
                            pathPoints[pathPoints.Count - 2] = GetSymmetryPoint(pathPoints[pathPoints.Count - 1], e.Location);

                            PointF[] newHandle;
                            newHandle = GetthrBezirPoint(pathPoints[pathPoints.Count - 4], pathPoints[pathPoints.Count - 3], pathPoints[pathPoints.Count - 1]);
                            pathPoints[pathPoints.Count - 3] = newHandle[0];
                            pathPoints[pathPoints.Count - 2] = newHandle[1];
                        }
                        connPath = new GraphicsPath(pathPoints.ToArray(), pathByte.ToArray());
                        break;
                    case WShape.MULTI:
                        if (e.Button == MouseButtons.Middle)
                        {
                            end = e.Location;
                            seltempPath = new GraphicsPath();
                            seltempPath.AddRectangle(mCanvas.CreateRect(start.X, start.Y, end.X, end.Y));
                            mCanvas.PictureBox1.Refresh();
                        }
                        break;
                    default:
                        break;
                }

                lbl_size.Text = String.Format("크기 : {0}cm x {1}cm", ConvertToCm(currentPath.GetBounds().Width), ConvertToCm(currentPath.GetBounds().Height));
            }
        }

        public void work_MouseUp(object sender, MouseEventArgs e)
        {
            if (wShape == WShape.PATH)
            {
                connPath = new GraphicsPath(pathPoints.ToArray(), pathByte.ToArray());
                connPathList.Add((GraphicsPath)connPath.Clone());
                if (connPathList.Count > 0)
                {
                    btn_del.Enabled = true;
                    if (connPathList.Count > 2)
                    {
                        btn_close.Enabled = true;
                    }
                }
                isFirst = true;
                isClick = false;
                mCanvas.PictureBox1.Refresh();

                lbl_size.Text = String.Format("Size : {0}cm x {1}cm", ConvertToCm(connPath.GetBounds().Width), ConvertToCm(connPath.GetBounds().Height));
            }
            else if (wShape == WShape.FREE)
            {
                AddArea(currentPath);
                mCanvas.PictureBox1.Refresh();
                mCanvas.drawing = false;
            }
            else if (wShape == WShape.MULTI)
            {
                if (e.Button == MouseButtons.Middle)
                {
                    end = e.Location;
                    Rectangle rect = mCanvas.CreateRect(start.X, start.Y, end.X, end.Y);

                    for (int row = rect.Top; row < rect.Top + rect.Height - 1; row++)
                    {
                        for (int col = rect.Left; col < rect.Left + rect.Width - 1; col++)
                        {
                            Color c = Color.FromArgb(mCanvas.dst.At<int>(row, col));
                            if (c.A == 255)
                            {
                                if (!isDelete && !selColors.Contains(mCanvas.dst.At<int>(row, col)))
                                {
                                    selColors.Add(mCanvas.dst.At<int>(row, col));
                                }
                                else if (isDelete && selColors.Contains(mCanvas.dst.At<int>(row, col)))
                                {
                                    selColors.Remove(mCanvas.dst.At<int>(row, col));
                                }
                            }
                        }
                    }

                    pnl_color.Controls.Clear();
                    for (int i = 0; i < selColors.Count; i++)
                    {
                        Panel panel = new Panel();
                        panel.Click += panelClick;
                        panel.Width = 17;
                        panel.Height = 17;
                        panel.BackColor = Color.FromArgb((int)selColors[i]);
                        panel.BorderStyle = BorderStyle.FixedSingle;
                        pnl_color.Controls.Add(panel);
                    }

                    ContourPath();
                    seltempPath = null;
                    mCanvas.PictureBox1.Refresh();
                    mCanvas.drawing = false;
                }
            }
        }

        private void WorkArea_Load(object sender, EventArgs e)
        {
            this.Height = pnl_workarea.Height + 38;

            ToolTip toolTip = new ToolTip();
            toolTip.ShowAlways = true;
            toolTip.SetToolTip(btn_rect, "사각형");
            toolTip.SetToolTip(btn_ellipse, "원형");
            toolTip.SetToolTip(btn_path, "경로");
            toolTip.SetToolTip(btn_outline, "윤곽선");
            toolTip.SetToolTip(btn_full, "전체 선택");
            toolTip.SetToolTip(btn_off, "전체 해제");
            toolTip.SetToolTip(btn_restore, "환원");

            wShape = WShape.RECT;
            mCanvas.tsl_sub.Text = wShape.ToString().ToLower();
            UpdateDisplay();

        }

        public void WorkShapeClick(object sender, EventArgs e)
        {
            mCanvas.drawing = false;

            String name;
            try
            {
                name = ((ToolStripMenuItem)sender).Name;
            }
            catch
            {
                name = ((Button)sender).Name;
            }

            switch (name)
            {
                case "btn_rect":
                    wShape = WShape.RECT;
                    break;
                case "btn_ellipse":
                    wShape = WShape.ELLIPSE;
                    break;
                case "tsm_free":
                    wShape = WShape.FREE;
                    break;
                case "tsm_line":
                    wShape = WShape.PATH;
                    break;
                case "tsm_single":
                    wShape = WShape.SINGLE;
                    break;
                case "tsm_multi":
                    wShape = WShape.MULTI;
                    break;
                case "tsm_border":
                    wShape = WShape.BORDER;
                    break;
                case "btn_path":
                    //wShape = WShape.PATH;
                    cms_irregular.Show(MousePosition);
                    return;
                case "btn_full":
                    SelAll();
                    break;
                case "btn_restore":
                    wShape = WShape.UNDO;
                    break;
                case "btn_outline":
                    wShape = WShape.SINGLE;
                    cms_workarea.Show(MousePosition);
                    break;
                default:
                    break;
            }

            mCanvas.tsl_sub.Text = wShape.ToString().ToLower();
            UpdateDisplay();
        }

        public void AddArea(GraphicsPath currentPath)
        {
            if (currentPath != null)
            {
                if (rdo_set.Checked)
                {
                    workList.Clear();
                    totalPath = currentPath;
                }
                else if (rdo_add.Checked)
                {
                    workList.Add(currentPath);
                    totalPath = (GraphicsPath)PathFinder(workList, ClipType.ctUnion).Clone();
                }
                else if (rdo_sub.Checked)
                {
                    workList.Add(currentPath);
                    totalPath = (GraphicsPath)PathFinder(workList, ClipType.ctDifference).Clone();
                }

                if (totalPath.GetBounds().Width == 0 || totalPath.GetBounds().Height == 0)
                {
                    workList.Clear();
                    totalPath = null;
                    this.currentPath = null;
                    mCanvas.drawing = false;
                    return;
                }
            }

            undoList.Add(totalPath);
            undoList.RemoveRange(history + 1, undoList.Count - 1 - history);
            history = undoList.Count;
        }

        public void ContourPath()
        {
            workList.Clear();
            totalPath = null;
            currentPath = null;

            if (!(selColors.Count > 0))
            {
                mCanvas.PictureBox1.Refresh();
                return;
            }

            cvPoint[][] contours;
            HierarchyIndex[] hierarchy;

            Mat hsv = new Mat();

            if (chk_similar.Checked)
            {
                for (int i = 0; i < selColors.Count; i++)
                {
                    Color c = Color.FromArgb((int)selColors[i]);
                    Scalar selColor = new Scalar(c.B, c.G, c.R, c.A);
                    Mat ori = mCanvas.dst.Clone();
                    Cv2.CvtColor(ori, hsv, ColorConversionCodes.BGR2HSV);
                    Cv2.InRange(ori, selColor, selColor, hsv);
                    if (i == 0)
                    {
                        tempMat = new Mat();
                        tempMat.Add(hsv);
                    }
                    else
                    {
                        tempMat += hsv;
                    }
                }
            }
            else
            {
                tempMat = FloodFill(mCanvas.dst, new cvPoint(start.X, start.Y));
            }

            Cv2.FindContours(tempMat, out contours, out hierarchy, RetrievalModes.Tree, ContourApproximationModes.ApproxTC89KCOS);

            List<cvPoint[]> new_contours = new List<cvPoint[]>();
            currentPath = new GraphicsPath();

            foreach (cvPoint[] p in contours)
            {
                new_contours.Add(p);
                GraphicsPath tempPath = new GraphicsPath();
                for (int i = 0; i < p.Length; i++)
                {
                    if (i == p.Length - 1)
                    {
                        tempPath.CloseAllFigures();
                    }
                    else
                    {
                        tempPath.AddLine(new Point(p[i].X, p[i].Y), new Point(p[i + 1].X, p[i + 1].Y));
                    }
                }

                if (rdo_set.Checked)
                {
                    currentPath.AddPath(tempPath, false);
                }
                else
                {
                    currentPath = new GraphicsPath();
                    workList.Add(tempPath);
                }
            }

            AddArea(currentPath);
            mCanvas.PictureBox1.Refresh();
        }

        public void AddColorChip(Point point)
        {
            pnl_color.Controls.Clear();

            if (!selColors.Contains(mCanvas.dst.At<int>(point.Y, point.X)))
            {
                selColors.Add(mCanvas.dst.At<int>(point.Y, point.X));
            }

            for (int i = 0; i < selColors.Count; i++)
            {
                Panel panel = new Panel();
                panel.Click += panelClick;
                panel.Width = 17;
                panel.Height = 17;
                panel.BackColor = Color.FromArgb((int)selColors[i]);
                panel.BorderStyle = BorderStyle.FixedSingle;
                pnl_color.Controls.Add(panel);
            }
        }

        public void panelClick(object sender, EventArgs e)
        {
            selColors.Remove(((Panel)sender).BackColor.ToArgb());
            pnl_color.Controls.Remove((Panel)sender);
            ContourPath();
        }

        private PointF GetSymmetryPoint(PointF p1, PointF p2, bool reverse = false)
        {
            float x, y;
            if (!reverse)
            {
                x = 2 * p1.X - p2.X;
                y = 2 * p1.Y - p2.Y;
            }
            else
            {
                x = 2 * p2.X - p1.X;
                y = 2 * p2.Y - p1.Y;
            }

            PointF result = new PointF(x, y);
            return result;
        }

        public PointF[] GetthrBezirPoint(PointF s, PointF c, PointF e)
        {
            PointF[] controlPt = new PointF[2];
            //c1 = Control Point
            float bX = s.X + (2.0f / 3.0f) * (c.X - s.X);
            float bY = s.Y + (2.0f / 3.0f) * (c.Y - s.Y);
            float cX = e.X + (2.0f / 3.0f) * (c.X - e.X);
            float cY = e.Y + (2.0f / 3.0f) * (c.Y - e.Y);

            controlPt[0] = new PointF(bX, bY);
            controlPt[1] = new PointF(cX, cY);

            return controlPt;
        }

        public double ConvertToCm(float pixel)
        {
            return Math.Round(pixel * 0.026458333, 2);
        }

        public void WorkArea_Paint(Graphics gr)
        {
            if (seltempPath != null)
            {
                gr.DrawPath(Pens.Black, seltempPath);
            }

            if (totalPath != null)
            {
                gr.DrawPath(blankPen, totalPath);
                gr.DrawPath(dashPen, totalPath);
                mCanvas.menu.tsb_pattern.Enabled = true;
                mCanvas.menu.tsb_move.Enabled = true;
                mCanvas.menu.tsb_repeat.Enabled = true;
                mCanvas.menu.tsb_zoom.Enabled = true;
                mCanvas.menu.tsb_shadow.Enabled = true;
                mCanvas.mFill.btn_s_pattern.Enabled = true;
                mCanvas.mAssem.tabPage1.Enabled = true;

                //# warp
                //mCanvas.mAssem.btn_workarea.Enabled = false;
                mCanvas.mAssem.btn_add.Enabled = true;
                mCanvas.mAssem.btn_modify.Enabled = true;
                mCanvas.mAssem.btn_warp.Enabled = true;

                mCanvas.mStipple.btn_pattern.Enabled = true;
            }
            else
            {
                mCanvas.menu.tsb_pattern.Enabled = false;
                mCanvas.menu.tsb_move.Enabled = false;
                mCanvas.menu.tsb_repeat.Enabled = false;
                mCanvas.menu.tsb_zoom.Enabled = false;
                mCanvas.menu.tsb_shadow.Enabled = false;
                mCanvas.mFill.btn_s_pattern.Enabled = false;
                mCanvas.mAssem.tabPage1.Enabled = false;

                //# warp
                //mCanvas.mAssem.btn_workarea.Enabled = true;
                mCanvas.mAssem.btn_add.Enabled = false;
                mCanvas.mAssem.btn_modify.Enabled = false;
                mCanvas.mAssem.btn_warp.Enabled = false;

                mCanvas.mStipple.btn_pattern.Enabled = false;
            }
            if (currentPath != null)
            {
                gr.DrawPath(blankPen, currentPath);
                gr.DrawPath(dashPen, currentPath);
            }
            if (connPath != null)
            {
                gr.DrawPath(blankPen, connPath);
                gr.DrawPath(dashPen, connPath);
            }

        }

        private void rdo_add_CheckedChanged(object sender, EventArgs e)
        {
            if (rdo_add.Checked)
            {
                workList.Clear();
                if (totalPath != null)
                {
                    workList.Add(totalPath);
                }
            }
        }

        private void rdo_sub_CheckedChanged(object sender, EventArgs e)
        {
            if (rdo_sub.Checked)
            {
                workList.Clear();
                if (totalPath != null)
                {
                    workList.Add(totalPath);
                }
            }
        }

        private void btn_close_Click(object sender, EventArgs e)
        {
            if (wShape == WShape.PATH && connPath != null && pathPoints.Count > 2)
            {
                connPath.CloseAllFigures();
                AddArea(connPath);

                pathPoints = null;
                pathByte = null;
                connPath = null;
                connPathList.Clear();

                mCanvas.PictureBox1.Refresh();
                mCanvas.drawing = false;
            }

            if (connPathList.Count == 0)
            {
                btn_del.Enabled = false;
                btn_close.Enabled = false;
            }
        }

        private void btn_del_Click(object sender, EventArgs e)
        {

            if (wShape == WShape.PATH && connPathList.Count > 0)
            {
                if (connPathList.Count == 1)
                    connPath = (GraphicsPath)connPathList[connPathList.Count - 1].Clone();
                else
                {
                    connPath = (GraphicsPath)connPathList[connPathList.Count - 2].Clone();
                    connPathList.RemoveAt(connPathList.Count - 1);
                }

                pathPoints = connPath.PathPoints.ToList();
                pathByte = connPath.PathTypes.ToList();

                mCanvas.drawing = true;
                mCanvas.PictureBox1.Refresh();
            }

        }

        public void SelAll()
        {
            currentPath = new GraphicsPath();
            totalPath = new GraphicsPath();
            currentPath.AddRectangle(new Rectangle(0, 0, mCanvas.PictureBox1.Width - 1, mCanvas.PictureBox1.Height - 1));

            AddArea(currentPath);
            currentPath = null;

            mCanvas.drawing = true;
            mCanvas.PictureBox1.Refresh();
            mCanvas.drawing = false;
        }

        private void btn_off_Click(object sender, EventArgs e)
        {
            totalPath = null;
            currentPath = null;
            connPath = null;

            AddArea(currentPath);

            mCanvas.drawing = true;
            mCanvas.PictureBox1.Refresh();
            mCanvas.drawing = false;
        }

        private void btn_undo_Click(object sender, EventArgs e)
        {
            if (history > 0)
            {
                history--;
                if (history == 0)
                {
                    totalPath = null;
                }
                else
                {
                    totalPath = (GraphicsPath)undoList[history - 1].Clone();
                }
                mCanvas.PictureBox1.Refresh();
            }

            UpdateDisplay();
        }

        private void btn_redo_Click(object sender, EventArgs e)
        {
            if (history < undoList.Count)
            {
                if (history == undoList.Count)
                {
                    totalPath = (GraphicsPath)undoList[history - 1].Clone();
                }
                else
                {
                    if (undoList[history] == null)
                    {
                        totalPath = null;
                    }
                    else
                    {
                        totalPath = (GraphicsPath)undoList[history].Clone();
                    }
                }
                history++;
                mCanvas.PictureBox1.Refresh();
            }

            UpdateDisplay();
        }

        private void btn_new_Click(object sender, EventArgs e)
        {
            multi.SelectClear(pnl_color);
            ContourPath();
            mCanvas.PictureBox1.Refresh();
        }

        private void btn_all_Click(object sender, EventArgs e)
        {
            multi.SelectAll(pnl_color, mCanvas.dst, totalPath);
            ContourPath();
            mCanvas.PictureBox1.Refresh();
        }

        private void btn_area_Click(object sender, EventArgs e)
        {
            multi.SetSelectState(btn_area);
        }

        private void btn_insert_Click(object sender, EventArgs e)
        {
            multi.SetSelectState(btn_insert);
        }

        private void btn_pre_Click(object sender, EventArgs e)
        {
            multi.SelectRestore(pnl_color);
        }

        private void WorkArea_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = true;
            this.Hide();
        }

        private void WorkArea_Activated(object sender, EventArgs e)
        {
            MainMenu.mode = MainMode.WORKAREA;
            mCanvas.tsl_mode.Text = MainMenu.mode.ToString().ToLower();
        }

        public GraphicsPath PathFinder(List<GraphicsPath> pathList, ClipType type)
        {
            //# 마지막점 연결 Error
            Clipper c = new Clipper();
            PolyTree polyTree = new PolyTree();

            for (int i = 0; i < pathList.Count; i++)
            {
                GraphicsPath path = pathList[i];
                Matrix mat = new Matrix();
                path.Flatten(mat, 0.01F);
                Polygon poly1 = new Polygon();
                for (int j = 0; j < path.PointCount; j++)
                {
                    poly1.Add(new IntPoint((long)path.PathPoints[j].X, (long)path.PathPoints[j].Y));
                }

                if (i == 0)
                {
                    c.AddPolygon(poly1, PolyType.ptSubject);
                }
                else
                {
                    c.AddPolygon(poly1, PolyType.ptClip);
                }
            }

            c.Execute(type, polyTree, PolyFillType.pftNonZero, PolyFillType.pftNonZero);

            List<byte> pathByte = new List<byte>();
            List<PointF> pathPoints = new List<PointF>();

            for (int i = 0; i < polyTree.ChildCount; i++)
            {
                PolyNode nodeList = polyTree.Childs[i];

                for (int j = 0; j < nodeList.Childs.Count; j++)
                {
                    PolyNode node = nodeList.Childs[j];

                    if (node.IsHole)
                    {
                        for (int k = 0; k < node.Contour.Count; k++)
                        {
                            if (k == node.Contour.Count - 1)
                            {
                                pathByte.Add(161);
                            }
                            else if (k == 0)
                            {
                                pathByte.Add(0);
                            }
                            else
                            {
                                pathByte.Add(1);
                            }
                            pathPoints.Add(new Point((int)node.Contour[k].X, (int)node.Contour[k].Y));
                        }
                    }
                }

                for (int j = 0; j < nodeList.Contour.Count; j++)
                {
                    if (j == nodeList.Contour.Count - 1)
                    {
                        pathByte.Add(129);
                    }
                    else if (j == 0)
                    {
                        pathByte.Add(0);
                    }
                    else
                    {
                        pathByte.Add(1);
                    }

                    pathPoints.Add(new Point((int)nodeList.Contour[j].X, (int)nodeList.Contour[j].Y));
                }
            }

            GraphicsPath result = new GraphicsPath(pathPoints.ToArray(), pathByte.ToArray());
            return result;
        }
    }
}
