﻿using OpenCvSharp;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Linq;
using System.Windows.Forms;
using TexMaster.Source;
using cvPoint = OpenCvSharp.Point;
using cvSize = OpenCvSharp.Size;
using Point = System.Drawing.Point;

namespace TexMaster
{
    public partial class PenManager : Form
    {
        public PenShape penMode = PenShape.NORMAL;

        //# 보호펜
        public Mat oriMat;  //# 원본 이미지
        public Mat revMat;  //# 역상 이미지

        public Color penColor;
        public int penSize;
        public LineCap gFreeCap;

        public List<Color> multiSpray;

        private bool isPenMenuOpen = false;

        private List<Color> allColor;               //Canvas에 있는 색을 담을 List
        private List<Panel> pnl_allColor;           //ColorChip
        private int clrChipW = 23;
        private int clrChipH = 12;

        public List<GraphicsPath> kasuriPaths;
        public List<List<PointF>> curvePts;
        public PointF[,] AllPoints;
        public GraphicsPath currKasuriPath;
        public List<PointF> currCurvePts;
        public List<PointF> KPointList;

        public int branchCount;
        public int pointCount;
        public int twigCount;
        public int allBranchCount;

        private bool isMove = false;
        public int iIdx;
        public int jIdx;

        private Point start, end;
        private int nowCount;
        private double dDist;
        private PointF[] twigPoints;

        public KasuriPen mKasuri;

        Bitmap bit_color;
        Color currentColor;
        MultiColor multi;

        public enum PenShape
        {
            NORMAL, AIRBRUSH, CRAYON, WATER, SPATTER, PURE, DIFFUSE, BLUR, SCRATCH, KASURI, MASK, ERASER
        }

        //private bool mRound = true;
        private Canvas mCanvas;

        public PenManager(Canvas canvas)
        {
            InitializeComponent();
            mCanvas = canvas;
            mCanvas.mPen = this;
            /* set default value */
            pnl_outline.BackColor = Color.White;
            nud_size.Value = 5;
            penColor = Color.Black;
            penSize = (int)nud_size.Value;

            allColor = new List<Color>();
            pnl_allColor = new List<Panel>();
            multiSpray = new List<Color>();
            kasuriPaths = new List<GraphicsPath>();         //가지 Point 를 커브로 담은 리스트
            curvePts = new List<List<PointF>>();            //가지 Point 만 담은 리스트
            currKasuriPath = new GraphicsPath();            //현재 그리고 있는 Path(Free Drawing)
            KPointList = new List<PointF>();

            dDist = 0.2f;
            nowCount = 0;
            branchCount = (int)nud_branch.Value;
            pointCount = (int)nud_point.Value;
            twigCount = int.Parse(txt_twigs.Text);

            oriMat = new Mat();
            revMat = new Mat();

            mKasuri = new KasuriPen(this);
        }

        private void nud_size_ValueChanged(object sender, EventArgs e)
        {
            penSize = (int)nud_size.Value;
            trb_diameter.Value = penSize;
            pnl_outline.Refresh();
            mCanvas.SetCursor();
        }

        public void pen_MouseDown(MouseEventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.MouseDown(mCanvas.dst, mCanvas.mWorkarea.totalPath, pnl_color, e);
                mCanvas.drawing = true;
            }
        }

        public void pen_MouseMove(MouseEventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.MouseMove(e);
                mCanvas.PictureBox1.Refresh();
            }
            else
            {
                Mat tempMat = new Mat();
                Random rand = new Random();

                Scalar penScalar = new Scalar(penColor.B, penColor.G, penColor.R, penColor.A);

                switch (penMode)
                {
                    case PenShape.AIRBRUSH:
                        tempMat = new Mat(new cvSize((int)nud_spray.Value, (int)nud_spray.Value), MatType.CV_8UC4, 0);
                        Cv2.Circle(tempMat, (int)nud_spray.Value / 2, (int)nud_spray.Value / 2, (int)nud_spray.Value / 2, Scalar.White, -1, LineTypes.Link8);
                        List<cvPoint> ptList = new List<cvPoint>();
                        for (int i = 0; i < (int)nud_spray.Value; i++)
                        {
                            for (int j = 0; j < (int)nud_spray.Value; j++)
                            {
                                Color c = Color.FromArgb(tempMat.At<int>(i, j));
                                if (c.R == 255 && c.G == 255 && c.B == 255)
                                {
                                    cvPoint pt = new cvPoint(e.Y - (int)nud_spray.Value / 2 + j, e.X - (int)nud_spray.Value / 2 + i);
                                    /*
                                    int radiusX = Math.Abs(i - (int)nud_spray.Value / 2);
                                    int radiusY = Math.Abs(j - (int)nud_spray.Value / 2);
                                    int spreadDegree;
                                    if (radiusX != 0 && radiusY != 0)
                                    {
                                        spreadDegree = ((int)nud_spray.Value / 2 * (int)nud_spray.Value / 2) / (radiusX * radiusY);
                                    }
                                    else
                                    {
                                        continue;
                                    }
                                    */
                                    if (rand.Next(0, (int)nud_density.Maximum - (int)nud_density.Value + 1) == 0)
                                    {
                                        ptList.Add(pt);
                                    }
                                    else
                                    {
                                        mCanvas.dst.Set<Vec3b>(pt.X, pt.Y, mCanvas.dst.At<Vec3b>(pt.X, pt.Y));
                                    }
                                }
                            }
                        }

                        if (rdo_single.Checked)
                        {

                            for (int i = 0; i < ptList.Count; i++)
                            {
                                float irr = rand.Next(1, (int)nud_irregularity.Value + 1);
                                float nozz = (int)nud_dropsize.Value;
                                Scalar s = new Scalar(penColor.B, penColor.G, penColor.R, penColor.A);
                                Cv2.Circle(mCanvas.dst, ptList[i].Y, ptList[i].X, (int)(irr / 2 * nozz / 2), s, -1, LineTypes.Link8);
                            }
                        }
                        else if (rdo_multi.Checked)
                        {
                            for (int i = 0; i < ptList.Count; i++)
                            {
                                float irr = rand.Next(1, (int)nud_irregularity.Value + 1);
                                float nozz = (int)nud_dropsize.Value;
                                Color multi = multiSpray[rand.Next(0, multiSpray.Count)];
                                Scalar s = new Scalar(multi.B, multi.G, multi.R, multi.A);
                                Cv2.Circle(mCanvas.dst, new cvPoint(ptList[i].Y, ptList[i].X), (int)(irr / 2 * nozz / 2), s, -1, LineTypes.Link8);
                            }
                        }
                        break;
                    case PenShape.CRAYON:
                        tempMat = new Mat(new cvSize(penSize, penSize), MatType.CV_8UC4, 0);
                        Cv2.Circle(tempMat, penSize / 2, penSize / 2, penSize / 2, Scalar.White, -1, LineTypes.Link8);
                        for (int i = 0; i < penSize - 1; i++)
                        {
                            for (int j = 0; j < penSize - 1; j++)
                            {
                                Color c = Color.FromArgb(tempMat.At<int>(i, j));
                                if (c.R == 255 && c.G == 255 && c.B == 255)
                                {
                                    cvPoint pt = new cvPoint(e.Y - penSize / 2 + j, e.X - penSize / 2 + i);
                                    if (rand.Next(0, (int)(1 / (0.05 * (float)(nud_density2.Value)))) == 0)
                                    {
                                        mCanvas.dst.Set<Vec3b>(pt.X, pt.Y, penScalar.ToVec3b());
                                    }
                                    else
                                    {
                                        mCanvas.dst.Set<Vec3b>(pt.X, pt.Y, mCanvas.dst.At<Vec3b>(pt.X, pt.Y));
                                    }
                                }
                            }
                        }
                        break;
                }

                mCanvas.PictureBox1.ImageIpl = mCanvas.dst;
            }
        }

        public void pen_MouseUp(MouseEventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.MouseUp(mCanvas.dst, pnl_color, e);
                mCanvas.drawing = false;
                mCanvas.PictureBox1.Refresh();
            }
        }

        public void pen_Paint(PaintEventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.Paint(e);
            }
        }

        private void PenManager_Paint(object sender, PaintEventArgs e)
        {

        }

        private void pnl_outline_Paint(object sender, PaintEventArgs e)
        {
            int x = (pnl_outline.Width - 2) / 2 - (int)nud_size.Value / 2;
            int y = (pnl_outline.Height - 2) / 2 - (int)nud_size.Value / 2;

            penColor = mCanvas.mPalette.color;
            e.Graphics.FillEllipse(new SolidBrush(penColor), new Rectangle(x, y, (int)nud_size.Value, (int)nud_size.Value));
            gFreeCap = LineCap.Round;

            pnl_outline.BackColor = mCanvas.gBackColor;
        }

        private void trb_diameter_Scroll(object sender, EventArgs e)
        {
            nud_size.Value = trb_diameter.Value;
        }

        private void btn_normal_Click(object sender, EventArgs e)
        {
            penMode = PenShape.NORMAL;
            UpdateDisplay(penMode);

            //NormalPenInit();
        }

        private void btn_airbrush_Click(object sender, EventArgs e)
        {
            penMode = PenShape.AIRBRUSH;
            UpdateDisplay(penMode);
            //AirbrushPenInit();
        }

        private void btn_crayon_Click(object sender, EventArgs e)
        {
            penMode = PenShape.CRAYON;
            UpdateDisplay(penMode);

            //CrayonPenInit();
        }

        private void btn_water_Click(object sender, EventArgs e)
        {
            penMode = PenShape.WATER;
            UpdateDisplay(penMode);

            //WaterPenInit();
        }

        private void btn_scratch_Click(object sender, EventArgs e)
        {
            cms_scratch.Visible = true;
            cms_scratch.Left = this.Left + btn_scratch.Left + (int)(btn_scratch.Width / 2);
            cms_scratch.Top = this.Top + btn_scratch.Top + (int)(btn_scratch.Top / 2);
        }

        private void btn_mask_Click(object sender, EventArgs e)
        {
            penMode = PenShape.MASK;
            UpdateDisplay(penMode);
            oriMat = mCanvas.PictureBox1.ImageIpl.Clone();

            if (mCanvas.mask != null)
            {
                //# mask영역과 toggle 영역은 MouseMove 때마다 변경되므로 때마다 계산해준다. (MouseMove시 최소 1번의 Cv2~함수 사용)
                Mat toggle = new Mat();
                Cv2.BitwiseNot(mCanvas.mask, toggle);
                oriMat.CopyTo(mCanvas.protect, toggle);

                //# 보호 영역 역상
                revMat = oriMat.Clone();
                Cv2.BitwiseNot(revMat, revMat);

                Mat tempMat = revMat.Clone();
                tempMat.SetTo(new Scalar(0, 0, 0, 255));
                revMat += tempMat;
                revMat.CopyTo(revMat, mCanvas.mask);

                mCanvas.dst = oriMat.Clone();
                mCanvas.dst.SetTo(new Scalar(0, 0, 0, 255), mCanvas.mask);
                mCanvas.dst = mCanvas.dst + revMat;
            }

        }

        private void kasuriToolStripMenuItem_Click(object sender, EventArgs e)
        {
            penMode = PenShape.KASURI;
            UpdateDisplay(penMode);

            //KasuriPenInit();
        }

        private void scratchToolStripMenuItem_Click(object sender, EventArgs e)
        {
            penMode = PenShape.SCRATCH;
            UpdateDisplay(penMode);

            //ScratchPenInit();
        }

        private void btn_open_Click(object sender, EventArgs e)
        {
            if (isPenMenuOpen)
            {
                this.Height = pnl_pen.Top + pnl_pen.Height + 38;
                btn_open.BackgroundImage = img_topdown.Images[0];
                isPenMenuOpen = false;
            }
            else
            {
                UpdateDisplay(penMode);
                btn_open.BackgroundImage = img_topdown.Images[1];
            }
        }

        private void btn_all_Click(object sender, EventArgs e)
        {
            if (chk_select.Checked)
            {
                Bitmap bt = (Bitmap)mCanvas.PictureBox1.Image;
                Color temp_c;
                Panel temp_p = new Panel();
                temp_p.Visible = false;
                temp_p.Width = clrChipW;
                temp_p.Height = clrChipH;
                temp_p.BorderStyle = BorderStyle.FixedSingle;
                temp_p.Parent = pnl_color;

                pnl_allColor.Clear();
                for (int i = 0; i < bt.Height; i++)
                {
                    for (int j = 0; j < bt.Width; j++)
                    {
                        temp_c = bt.GetPixel(j, i);
                        if (!allColor.Contains(temp_c))
                        {
                            allColor.Add(temp_c);
                            pnl_allColor.Add(temp_p);
                        }
                    }
                }

                DrawColorChip();
            }
        }

        private void DrawColorChip()
        {
            int row = -1;
            for (int i = 0; i < pnl_allColor.Count; i++)
            {
                if (i % 3 == 0) row++;

                pnl_allColor[i].BackColor = allColor[i];
                pnl_allColor[i].Top = row * (clrChipH + 3) + 1;
                pnl_allColor[i].Left = (i % 3) * (clrChipW + 4) + 1;
                pnl_allColor[i].Visible = true;
            }

            for (int i = 0; i < pnl_allColor.Count; i++)
            {
                Console.WriteLine(i + " =  " + pnl_allColor[i].Left + ", " + pnl_allColor[i].Top);
                Console.WriteLine(pnl_allColor[i].BackColor);
            }
        }

        public void UpdateDisplay(PenShape pen)
        {
            List<Panel> pnlList = new List<Panel>();
            pnlList.Add(pnl_pen);

            nud_size.Enabled = true;
            switch (pen)
            {
                case PenShape.NORMAL:
                    pnlList.Add(pnl_etc);
                    break;
                case PenShape.AIRBRUSH:
                    pnlList.Add(pnl_airbrush);
                    nud_size.Enabled = false;
                    break;
                case PenShape.CRAYON:
                    pnlList.Add(pnl_etc);
                    pnlList.Add(pnl_density);
                    break;
                case PenShape.WATER:
                case PenShape.PURE:
                    pnlList.Add(pnl_water);
                    pnlList.Add(pnl_setting);
                    lbl_color.Text = "색";
                    lbl_effect.Text = "효과";
                    break;
                case PenShape.DIFFUSE:
                    pnlList.Add(pnl_water);
                    pnlList.Add(pnl_setting);
                    lbl_color.Text = "색";
                    lbl_effect.Text = "모양";
                    break;
                case PenShape.SPATTER:
                    pnlList.Add(pnl_water);
                    pnlList.Add(pnl_setting);
                    lbl_color.Text = "내부밀도";
                    lbl_effect.Text = "외부밀도";
                    pnlList.Add(pnl_density);
                    break;
                case PenShape.BLUR:
                    pnlList.Add(pnl_density);
                    break;
                case PenShape.KASURI:
                    pnlList.Add(pnl_kasuri);
                    break;
                case PenShape.SCRATCH:
                    pnlList.Add(pnl_etc);
                    pnlList.Add(pnl_scrape);
                    break;
                case PenShape.MASK:
                    pnlList.Add(pnl_mask);
                    break;
            }

            if (pnlList.Count > 0)
            {
                int top = 0;
                for (int i = 0; i < pnlList.Count; i++)
                {
                    pnlList[i].Visible = true;
                    pnlList[i].Location = new Point(0, top);
                    pnlList[i].BringToFront();
                    top += pnlList[i].Height;
                }

                this.Height = pnlList[pnlList.Count - 1].Top + pnlList[pnlList.Count - 1].Height + 38;  //# 38 - Topbar Height
            }

            isPenMenuOpen = true;
            this.Refresh();

        }

        private void btn_simple_Click(object sender, EventArgs e)
        {
            //# 수채화
            penMode = PenShape.WATER;
            UpdateDisplay(penMode);
        }

        private void btn_spatter_Click(object sender, EventArgs e)
        {
            //# 물방울
            penMode = PenShape.SPATTER;
            UpdateDisplay(penMode);
        }

        private void btn_pure_Click(object sender, EventArgs e)
        {
            //# 물혼합
            penMode = PenShape.PURE;
            UpdateDisplay(penMode);
        }

        private void btn_diffuse_Click(object sender, EventArgs e)
        {
            //# 번지기
            penMode = PenShape.DIFFUSE;
            UpdateDisplay(penMode);
        }

        private void btn_avg_Click(object sender, EventArgs e)
        {
            //# 흐리기
            penMode = PenShape.BLUR;
            UpdateDisplay(penMode);
        }

        private void pic_color_Paint(object sender, PaintEventArgs e)
        {
            bit_color = new Bitmap(pic_color.Width, pic_color.Height);
            Graphics gr = Graphics.FromImage(bit_color);
            if (rdo_single.Checked)
            {
                gr.FillRectangle(new SolidBrush(mCanvas.mPalette.color), new RectangleF(0, 0, pic_color.Width, pic_color.Height));
            }
            else if (rdo_multi.Checked)
            {
                for (int i = 0; i < multiSpray.Count; i++)
                {
                    gr.FillRectangle(new SolidBrush(multiSpray[i]), new RectangleF(i * (float)pic_color.Width / multiSpray.Count, 0, (float)pic_color.Width / multiSpray.Count, pic_color.Height));
                }
            }

            pic_color.Image = bit_color;
        }

        private void txt_scale_KeyDown(object sender, KeyEventArgs e)
        {
            if (!int.TryParse(txt_scale.Text, out int scale))
            {
                MessageBox.Show("정수값을 입력하세요!");
            }
        }

        private void PenManager_Activated(object sender, EventArgs e)
        {
            MainMenu.mode = MainMode.DRAW;
            mCanvas.tsl_mode.Text = MainMenu.mode.ToString().ToLower();
            mCanvas.mDrawObject = CanvasType.BITMAP;
            if (mCanvas.mDraw != null)
            {
                mCanvas.mDraw.shape = TShape.PATH;
            }
        }

        private void btn_toggle_Click(object sender, EventArgs e)
        {
            Mat oriMat = mCanvas.mask.Clone();
            Cv2.BitwiseNot(oriMat, mCanvas.mask);
            mCanvas.DrawMask();

            mCanvas.dst.SetTo(new Scalar(0, 0, 0, 255), oriMat);
            mCanvas.dst = mCanvas.dst + mCanvas.protect;
            mCanvas.PictureBox1.ImageIpl = mCanvas.dst;

            mCanvas.protect = this.oriMat.Clone();
            mCanvas.protect.SetTo(0, oriMat);
        }

        private void trb_twigs_ValueChanged(object sender, EventArgs e)
        {
            txt_twigs.Text = trb_twigs.Value.ToString();
            twigCount = trb_twigs.Value;
            CalculatePoints();

            mCanvas.drawing = true;
            mCanvas.PictureBox1.Refresh();
            mCanvas.drawing = false;
        }

        private void trb_distance_ValueChanged(object sender, EventArgs e)
        {
            txt_distance.Text = trb_distance.Value.ToString();
            SetDistance(trb_distance.Value);
            CalculatePoints();

            mCanvas.drawing = true;
            mCanvas.PictureBox1.Refresh();
            mCanvas.drawing = false;
        }

        private void txt_twigs_KeyDown(object sender, KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.Enter:
                    if (int.TryParse(txt_twigs.Text, out int result))
                    {
                        if (result < trb_twigs.Minimum || result > trb_twigs.Maximum)
                        {
                            MessageBox.Show("Out of Range(1 ~ " + trb_twigs.Maximum.ToString() + ")");
                        }
                        else
                        {
                            trb_twigs.Value = result;
                            twigCount = result;
                            CalculatePoints();

                            mCanvas.drawing = true;
                            mCanvas.PictureBox1.Refresh();
                            mCanvas.drawing = false;
                        }
                    }
                    else
                    {
                        MessageBox.Show("Input Number!");
                    }
                    break;
            }
        }

        private void txt_distance_KeyDown(object sender, KeyEventArgs e)
        {
            switch (e.KeyCode)
            {
                case Keys.Enter:
                    if (int.TryParse(txt_distance.Text, out int result))
                    {
                        if (result < trb_distance.Minimum || result > trb_distance.Maximum)
                        {
                            MessageBox.Show("Out of Range(1 ~ 5)");
                        }
                        else
                        {
                            trb_distance.Value = result;
                            SetDistance(result);
                            CalculatePoints();

                            mCanvas.drawing = true;
                            mCanvas.PictureBox1.Refresh();
                            mCanvas.drawing = false;
                        }
                    }
                    else
                    {
                        MessageBox.Show("Input Number!");
                    }
                    break;
            }
        }

        public void Kasuri_MouseDown(MouseEventArgs e)
        {
            mCanvas.drawing = true;
            if (kasuriPaths.Count < nud_branch.Value)//가지를 더 그려야 할 때
            {
                start = e.Location;
                currKasuriPath = new GraphicsPath();
                currCurvePts = new List<PointF>();
            }
            else//가지를 Count 만큼 다 그렸을 때 포인트 이동
            {
                for (int i = 0; i < curvePts.Count; i++)
                {
                    for (int j = 0; j < curvePts[i].Count; j++)
                    {
                        int size = mCanvas.mVector.handleSize;
                        RectangleF ptRect = new RectangleF(curvePts[i][j].X - size, curvePts[i][j].Y - size, size * 2, size * 2);
                        if (ptRect.Contains(e.Location))
                        {
                            isMove = true;
                            iIdx = i * (twigCount + 1);
                            jIdx = j;
                            mCanvas.drawing = true;
                        }
                    }
                }
            }
        }

        public void Kasuri_MouseMove(MouseEventArgs e)
        {
            if (kasuriPaths.Count < nud_branch.Value)//가지를 더 그려야 할 때
            {
                end = e.Location;
                currKasuriPath.AddLine(start, end);
                start = e.Location;

                mCanvas.PictureBox1.Refresh();
            }
            else//가지를 Count 만큼 다 그렸을 때 포인트 이동
            {
                if (isMove)
                {
                    AllPoints[iIdx, jIdx] = new PointF(e.Location.X, e.Location.Y);
                    curvePts[iIdx / (twigCount + 1)][jIdx] = new PointF(e.Location.X, e.Location.Y);

                    mCanvas.PictureBox1.Refresh();
                }
            }
        }

        public void Kasuri_MouseUp(MouseEventArgs e)
        {
            if (kasuriPaths.Count < nud_branch.Value)//그릴 가지가 남았을 때
            {
                end = e.Location;
                currKasuriPath.AddLine(start, end);
                if (KasuriPathPointCalculate((int)nud_point.Value - 2))
                {
                    AddPoints();

                    GraphicsPath temp = new GraphicsPath();
                    temp.AddCurve(currCurvePts.ToArray());
                    curvePts.Add(currCurvePts.ToList());
                    kasuriPaths.Add((GraphicsPath)temp.Clone());

                    currKasuriPath = null;
                    mCanvas.PictureBox1.Refresh();
                }
            }
            else //가지 다 그리고 포인트 이동
            {
                isMove = false;

                AllPoints[iIdx, jIdx] = new PointF(e.Location.X, e.Location.Y);

                //curvePts = KPOintList
                curvePts[iIdx / (twigCount + 1)][jIdx] = new PointF(e.Location.X, e.Location.Y);
                KPointList[iIdx / (twigCount + 1) * 3 + jIdx] = new PointF(e.Location.X, e.Location.Y);
                CalculatePoints();

                mCanvas.PictureBox1.Refresh();
            }
            mCanvas.drawing = false;
        }

        private void nud_point_ValueChanged(object sender, EventArgs e)
        {
            KasuriDataInit();
        }

        private void nud_branch_ValueChanged(object sender, EventArgs e)
        {
            KasuriDataInit();
        }
        
        private void btn_cancel_Click(object sender, EventArgs e)
        {
            KasuriDataInit();
        }

        private void KasuriDataInit()
        {
            kasuriPaths.Clear();
            curvePts.Clear();
            KPointList.Clear();
            currCurvePts = null;
            currKasuriPath = null;
            nowCount = 0;

            pointCount = (int)nud_point.Value;
            branchCount = (int)nud_branch.Value;

            mCanvas.drawing = true;
            mCanvas.PictureBox1.Refresh();
            mCanvas.drawing = false;
        }

        private bool KasuriPathPointCalculate(int nPointCount)
        {
            //그린 곡선의 길이에 따라 포인트 개수대로 나눔
            int nDistance = 0;
            int nTempDistance;
            int nBeforeTemp, nNowTemp;
            PointF TempPoint, TempPoint1, TempPoint2;
            PointF CorrectPoint;
            PointF pBeforePoint = new PointF();
            PointF pNowPoint = new PointF();
            currCurvePts = new List<PointF>();

            if (currKasuriPath.PointCount < 10)
            {
                currKasuriPath = null;
                return false;
            }

            TempPoint1 = currKasuriPath.PathPoints[0];
            PointF first = new PointF();
            first.X = TempPoint1.X;
            first.Y = TempPoint1.Y;
            currCurvePts.Add(first);  //맨 첫 점을 넣어준다

            TempPoint2 = currKasuriPath.PathPoints[1];
            pNowPoint.X = TempPoint2.X;
            pNowPoint.Y = TempPoint2.Y;

            for (int i = 2; i < currKasuriPath.PointCount - 1; i++)
            {
                TempPoint = currKasuriPath.PathPoints[i];
                pBeforePoint.X = pNowPoint.X;
                pBeforePoint.Y = pNowPoint.Y;
                pNowPoint.X = TempPoint.X;
                pNowPoint.Y = TempPoint.Y;
                nDistance += (int)Math.Sqrt((pNowPoint.X - pBeforePoint.X) * (pNowPoint.X - pBeforePoint.X) + (pNowPoint.Y - pBeforePoint.Y) * (pNowPoint.Y - pBeforePoint.Y));
            }

            for (int k = 0; k < nPointCount; k++)
            {
                CorrectPoint = new PointF();
                nTempDistance = 0;
                TempPoint = currKasuriPath.PathPoints[1];
                pNowPoint.X = TempPoint.X;
                pNowPoint.Y = TempPoint.Y;

                nNowTemp = ((k + 1) * (nDistance / (nPointCount + 1)));

                for (int i = 2; i < currKasuriPath.PointCount - 1; i++)
                {
                    TempPoint = currKasuriPath.PathPoints[i];
                    pBeforePoint.X = pNowPoint.X;
                    pBeforePoint.Y = pNowPoint.Y;
                    pNowPoint.X = TempPoint.X;
                    pNowPoint.Y = TempPoint.Y;
                    nTempDistance += (int)Math.Sqrt((pNowPoint.X - pBeforePoint.X) * (pNowPoint.X - pBeforePoint.X) + (pNowPoint.Y - pBeforePoint.Y) * (pNowPoint.Y - pBeforePoint.Y));
                    nBeforeTemp = nNowTemp;
                    nNowTemp = Math.Abs(((k + 1) * (nDistance / (nPointCount + 1))) - nTempDistance);
                    if (nNowTemp < nBeforeTemp)
                    {
                        CorrectPoint.X = pNowPoint.X;
                        CorrectPoint.Y = pNowPoint.Y;
                    }
                }
                currCurvePts.Add(CorrectPoint);
            }

            TempPoint = currKasuriPath.PathPoints[currKasuriPath.PointCount - 1];
            PointF last = new PointF();
            last.X = TempPoint.X;
            last.Y = TempPoint.Y;
            currCurvePts.Add(last);  //맨 마지막 점을 넣어준다

            return true;
        }

        private void CalculateSegment(int nBranchNum)
        {
            twigPoints = new PointF[twigCount];

            PointF pBranchS, pBranchE;
            pBranchS = new PointF();
            pBranchE = new PointF();

            for (int order = 0; order < pointCount; order++)
            {
                for (int j = 0; j < twigCount; j++)
                {
                    twigPoints[j].X = 0;
                    twigPoints[j].Y = 0;
                }
                FindBranchPoints(nBranchNum, order, ref pBranchS, ref pBranchE);
                if (order == 0)
                {
                    int tempx, tempy;
                    int count = twigCount + 1;
                    int sx, sy, ex, ey;

                    sx = (int)pBranchS.X;
                    sy = (int)pBranchS.Y;
                    ex = (int)pBranchE.X;
                    ey = (int)pBranchE.Y;
                    int m = 1;
                    int n = count - m;

                    for (int k = 0; k < twigCount; k++)
                    {
                        tempx = (sx * n + ex * m) / (m + n);
                        tempy = (sy * n + ey * m) / (m + n);
                        twigPoints[k].X = tempx;
                        twigPoints[k].Y = tempy;

                        m++;
                        n--;
                        if (m < 1 || n < 1) break;
                    }
                }
                else
                {
                    CalculateSegmentTwigPositions(nBranchNum, ref pBranchS, ref pBranchE, ref twigPoints);  // 사이마다 위치할 점들을 계산한다.
                    for (int i = 0; i < twigCount; i++)
                    {
                        if (i != 0)
                        {
                            if (twigPoints[i].X == 0 && twigPoints[i].Y == 0)
                            {
                                twigPoints[i].X = twigPoints[i - 1].X;
                                twigPoints[i].Y = twigPoints[i - 1].Y;
                            }
                        }
                    }
                }
                AddSegmentTwigPoints(nBranchNum, order, twigPoints);
            }
        }

        private void AddSegmentTwigPoints(int nBranchNum, int nOrder, PointF[] twigPoints)
        {
            for (int i = 0; i < twigCount; i++)
            {
                AllPoints[nBranchNum * (twigCount + 1) + i + 1, nOrder].X = twigPoints[i].X;
                AllPoints[nBranchNum * (twigCount + 1) + i + 1, nOrder].Y = twigPoints[i].Y;
            }
        }

        private void FindBranchPoints(int nBranchNum, int nOrder, ref PointF pS, ref PointF pE)
        {
            PointF pointS, pointE;

            pointS = KPointList[pointCount * nBranchNum + nOrder];
            pointE = KPointList[pointCount * (nBranchNum + 1) + nOrder];

            pS.X = pointS.X;
            pS.Y = pointS.Y;
            pE.X = pointE.X;
            pE.Y = pointE.Y;
        }

        private void SetDistance(int Value)
        {
            dDist = (double)Value / 10;
            if (dDist <= 0) dDist = 0.05;
        }

        private void GetMiddlePoint(ref PointF pointS, ref PointF pointE, ref PointF pMiddle, int nNum, double dBend)
        {
            double m = dBend * 10;
            double n = (1 - dBend) * 10;

            int startx, starty;
            int tempx, tempy;

            //CalculatePoints();//추가
            startx = (int)((AllPoints[nNum, 0].X + AllPoints[nNum + 1, 0].X) / 2);
            starty = (int)((AllPoints[nNum, 0].Y + AllPoints[nNum + 1, 0].Y) / 2);

            tempx = (int)((pointS.X + pointE.X) / 2);
            tempy = (int)((pointS.Y + pointE.Y) / 2);

            if (m == 0)
            {
                pMiddle.X = tempx;
                pMiddle.Y = tempy;
            }
            else
            {
                pMiddle.X = (float)((tempx * n + startx * m) / (m + n));
                pMiddle.Y = (float)((tempy * n + starty * m) / (m + n));
            }
        }

        private void CalculatePoints()
        {
            int oldBranchCount;
            allBranchCount = branchCount + twigCount * (branchCount - 1);

            if (branchCount > 1 && KPointList.Count > 0)
            {
                PointF point;

                oldBranchCount = allBranchCount;
                AllPoints = new PointF[allBranchCount, pointCount];

                //branch넣기.
                int count = 0;
                for (int y = 0; y < allBranchCount; y += (twigCount + 1))
                {
                    for (int x = 0; x < pointCount; x++)
                    {
                        point = KPointList[count];
                        AllPoints[y, x].X = point.X;
                        AllPoints[y, x].Y = point.Y;
                        count++;
                    }
                }

                //twig 넣기
                for (int i = 0; i < branchCount - 1; i++)
                {
                    CalculateSegment(i);
                }
            }
        }

        private void AddPoints()
        {
            int pCount = currCurvePts.Count;

            for (int i = 0; i < pCount; i++)
            {
                KPointList.Add(currCurvePts[i]);
            }

            nowCount += pCount;
            if (nowCount == pointCount * branchCount)
            {
                ChangeKasuriPoints(true);
                if (branchCount > 1)
                {
                    int maxtwigs = CalMaxTwigs();

                    trb_twigs.Maximum = maxtwigs;
                    if (maxtwigs < twigCount)
                    {
                        trb_twigs.Value = trb_twigs.Maximum;
                        txt_twigs.Text = trb_twigs.Maximum.ToString();
                    }
                    else
                    {
                        trb_twigs.Value = twigCount;
                    }
                }
            }
        }

        private int CalMaxTwigs()
        {
            int result = 0;
            int diff = 10000;
            int finda = 0, findb = 0;
            List<PointF> BranchPointList = new List<PointF>();
            for (int i = 1; i < pointCount; i++)
            {
                MakeSameOrderBranchPointList(i, BranchPointList); //각 branch마다 같은 위치에 있는 점들을 가져온다

                //점사이의 거리가 가장 좁은 곳을 찾는다
                for (int j = 0; j < BranchPointList.Count - 1; j++)
                {
                    PointF point = BranchPointList[j];
                    PointF pointn = BranchPointList[j + 1];
                    int temp = (int)Math.Sqrt((point.X - pointn.X) * (point.X - pointn.X) + (point.Y - pointn.Y) * (point.Y - pointn.Y));
                    if (temp < diff)
                    {
                        diff = temp;
                        finda = i;
                        findb = j;
                    }
                }
            }

            result = (int)(diff * 0.3);
            return result;
        }

        private void MakeSameOrderBranchPointList(int nOrder, List<PointF> BranchPointList)
        {
            PointF point;
            int count = 0;
            if (KPointList != null && KPointList.Count == pointCount * branchCount)
            {
                for (int i = nOrder; i < KPointList.Count; i += pointCount)
                {
                    point = KPointList[i];
                    PointF branchPoint = new PointF();
                    branchPoint.X = point.X;
                    branchPoint.Y = point.Y;
                    BranchPointList.Add(branchPoint);
                    count++;
                }
            }
        }

        private void ChangeKasuriPoints(bool IsCrayonChange)
        {
            CalculatePoints();
        }

        private void btn_penShape_Click(object sender, EventArgs e)
        {
            if (mKasuri.Visible)
            {
                mKasuri.Visible = false;
                mKasuri.Hide();
            }
            else
            {
                mKasuri.Visible = true;
                mKasuri.Show();
            }
        }

        private void trb_density_Scroll(object sender, EventArgs e)
        {
            nud_density.Value = trb_density.Value;
        }

        private void nud_density_ValueChanged(object sender, EventArgs e)
        {
            trb_density.Value = (int)nud_density.Value;
        }

        private void trb_irregularity_Scroll(object sender, EventArgs e)
        {
            nud_irregularity.Value = trb_irregularity.Value;
        }

        private void nud_irregularity_ValueChanged(object sender, EventArgs e)
        {
            trb_irregularity.Value = (int)nud_irregularity.Value;
        }

        private void trb_dropsize_Scroll(object sender, EventArgs e)
        {
            nud_dropsize.Value = trb_dropsize.Value;
        }

        private void nud_dropsize_ValueChanged(object sender, EventArgs e)
        {
            trb_dropsize.Value = (int)nud_dropsize.Value;
        }

        private void rdo_single_CheckedChanged(object sender, EventArgs e)
        {
            pic_color.Refresh();
        }

        private void tsm_delete_Click(object sender, EventArgs e)
        {
            multiSpray.Remove(currentColor);
            pic_color.Refresh();
        }

        private void tsm_clear_Click(object sender, EventArgs e)
        {
            multiSpray.Clear();
            pic_color.Refresh();
        }

        private void tsm_reset_Click(object sender, EventArgs e)
        {

        }

        private void pic_color_MouseClick(object sender, MouseEventArgs e)
        {
            if (e.Button == MouseButtons.Right)
            {
                if (rdo_multi.Checked)
                {
                    cms_revise.Show(MousePosition);
                    currentColor = bit_color.GetPixel(e.X, e.Y);
                }
            }
        }

        private void btn_clear_Click(object sender, EventArgs e)
        {
            mCanvas.mask.SetTo(0);
            mCanvas.DrawMask();

            mCanvas.PictureBox1.ImageIpl = mCanvas.dst;
        }

        private void CalculateSegmentTwigPositions(int nBranchNum, ref PointF pointS, ref PointF pointE, ref PointF[] twigPoints)
        {
            int sx, mx, ex, sy, my, ey;
            int Lsx, Lsy;
            int Lex = 0, Ley = 0;
            double d1, d2, d3, a, b;
            double ct, st, ss, theta, alpha;
            double txs, tys, txe, tye, x, y, count;
            PointF pointC;

            pointC = new PointF();
            GetMiddlePoint(ref pointS, ref pointE, ref pointC, nBranchNum, dDist);

            sx = (int)pointS.X;
            sy = (int)pointS.Y;
            mx = (int)pointC.X;
            my = (int)pointC.Y;
            ex = (int)pointE.X;
            ey = (int)pointE.Y;

            d1 = Math.Sqrt((double)(sx - mx) * (sx - mx) + (sy - my) * (sy - my));
            d2 = Math.Sqrt((double)(ex - mx) * (ex - mx) + (ey - my) * (ey - my));
            d3 = Math.Sqrt((double)(ex - sx) * (ex - sx) + (ey - sy) * (ey - sy));

            if (d1 > d2)
            {
                a = d1; b = d2;
            }
            else
            {
                a = d2; b = d1;
            }

            if ((a > b * 1.8) && (d3 < a))
            {
                d1 = Math.Sqrt(d1); d2 = Math.Sqrt(d2);
            }

            txs = (ex * d1 + sx * d2) / (d1 + d2);
            tys = (ey * d1 + sy * d2) / (d1 + d2);

            if (txs == mx) ss = tys - my;
            else ss = (double)(tys - my) / (txs - mx);

            theta = Math.PI / 2.0 - Math.Atan(ss);
            ct = Math.Cos(theta);
            st = Math.Sin(theta);

            txs = ct * (sx - mx) - st * (sy - my);
            tys = st * (sx - mx) + ct * (sy - my);
            txe = ct * (ex - mx) - st * (ey - my);
            tye = st * (ex - mx) + ct * (ey - my);

            alpha = tys / (txs * txs);
            if (twigCount % 2 == 1)
            {
                count = Math.Abs(txs) / (twigCount / 2 + 1);
            }
            else
            {
                count = Math.Abs(txs) / (twigCount + 1);
            }
            if (count == 0)
            {
                count = 1;
            }

            Lsx = mx; Lsy = my;
            int tempcount;
            if (txs < 0)
            {
                x = 0;
                bool first = true;
                tempcount = twigCount / 2 - 1;
                while (x >= txs)
                {
                    y = alpha * (x * x);
                    Lex = (int)(ct * x + st * y + mx);
                    Ley = (int)(ct * y - st * x + my);
                    if ((Lex != sx || Ley != sy) && (Lex != mx || Ley != my))
                    {
                        twigPoints[tempcount].X = Lex;
                        twigPoints[tempcount].Y = Ley;
                        tempcount--;
                    }
                    Lsx = Lex;
                    Lsy = Ley;
                    if (first || twigCount % 2 == 1)
                    {
                        x = x - count;
                        first = false;
                    }
                    else
                    {
                        x = x - count * 2;
                    }
                    if (tempcount < 0) break;
                }
                if (tempcount >= 0)
                {
                    for (int j = tempcount; j >= 0; j--)
                    {
                        twigPoints[j].X = Lex;
                        twigPoints[j].Y = Ley;
                    }
                }

            }
            else
            {
                x = 0;
                tempcount = twigCount / 2 - 1;
                bool first = true;
                while (x <= txs)
                {
                    y = alpha * (x * x);
                    Lex = (int)(ct * x + st * y + mx);
                    Ley = (int)(ct * y - st * x + my);
                    if ((Lex != sx || Ley != sy) && (Lex != mx || Ley != my))
                    {
                        twigPoints[tempcount].X = Lex;
                        twigPoints[tempcount].Y = Ley;
                        tempcount--;
                    }
                    Lsx = Lex;
                    Lsy = Ley;
                    if (first || twigCount % 2 == 1)
                    {
                        x = x + count;
                        first = false;
                    }
                    else
                    {
                        x = x + count * 2;
                    }
                    if (tempcount < 0) break;
                }

                if (tempcount >= 0)
                {
                    for (int j = tempcount; j >= 0; j--)
                    {
                        twigPoints[j].X = Lex;
                        twigPoints[j].Y = Ley;
                    }
                }
            }
            //////////////////////////////////////
            alpha = tye / (txe * txe);
            if (twigCount % 2 == 1)
            {
                count = Math.Abs(txe) / (twigCount / 2 + 1);
            }
            else
            {
                count = Math.Abs(txe) / (twigCount + 1);
            }

            if (count == 0)
            {
                count = 1;
            }

            Lsx = mx; Lsy = my;
            tempcount = 0;
            if (txe < 0)
            {
                x = 0;
                tempcount = twigCount / 2;
                bool first = true;
                while (x >= txe)
                {
                    y = alpha * (x * x);
                    Lex = (int)(ct * x + st * y + mx);
                    Ley = (int)(ct * y - st * x + my);
                    if (Lex != sx || Ley != sy)
                    {
                        if (twigCount % 2 == 1)
                        {
                            twigPoints[tempcount].X = Lex;
                            twigPoints[tempcount].Y = Ley;
                            tempcount++;
                        }
                        else
                        {
                            if (Lex != mx || Ley != my)
                            {
                                twigPoints[tempcount].X = Lex;
                                twigPoints[tempcount].Y = Ley;
                                tempcount++;
                            }
                        }
                    }
                    Lsx = Lex;
                    Lsy = Ley;
                    if (first || twigCount % 2 == 1)
                    {
                        x = x - count;
                        first = false;
                    }
                    else
                    {
                        x = x - count * 2;
                    }
                    if (tempcount >= twigCount) break;
                }
                if (tempcount < twigCount)
                {
                    for (int j = 0; j < twigCount - tempcount; j++)
                    {
                        twigPoints[j].X = Lex;
                        twigPoints[j].Y = Ley;
                    }
                }

            }
            else
            {
                x = 0;
                tempcount = twigCount / 2;
                bool first = true;
                while (x <= txe)
                {
                    y = alpha * (x * x);
                    Lex = (int)(ct * x + st * y + mx);
                    Ley = (int)(ct * y - st * x + my);
                    if (Lex != sx || Ley != sy)
                    {
                        if (twigCount % 2 == 1)
                        {
                            twigPoints[tempcount].X = Lex;
                            twigPoints[tempcount].Y = Ley;
                            tempcount++;
                        }
                        else
                        {
                            if (Lex != mx || Ley != my)
                            {
                                twigPoints[tempcount].X = Lex;
                                twigPoints[tempcount].Y = Ley;
                                tempcount++;
                            }
                        }
                    }
                    Lsx = Lex;
                    Lsy = Ley;
                    if (first || twigCount % 2 == 1)
                    {
                        x = x + count;
                        first = false;
                    }
                    else
                    {
                        x = x + count * 2;
                    }
                    if (tempcount >= twigCount) break;
                }
                if (tempcount < twigCount)
                {
                    for (int j = 0; j < twigCount - tempcount; j++)
                    {
                        twigPoints[j].X = Lex;
                        twigPoints[j].Y = Ley;
                    }
                }
            }
        }

        //# multi color
        private void btn_multi_new_Click(object sender, EventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.SelectClear(pnl_color);
            }
        }

        private void btn_multi_all_Click(object sender, EventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.SelectAll(pnl_color, mCanvas.dst, mCanvas.mWorkarea.totalPath);
            }
        }

        private void btn_multi_area_Click(object sender, EventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.SetSelectState(btn_multi_area);
            }
        }

        private void btn_multi_insert_Click(object sender, EventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.SetInsertState(btn_multi_insert);
            }
        }

        private void btn_multi_pre_Click(object sender, EventArgs e)
        {
            if (chk_select.Checked)
            {
                multi.SelectRestore(pnl_color);
            }
        }
    }
}
