﻿using ClipperLib;
using Svg;
using Svg.Pathing;
using Svg.Transforms;
using System;
using System.Collections.Generic;
using System.Drawing;
using System.Drawing.Drawing2D;
using System.Drawing.Imaging;
using System.IO;
using System.Linq;
using System.Runtime.Serialization.Formatters.Binary;
using System.Threading;
using System.Windows.Forms;
using System.Windows.Input;
using System.Xml;
using KeyEventArgs = System.Windows.Forms.KeyEventArgs;
using MouseEventArgs = System.Windows.Forms.MouseEventArgs;
using Polygon = System.Collections.Generic.List<ClipperLib.IntPoint>;
using WinVector = System.Windows.Vector;
using VECTOR = System.Numerics.Vector2;
using WinPoint = System.Windows.Point;
using burningmime.curves;

namespace TexMaster
{

    public enum Direction
    {
        NONE, X, Y
    }
    public enum Change
    {
        NONE, PLUS, MINUS
    }

    public enum ModifyMode
    {
        FONT, PATH, PEN, BRUSH, DATA, PATHDATA
    }

    /// <summary>
    /// S  = Start, C1 = Control1, C2 = Control2, N = None, CE = CurveEnd, LE = LineEnd, CC = ClosedCurve, CL = ClosedLine
    /// S  = 직선, 곡선의 시작점
    /// C1 = 4 Bezier 곡선의 첫번째 Handle
    /// C2 = 4 Bezier 곡선의 두번째 Handle
    /// N  = 3 Bezier 곡선을 만들기 위해 C1 or C2 Handle중 하나를 N으로 바꿈
    /// CE = Path의 마지막 점이 곡선으로 끝났을 때
    /// LE = Path의 마지막 점이 직선으로 끝났을 때
    /// CC = Path의 마지막 점이 닫힌 곡선으로 끝났을 때
    /// CL = Path의 마지막 점이 닫힌 직선으로 끝났을 때
    /// </summary>
    public enum PathTypes
    {
        S, C1, C2, N, CE, LE, CC = 131, CL = 129,
    }

    public enum Mode
    {
        //# draw
        LINE, CURVE, FREEDRAW, RECT, SQUARE, ELLIPSE, CIRCLE, TEXT, SPRAY, REPEAT,
        //# select
        SELECT, SELLASSO,
        //# transform
        ZOOM, ROTATE, FOUNDATION,
        //# point
        POINTSEL, PENTOOL, POINTADD, POINTDEL, POINTCHANGE, KNIFE,
        //# reverse
        REVHORI, REVVERT,
        //# etc
        STROKE, PATH, PATHTEXT, FILL, UNDO, REDO
    }

    public struct VectorData
    {
        public GraphicsPath path;       //# 오브젝트의 path
        public Brush brush;             //# 오브젝트를 채우는 Brush
        public Pen pen;                 //# 오브젝트를 그리는 Pen

        //# vector text
        public bool isString;           //# 현재 저장할 데이터가 문자이면 True, 그러지 않으면 False
        public string str;              //# 그려질 문자열
        public Font font;               //# 그려질 문자열의 글꼴

        //# path text
        public bool isPathText;         //# 문자 오브젝트가 패스 문자이면 True, 그러지 않으면 False

    }

    [Serializable()]
    public struct FileData
    {
        public string xmlPath;          //# 오브젝트의 Path, xmlPath 형식으로 변환된 문자열로 내보낸다(Svg 동일)
        public bool isString;           //# VectorData의 isString과 동일

        //# brush
        public int brushType;           //# 0 : SolidBrush, 1 : GradientBrush, 2: Texture
        public int color;               //# SolidBrush Color
        public int sColor;              //# GradientBrush Start Color
        public int eColor;              //# GradientBrusn End Color
        public Image pattern;           //# TextureBrush Image

        //# pen
        public float penWidth;          //# Pen 굵기
        public DashStyle dash;          //# Pen의 Dash 종류
        public LineCap sCap;            //# Pen의 Start LineCap Type
        public LineCap eCap;            //# Pen의 End LineCap Type
        public LineJoin lCap;           //# Pen의 결합선 스타일
        public int pColor;              //# Pen 색상

        //# text
        public string str;              //# VectorData의 str과 동일
        public string fontName;         //# 글꼴 이름
        public float fontSize;          //# 글꼴 크기
        public FontStyle fontStyle;     //# 글꼴 스타일(Bold, Italic 등)

        public bool isPathText;         //# VectorData의 isPathText와 동일
    }

    public partial class Vector : Form
    {
        //# drawing control
        public Canvas bitCanvas;                //# Bitmap Mode에 접근하기 위한 변수 (MainMenu에서 설정)
        public PictureBox pictureBox1;              //# 그림이 그려질 picturebox (MainMenu에서 설정)
        public List<VectorData> clipboard;          //# 복사 / 붙여넣기 하기 위한 클립보드 데이터
        public List<List<PathTypes>> clipPath;      //# clipboard와 쌍을 이룰 PathTypes(For Three Bezier)
        public Mode mode;                           //# 그리기 모드 저장
        public Point start;                         //# 오브젝트를 그릴 시작점
        public Point end;                           //# 오브젝트를 그릴 끝점
        public bool drawing;                        //# Paint 이벤트 실행 여부
        public Color pathColor;                     //# 선택 박스의 색상
        public bool isClicked;                      //# 마우스가 클릭이 되어있는 상태인지 확인 (Text는 이동 시에만 DeepBlue 색상으로 Highlight 해주기 위함)
        public Point oldPosition;                   //# 이동 / 복사 시 Offset 계산을 위해 이전 좌표를 기억해 둠
        public bool isMirror;                       //# 현재 모드가 반전 모드이면 True, 아니면 False
        public List<ToolStripButton> tsbList;       //# 메뉴 버튼 List

        //# normal vector object drawing
        public List<VectorData> vecList;            //# 기본 모드에서의 전체 DataList
        public int doIdx;                           //# History Index
        public List<List<VectorData>> doVector;     //# vecList들의 List, 환원 구현을 위함

        //# reverse vector object drawing
        public List<VectorData> oriList;            //# 반전모드에서 그려지는 부분의 DataList
        public List<VectorData> refList;            //# 반전모드에서 반전되어 보여지는 부분의 DataList
        public int refDoIdx;                        //# History Index
        public List<List<VectorData>> oriDoVector;  //# oriList들의 List, 환원 구현을 위함
        public List<List<VectorData>> refDoVector;  //# refList들의 List, 환원 구현을 위함
        public Point baseLine;                      //# 기준선을 그릴 위치 좌표
        public bool isHori;                         //# 가로 반전이면 True, 세로 반전이면 False
        public bool moveBaseLine;                   //# 가로 반전이면 True, 세로 반전이면 False

        //# select
        public List<int> selPathIndex;              //# 선택된 Data의 Index List
        public List<GraphicsPath> selTempPath;      //# Data를 변화시킬 때 미리보기 할 Path의 List
        public List<GraphicsPath> selRefPath;       //# Data를 변화시킬 때 반전하여 미리보기 할 Path의 List
        public List<int> selOriIdx;                 //# 선택 상태에서 Mirror 모드 후 원래 DataList의 Index로 Insert 해주기 위해 원래의 index를 기억
        public RectangleF selRects;                 //# 선택된 영역
        public RectangleF selRefRects;              //# 반전 쪽의 선택된 영역
        public RectangleF[] selRectRect;            //# 선택 영역의 Control 영역
        public PointF[] selRectPoints;              //# 선택 영역의 Control Point
        public GraphicsPath selRegion;              //# 선택 구역 설정
        public bool isRectLasso;

        //# current data
        public List<VectorData> currentDataList;    //# currentData들의 List, 반복 그리기 등에서 여러 개의 데이터를 동시에 그리기 위함
        public VectorData currentData;              //# 현재 그려지고 있는 데이터

        //# vector text
        public RichTextBoxIme richTextBox;          //#문자열을 입력 받기 위해 Custom한 RichTextBox
        public int nTextIndex;                      //# 현재 수정 중인 데이터의 Index를 기억

        //# Zoom
        public RectangleF zoomRect;                 //# 확대 / 축소할 전체 사각형
        public PointF zoomRectPt;                   //# Scale 기준점(선택한 점의 반대점)
        public int zoomIdx;                         //# Scale 기준점의 Index(0~7)
        public List<GraphicsPath> zoomPath;         //# 미리보기 Path
        public RectangleF zoomRefRect;              //# 반전그리기 - 확대 / 축소할 전체 사각형
        public PointF zoomRefRectPt;                //# 반전그리기 - Scale 기준점(선택한 점의 반대점)
        public int zoomRefIdx;                      //# 반전그리기 - Scale 기준점(선택한 점의 반대점)
        public List<GraphicsPath> zoomRefPath;      //# 반전그리기 - 미리보기 Path

        //# Pen Tool
        public List<PointF> pathPoints;             //# 경로의 지점
        public List<PathTypes> pathTypes;           //# 해당 지점의 종류
        public List<byte> pathByte;                 //# 해당 지점의 종류의 Byte값
        public int pathPointIdx;                    //# 선택 점의 Index
        public int pathPointSymIdx;                 //# 선택 점의 반대 Index

        //# etc
        public List<PointF> curveList;              //# Bezier Points ( Point Count = 4 )
        public GraphicsPath connPath;               //# PenTool로 그릴 때 직선, 곡선을 Add한 Path
        public List<GraphicsPath> connPathList;     //# connPath의 리스트 ( 한 점 뒤로 기능할 때 List를 하나씩 뺀다)
        public Bitmap patternImg;                   //# TextureBrush Image
        public bool isLasso;                        //# 올가미 영역을 그리고 있는 중인지 여부
        public int xOffset, yOffset;                //# Path를 이동시킬 때의 x 이동 변화량과 Y이동 변화량
        public Direction dir;                       //# 수평 / 수직 이동 구현을 위한 이동 방향
        public List<List<int>> groupList;           //# 한 그룹 : DataList의 index의 List로 구성, groupList : 그룹들의 List

        //CurvePoint
        public List<PointF> thrBezirData;           //# 3 point Bezier의 Control Point를 담을 List
        public int thrBezirIdx;                     //# 선택한 점이 3 point Bezier일 때 그 점의 Idx
        public int thrBezirSymIdx;                  //# 선택한 점의 맞은 편 점이 3 point Bezier 일 때 그 점의 Idx
        public bool isThrBezir;                     //# 선택한 점이 3 point Bezier인지 여부
        public PointF[] thrBezirPts;                //# 3Point Bezier를 나타내기 위해 저장하는 배열 이 값을 4Point Bezier로 변환함
        public List<List<PathTypes>> pathTypesList; //# 각 VectorPath의 PathType을 저장할 List(3 Point Bezier를 나타내기 위해 사용)
        public List<List<PathTypes>> refTypesList;  //# 반전 모드에 사용할 PathTypesList
        public bool isFirst;                        //# MouseMove 시 Data를 삭제, 추가, 수정이 한 번만 필요한 경우 사용
        public int handleSize;

        //TestValue
        public PointF addedPt;                      //# 추가된 새로운 포인트
        public List<PointF> newC1C2;                //# 포인트 추가 시 해당 점의 새로운 Handle
        public float t_value;                       //# 포인트 추가 시 해당 점의 t값을 저장(t = 0 ~ 1)

        public Change changeX = Change.NONE;
        public Change changeY = Change.NONE;
        public List<PointF> changePt;
        public List<byte> curvedByte;

        public struct PathData
        {
            public RectangleF pBounds;              //# Object의 Line or Curve의 Bounds를 저장할 리스트
            public int pIdx;                        //# Object의 Line or Curve의 Index
            public bool isCurve;                    //# Object의 Line or Curve
        }

        public List<PathData> pData;

        public Vector() //# 생성자
        {
            clipboard = new List<VectorData>();
            clipPath = new List<List<PathTypes>>();
            mode = Mode.LINE;
            start = new Point();
            end = new Point();
            drawing = false;
            pathColor = Color.Red;
            isClicked = false;

            isMirror = false;
            moveBaseLine = false;
            baseLine = new Point(0, 0);

            tsbList = new List<ToolStripButton>();

            currentDataList = new List<VectorData>();
            currentData = new VectorData();

            vecList = new List<VectorData>();
            doVector = new List<List<VectorData>>();
            doIdx = -1;
            oriDoVector = new List<List<VectorData>>();
            refDoVector = new List<List<VectorData>>();
            refDoIdx = -1;

            selPathIndex = new List<int>();
            selTempPath = new List<GraphicsPath>();
            selRefPath = new List<GraphicsPath>();
            selOriIdx = new List<int>();
            selRectRect = new RectangleF[8];
            selRectPoints = new PointF[8];

            zoomRect = new RectangleF();
            zoomRectPt = new PointF();
            zoomPath = new List<GraphicsPath>();
            zoomRefRect = new RectangleF();
            zoomRefRectPt = new PointF();
            zoomRefPath = new List<GraphicsPath>();
            nTextIndex = -1;

            curveList = new List<PointF>();
            connPathList = new List<GraphicsPath>();
            dir = Direction.NONE;
            groupList = new List<List<int>>();
            isFirst = false;

            isThrBezir = false;
            pathPointIdx = -1;
            pathPointSymIdx = -1;
            thrBezirIdx = -1;
            thrBezirSymIdx = -1;
            thrBezirData = new List<PointF>();
            pathTypesList = new List<List<PathTypes>>();
            refTypesList = new List<List<PathTypes>>();
            handleSize = 5;
            isRectLasso = false;

            //Test Value
            addedPt = new PointF();
            newC1C2 = new List<PointF>();
            pData = new List<PathData>();

            changePt = new List<PointF>();
            curvedByte = new List<byte>();

            InitializeComponent();
        }

        private void Vector_Load(object sender, EventArgs e)
        {
            //# ui 설정
            tsb_line.Checked = true;
            UpdateDisplay();

            //# text input control set
            richTextBox = new RichTextBoxIme();
            richTextBox.Visible = false;
            richTextBox.Parent = this;
            richTextBox.Size = new Size(0, 0);  //# To make it not appear
            richTextBox.SendToBack();
            richTextBox.TextChanged += rtb_text_TextChanged;

            //# size panel init
            nud_x.Minimum = int.MinValue;
            nud_y.Minimum = int.MinValue;
            nud_w.Minimum = int.MinValue;
            nud_h.Minimum = int.MinValue;
            nud_x.Maximum = int.MaxValue;
            nud_y.Maximum = int.MaxValue;
            nud_w.Maximum = int.MaxValue;
            nud_h.Maximum = int.MaxValue;

            //# Font combo box setting
            foreach (FontFamily family in FontFamily.Families)
            {
                cmb_fontName.Items.Add(family.Name);
            }
            cmb_fontName.SelectedIndex = cmb_fontName.Items.IndexOf("굴림");
            cmb_fontName.DrawMode = DrawMode.OwnerDrawVariable;
            cmb_fontSize.SelectedIndex = cmb_fontSize.Items.IndexOf("20 pt");
            cmb_fontStyle.SelectedIndex = 0;

            //# Draw combo box setting
            tscb_width.SelectedIndex = 5;                                  //# default Pen width
            cmb_dashStyle.SelectedIndex = 0;
            cmb_startCap.SelectedIndex = 0;
            cmb_endCap.SelectedIndex = 0;
            cmb_lineJoin.SelectedIndex = 0;

            //# make toolstripbutton list
            //# draw
            tsbList.Add(tsb_line);
            tsbList.Add(tsb_curve);
            tsbList.Add(tsb_freedraw);
            tsbList.Add(tsb_rect);
            tsbList.Add(tsb_sqare);
            tsbList.Add(tsb_ellipse);
            tsbList.Add(tsb_circle);
            tsbList.Add(tsb_text);
            tsbList.Add(tsb_spray);
            tsbList.Add(tsb_repeat);
            //# sel
            tsbList.Add(tsb_region);
            tsbList.Add(tsb_lasso);
            tsbList.Add(tsb_zoom);
            tsbList.Add(tsb_rotate);
            tsbList.Add(tsb_foundation);
            //# point
            tsbList.Add(tsb_PointSel);
            tsbList.Add(tsb_penTool);
            tsbList.Add(tsb_PointAdd);
            tsbList.Add(tsb_PointDel);
            tsbList.Add(tsb_PointChange);
            tsbList.Add(tsb_knife);
            //# reverse
            tsbList.Add(tsb_hori_reverse);
            tsbList.Add(tsb_vert_reverse);
            //# etc
            tsbList.Add(tsb_stroke);
            tsbList.Add(tsb_selColor);
            tsbList.Add(tsb_type_path);
            tsbList.Add(tsb_paint);

            btn_solid_apply.Enabled = true;

            cmb_type.SelectedIndex = 0;
            cmb_method.SelectedIndex = 0;
            cmb_location.SelectedIndex = 0;
        }

        private void Vector_FormClosing(object sender, FormClosingEventArgs e)
        {
            e.Cancel = true; //리소스를 해제하지 않아 데이터를 유지할 수 있다
            this.Hide();
            bitCanvas.mDrawObject = CanvasType.BITMAP;
        }

        private void Vector_KeyDown(object sender, KeyEventArgs e)
        {
            HotKey(e.KeyCode, e.Modifiers);
        }

        /// <summary>
        /// 단축키
        /// </summary>
        /// <param name="key"></param>
        int keystate; //# modifier key 0:alt, 1:shift, 2:ctrl, -1:none
        public void HotKey(Keys key, Keys modifier)
        {
            /*디버깅.... 편하게*/
            if (mode == Mode.TEXT || mode == Mode.PATHTEXT) return;



            if (modifier == Keys.Control)
            {
                keystate = 2;
                switch (key)
                {
                    case Keys.E:
                        if (selPathIndex.Count == 0) return;

                        clipboard.Clear();
                        clipPath.Clear();

                        if (!isMirror) SetClipboardData(vecList, pathTypesList);
                        else SetClipboardData(oriList, refTypesList);

                        break;
                    case Keys.R:
                        if (clipboard.Count == 0) return;

                        selPathIndex.Clear();
                        selTempPath.Clear();
                        //mVectorList.dgv_vector.ClearSelection();

                        if (!isMirror) GetClipboardData(vecList, pathTypesList);
                        else
                        {
                            selRefPath.Clear();
                            GetClipboardData(oriList, refTypesList);
                        }
                        Commit();

                        mode = Mode.SELECT;
                        tsb_region.Checked = true;

                        drawing = true;
                        pictureBox1.Refresh();
                        drawing = false;
                        break;
                    case Keys.A:
                        ClearSelList();

                        if (!isMirror)
                        {
                            if (vecList.Count == 0) return;
                            for (int i = vecList.Count - 1; i >= 0; i--)
                            {
                                AddSelList(i);
                            }
                        }
                        else
                        {
                            if (oriList.Count == 0) return;
                            for (int i = oriList.Count - 1; i >= 0; i--)
                            {
                                AddSelList(i);
                            }
                        }

                        RestoreSelMode();
                        break;
                    case Keys.Z:
                        Undo();
                        break;
                    case Keys.Y:
                        Redo();
                        break;
                }
            }
            else if (modifier == Keys.Alt)
            {
                keystate = 0;
            }
            else if (modifier == Keys.Shift)
            {
                keystate = 1;
            }
            else if (modifier == Keys.None)
            {
                keystate = -1;
                switch (key)
                {
                    case Keys.R://Draw Rect
                        ButtonClick(tsb_rect, EventArgs.Empty);
                        break;
                    case Keys.A://PointSel
                        ButtonClick(tsb_PointSel, EventArgs.Empty);
                        break;
                    case Keys.V://Sel Region
                        ButtonClick(tsb_region, EventArgs.Empty);
                        break;
                    case Keys.P://PenTool
                        ButtonClick(tsb_penTool, EventArgs.Empty);
                        break;
                    case Keys.OemMinus://PointDel
                        ButtonClick(tsb_PointDel, EventArgs.Empty);
                        break;
                    case Keys.Oemplus://PointAdd
                        ButtonClick(tsb_PointAdd, EventArgs.Empty);
                        break;
                    case Keys.Oem5://PointChange
                        ButtonClick(tsb_PointChange, EventArgs.Empty);
                        break;
                    case Keys.F://Fill
                        ButtonClick(tsb_paint, EventArgs.Empty);
                        btn_solid_apply.PerformClick();
                        break;
                    case Keys.B:
                        if (mode == Mode.PENTOOL && connPathList.Count > 0)
                        {
                            if (connPathList.Count == 1)
                                connPath = (GraphicsPath)connPathList[connPathList.Count - 1].Clone();
                            else
                            {
                                connPath = (GraphicsPath)connPathList[connPathList.Count - 2].Clone();
                                connPathList.RemoveAt(connPathList.Count - 1);
                            }

                            pathPoints = connPath.PathPoints.ToList();
                            pathByte = connPath.PathTypes.ToList();

                            drawing = true;
                            pictureBox1.Refresh();
                            drawing = false;
                        }
                        break;
                    case Keys.C:
                        if (mode == Mode.PENTOOL && connPath != null && pathPoints.Count > 2)
                        {
                            SetPenToolVector(true);
                        }
                        break;
                    case Keys.X:
                        if (mode == Mode.PENTOOL && connPath != null && pathPoints.Count > 1)
                        {
                            SetPenToolVector(false);
                        }
                        break;
                    case Keys.Delete:
                        for (int i = 0; i < selPathIndex.Count; i++)
                        {
                            DeleteData(selPathIndex[i]);
                        }

                        ClearSelList();
                        Commit();

                        drawing = true;
                        pictureBox1.Refresh();
                        drawing = false;
                        break;
                }
            }
        }

        /// <summary>
        /// ToolStripButton Click Event
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void ButtonClick(object sender, EventArgs e)
        {
            //# mode init
            mode = Mode.LINE;  //# default(no select)
            //# check clicked button
            for (int i = 0; i < tsbList.Count; i++)
            {
                tsbList[i].Checked = false;
                if (sender.Equals(tsbList[i]))
                {
                    tsbList[i].Checked = true;
                    mode = (Mode)i;
                }
            }

            Console.WriteLine(mode);
            UpdateDisplay();
            SetPenToolVector(false);
            switch (mode)
            {
                case Mode.FREEDRAW:
                case Mode.RECT:
                case Mode.SQUARE:
                case Mode.ELLIPSE:
                case Mode.CIRCLE:
                case Mode.TEXT:
                    ClearSelList();
                    drawing = true;
                    pictureBox1.Refresh();
                    drawing = false;
                    break;
                case Mode.PATHTEXT:
                    nTextIndex = selPathIndex[0];
                    richTextBox.Visible = true;
                    if (vecList[nTextIndex].str == null)
                    {
                        richTextBox.Text = "";
                    }
                    else
                    {
                        richTextBox.Text = vecList[nTextIndex].str;

                    }
                    richTextBox.Focus();
                    richTextBox.Select(richTextBox.Text.Length, 0);

                    currentData.path = (GraphicsPath)vecList[selPathIndex[0]].path;
                    currentData.isPathText = true;
                    currentData.str = richTextBox.Text;

                    ClearSelList();

                    drawing = true;
                    pictureBox1.Refresh();
                    break;
                case Mode.SELLASSO:
                    drawing = false;
                    isLasso = false;
                    break;
                case Mode.ROTATE:
                case Mode.FILL:
                case Mode.ZOOM:
                case Mode.REPEAT:
                    selPathIndex.Sort((x1, x2) => x1.CompareTo(x2));
                    drawing = true;
                    break;
                case Mode.STROKE:
                    ChangePen();
                    if (!(selPathIndex.Count > 0)) return;
                    for (int i = 0; i < selPathIndex.Count; i++)
                    {
                        ModifyData(ModifyMode.PEN, selPathIndex[i], (Pen)currentData.pen.Clone());
                    }
                    RestoreSelMode();
                    break;
                case Mode.PATH:
                    pathColor = bitCanvas.mPalette.color;
                    drawing = true;
                    pictureBox1.Refresh();
                    drawing = false;
                    //RestoreSelMode();
                    break;
                case Mode.REVHORI:
                case Mode.REVVERT:
                    if (oriList == null && refList == null)
                    {
                        oriList = new List<VectorData>();
                        refList = new List<VectorData>();
                        if (mode == Mode.REVHORI)
                        {
                            isHori = true;
                            tsb_hori_reverse.Checked = true;
                            tsb_vert_reverse.Enabled = false;
                        }
                        else //# REVVERT
                        {
                            isHori = false;
                            tsb_vert_reverse.Checked = true;
                            tsb_hori_reverse.Enabled = false;
                        }
                        drawing = true;
                        moveBaseLine = true;
                    }
                    else
                    {
                        DialogResult result = MessageBox.Show("Do you want to apply?", "Question", MessageBoxButtons.YesNoCancel);
                        if (result == DialogResult.Yes)
                        {
                            isMirror = false;
                            for (int i = 0; i < oriList.Count; i++)
                            {
                                //# Add oriList
                                currentData = oriList[i];
                                if (i < selOriIdx.Count) //# 원래 DataList Idx로 Insert
                                {
                                    CreateData(selOriIdx[selOriIdx.Count - 1 - i]);
                                }
                                else
                                {
                                    CreateData();
                                }
                                //# Add refList
                                currentData = refList[i];
                                CreateData();
                            }
                            tsb_hori_reverse.Enabled = true;
                            tsb_vert_reverse.Enabled = true;
                            tsb_text.Enabled = true;
                            tab_gradient.Enabled = true;
                            tab_texture.Enabled = true;
                            ClearSelList();
                            Commit();
                        }
                        else if (result == DialogResult.No)
                        {
                            isMirror = false;
                            for (int i = 0; i < oriList.Count; i++)
                            {
                                //# Add oriList
                                currentData = oriList[i];
                                if (i < selOriIdx.Count) //# 원래 DataList Idx로 Insert
                                {
                                    CreateData(selOriIdx[selOriIdx.Count - 1 - i]);
                                }
                                else
                                {
                                    CreateData();
                                }
                            }
                            tsb_hori_reverse.Enabled = true;
                            tsb_vert_reverse.Enabled = true;
                            tsb_text.Enabled = true;
                            tab_gradient.Enabled = true;
                            tab_texture.Enabled = true;
                            ClearSelList();
                            Commit();
                        }
                        else if (result == DialogResult.Cancel)
                        {
                            RestoreSelMode();
                            return;
                        }

                        //# clear mirror data
                        oriList.Clear();
                        refList.Clear();
                        oriList = null;
                        refList = null;
                        selOriIdx.Clear();

                        RestoreSelMode();
                    }
                    break;
                case Mode.POINTSEL:
                case Mode.POINTCHANGE:
                case Mode.POINTADD:
                case Mode.POINTDEL:
                    if (selPathIndex.Count == 0)
                    {
                        MessageBox.Show("Plz Select Object!");
                        mode = Mode.SELECT;
                        tsb_PointSel.Checked = false;
                        tsb_region.Checked = true;
                        return;
                    }
                    else if (selPathIndex.Count == 1)
                    {
                        if (!isMirror) GetPathDatas(vecList, pathTypesList);
                        else GetPathDatas(oriList, refTypesList);

                        SetthrBezirAndLinePath();
                    }
                    drawing = true;
                    pictureBox1.Refresh();
                    drawing = false;
                    break;
                default:
                    drawing = false;
                    break;
            }
        }

        /// <summary>
        /// Mode에 따른 Panel List 생성
        /// </summary>
        public void UpdateDisplay()
        {
            List<Panel> pnlList = new List<Panel>();

            switch (mode)
            {
                //# draw
                case Mode.LINE:
                case Mode.CURVE:
                case Mode.FREEDRAW:
                case Mode.RECT:
                case Mode.SQUARE:
                case Mode.ELLIPSE:
                case Mode.CIRCLE:
                    pnlList.Add(pnl_draw);
                    pnlList.Add(pnl_size);
                    pnlList.Add(pnl_pen);
                    break;
                //# text
                case Mode.TEXT:
                case Mode.PATHTEXT:
                    pnlList.Add(pnl_text);
                    break;
                case Mode.SELLASSO:
                case Mode.SELECT:
                    pnlList.Add(pnl_edit);
                    if (selPathIndex.Count > 0)
                    {
                        pnlList.Add(pnl_size);
                        List<VectorData> selList;
                        if (!isMirror)
                        {
                            selList = vecList;
                        }
                        else
                        {
                            selList = oriList;
                        }
                        bool isAllString = true;
                        bool isAllFigure = true;
                        for (int i = 0; i < selPathIndex.Count; i++)
                        {
                            if (selList[selPathIndex[i]].isString)
                            {
                                tls_edit.Enabled = false;
                                if (!isMirror)
                                {
                                    tsb_hori_reverse.Enabled = false;
                                    tsb_vert_reverse.Enabled = false;
                                }
                                tsb_lock.Enabled = false;
                                tsb_PointSel.Enabled = false;
                                tsb_PointAdd.Enabled = false;
                                tsb_PointDel.Enabled = false;
                                tsb_knife.Enabled = false;
                                isAllString &= true;
                                isAllFigure &= false;
                            }
                            else
                            {
                                tls_edit.Enabled = true;
                                if (selPathIndex.Count != 2)
                                {
                                    tsb_foundation.Enabled = false;
                                }
                                else
                                {
                                    tsb_foundation.Enabled = true;
                                }
                                if (!isMirror)
                                {
                                    tsb_hori_reverse.Enabled = true;
                                    tsb_vert_reverse.Enabled = true;
                                }
                                tsb_lock.Enabled = true;
                                tsb_PointSel.Enabled = true;
                                tsb_PointAdd.Enabled = true;
                                tsb_PointDel.Enabled = true;
                                tsb_knife.Enabled = true;
                                isAllString &= false;
                                isAllFigure &= true;
                            }
                        }

                        if (isAllString)
                        {
                            pnlList.Add(pnl_text);
                        }

                        if (isAllFigure)
                        {
                            pnlList.Add(pnl_pen);
                        }
                    }
                    else
                    {
                        tls_edit.Enabled = false;
                    }
                    break;
                case Mode.SPRAY:
                    pnlList.Add(pnl_spray);
                    break;
                case Mode.REPEAT:
                    pnlList.Add(pnl_repeat);
                    break;
                case Mode.ROTATE:
                    pnlList.Add(pnl_rotate);
                    break;
                case Mode.FILL:
                    pnlList.Add(pnl_brush);
                    break;
                case Mode.FOUNDATION:
                    pnlList.Add(pnl_foundation);
                    break;
                case Mode.REVHORI:
                case Mode.REVVERT:
                    tsb_text.Enabled = false;
                    tab_gradient.Enabled = false;
                    tab_texture.Enabled = false;
                    break;
                default:
                    break;

            }

            if (pnlList.Count > 0)
            {
                int top = pnl_vector.Height;
                for (int i = 0; i < pnlList.Count; i++)
                {
                    pnlList[i].Visible = true;
                    pnlList[i].Location = new Point(0, top);
                    pnlList[i].BringToFront();
                    top += pnlList[i].Height;
                }

                this.Height = pnlList[pnlList.Count - 1].Top + pnlList[pnlList.Count - 1].Height + 38;  //# 38 - Topbar Height
            }

            this.Refresh();
        }

        /// <summary>
        /// SELECT Mode에서 Vector object를 수정한 후에 다시 SELECT Mode로 돌아가기 위한 루틴
        /// </summary>
        public void RestoreSelMode()
        {
            mode = Mode.SELECT;
            ButtonClick(tsb_region, EventArgs.Empty);

            currentData.path = null;

            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
        }

        //=======================================================================
        //# Data Control
        //=======================================================================

        /// <summary>
        /// Data(Current Data) 저장
        /// </summary>
        /// <param name="idx">삽입할 위치</param>
        public void CreateData(int idx = -1)
        {
            //# add currentData
            SetPathTypes(currentData.path);
            if (!isMirror)
            {

                if (idx < 0)
                {
                    pathTypesList.Add(pathTypes.ToList());
                    vecList.Add(currentData);
                }
                else
                {
                    vecList.Insert(idx, currentData);
                    pathTypesList.Insert(idx, pathTypes.ToList());
                }
                SetVectorList(currentData);
            }
            else
            {
                refTypesList.Add(pathTypes.ToList());
                oriList.Add(currentData);
                refList.Add(GetMirrorVector(currentData, baseLine));
            }

        }

        /// <summary>
        /// Data 삭제
        /// </summary>
        /// <param name="idx">삭제할 위치</param>
        public void DeleteData(int idx)
        {
            if (!isMirror)
            {
                vecList.RemoveAt(idx);
                pathTypesList.RemoveAt(idx);
            }
            else
            {
                oriList.RemoveAt(idx);
                refList.RemoveAt(idx);
                refTypesList.RemoveAt(idx);
            }
        }

        /// <summary>
        /// Data 수정 한 후 적용
        /// </summary>
        /// <param name="mode">변경시키고자 하는 Data의 종류</param>
        /// <param name="idx">변경할 위치</param>
        /// <param name="val">변경할 값</param>
        public void ModifyData(ModifyMode mode, int idx, object val)
        {
            VectorData newData;
            if (!isMirror)
            {
                if (mode != ModifyMode.DATA)
                {
                    newData = GetModifyData(ref vecList, mode, idx, val);
                }
                else
                {
                    newData = (VectorData)val;
                }
                //pathTypesList[idx] = pathTypes;
                vecList.RemoveAt(idx);
                vecList.Insert(idx, newData);
                SetVectorList(idx, newData);
            }
            else
            {
                if (mode != ModifyMode.DATA)
                {
                    newData = GetModifyData(ref oriList, mode, idx, val);
                }
                else
                {
                    newData = (VectorData)val;
                }
                refTypesList[idx] = pathTypes;
                oriList.RemoveAt(idx);
                oriList.Insert(idx, newData);
                refList.RemoveAt(idx);
                refList.Insert(idx, GetMirrorVector(newData, baseLine));
            }
        }

        public void SetPenToolVector(bool isConnect)
        {
            if (connPath == null) return;

            if (isConnect) connPath.CloseAllFigures();

            currentData.path = (GraphicsPath)connPath.Clone();
            CreateData();
            Commit();

            pathPoints = null;
            pathByte = null;
            connPath = null;
            connPathList.Clear();
            currentData.path = null;

            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
        }

        public void GetPathDatas(List<VectorData> datas, List<List<PathTypes>> types, int selIdx = 0)
        {
            pathPoints = datas[selPathIndex[selIdx]].path.PathPoints.ToList();
            pathByte = datas[selPathIndex[selIdx]].path.PathTypes.ToList();
            pathTypes = types[selPathIndex[selIdx]];
        }

        public void SetClipboardData(List<VectorData> vector, List<List<PathTypes>> paths)
        {
            GraphicsPath path = new GraphicsPath();
            for (int i = 0; i < selPathIndex.Count; i++)
            {
                path.AddPath(vector[selPathIndex[i]].path, false);
            }

            float x, y;
            for (int i = 0; i < selPathIndex.Count; i++)
            {
                x = (pictureBox1.Width / 2) - path.GetBounds().X - (path.GetBounds().Width / 2);
                y = (pictureBox1.Height / 2) - path.GetBounds().Y - (path.GetBounds().Height / 2);
                Matrix mat = new Matrix();
                mat.Translate(x, y);

                GraphicsPath tempPath = (GraphicsPath)vector[selPathIndex[i]].path.Clone();
                tempPath.Transform(mat);

                VectorData data = GetModifyData(ref vector, ModifyMode.PATH, selPathIndex[i], tempPath);

                string brushClass = data.brush.ToString();
                if (brushClass == "System.Drawing.TextureBrush")
                {
                    ((TextureBrush)data.brush).TranslateTransform(x, y, MatrixOrder.Append);
                }
                else if (brushClass == "System.Drawing.Drawing2D.LinearGradientBrush")
                {
                    ((LinearGradientBrush)data.brush).TranslateTransform(x, y, MatrixOrder.Append);
                }
                clipboard.Add(data);
                clipPath.Add(paths[selPathIndex[i]].ToList());
            }
        }

        public void GetClipboardData(List<VectorData> vector, List<List<PathTypes>> paths)
        {
            for (int i = 0; i < clipboard.Count; i++)
            {
                VectorData data = clipboard[i];

                vector.Add(data);
                paths.Add(clipPath[i].ToList());
                SetVectorList(data);

                selPathIndex.Add(vector.Count - 1); //마지막 index 추가
                selTempPath.Add((GraphicsPath)data.path.Clone());

                if (isMirror)
                {
                    refList.Add(GetMirrorVector(data, baseLine));
                    selRefPath.Add(GetMirrorPath((GraphicsPath)data.path.Clone(), baseLine));
                }
                //mVectorList.dgv_vector.Rows[vecList.Count - 1].Selected = true;
            }
        }

        public void SetSelPoint(List<VectorData> datas, Point mPoint, int selIdx = 0)
        {
            RectangleF pathPointsRect;
            for (int i = 0; i < thrBezirData.Count; i++)
            {
                pathPointsRect = new RectangleF(thrBezirData[i].X - handleSize, thrBezirData[i].Y - handleSize, handleSize * 2, handleSize * 2);
                if (pathPointsRect.Contains(mPoint))
                {
                    isThrBezir = true;
                    thrBezirIdx = i;
                    int nCount = 0;
                    for (int j = 0; j < pathTypes.Count; j++)
                    {
                        if (pathTypes[j] == PathTypes.N)
                        {
                            if (nCount == thrBezirIdx)
                            {
                                pathPointIdx = j;
                                SetPathPointSymIdx();
                                break;
                            }
                            nCount++;
                        }
                    }
                }
            }

            if (!isThrBezir)
            {
                for (int i = 0; i < datas[selPathIndex[selIdx]].path.PointCount; i++)
                {
                    pathPointsRect = new RectangleF(datas[selPathIndex[selIdx]].path.PathPoints[i].X - handleSize, datas[selPathIndex[selIdx]].path.PathPoints[i].Y - handleSize, handleSize * 2, handleSize * 2);
                    if (pathPointsRect.Contains(mPoint))
                    {
                        pathPointIdx = i;
                        if (pathTypes[i] == PathTypes.C2 && pathTypes[i + 1] == PathTypes.CC)
                        {
                            if (pathTypes[1] == PathTypes.C1 && pathTypes[2] != PathTypes.N)
                                pathPointSymIdx = 1;
                            else if (pathTypes[1] == PathTypes.C1 && pathTypes[2] == PathTypes.N)
                                thrBezirIdx = GetThrBezierIdx(2);
                        }
                        else if (pathTypes[i] == PathTypes.C1 && pathTypes[pathTypes.Count - i] == PathTypes.CC)
                        {
                            if (pathTypes[pathTypes.Count - 2] == PathTypes.C2 && pathTypes[pathTypes.Count - 3] != PathTypes.N)
                                pathPointSymIdx = pathTypes.Count - 2;
                            else if (pathTypes[pathTypes.Count - 2] == PathTypes.C2 && pathTypes[pathTypes.Count - 3] == PathTypes.N)
                                thrBezirIdx = GetThrBezierIdx(pathTypes.Count - 3);
                        }
                        else if (pathTypes[i] == PathTypes.C1 && i - 2 > -1)
                        {
                            if (pathTypes[i - 2] == PathTypes.C2 && pathTypes[i - 3] != PathTypes.N)
                                pathPointSymIdx = i - 2;
                            else if (pathTypes[i - 2] == PathTypes.C2 && pathTypes[i - 3] == PathTypes.N)
                                thrBezirIdx = GetThrBezierIdx(i - 3);
                        }
                        else if (pathTypes[i] == PathTypes.C2 && i + 2 < datas[selPathIndex[selIdx]].path.PointCount)
                        {
                            if (pathTypes[i + 2] == PathTypes.C1 && pathTypes[i + 3] != PathTypes.N)
                                pathPointSymIdx = i + 2;
                            else if (pathTypes[i + 2] == PathTypes.C1 && pathTypes[i + 3] == PathTypes.N)
                                thrBezirIdx = GetThrBezierIdx(i + 3);
                        }
                        break;
                    }
                }
            }
        }


        /// <summary>
        /// 수정할 Data를 적용하지 않고 반환
        /// </summary>
        /// <param name="list">변경하고자 하는 Data List, 기본 모드 : vecList, 반전 모드 : oriList</param>
        /// <param name="mode">변경시키고자 하는 Data의 종류</param>
        /// <param name="idx">변경할 위치</param>
        /// <param name="value">변경할 값</param>
        /// <returns>수정된 Vector Data</returns>
        public VectorData GetModifyData(ref List<VectorData> list, ModifyMode mode, int idx, object value)
        {
            VectorData oridata = list[idx];
            VectorData newData = new VectorData();

            //# copy oridata - 그래픽 개체는 Clone 사용하여 복사
            newData.path = (GraphicsPath)oridata.path.Clone();
            newData.pen = (Pen)oridata.pen.Clone();
            //# casting
            string brushClass = oridata.brush.ToString();
            if (brushClass == "System.Drawing.TextureBrush")
            {
                newData.brush = (TextureBrush)oridata.brush.Clone();
            }
            else if (brushClass == "System.Drawing.Drawing2D.LinearGradientBrush")
            {
                newData.brush = (LinearGradientBrush)oridata.brush.Clone();
            }
            else
            {
                newData.brush = (SolidBrush)oridata.brush.Clone();
            }
            newData.isString = oridata.isString;
            if (newData.isString)
            {
                newData.font = (Font)oridata.font.Clone();
                newData.str = oridata.str;
                newData.isPathText = oridata.isPathText;
            }

            //# ori data replace with current data
            switch (mode)
            {
                case ModifyMode.PEN:
                    newData.pen = (Pen)value;
                    break;
                case ModifyMode.FONT:
                    newData.font = (Font)value;
                    if (!newData.isPathText)
                    {
                        newData.path = (GraphicsPath)GetBaseLine(newData.str, newData.font, Point.Round(newData.path.PathPoints[0])).Clone();
                    }
                    else
                    {
                        newData.path = (GraphicsPath)vecList[idx].path.Clone();
                    }
                    break;
                case ModifyMode.PATH:
                    newData.path = (GraphicsPath)value;
                    break;
                case ModifyMode.BRUSH:
                    newData.brush = (Brush)value;
                    break;
            }

            return newData;
        }

        /// <summary>
        /// 원래의 Data를 통해 반전된 Data를 반환
        /// </summary>
        /// <param name="recieve">원래의 Data</param>
        /// <param name="baseLine">반전 시킬 지점</param>
        /// <returns>반전된 Vector Data</returns>
        private VectorData GetMirrorVector(VectorData recieve, Point baseLine)
        {
            VectorData data = new VectorData();
            data.path = (GraphicsPath)recieve.path.Clone();
            data.pen = (Pen)recieve.pen.Clone();
            data.isString = recieve.isString;
            if (data.isString)
            {
                data.isString = recieve.isString;
                data.str = recieve.str;
                data.font = (Font)recieve.font.Clone();
            }

            if (isHori)
            {
                Matrix horiMat = new Matrix(-1, 0, 0, 1, 2 * baseLine.X, 0);
                data.path.Transform(horiMat);
                data.brush = (Brush)CreateNewBrush(recieve.brush, data.path.GetBounds());
            }
            else
            {
                Matrix vertMat = new Matrix(1, 0, 0, -1, 0, 2 * baseLine.Y);
                data.path.Transform(vertMat);
                data.brush = (Brush)CreateNewBrush(recieve.brush, data.path.GetBounds());
            }

            return data;
        }
        private GraphicsPath GetMirrorPath(GraphicsPath recieve, Point baseLine)
        {
            if (isHori)
            {
                Matrix horiMat = new Matrix(-1, 0, 0, 1, 2 * baseLine.X, 0);
                recieve.Transform(horiMat);
            }
            else
            {
                Matrix vertMat = new Matrix(1, 0, 0, -1, 0, 2 * baseLine.Y);
                recieve.Transform(vertMat);
            }
            return recieve;
        }

        /// <summary>
        /// Data 이동
        /// </summary>
        /// <param name="xOffset">Matrix를 이동시킬 x 방향 Offset</param>
        /// <param name="yOffset">Matrix를 이동시킬 y 방향 Offset</param>
        /// <param name="isApply">DataList에 적용시키려면 true, 적용시키지 않으려면 false</param>
        public void MoveData(int xOffset, int yOffset, bool isApply = false)
        {
            Matrix mvMat = new Matrix();        //# 기존 data matrix
            Matrix refMat = new Matrix();       //# 반대 data matrix
            Direction dir = GetDirection(start, end);
            switch (dir)
            {
                case Direction.X:
                    mvMat.Translate(xOffset, 0);
                    break;
                case Direction.Y:
                    mvMat.Translate(0, yOffset);
                    break;
                case Direction.NONE:
                    mvMat.Translate(xOffset, yOffset);
                    break;
            }

            if (isMirror)
            {
                switch (dir)
                {
                    case Direction.X:
                        if (isHori)
                            refMat.Translate(-xOffset, 0);
                        else
                            refMat.Translate(xOffset, 0);
                        break;
                    case Direction.Y:
                        if (isHori)
                            refMat.Translate(0, yOffset);
                        else
                            refMat.Translate(0, -yOffset);

                        break;
                    case Direction.NONE:
                        if (isHori)
                            refMat.Translate(-xOffset, yOffset);
                        else
                            refMat.Translate(xOffset, -yOffset);
                        break;
                }
            }

            GraphicsPath tempPath, tempRefPath;
            for (int i = 0; i < selPathIndex.Count; i++)
            {
                if (!isMirror)
                {
                    tempPath = (GraphicsPath)vecList[selPathIndex[i]].path.Clone();
                    tempPath.Transform(mvMat);
                    selTempPath[i] = tempPath;
                }
                else
                {
                    tempPath = (GraphicsPath)oriList[selPathIndex[i]].path.Clone();
                    tempPath.Transform(mvMat);
                    selTempPath[i] = tempPath;

                    tempRefPath = (GraphicsPath)refList[selPathIndex[i]].path.Clone();
                    tempRefPath.Transform(refMat);
                    selRefPath[i] = tempRefPath;
                }

                if (isApply)
                {
                    ModifyData(ModifyMode.PATH, selPathIndex[i], (GraphicsPath)selTempPath[i].Clone());
                    Brush tempBrush;
                    if (isMirror)
                    {
                        tempBrush = (Brush)oriList[selPathIndex[i]].brush.Clone();
                    }
                    else
                    {
                        tempBrush = (Brush)vecList[selPathIndex[i]].brush.Clone();
                    }
                    ModifyData(ModifyMode.BRUSH, selPathIndex[i], (Brush)CreateNewBrush(tempBrush, selTempPath[i].GetBounds()));
                }
            }
        }

        /// <summary>
        /// Data 선택
        /// </summary>
        /// <param name="searchBound">선택 범위</param>
        public void SelectData(RectangleF searchBound)
        {
            List<VectorData> selList;
            if (!isMirror)
            {
                selList = vecList;
            }
            else
            {
                selList = oriList;
            }

            bool isContain = false;
            //# 여러개를 이미 선택하였을 때(다중 이동)
            if ((Keyboard.GetKeyStates(Key.LeftCtrl) & KeyStates.Down) <= 0 && selPathIndex.Count > 1)
            {
                Region selListRegion = new Region();
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    if (i == 0)
                    {
                        selListRegion = new Region(selList[selPathIndex[i]].path.GetBounds());
                    }
                    else
                    {
                        selListRegion.Union(selList[selPathIndex[i]].path.GetBounds());
                    }

                }
                if (selListRegion.IsVisible(searchBound))
                {
                    return;
                }
                else
                {
                    ClearSelList();
                }
                return;
            }

            //# 선택
            for (int i = selList.Count - 1; i >= 0; i--)
            {
                RectangleF bound = selList[i].path.GetBounds();

                if (selList[i].isString && !selList[i].isPathText)
                {
                    bound = new RectangleF(bound.X, bound.Y - selList[i].font.Height, bound.Width, bound.Height + selList[i].font.Height);
                }

                if (bound.Contains(searchBound))
                {
                    isContain = true;
                    //# 여러개 선택 혹은 해제
                    if ((Keyboard.GetKeyStates(Key.LeftCtrl) & KeyStates.Down) > 0)
                    {
                        //# 삭제
                        if (selPathIndex.Contains(i))
                        {
                            if (isMirror)
                            {
                                selRefPath.RemoveAt(selPathIndex.IndexOf(i));
                            }
                            selTempPath.RemoveAt(selPathIndex.IndexOf(i));
                            selPathIndex.Remove(i);
                        }
                        //# 추가
                        else
                        {
                            AddSelList(i);
                        }
                    }
                    //# 하나만 선택할 때
                    else
                    {
                        ClearSelList();
                        AddSelList(i);
                    }
                    break;
                }
            }

            //# 네모 올가미
            if (selRegion != null)
            {
                ClearSelList();
                for (int i = selList.Count - 1; i >= 0; i--)
                {
                    for (int j = 0; j < selList[i].path.PointCount; j++)
                    {
                        if (searchBound.Contains(selList[i].path.PathPoints[j]))
                        {
                            isContain = true;
                            AddSelList(i);
                            break;
                        }
                    }
                }
                selRegion = null;
                isRectLasso = true;
            }

            //# 빈 공간 선택 - select 전부 해제
            if (!isContain)
            {
                ClearSelList();
            }
        }

        /// <summary>
        /// Select 관련 List 초기화
        /// </summary>
        public void ClearSelList()
        {
            //# vector view 선택 표시 해제
            //bitCanvas.mVectorList.dgv_vector.ClearSelection();

            //# 선택 list 해제
            selPathIndex.Clear();
            selTempPath.Clear();
            selRefPath.Clear();
        }

        /// <summary>
        /// data를 Select List에 추가
        /// </summary>
        /// <param name="idx">선택한 List의 Index</param>
        public void AddSelList(int idx)
        {
            //# vecList에 있는 idx의 정보를 selList에 저장
            //# 선택 list 추가
            selPathIndex.Add(idx);

            if (isMirror)
            {
                //# 임시 경로 추가
                selTempPath.Add((GraphicsPath)oriList[idx].path.Clone());
                selRefPath.Add((GraphicsPath)refList[idx].path.Clone());
            }
            else
            {
                //# vector view 선택 표시 - System.ArgumentOutOfRangeException 예외가 처리되지 않아 막아둠
                //bitCanvas.mVectorList.dgv_vector.Rows[idx].Selected = true;

                //# 임시 경로 추가
                selTempPath.Add((GraphicsPath)vecList[idx].path.Clone());
            }
        }

        /// <summary>
        /// Data를 반복하여 저장
        /// </summary>
        /// <param name="rowCnt">행 개수</param>
        /// <param name="colCnt">열 개수</param>
        /// <param name="xOffset">x 방향 간격</param>
        /// <param name="yOffset">y 방향 간격</param>
        /// <param name="isApply">Data List에 적용 여부</param>
        private void RepeatData(int rowCnt, int colCnt, int xOffset, int yOffset, bool isApply = false)
        {
            currentDataList = new List<VectorData>();

            //# 현재 선택 된 개체 수 - Sel 동작을 하면서 count가 변하기 때문에 기억해 둠
            int selCount = selPathIndex.Count;

            for (int row = 0; row < rowCnt; row++)
            {
                for (int col = 0; col < colCnt; col++)
                {
                    //# 기존 데이터는 제외하고 추가
                    if (row == 0 && col == 0) continue;

                    //# Offset 만큼 이동
                    Matrix mat = new Matrix();
                    mat.Translate(xOffset * row, yOffset * col);

                    for (int i = 0; i < selCount; i++)
                    {

                        if (!isMirror)
                        {
                            //# 현재 경로 변경
                            currentData.path = (GraphicsPath)vecList[selPathIndex[i]].path.Clone();
                            currentData.path.Transform(mat);

                            //# 데이터 정보에 변경된 경로 적용
                            currentData = GetModifyData(ref vecList, ModifyMode.PATH, selPathIndex[i], (GraphicsPath)currentData.path.Clone());

                            //# 적용 전 Preview
                            if (!isApply)
                            {
                                currentDataList.Add(currentData);
                            }
                            else
                            {
                                //# 선택된 경로를 변경하여 vecList에 추가한다.
                                CreateData();

                                //# 개체 선택
                                AddSelList(vecList.Count - 1);
                            }
                        }
                        else
                        {
                            //# 현재 경로 변경
                            currentData.path = (GraphicsPath)oriList[selPathIndex[i]].path.Clone();
                            currentData.path.Transform(mat);

                            //# 데이터 정보에 변경된 경로 적용
                            currentData = GetModifyData(ref oriList, ModifyMode.PATH, selPathIndex[i], (GraphicsPath)currentData.path.Clone());

                            //# 적용 전 Preview
                            if (!isApply)
                            {
                                currentDataList.Add(currentData);
                            }
                            else
                            {
                                //# 선택된 경로를 변경하여 vecList에 추가한다.
                                CreateData();

                                //# 개체 선택
                                AddSelList(oriList.Count - 1);
                            }
                        }
                    }
                }
            }
        }

        /// <summary>
        /// Data를 반복하여 저장(row, col 입력 받았을 때)
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btn_repeat_Click(object sender, EventArgs e)
        {
            GetBounds();
            int xOffset = (int)selRects.Width + Convert.ToInt32(txt_Xinvert.Text);
            int yOffset = (int)selRects.Height + Convert.ToInt32(txt_Yinvert.Text);
            RepeatData(Convert.ToInt32(txt_row.Text), Convert.ToInt32(txt_col.Text), xOffset, yOffset, true);
            RestoreSelMode();
        }

        private void chk_repeatCnt_CheckedChanged(object sender, EventArgs e)
        {
            if (chk_repeatCnt.Checked)
            {
                txt_row.Enabled = true;
                txt_col.Enabled = true;
                btn_repeat.Enabled = true;
            }
            else
            {
                txt_row.Enabled = false;
                txt_col.Enabled = false;
                btn_repeat.Enabled = false;
            }
        }

        /// <summary>
        /// Data를 회전하여 저장
        /// </summary>
        /// <param name="angle">회전 시킬 각도</param>
        /// <param name="center">회전축(중점)</param>
        public void RotateData(float angle, PointF center)
        {
            if (selPathIndex.Count != 0)
            {
                Matrix rotMat = new Matrix();
                rotMat.RotateAt(angle, center);
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    selTempPath[i].Transform(rotMat);
                }

                if (isMirror)
                {
                    Matrix refMat = new Matrix();
                    PointF refCenter = new PointF(selRefRects.Left + selRefRects.Width / 2, selRefRects.Top + selRefRects.Height / 2);
                    refMat.RotateAt(360 - angle, refCenter); // Todo 중심점이 여러개 선택하면 왜그래 ..
                    for (int i = 0; i < selPathIndex.Count; i++)
                    {
                        selRefPath[i].Transform(refMat);
                    }
                }
            }
        }

        /// <summary>
        /// Data를 회전하여 저장(각도를 입력 받았을 때)
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void txt_rotate_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                GetBounds();
                PointF center = new PointF(selRects.Left + selRects.Width / 2, selRects.Top + selRects.Height / 2);
                if (float.TryParse(txt_rotate.Text, out float result))
                {
                    RotateData(result, center);
                }
                else
                {
                    MessageBox.Show("Invalid Value!");
                    txt_rotate.Focus();
                }

                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    ModifyData(ModifyMode.PATH, selPathIndex[i], (GraphicsPath)selTempPath[i].Clone());
                }

                RestoreSelMode();
            }
        }

        /// <summary>
        /// 지정된 Path로 뿌리기
        /// </summary>
        /// <param name="path">Motive가 뿌려질 path</param>
        public void SprayData(GraphicsPath path)
        {
            List<VectorData> selList;
            if (isMirror) selList = oriList;
            else selList = vecList;
            int idx = 0;
            float dx, dy, dist;

            if (cmb_method.Text == "Line" || cmb_method.Text == "Rectangle")
            {
                List<float> angleList = new List<float>();
                List<PointF> pointList = new List<PointF>();
                GetSprayPoint(path, ref angleList, ref pointList);
                GraphicsPath tempPath = new GraphicsPath();
                for (int j = 0; j < selPathIndex.Count; j++)
                {
                    tempPath.AddPath(selList[selPathIndex[j]].path, false);
                }

                PointF cPoint = new PointF();

                switch (cmb_location.SelectedIndex)
                {
                    case 0: //# center
                        cPoint = new PointF(tempPath.GetBounds().Left + tempPath.GetBounds().Width / 2, tempPath.GetBounds().Top + tempPath.GetBounds().Height / 2);
                        break;
                    case 1: //# left
                        cPoint = new PointF(tempPath.GetBounds().Left + tempPath.GetBounds().Width / 2, tempPath.GetBounds().Bottom);
                        break;
                    case 2: //# right
                        cPoint = new PointF(tempPath.GetBounds().Left + tempPath.GetBounds().Width / 2, tempPath.GetBounds().Top);
                        break;
                }

                //# 직선
                for (int j = 0; j < pointList.Count; j++)
                {
                    Matrix mat = new Matrix();
                    mat.Translate(pointList[j].X - cPoint.X, pointList[j].Y - cPoint.Y);
                    mat.RotateAt(-(float)(180 * angleList[j] / Math.PI), cPoint);

                    GraphicsPath copyPath = (GraphicsPath)tempPath.Clone();
                    copyPath.Transform(mat);
                    currentData = GetModifyData(ref selList, ModifyMode.PATH, selPathIndex[0], (GraphicsPath)copyPath.Clone());
                    CreateData();
                }
            }
            else
            {
                List<int> indexList = GetSprayIndex(path);

                for (int i = 0; i < path.PointCount - 1; i++)
                {
                    dx = path.PathPoints[i + 1].X - path.PathPoints[i].X;
                    dy = path.PathPoints[i + 1].Y - path.PathPoints[i].Y;
                    dist = (float)Math.Sqrt(dx * dx + dy * dy);
                    dx /= dist;
                    dy /= dist;

                    float angle = (float)(180 * Math.Atan2(dy, dx) / Math.PI);
                    GraphicsPath tempPath = new GraphicsPath();

                    for (int j = 0; j < selPathIndex.Count; j++)
                    {
                        tempPath.AddPath(selList[selPathIndex[j]].path, false);
                    }

                    //# 곡선
                    if (idx < indexList.Count && i == indexList[idx])
                    {
                        Matrix mat = new Matrix();
                        PointF cPoint = new PointF();

                        switch (cmb_location.SelectedIndex)
                        {
                            case 0: //# center
                                cPoint = new PointF(tempPath.GetBounds().Left + tempPath.GetBounds().Width / 2, tempPath.GetBounds().Top + tempPath.GetBounds().Height / 2);
                                break;
                            case 1: //# left
                                cPoint = new PointF(tempPath.GetBounds().Left + tempPath.GetBounds().Width / 2, tempPath.GetBounds().Bottom);
                                break;
                            case 2: //# right
                                cPoint = new PointF(tempPath.GetBounds().Left + tempPath.GetBounds().Width / 2, tempPath.GetBounds().Top);
                                break;
                        }

                        mat.Translate(path.PathPoints[i].X - cPoint.X, path.PathPoints[i].Y - cPoint.Y);
                        mat.RotateAt(angle, cPoint);

                        GraphicsPath copyPath = (GraphicsPath)tempPath.Clone();
                        copyPath.Transform(mat);
                        currentData = GetModifyData(ref selList, ModifyMode.PATH, selPathIndex[0], (GraphicsPath)copyPath.Clone());
                        CreateData();
                        idx++;
                    }
                }
            }
        }

        private List<int> GetSprayIndex(GraphicsPath path)
        {
            List<int> indexList = new List<int>();
            float totLength = 0;

            //# 전체 길이 구하기
            for (int i = 0; i < path.PointCount - 1; i++)
            {
                float dx = path.PathPoints[i + 1].X - path.PathPoints[i].X;
                float dy = path.PathPoints[i + 1].Y - path.PathPoints[i].Y;
                totLength += (float)Math.Sqrt(dx * dx + dy * dy);
            }

            //# 간격 구하기
            float unitLength = totLength / Convert.ToSingle(txt_spray_cnt.Text);

            totLength = 0;

            //# 간격마다의 점 구하기
            for (int i = 0; i < path.PointCount - 1; i++)
            {
                float dx = path.PathPoints[i + 1].X - path.PathPoints[i].X;
                float dy = path.PathPoints[i + 1].Y - path.PathPoints[i].Y;
                totLength += (float)Math.Sqrt(dx * dx + dy * dy);

                while (totLength > unitLength || totLength > 10F)
                {
                    totLength = totLength - unitLength;
                    indexList.Add(i);
                }
            }
            return indexList;
        }

        private void GetSprayPoint(GraphicsPath path, ref List<float> angleList, ref List<PointF> pointList)
        {
            if (!(path.PointCount > 0)) return; //# path가 없으면 return

            //# create local variable
            pointList = new List<PointF>();     //# 좌표 list
            angleList = new List<float>();      //# 각도 list
            float totLength = 0;                //# 길이
            PointF sPoint;                      //# 계산된 좌표
            float dx;                           //# x 변화량
            float dy;                           //# y 변화량

            //# 전체 길이 구하기
            for (int i = 0; i < path.PointCount; i++)
            {
                //# 마지막 점이 닫힌 패스인 경우 첫점과의 길이도 더해준다.
                if (i == path.PointCount - 1)
                {
                    if (path.PathTypes[i] == 129)
                    {
                        dx = path.PathPoints[0].X - path.PathPoints[i].X;
                        dy = path.PathPoints[0].Y - path.PathPoints[i].Y;
                    }
                    else
                    {
                        //# 열린 패스이면 더이상 반복하지 않고 빠져나간다.
                        break;
                    }
                }
                else
                {
                    dx = path.PathPoints[i + 1].X - path.PathPoints[i].X;
                    dy = path.PathPoints[i + 1].Y - path.PathPoints[i].Y;
                }

                totLength += (float)Math.Sqrt(dx * dx + dy * dy);

            }

            //# 간격 구하기
            float unitLength = totLength / Convert.ToSingle(txt_spray_cnt.Text);

            totLength = 0;
            float extraLength = 0;

            for (int i = 0; i < path.PointCount; i++)
            {
                if (i == path.PointCount - 1)
                {
                    if (path.PathTypes[i] == 129)
                    {
                        dx = path.PathPoints[0].X - path.PathPoints[i].X;
                        dy = path.PathPoints[0].Y - path.PathPoints[i].Y;
                    }
                    else
                    {
                        //# 열린 패스이면 더이상 반복하지 않고 빠져나간다.
                        break;
                    }
                }
                else
                {
                    dx = path.PathPoints[i + 1].X - path.PathPoints[i].X;
                    dy = path.PathPoints[i + 1].Y - path.PathPoints[i].Y;
                }

                totLength += (float)Math.Sqrt(dx * dx + dy * dy);
                float angle = (float)(2 * Math.PI - Math.Atan2(dy, dx));

                //# 전체 길이가 단위길이보다 짧으면 전체길이를 반복문을 돌며 추가해준다.
                if (totLength <= unitLength) continue;

                //# 변화량 초기화
                dx = 0;
                dy = 0;

                //# 한 변을 단위 길이만큼 증가시키며 반복
                while (totLength >= unitLength)
                {
                    //# 직전 변에서 남은 길이만큼 빼주어 거리 계산
                    if (extraLength > 0)
                    {
                        dx += (int)((unitLength - extraLength) * (float)Math.Cos(angle));
                        dy += (int)((unitLength - extraLength) * -(float)Math.Sin(angle));
                        extraLength = 0;
                    }
                    //# 단위길이만큼의 변화량을 계산
                    else
                    {
                        dx += (int)(unitLength * (float)Math.Cos(angle));
                        dy += (int)(unitLength * -(float)Math.Sin(angle));
                    }

                    //# 포인트 계산 / 추가
                    sPoint = new PointF(path.PathPoints[i].X + dx, path.PathPoints[i].Y + dy);
                    pointList.Add(sPoint);
                    angleList.Add(angle);

                    //# 변화한 만큼 전체 길이 조정
                    totLength -= unitLength;
                }

                //# 남은 길이 계산
                extraLength = totLength % unitLength;
            }

            //# 추가된 포인트의 개수와 개수가 일치하지 않을 경우의 처리
            while (pointList.Count != Convert.ToInt32(txt_spray_cnt.Text))
            {
                //# 개수가 부족할 경우 첫점을 추가
                if (pointList.Count < Convert.ToInt32(txt_spray_cnt.Text))
                {
                    pointList.Add(path.PathPoints[0]);
                    angleList.Add(angleList[0]);
                }
                //# 개수가 클 경우 마지막 점을 제거
                else if (pointList.Count > Convert.ToInt32(txt_spray_cnt.Text))
                {
                    pointList.RemoveAt(pointList.Count - 1);
                    angleList.RemoveAt(angleList.Count - 1);
                }
            }
        }

        /// <summary>
        /// DataList Control 완료 후 데이터 적용
        /// </summary>
        public void Commit()
        {
            //# add current vecList
            if (!isMirror)
            {
                doVector.Add(vecList.ToList());
                doIdx = doVector.Count - 1;
            }
            else
            {
                oriDoVector.Add(oriList.ToList());
                refDoVector.Add(refList.ToList());
                refDoIdx = oriDoVector.Count - 1;
            }
        }

        //=======================================================================
        //# pictureBox Event
        //=======================================================================

        public void pictureBox1_MouseDoubleClick(object sender, MouseEventArgs e)
        {
            switch (mode)
            {
                case Mode.LINE:
                case Mode.RECT:
                case Mode.SQUARE:
                case Mode.CIRCLE:
                case Mode.ELLIPSE:
                    ClearSelList();
                    start = e.Location;
                    InputSize size = new InputSize(this, mode);
                    size.Location = MousePosition;
                    size.ShowDialog();
                    drawing = false;
                    return;
                case Mode.SELECT:
                    for (int i = 0; i < vecList.Count; i++)
                    {
                        RectangleF rect = vecList[i].path.GetBounds();
                        {
                            if (rect.Contains(e.Location) && vecList[i].isString)
                            {
                                //# set flag
                                if (mode == Mode.TEXT)
                                {
                                    nTextIndex = -1;
                                }
                                else
                                {
                                    nTextIndex = selPathIndex[0];
                                }

                                currentData.font = (Font)vecList[selPathIndex[0]].font.Clone();

                                //# init richTextBox
                                richTextBox.Visible = true;
                                richTextBox.Text = vecList[i].str;
                                richTextBox.Focus();
                                richTextBox.Select(richTextBox.Text.Length, 0);

                                selPathIndex.Add(i);
                                selTempPath.Add((GraphicsPath)vecList[selPathIndex[0]].path.Clone());
                                //# set current data
                                currentData = vecList[selPathIndex[0]];
                                currentData.path = (GraphicsPath)vecList[selPathIndex[0]].path.Clone();
                                currentData.isPathText = vecList[selPathIndex[0]].isPathText;

                                mode = Mode.PATHTEXT;
                                pnl_text.Visible = true;
                                tsb_type_path.Checked = true;
                                break;
                            }
                        }
                    }

                    pictureBox1.Refresh();
                    drawing = false;
                    return;
            }
        }

        public void pictureBox1_MouseDown(object sender, MouseEventArgs e)
        {
            if (bitCanvas.mDrawObject != CanvasType.VECTOR) return;
            if (e.Button != MouseButtons.Left) return;
            isClicked = true;
            pictureBox1.Focus();
            GraphicsPath tempPath;
            RectangleF pathPointsRect;

            //나중에 mode 조건 넣어줘야할 수도 있음 때에 따라
            if (doIdx != doVector.Count - 1) doVector.RemoveRange(doIdx + 1, doVector.Count - 1 - doIdx);

            if (refDoIdx != oriDoVector.Count - 1)
            {
                oriDoVector.RemoveRange(refDoIdx + 1, oriDoVector.Count - 1 - refDoIdx);
                refDoVector.RemoveRange(refDoIdx + 1, refDoVector.Count - 1 - refDoIdx);
            }

            if (mode == Mode.CIRCLE || mode == Mode.CURVE || mode == Mode.ELLIPSE || mode == Mode.FREEDRAW || mode == Mode.LINE ||
                mode == Mode.RECT || mode == Mode.SQUARE)
            {
                selPathIndex.Clear();
                selTempPath.Clear();
            }

            if (isMirror && !tsb_lock.Checked && !moveBaseLine)
            {
                if (GetPointToRegion(new Point(baseLine.X, 0), new Point(baseLine.X, pictureBox1.Height)).IsVisible(e.Location) && isHori)
                {
                    if (moveBaseLine) moveBaseLine = false;
                    else
                    {
                        drawing = true;
                        moveBaseLine = true;
                        ClearSelList();
                    }

                    mode = Mode.REVHORI;
                    oldPosition = e.Location;
                    return;
                }
                else if (GetPointToRegion(new Point(0, baseLine.Y), new Point(pictureBox1.Width, baseLine.Y)).IsVisible(e.Location) && !isHori)
                {
                    if (moveBaseLine) moveBaseLine = false;
                    else
                    {
                        drawing = true;
                        moveBaseLine = true;

                        ClearSelList();
                    }
                    mode = Mode.REVVERT;
                    oldPosition = e.Location;
                    return;
                }
            }
            else
            {
                moveBaseLine = false;
            }

            List<VectorData> selList;
            if (isMirror)
            {
                selList = oriList;
            }
            else
            {
                selList = vecList;
            }

            start = end = e.Location;

            switch (mode)
            {
                case Mode.SPRAY:
                    if (tcl_spray.SelectedTab.Text == "Spray")
                    {
                        if (cmb_method.Text == "Curve")
                        {
                            if (currentData.path == null)
                            {
                                currentData.path = new GraphicsPath();
                                curveList.Add(e.Location);
                                drawing = true;
                            }
                            //# connected drawing
                            else
                            {
                                curveList.Add(e.Location);
                                if (curveList.Count == 2)
                                {
                                    curveList.Add(e.Location);
                                    curveList.Add(e.Location);
                                    curveList[curveList.Count - 1] = curveList[curveList.Count - 3];
                                }
                                else if (curveList.Count == 5)
                                {
                                    curveList.RemoveAt(curveList.Count - 1);
                                    currentData.path = new GraphicsPath();
                                    currentData.path.AddBeziers(curveList.ToArray());
                                    currentData.path.Flatten();
                                    SprayData(currentData.path);
                                    currentData.path = null;
                                    curveList.Clear();
                                    pictureBox1.Refresh();
                                    drawing = false;
                                }
                            }
                        }
                        else
                        {
                            currentData.path = new GraphicsPath();
                            drawing = true;
                        }
                    }
                    break;
                case Mode.REVHORI:
                case Mode.REVVERT:
                    isMirror = true;
                    if (oriList.Count != 0)
                    {
                        //todo BaseLIne 다시조절하고 내려놓을 때
                        moveBaseLine = false;
                    }
                    else
                    {
                        //# 선택되어있는 항목이 있으면 oriList, refList에 저장
                        for (int i = 0; i < selPathIndex.Count; i++)
                        {
                            currentData = vecList[selPathIndex[i]];
                            CreateData();
                        }
                        selPathIndex.Sort((x1, x2) => x2.CompareTo(x1));

                        for (int i = 0; i < selPathIndex.Count; i++)
                        {
                            selOriIdx.Add(selPathIndex[i]);
                            //# oridata에 들어간 data는 지워줌
                            vecList.RemoveAt(selPathIndex[i]);

                        }

                        for (int i = 0; i < selPathIndex.Count; i++)
                        {
                            selPathIndex[i] = i;
                            selRefPath.Add((GraphicsPath)refList[i].path.Clone());
                        }

                        RestoreSelMode();
                        moveBaseLine = false;
                    }
                    break;
                case Mode.REPEAT:
                    //# selRect의 크기를 구함
                    GetBounds();
                    Console.WriteLine("{0}, {1}", selRects.Width, selRects.Height);

                    //# 반복 개수 계산
                    int rowCnt = (int)(Math.Abs(e.X - selRects.X) / selRects.Width) + 1;
                    int colCnt = (int)(Math.Abs(e.Y - selRects.Y) / selRects.Height) + 1;

                    //# 반복해서 그려질 Offset 계산
                    int xOffset, yOffset;
                    if (e.X - selRects.X > 0)
                        xOffset = (int)selRects.Width + Convert.ToInt32(txt_Xinvert.Text);
                    else
                        xOffset = (int)-selRects.Width - Convert.ToInt32(txt_Xinvert.Text);

                    if (e.Y - selRects.Y > 0)
                        yOffset = (int)selRects.Height + Convert.ToInt32(txt_Yinvert.Text);
                    else
                        yOffset = (int)-selRects.Height - Convert.ToInt32(txt_Yinvert.Text);

                    RepeatData(rowCnt, colCnt, xOffset, yOffset, true);
                    currentData.path = null;

                    RestoreSelMode();
                    break;
                case Mode.ZOOM:
                    drawing = true;
                    GetBounds();
                    zoomRect = selRects;
                    zoomIdx = -1;
                    for (int i = 0; i < selRectRect.Length; i++)
                    {
                        if (selRectRect[i].Contains(e.Location))
                        {
                            //대각선에 있는 점을 선택하기 위해서
                            zoomRectPt = new PointF(selRectRect[(i + 4) % 8].X + selRectRect[(i + 4) % 8].Width / 2, selRectRect[(i + 4) % 8].Y + selRectRect[(i + 4) % 8].Height / 2);
                            zoomIdx = i;
                            if (isMirror)
                            {
                                zoomRefIdx = GetRefRectPointIdx(i);
                                zoomRefRectPt = GetRefRectStPoint(zoomRectPt, isHori);
                            }
                            break;
                        }
                    }

                    if (zoomIdx == -1) return;
                    oldPosition = e.Location;
                    break;
                case Mode.TEXT:
                case Mode.PATHTEXT:
                    if (currentData.path == null)
                    {
                        //# reset current information
                        currentData.path = new GraphicsPath();

                        //# set position
                        richTextBox.Visible = true;
                        richTextBox.ResetText();
                        richTextBox.Focus();

                        drawing = true;
                        pictureBox1.Refresh();
                    }
                    else //# Click on empty space
                    {
                        richTextBox.Visible = false;
                        currentData.isString = true;

                        if (!(nTextIndex < 0))
                        {
                            //# Change existing data
                            ModifyData(ModifyMode.DATA, nTextIndex, currentData);
                            nTextIndex = -1;
                        }
                        else
                        {
                            CreateData();
                            Brush textBrush = new SolidBrush(Color.Black);
                            ModifyData(ModifyMode.BRUSH, vecList.Count - 1, textBrush);
                        }

                        currentData.path = null;
                        ClearSelList();
                        RestoreSelMode();
                    }
                    break;
                case Mode.SELECT:
                    oldPosition = e.Location;

                    //# Region selection part of path
                    SelectData(new RectangleF(e.X, e.Y, 1, 1));

                    if (!(selPathIndex.Count > 0))
                    {
                        if (keystate != 2)
                        {
                            ClearSelList();
                        }
                        selRegion = new GraphicsPath();
                    }

                    //# Remember current location
                    drawing = true;
                    pictureBox1.Refresh();
                    break;
                case Mode.PENTOOL:
                    drawing = true;
                    if (connPath == null)
                    {
                        connPath = new GraphicsPath();
                        pathByte = new List<byte>();
                        pathPoints = new List<PointF>();
                        if (!tsb_fill.Checked)
                        {
                            currentData.brush = new SolidBrush(Color.Transparent);
                        }
                        pathByte.Add(0);
                        pathPoints.Add(e.Location);
                    }
                    else
                    {
                        isFirst = true;
                        pathByte.Add(1);
                        pathPoints.Add(e.Location);
                    }
                    pictureBox1.Refresh();
                    break;
                case Mode.POINTSEL:
                    drawing = true;
                    oldPosition = e.Location;
                    if (selPathIndex.Count == 1)
                    {
                        if (!isMirror) SetSelPoint(vecList, e.Location);
                        else SetSelPoint(oriList, e.Location);
                        pictureBox1.Refresh();
                    }
                    break;
                case Mode.POINTCHANGE:
                    if (selPathIndex.Count != 1)
                    {
                        MessageBox.Show("Plz One Object!");
                        return;
                    }

                    drawing = true;
                    isFirst = true;
                    for (int i = 0; i < pathPoints.Count; i++)
                    {
                        pathPointsRect = new RectangleF(pathPoints[i].X - handleSize, pathPoints[i].Y - handleSize, handleSize * 2, handleSize * 2);
                        if (pathPointsRect.Contains(start))
                        {
                            PointF[] newHandle;
                            pathPointIdx = i;
                            if (pathTypes[i] == PathTypes.S && i > 0)
                            {
                                if (pathTypes[i - 1] == PathTypes.C2)
                                {
                                    pathTypes[i - 1] = PathTypes.N;
                                    newHandle = GetthrBezirPoint(pathPoints[i - 3], pathPoints[i - 2], pathPoints[i]);
                                    pathPoints[i - 2] = newHandle[0];
                                    pathPoints[i - 1] = newHandle[1];
                                }
                                if (pathTypes[i + 1] == PathTypes.C1)
                                {
                                    pathTypes[i + 1] = PathTypes.N;
                                    newHandle = GetthrBezirPoint(pathPoints[i], pathPoints[i + 2], pathPoints[i + 3]);
                                    pathPoints[i + 1] = newHandle[0];
                                    pathPoints[i + 2] = newHandle[1];
                                }
                            }
                            else if (pathTypes[i] == PathTypes.S && i == 0)
                            {
                                if (pathTypes[pathTypes.Count - 2] == PathTypes.C2)
                                {
                                    pathTypes[pathTypes.Count - 2] = PathTypes.N;
                                    newHandle = GetthrBezirPoint(pathPoints[pathTypes.Count - 4], pathPoints[pathTypes.Count - 3], pathPoints[pathTypes.Count - 1]);
                                    pathPoints[pathTypes.Count - 3] = newHandle[0];
                                    pathPoints[pathTypes.Count - 2] = newHandle[1];
                                }
                                if (pathTypes[i + 1] == PathTypes.C1)
                                {
                                    pathTypes[i + 1] = PathTypes.N;
                                    newHandle = GetthrBezirPoint(pathPoints[i], pathPoints[i + 2], pathPoints[i + 3]);
                                    pathPoints[i + 1] = newHandle[0];
                                    pathPoints[i + 2] = newHandle[1];
                                }
                            }
                            else if (i == pathPoints.Count - 1)//마지막 점까지 검사했을 때
                            {

                            }
                            SetthrBezirAndLinePath();
                            selTempPath[0] = new GraphicsPath(pathPoints.ToArray(), pathByte.ToArray());
                            if (isMirror) selRefPath[0] = GetMirrorPath((GraphicsPath)selTempPath[0].Clone(), baseLine);
                            break;
                        }
                    }
                    break;
                case Mode.POINTADD:
                    if (selPathIndex.Count != 1)
                    {
                        MessageBox.Show("Plz One Object!");
                        return;
                    }

                    drawing = true;
                    newC1C2.Clear();
                    pData.Clear();
                    //1. 포인트 추가하는 부분이 Line인지 Curve인지 판단해야함
                    //2. 선택한 Object의 가장 가까운 Line or Curve에 점을 추가해야 함
                    //Line Curve 조건문 걸어서 나눠야함
                    PathData data;
                    for (int i = 0; i < selTempPath[0].PointCount; i++)
                    {
                        if (i == selTempPath[0].PointCount - 1) break;
                        if (i % 3 == 0)
                        {
                            PointF tc;
                            for (float t = 0.0f; t < 1.0f; t += 0.01f)
                            {
                                tc = new PointF();
                                tc.X = GetBezierPosition(t, selTempPath[0].PathPoints[i].X,
                                                            selTempPath[0].PathPoints[i + 1].X,
                                                            selTempPath[0].PathPoints[i + 2].X,
                                                            selTempPath[0].PathPoints[i + 3].X);
                                tc.Y = GetBezierPosition(t, selTempPath[0].PathPoints[i].Y,
                                                            selTempPath[0].PathPoints[i + 1].Y,
                                                            selTempPath[0].PathPoints[i + 2].Y,
                                                            selTempPath[0].PathPoints[i + 3].Y);

                                data.pBounds = new RectangleF(tc.X - 1.5f, tc.Y - 1.5f, 3, 3);
                                data.pIdx = i;
                                data.isCurve = true;
                                pData.Add(data);
                            }
                        }
                    }
                    //일단 Curve 라는 가정 하에 
                    for (int i = 0; i < selList.Count; i++)
                    {
                        for (int j = 0; j < pData.Count; j++)
                        {
                            if (pData[j].pBounds.Contains(e.Location))
                            {
                                addedPt = GetAddedBezierPoint(e.Location, selList[i].path.PathPoints[pData[j].pIdx],
                                                                          selList[i].path.PathPoints[pData[j].pIdx + 1],
                                                                          selList[i].path.PathPoints[pData[j].pIdx + 2],
                                                                          selList[i].path.PathPoints[pData[j].pIdx + 3]);
                                if (addedPt.Y == 0) return;

                                PointF[] tangentLine = GetTangentLine(t_value, selList[i].path.PathPoints[pData[j].pIdx],
                                                                               selList[i].path.PathPoints[pData[j].pIdx + 1],
                                                                               selList[i].path.PathPoints[pData[j].pIdx + 2],
                                                                               selList[i].path.PathPoints[pData[j].pIdx + 3]).ToArray();

                                newC1C2.Add(GetIntersectPoint2D(selList[i].path.PathPoints[pData[j].pIdx],
                                                                selList[i].path.PathPoints[pData[j].pIdx + 1],
                                                                tangentLine[0], tangentLine[1]));
                                newC1C2.Add(GetIntersectPoint2D(selList[i].path.PathPoints[pData[j].pIdx + 2],
                                                                selList[i].path.PathPoints[pData[j].pIdx + 3],
                                                                tangentLine[0], tangentLine[1]));

                                pathPoints = selList[i].path.PathPoints.ToList();
                                pathByte = selList[i].path.PathTypes.ToList();
                                pathTypes = pathTypesList[selPathIndex[0]].ToList();

                                pathPoints.Insert(pData[j].pIdx + 2, newC1C2[0]);
                                pathPoints.Insert(pData[j].pIdx + 3, addedPt);
                                pathPoints.Insert(pData[j].pIdx + 4, newC1C2[1]);

                                pathByte.Insert(pData[j].pIdx + 2, 3);
                                pathByte.Insert(pData[j].pIdx + 3, 3);
                                pathByte.Insert(pData[j].pIdx + 4, 3);

                                pathTypes.Insert(pData[j].pIdx + 2, PathTypes.C2);
                                pathTypes.Insert(pData[j].pIdx + 3, PathTypes.S);
                                pathTypes.Insert(pData[j].pIdx + 4, PathTypes.C1);

                                PointF[] newPt1 = GetthrBezirPoint(pathPoints[pData[j].pIdx], pathPoints[pData[j].pIdx + 2], pathPoints[pData[j].pIdx + 3]);
                                PointF[] newPt2 = GetthrBezirPoint(pathPoints[pData[j].pIdx + 3], pathPoints[pData[j].pIdx + 4], pathPoints[pData[j].pIdx + 6]);

                                pathPoints[pData[j].pIdx + 1] = newPt1[0];
                                pathPoints[pData[j].pIdx + 2] = newPt1[1];
                                pathPoints[pData[j].pIdx + 4] = newPt2[0];
                                pathPoints[pData[j].pIdx + 5] = newPt2[1];

                                ModifyData(ModifyMode.PATH, 0, new GraphicsPath(pathPoints.ToArray(), pathByte.ToArray()));
                                pathTypesList[selPathIndex[0]] = pathTypes.ToList();
                                selTempPath[0] = new GraphicsPath(pathPoints.ToArray(), pathByte.ToArray());
                                break;
                            }
                        }
                    }

                    pictureBox1.Refresh();
                    break;
                case Mode.POINTDEL:
                    if (selPathIndex.Count != 1)
                    {
                        MessageBox.Show("Plz One Object!");
                        return;
                    }

                    drawing = true;
                    for (int i = 0; i < vecList[selPathIndex[0]].path.PointCount; i++)
                    {
                        pathPointsRect = new RectangleF(vecList[selPathIndex[0]].path.PathPoints[i].X - handleSize, vecList[selPathIndex[0]].path.PathPoints[i].Y - handleSize, handleSize * 2, handleSize * 2);
                        if (pathPointsRect.Contains(e.Location))
                        {
                            if (pathByte.Count == 2) return; //점이 2개 남거나
                            else if (pathByte.Count == 4 && (pathByte[pathByte.Count - 1] == 3 || pathByte[pathByte.Count - 1] == 131)) return;//4개인데 곡선이면 return

                            switch (pathByte[i])
                            {
                                case 0:
                                    if (pathByte[pathByte.Count - 1] == 131)
                                    {
                                        pathByte.RemoveRange(pathByte.Count - 2, 2);
                                        pathPoints.RemoveRange(pathPoints.Count - 2, 2);
                                        pathTypes.RemoveRange(pathTypes.Count - 2, 2);

                                        pathByte.RemoveRange(0, 2);
                                        pathPoints.RemoveRange(0, 2);
                                        pathTypes.RemoveRange(0, 2);

                                        pathByte.Add(pathByte[0]);
                                        pathPoints.Add(pathPoints[0]);
                                        pathTypes.Add(pathTypes[0]);

                                        pathByte.Add(pathByte[1]);
                                        pathPoints.Add(pathPoints[1]);
                                        pathTypes.Add(pathTypes[1]);

                                        pathByte.RemoveAt(0);
                                        pathPoints.RemoveAt(0);
                                        pathTypes.RemoveAt(0);

                                        pathByte[0] = 0;
                                        pathByte[pathByte.Count - 1] = 131;

                                        pathTypes[0] = PathTypes.S;
                                        pathTypes[pathTypes.Count - 1] = PathTypes.CC;
                                    }
                                    else if (pathByte[i + 1] == 1)
                                    {
                                        pathByte[i + 1] = 0;
                                        pathByte.RemoveAt(i);
                                        pathTypes[i + 1] = PathTypes.S;
                                        pathTypes.RemoveAt(i);
                                        pathPoints.RemoveAt(i);
                                    }
                                    else if (pathByte[i + 1] == 3)
                                    {
                                        pathByte[i + 3] = 0;
                                        pathByte.RemoveRange(i, 3);
                                        pathTypes[i + 3] = PathTypes.S;
                                        pathTypes.RemoveRange(i, 3);
                                        pathPoints.RemoveRange(i, 3);
                                    }
                                    break;
                                case 1:
                                    pathByte.RemoveAt(i);
                                    pathPoints.RemoveAt(i);
                                    pathTypes.RemoveAt(i);
                                    break;
                                case 3:
                                    if (pathTypes[i] == PathTypes.S)//3이면서 Control Point가 아닐 때
                                    {
                                        if (pathByte[i - 1] == 3 && pathByte[i + 1] == 3)// 앞 뒤가 3이면 앞 c2, 뒤 c1도 같이 날림
                                        {
                                            pathByte.RemoveRange(i - 1, 3);
                                            pathPoints.RemoveRange(i - 1, 3);
                                        }
                                        else if (pathByte[i + 1] == 1)//다음 게 1일 경우 곡선으로 속성 바꿈
                                        {
                                            pathByte[i + 1] = 3;
                                            pathByte.RemoveAt(i);
                                            pathPoints.RemoveAt(i);
                                        }
                                    }
                                    break;
                                case 129:
                                    pathByte[i - 1] = 129;
                                    pathByte.RemoveAt(i);
                                    pathPoints.RemoveAt(i);
                                    break;
                                case 131:
                                    break;
                                default:
                                    break;
                            }
                            GraphicsPath delPath = new GraphicsPath(pathPoints.ToArray(), pathByte.ToArray());
                            selTempPath[0] = (GraphicsPath)delPath.Clone();
                            ModifyData(ModifyMode.PATH, selPathIndex[0], delPath);
                            pathTypesList[selPathIndex[0]] = pathTypes;
                            Commit();
                            pictureBox1.Refresh();
                            break;
                        }
                    }
                    break;
                case Mode.KNIFE:
                    drawing = true;

                    if (selPathIndex.Count == 0) return;

                    for (int i = 0; i < selList.Count; i++)
                    {
                        tempPath = (GraphicsPath)selList[i].path.Clone();
                        tempPath.Flatten();
                        for (int j = 0; j < tempPath.PointCount; j++)
                        {
                            Region region;
                            if (j == tempPath.PointCount - 1)
                                region = GetPointToRegion(Point.Round(tempPath.PathPoints[j]), Point.Round(tempPath.PathPoints[0]));
                            else
                                region = GetPointToRegion(Point.Round(tempPath.PathPoints[j]), Point.Round(tempPath.PathPoints[j + 1]));

                            if (region.IsVisible(e.Location))
                            {
                                if ((Keyboard.GetKeyStates(Key.LeftCtrl) & KeyStates.Down) <= 0)                //CTRL 안눌리면 리스트 지우고
                                {
                                    selPathIndex.Clear();
                                    selTempPath.Clear();
                                }

                                if (selPathIndex.Contains(i))
                                {
                                    selTempPath.RemoveAt(selPathIndex.IndexOf(i));
                                    selPathIndex.Remove(i);
                                    break;
                                }
                                selPathIndex.Add(i);
                                selTempPath.Add((GraphicsPath)selList[i].path.Clone());

                                pictureBox1.Refresh();
                                break;
                            }
                        }
                    }

                    currentData.path = new GraphicsPath();      //# 변경된 기존 경로
                    GraphicsPath newPath = new GraphicsPath();  //# 추가되는 새 경로

                    bool isCutting = false;

                    tempPath = (GraphicsPath)selList[selPathIndex[0]].path.Clone();
                    tempPath.Flatten();
                    for (int i = 0; i < tempPath.PointCount; i++)
                    {
                        Region region = new Region();
                        if (i == tempPath.PointCount - 1)
                        {
                            if (tempPath.PathTypes[i] == 129)
                            {
                                region = GetPointToRegion(Point.Round(tempPath.PathPoints[i]), Point.Round(tempPath.PathPoints[0]));
                            }
                        }
                        else
                        {
                            region = GetPointToRegion(Point.Round(tempPath.PathPoints[i]), Point.Round(tempPath.PathPoints[i + 1]));
                        }
                        if (region.IsVisible(e.Location))
                        {
                            //# sjpark - 포인트 선택 시 패스위의 점이 아닌 마우스 포인트로 추가되어 패스가 일그러지던 현상 개선
                            PointF tp = e.Location;
                            if (Math.Abs(e.X - tempPath.PathPoints[i].X) < 10)   //# 10 : offset
                            {
                                tp.X = tempPath.PathPoints[i].X;
                            }

                            if (Math.Abs(e.Y - tempPath.PathPoints[i].Y) < 10)
                            {
                                tp.Y = tempPath.PathPoints[i].Y;
                            }

                            if (!isCutting)
                            {
                                currentData.path.AddLine(tempPath.PathPoints[i], tp);
                            }

                            if (i == tempPath.PointCount - 1)
                            {
                                if (tempPath.PathTypes[i] == 129)
                                {
                                    newPath.AddLine(tp, tempPath.PathPoints[0]);
                                }
                            }
                            else
                            {
                                newPath.AddLine(tp, tempPath.PathPoints[i + 1]);
                            }

                            isCutting = true;
                            continue;
                        }

                        if (isCutting)  //# 잘린 후의 경로는 new 경로에
                        {
                            if (i == tempPath.PointCount - 1)
                            {
                                if (tempPath.PathTypes[i] == 129)
                                {
                                    //# 마지막점이 닫힌 패스이면 current Data의 경로를 new data에 추가 -> current 삭제
                                    newPath.AddLine(tempPath.PathPoints[i], tempPath.PathPoints[0]);

                                    for (int j = 0; j < currentData.path.PointCount - 1; j++)
                                    {
                                        newPath.AddLine(currentData.path.PathPoints[j], currentData.path.PathPoints[j + 1]);
                                    }

                                    currentData.path = null;
                                }
                            }
                            else
                            {
                                newPath.AddLine(tempPath.PathPoints[i], tempPath.PathPoints[i + 1]);
                            }
                        }
                        else            //# 잘리기 전의 경로는 temp 경로에
                        {
                            if (i == tempPath.PointCount - 1)
                            {
                                if (tempPath.PathTypes[i] == 129)
                                {
                                    currentData.path.AddLine(tempPath.PathPoints[i], tempPath.PathPoints[0]);
                                }
                            }
                            else
                            {
                                currentData.path.AddLine(tempPath.PathPoints[i], tempPath.PathPoints[i + 1]);
                            }
                        }
                    }

                    VectorData temp = selList[selPathIndex[0]];

                    //# create new path
                    if (currentData.path != null)
                    {
                        currentData = GetModifyData(ref selList, ModifyMode.PATH, selPathIndex[0], (GraphicsPath)currentData.path.Clone());
                        CreateData(selPathIndex[0] + 1);
                    }

                    if (newPath != null)
                    {
                        currentData = GetModifyData(ref selList, ModifyMode.PATH, selPathIndex[0], (GraphicsPath)newPath.Clone());
                        CreateData(selPathIndex[0] + 1);
                    }

                    //# delete ori path
                    DeleteData(selPathIndex[0]);
                    drawing = true;
                    pictureBox1.Refresh();
                    drawing = false;
                    Commit();
                    break;
                case Mode.SELLASSO:
                    //# reset current information
                    currentData.path = new GraphicsPath();
                    //# set position
                    drawing = true;
                    isLasso = true;
                    break;
                case Mode.ROTATE:
                    for (int i = 0; i < selPathIndex.Count; i++)
                    {
                        ModifyData(ModifyMode.PATH, selPathIndex[i], (GraphicsPath)selTempPath[i].Clone());
                    }
                    RestoreSelMode();
                    break;
                case Mode.LINE:
                case Mode.CURVE:
                case Mode.RECT:
                case Mode.SQUARE:
                case Mode.ELLIPSE:
                case Mode.CIRCLE:
                case Mode.FREEDRAW:
                    //# reset current information
                    currentData.path = new GraphicsPath();
                    if (!tsb_fill.Checked)
                    {
                        currentData.brush = new SolidBrush(Color.Transparent);
                    }
                    else
                    {
                        currentData.brush = new SolidBrush(currentData.pen.Color);
                    }

                    if (mode == Mode.FREEDRAW)
                    {
                        changePt.Clear();
                        curvedByte.Clear();

                        /*
                        changePt.Add(start);
                        curvedByte.Add(0);
                        */
                    }
                    //# set position
                    drawing = true;
                    break;
                default:
                    //# set position
                    drawing = true;
                    break;
            }
        }

        public void pictureBox1_MouseMove(object sender, MouseEventArgs e)
        {
            if (bitCanvas.mDrawObject != CanvasType.VECTOR) return;
            if (!drawing) return;

            end = e.Location;

            if (moveBaseLine) baseLine = e.Location;

            switch (mode)
            {
                case Mode.SPRAY:
                    if (tcl_spray.SelectedTab.Text == "Spray")
                    {
                        if (cmb_method.Text == "Free")
                        {
                            if (currentData.path == null) return;
                            currentData.path.AddLine(start, end);

                            start = e.Location;
                        }
                        else if (cmb_method.Text == "Line")
                        {
                            if ((Keyboard.GetKeyStates(Key.LeftShift) & KeyStates.Down) > 0)
                            {
                                SetIntuitiveLine();
                            }
                            else
                            {
                                currentData.path = new GraphicsPath();
                                currentData.path.AddLine(start, end);
                            }
                        }
                        else if (cmb_method.Text == "Curve")
                        {
                            if (currentData.path == null) return;
                            currentData.path = new GraphicsPath();
                            if (curveList.Count > 1)
                            {
                                curveList[1] = GetSymmetryPoint(curveList[3], e.Location);
                                curveList[2] = GetSymmetryPoint(curveList[3], e.Location);

                                int bX = (int)(curveList[0].X + (2.0f / 3.0f) * (curveList[1].X - curveList[0].X));
                                int bY = (int)(curveList[0].Y + (2.0f / 3.0f) * (curveList[1].Y - curveList[0].Y));
                                int cX = (int)(curveList[3].X + (2.0f / 3.0f) * (curveList[1].X - curveList[3].X));
                                int cY = (int)(curveList[3].Y + (2.0f / 3.0f) * (curveList[1].Y - curveList[3].Y));

                                curveList[1] = new Point(bX, bY);
                                curveList[2] = new Point(cX, cY);

                                currentData.path.AddBeziers(curveList.ToArray());
                            }
                            else
                            {
                                currentData.path.AddLine(start, end);
                            }
                        }
                        else if (cmb_method.Text == "Rectangle")
                        {
                            currentData.path = new GraphicsPath();
                            if ((Keyboard.GetKeyStates(Key.LeftShift) & KeyStates.Down) > 0)
                                currentData.path.AddRectangle(GetSquare(start.X, start.Y, end.X, end.Y));

                            else
                                currentData.path.AddRectangle(GetRect(start.X, start.Y, end.X, end.Y));
                        }
                        else if (cmb_method.Text == "Ellipse")
                        {
                            currentData.path = new GraphicsPath();
                            if ((Keyboard.GetKeyStates(Key.LeftShift) & KeyStates.Down) > 0)
                                currentData.path.AddEllipse(GetSquare(start.X, start.Y, end.X, end.Y));
                            else
                                currentData.path.AddEllipse(GetRect(start.X, start.Y, end.X, end.Y));
                        }
                        pictureBox1.Refresh();
                    }
                    break;
                case Mode.REVHORI:
                case Mode.REVVERT:
                    Matrix translate = new Matrix();
                    if (moveBaseLine)
                    {
                        if (isHori)
                        {
                            translate.Translate(e.Location.X - oldPosition.X, 0);
                        }
                        else
                        {
                            translate.Translate(0, e.Location.Y - oldPosition.Y);
                        }

                        for (int i = 0; i < oriList.Count; i++)
                        {
                            GraphicsPath oriPath = (GraphicsPath)oriList[i].path.Clone();
                            oriPath.Transform(translate);
                            ModifyData(ModifyMode.PATH, i, oriPath);

                            GraphicsPath refPath = (GraphicsPath)refList[i].path.Clone();
                            refPath.Transform(translate);
                            ModifyData(ModifyMode.PATH, i, refPath);
                        }
                    }
                    oldPosition = e.Location;
                    pictureBox1.Refresh();
                    break;
                case Mode.REPEAT:
                    //# selRect의 크기를 구함
                    GetBounds();

                    //# 반복 개수 계산
                    int rowCnt = (int)(Math.Abs(e.X - selRects.X) / selRects.Width) + 1;
                    int colCnt = (int)(Math.Abs(e.Y - selRects.Y) / selRects.Height) + 1;

                    //# 반복해서 그려질 Offset 계산
                    int xOffset, yOffset;
                    if (e.X - selRects.X > 0)
                        xOffset = (int)selRects.Width + Convert.ToInt32(txt_Xinvert.Text);
                    else
                        xOffset = (int)-selRects.Width - Convert.ToInt32(txt_Xinvert.Text);

                    if (e.Y - selRects.Y > 0)
                        yOffset = (int)selRects.Height + Convert.ToInt32(txt_Yinvert.Text);
                    else
                        yOffset = (int)-selRects.Height - Convert.ToInt32(txt_Yinvert.Text);

                    //# 적용 전
                    RepeatData(rowCnt, colCnt, xOffset, yOffset, false);
                    pictureBox1.Refresh();
                    break;
                case Mode.ZOOM://와드
                    if (zoomIdx == -1) return;
                    GetBounds();
                    if (!isMirror)
                        selTempPath = GetZoomPath(e.Location, vecList, zoomPath, zoomRectPt, selRects, zoomIdx).ToList();
                    else
                    {
                        selTempPath = GetZoomPath(e.Location, oriList, zoomPath, zoomRectPt, selRects, zoomIdx).ToList();
                        if (isHori)
                            selRefPath = GetZoomPath(GetRefRectStPoint(e.Location, true), refList, zoomRefPath, zoomRefRectPt, selRefRects, zoomRefIdx).ToList();
                        else
                            selRefPath = GetZoomPath(GetRefRectStPoint(e.Location, false), refList, zoomRefPath, zoomRefRectPt, selRefRects, zoomRefIdx).ToList();
                    }

                    oldPosition = e.Location;
                    pictureBox1.Refresh();
                    break;
                case Mode.FREEDRAW:
                case Mode.SELLASSO:
                    if (currentData.path == null) return;
                    currentData.path.AddLine(start, end);
                    currentData.isString = false;

                    start = e.Location;
                    pictureBox1.Refresh();
                    break;
                case Mode.RECT:
                case Mode.SQUARE:
                case Mode.ELLIPSE:
                case Mode.CIRCLE:
                case Mode.LINE:
                case Mode.CURVE:
                    currentData.path = new GraphicsPath();
                    if (mode == Mode.RECT)
                    {
                        if ((Keyboard.GetKeyStates(Key.LeftShift) & KeyStates.Down) > 0)
                            currentData.path.AddRectangle(GetSquare(start.X, start.Y, end.X, end.Y));

                        else
                            currentData.path.AddRectangle(GetRect(start.X, start.Y, end.X, end.Y));
                    }
                    else if (mode == Mode.SQUARE)
                    {
                        currentData.path.AddRectangle(GetSquare(start.X, start.Y, end.X, end.Y));
                    }
                    else if (mode == Mode.ELLIPSE)
                    {
                        if ((Keyboard.GetKeyStates(Key.LeftShift) & KeyStates.Down) > 0)
                            currentData.path.AddEllipse(GetSquare(start.X, start.Y, end.X, end.Y));
                        else
                            currentData.path.AddEllipse(GetRect(start.X, start.Y, end.X, end.Y));
                    }
                    else if (mode == Mode.CIRCLE)
                    {
                        currentData.path.AddEllipse(GetSquare(start.X, start.Y, end.X, end.Y));
                    }
                    else if (mode == Mode.LINE)
                    {
                        if ((Keyboard.GetKeyStates(Key.LeftShift) & KeyStates.Down) > 0)
                        {
                            SetIntuitiveLine();
                        }
                        else
                        {
                            currentData.path.AddLine(start, end);
                        }
                    }
                    else if (mode == Mode.CURVE)
                    {
                        SetArcPath(start.X, start.Y, end.X, end.Y, currentData.path);
                    }
                    pictureBox1.Refresh();
                    break;
                case Mode.SELECT:
                    if (!isClicked) return;

                    if (selRegion != null)
                    {
                        selRegion = new GraphicsPath();
                        selRegion.AddRectangle(GetRect(start.X, start.Y, end.X, end.Y));
                        oldPosition = e.Location;
                        pictureBox1.Refresh();
                        return;
                    }

                    //xOffset = e.Location.X - oldPosition.X;
                    //yOffset = e.Location.Y - oldPosition.Y;

                    MoveData(e.Location.X - start.X, e.Location.Y - start.Y);
                    oldPosition = e.Location;

                    drawing = true;
                    pictureBox1.Refresh();
                    break;
                case Mode.PENTOOL:
                    if (!isClicked) return;
                    if (pathPoints.Count == 1) return;

                    if (isFirst)//Bezier Curve Make
                    {
                        isFirst = false;
                        pathByte.RemoveAt(pathByte.Count - 1);
                        pathPoints.RemoveAt(pathPoints.Count - 1);
                        for (int i = 0; i < 3; i++)
                        {
                            pathByte.Add(3);
                            pathPoints.Add(e.Location);
                        }
                    }
                    else
                    {
                        pathPoints[pathPoints.Count - 3] = GetSymmetryPoint(pathPoints[pathPoints.Count - 1], e.Location);
                        pathPoints[pathPoints.Count - 2] = GetSymmetryPoint(pathPoints[pathPoints.Count - 1], e.Location);

                        PointF[] newHandle;
                        newHandle = GetthrBezirPoint(pathPoints[pathPoints.Count - 4], pathPoints[pathPoints.Count - 3], pathPoints[pathPoints.Count - 1]);
                        pathPoints[pathPoints.Count - 3] = newHandle[0];
                        pathPoints[pathPoints.Count - 2] = newHandle[1];
                    }
                    connPath = new GraphicsPath(pathPoints.ToArray(), pathByte.ToArray());
                    pictureBox1.Refresh();
                    break;
                case Mode.POINTCHANGE:
                    if (!isClicked || pathPointIdx == -1) return;//선택한 점이 없거나 클릭상태가 아니면 return

                    RectangleF pathPointsRect = new RectangleF(pathPoints[pathPointIdx].X - handleSize, pathPoints[pathPointIdx].Y - handleSize, handleSize * 2, handleSize * 2);
                    if (pathPointsRect.Contains(e.Location)) return;

                    if (isFirst)//처음에만 Handle Point를 추가
                    {
                        //Mouse를 당길 때 Control Point는 항상 C1이 기준이 된다
                        //반시계방향 검사(내가 선택한 점의 대칭에 있는 점)
                        if (pathPointIdx == 0)//내가 선택한 점이 첫 점일 때
                        {
                            if (pathTypes[pathTypes.Count - 1] == PathTypes.CC)//닫힌 커브일 경우
                            {
                                if (pathTypes[pathTypes.Count - 3] == PathTypes.C1)
                                {
                                    thrBezirIdx = GetThrBezierIdx(pathTypes.Count - 2);
                                    pathPoints[pathTypes.Count - 2] = GetSymmetryPoint(e.Location, pathPoints[pathPointIdx]);
                                    pathPoints[pathTypes.Count - 3] = thrBezirData[thrBezirIdx];

                                    //그리고 이제 필요없어진 3Bezier는 제거
                                    thrBezirData.RemoveAt(thrBezirIdx);

                                    pathTypes[pathTypes.Count - 2] = PathTypes.C2; //N인 속성을 C2로 
                                    pathPointSymIdx = pathTypes.Count - 2;//대칭에 있는 4Bezir 이므로 pathPointSymIdx 갱신
                                }
                                else
                                {
                                    //핸들이 없으므로 3Bezier를 만들어야한다
                                    pathTypes.Insert(pathTypes.Count - 2, PathTypes.C2);
                                    pathTypes.Insert(pathTypes.Count - 3, PathTypes.N);

                                    //pathByte[pathPointIdx] = 3;
                                    pathByte.Insert(pathTypes.Count - 2, 3);
                                    pathByte.Insert(pathTypes.Count - 3, 3);

                                    pathPoints.Insert(pathTypes.Count - 2, pathPoints[pathPointIdx]);
                                    pathPoints.Insert(pathTypes.Count - 3, pathPoints[pathPointIdx]);

                                    //Handle이 없는경우는 3Bezier 이므로 thrBezierData에 추가해야함
                                    thrBezirSymIdx = GetThrBezierIdx(pathTypes.Count - 3) + 1;
                                    pathPointSymIdx = pathTypes.Count - 3;
                                    thrBezirData.Insert(thrBezirSymIdx, pathPoints[pathTypes.Count - 3]);
                                }
                            }
                            else if (pathTypes[pathTypes.Count - 1] == PathTypes.CL)//닫힌 직선일 경우
                            {
                                pathTypes[pathTypes.Count - 1] = PathTypes.S;
                                pathTypes.Add(PathTypes.N);
                                pathTypes.Add(PathTypes.C2);
                                pathTypes.Add(PathTypes.CC);

                                //pathByte[pathByte.Count - 1] = 1;
                                //바로 앞점이 곡선이면 곡선 시작점인3으로 값을 바꿈
                                if (pathByte[pathByte.Count - 2] == 3) pathByte[pathByte.Count - 1] = 3;
                                else pathByte[pathByte.Count - 1] = 1;

                                pathByte.Add(3);
                                pathByte.Add(3);
                                pathByte.Add(131);

                                pathPoints.Add(pathPoints[pathPointIdx]);
                                pathPoints.Add(pathPoints[pathPointIdx]);
                                pathPoints.Add(pathPoints[pathPointIdx]);

                                thrBezirSymIdx = GetThrBezierIdx(pathPoints.Count - 3);
                                pathPointSymIdx = pathTypes.Count - 3;
                                thrBezirData.Insert(thrBezirSymIdx, pathPoints[pathPointIdx]);
                            }
                            else//열려 있는 경우
                            {

                            }
                        }
                        else//선택한 점이 첫 점이 아닐 때 
                        {
                            //if (pathTypes[pathTypes.Count - 1] == PathTypes.CC)//닫힌 커브일 경우
                            //{
                            if (pathTypes[pathPointIdx - 1] == PathTypes.N)//3Bezir로 Handle이 남아있는 경우
                            {
                                thrBezirIdx = GetThrBezierIdx(pathPointIdx - 1);
                                pathPoints[pathPointIdx - 1] = GetSymmetryPoint(e.Location, pathPoints[pathPointIdx]);
                                pathPoints[pathPointIdx - 2] = thrBezirData[thrBezirIdx];

                                //그리고 이제 필요없어진 3Bezier는 제거
                                thrBezirData.RemoveAt(thrBezirIdx);

                                pathTypes[pathPointIdx - 1] = PathTypes.C2; //N인 속성을 C2로 
                                pathPointSymIdx = pathPointIdx - 1;//대칭에 있는 4Bezir 이므로 pathPointSymIdx 갱신
                            }
                            else//Handle이 없어서 추가해야 하는 경우
                            {
                                //핸들이 없으므로 3Bezier를 만들어야한다
                                pathTypes.Insert(pathPointIdx, PathTypes.C2);
                                pathTypes.Insert(pathPointIdx, PathTypes.N);

                                pathByte[pathPointIdx] = 3;
                                pathByte.Insert(pathPointIdx, 3);
                                pathByte.Insert(pathPointIdx, 3);

                                pathPoints.Insert(pathPointIdx, pathPoints[pathPointIdx]);
                                pathPoints.Insert(pathPointIdx, pathPoints[pathPointIdx]);
                                pathPointIdx = pathPointIdx + 2;

                                //pathByte[pathPointIdx] = 3;
                                //pathTypes[pathPointIdx] = PathTypes.S;

                                if (pathTypes[pathPointIdx] == PathTypes.CL)
                                {
                                    pathByte[pathPointIdx] = 131;
                                    pathTypes[pathPointIdx] = PathTypes.CC;
                                }

                                //Handle이 없는경우는 3Bezier 이므로 thrBezierData에 추가해야함
                                thrBezirSymIdx = GetThrBezierIdx(pathPointIdx);
                                //pathPointSymIdx = pathPointIdx + 2;
                                pathPointSymIdx = pathPointIdx - 2;
                                thrBezirData.Insert(thrBezirSymIdx, pathPoints[pathPointIdx]);
                            }
                            //}
                            /*
                            else if (pathTypes[pathTypes.Count - 1] == PathTypes.CL)//닫힌 직선일 경우
                            {

                            }
                            else//열려 있는 경우
                            {

                            }
                            */
                        }

                        //시계방향 검사(내가 선택한 점)
                        if (pathTypes[(pathPointIdx + 1) % pathTypes.Count] == PathTypes.N)//3Bezier로 handle이 남아있는 경우
                        {
                            thrBezirIdx = GetThrBezierIdx(pathPointIdx + 1);
                            pathPoints[pathPointIdx + 1] = e.Location;
                            pathPoints[pathPointIdx + 2] = thrBezirData[thrBezirIdx];

                            thrBezirData.RemoveAt(thrBezirIdx);

                            pathTypes[pathPointIdx + 1] = PathTypes.C1; //N인 속성을 C1로 바꿔주고
                            pathPointIdx = pathPointIdx + 1;//선택한 점을 S에서 C1으로 옮긴다

                            isThrBezir = false; //내가 선택한 점이 3 Bezier인지 확인
                        }
                        else//Handle이 없어서 추가해야하는 경우
                        {
                            if (pathTypes[pathPointIdx] == PathTypes.CC)
                            {
                                pathTypes[pathPointIdx] = PathTypes.S;
                                pathTypes.Insert(pathPointIdx + 1, PathTypes.C1);
                                pathTypes.Insert(pathPointIdx + 2, PathTypes.N);
                                pathTypes.Insert(pathPointIdx + 3, PathTypes.CC);

                                pathByte[pathPointIdx] = 3;
                                pathByte.Insert(pathPointIdx + 1, 3);
                                pathByte.Insert(pathPointIdx + 2, 3);
                                pathByte.Insert(pathPointIdx + 3, 131);

                                pathPoints.Insert(pathPointIdx + 1, pathPoints[pathPointIdx]);
                                pathPoints.Insert(pathPointIdx + 2, pathPoints[pathPointIdx]);
                                pathPoints.Insert(pathPointIdx + 3, pathPoints[0]);
                            }
                            else
                            {
                                if (pathTypes[pathTypes.Count - 1] == PathTypes.CL && pathPointIdx == pathPoints.Count - 1)
                                {
                                    pathTypes[pathTypes.Count - 1] = PathTypes.S;
                                    pathTypes.Insert(pathPointIdx + 1, PathTypes.C1);
                                    pathTypes.Insert(pathPointIdx + 2, PathTypes.N);
                                    pathTypes.Insert(pathPointIdx + 3, PathTypes.CC);

                                    pathByte[pathByte.Count - 1] = 3;
                                    pathByte.Insert(pathPointIdx + 1, 3);
                                    pathByte.Insert(pathPointIdx + 2, 3);
                                    pathByte.Insert(pathPointIdx + 3, 131);

                                    pathPoints.Insert(pathPointIdx + 1, pathPoints[pathPointIdx]);
                                    pathPoints.Insert(pathPointIdx + 2, pathPoints[pathPointIdx]);
                                    pathPoints.Insert(pathPointIdx + 3, pathPoints[0]);

                                    if (pathPointSymIdx > pathPointIdx) pathPointSymIdx = pathPointSymIdx + 3;
                                }
                                else
                                {
                                    pathTypes.Insert(pathPointIdx + 1, PathTypes.C1);
                                    pathTypes.Insert(pathPointIdx + 2, PathTypes.N);

                                    pathByte.Insert(pathPointIdx + 1, 3);
                                    pathByte.Insert(pathPointIdx + 2, 3);

                                    if (pathPointIdx + 3 == pathByte.Count - 1)
                                    {
                                        pathByte[pathPointIdx + 3] = 163;
                                    }
                                    else pathByte[pathPointIdx + 3] = 3;

                                    pathPoints.Insert(pathPointIdx + 1, pathPoints[pathPointIdx]);
                                    pathPoints.Insert(pathPointIdx + 2, pathPoints[pathPointIdx]);

                                    //Insert로 Index가 바뀌어서 처리
                                    if (pathPointSymIdx > pathPointIdx) pathPointSymIdx = pathPointSymIdx + 2;
                                }
                            }

                            thrBezirIdx = GetThrBezierIdx(pathPointIdx + 2); //선택한 점을 S에서 3Bezir Point로 옮긴다
                            thrBezirData.Insert(thrBezirIdx, pathPoints[pathPointIdx + 2]);

                            if (pathPointIdx == 0) thrBezirSymIdx = GetThrBezierIdx(pathPoints.Count - 3);

                            pathPointIdx = pathPointIdx + 2; //N으로 pathPointIdx 이동

                            isThrBezir = true;
                        }
                        isFirst = false;
                    }
                    else//Point를 움직이는 과정
                    {
                        PointF[] newHandle;
                        thrBezirSymIdx = GetThrBezierIdx(pathPointSymIdx);
                        if (isThrBezir)//내가 선택한 점이 3bezir일 때
                        {
                            //그 점의 위치와 C1, C2를 바꾸고
                            thrBezirData[thrBezirIdx] = e.Location;
                            newHandle = GetthrBezirPoint(pathPoints[pathPointIdx - 2], e.Location, pathPoints[pathPointIdx + 1]);
                            pathPoints[pathPointIdx - 1] = newHandle[0];
                            pathPoints[pathPointIdx] = newHandle[1];
                            if (pathPointIdx == 2) // 선택한 점이 첫번째 점일때 
                            {
                                if (pathTypes[pathTypes.Count - 3] == PathTypes.N) //반대편 점이 3bezir이면
                                {
                                    //thrBezirData[thrBezirSymIdx] = GetSymmetryPoint(pathPoints[pathPoints.Count - 3], e.Location);
                                    thrBezirData[thrBezirSymIdx] = GetSymmetryPoint(pathPoints[0], e.Location);
                                    newHandle = GetthrBezirPoint(pathPoints[pathPoints.Count - 4], thrBezirData[thrBezirSymIdx], pathPoints[pathPoints.Count - 1]);
                                    pathPoints[pathPoints.Count - 3] = newHandle[0];
                                    pathPoints[pathPoints.Count - 2] = newHandle[1];
                                }
                                else//4bezir이면
                                {
                                    pathPoints[pathPointSymIdx] = GetSymmetryPoint(pathPoints[pathPoints.Count - 1], e.Location);
                                    //pathPoints[pathPointSymIdx] = GetSymmetryPoint(pathPoints[pathPoints.Count - 2], e.Location);
                                }
                            }
                            else //이외의 점들
                            {
                                if (pathTypes[pathPointIdx - 4] == PathTypes.N)
                                {
                                    thrBezirData[thrBezirSymIdx] = GetSymmetryPoint(pathPoints[pathPointIdx - 2], e.Location);
                                    newHandle = GetthrBezirPoint(pathPoints[pathPointIdx - 5], thrBezirData[thrBezirSymIdx], pathPoints[pathPointIdx - 2]);
                                    pathPoints[pathPointIdx - 4] = newHandle[0];
                                    pathPoints[pathPointIdx - 3] = newHandle[1];
                                }
                                else
                                    pathPoints[pathPointSymIdx] = GetSymmetryPoint(pathPoints[pathPointIdx - 2], e.Location);
                            }
                        }
                        else
                        {
                            pathPoints[pathPointIdx] = e.Location;
                            if (pathPointIdx == 1)
                            {
                                if (pathTypes[pathTypes.Count - 3] == PathTypes.N)
                                {
                                    //thrBezirData[thrBezirSymIdx] = GetSymmetryPoint(pathPoints[pathPoints.Count - 3], e.Location);
                                    thrBezirData[thrBezirSymIdx] = GetSymmetryPoint(pathPoints[0], e.Location);
                                    newHandle = GetthrBezirPoint(pathPoints[pathPoints.Count - 4], thrBezirData[thrBezirSymIdx], pathPoints[pathPoints.Count - 1]);
                                    pathPoints[pathPoints.Count - 3] = newHandle[0];
                                    pathPoints[pathPoints.Count - 2] = newHandle[1];

                                }
                                else
                                    pathPoints[pathPointSymIdx] = GetSymmetryPoint(pathPoints[pathPointIdx - 1], e.Location);
                            }
                            else
                            {
                                if (pathTypes[pathPointIdx - 3] == PathTypes.N)
                                {
                                    thrBezirData[thrBezirSymIdx] = GetSymmetryPoint(pathPoints[pathPointIdx - 1], e.Location);
                                    newHandle = GetthrBezirPoint(pathPoints[pathPointIdx - 4], thrBezirData[thrBezirSymIdx], pathPoints[pathPointIdx - 1]);
                                    pathPoints[pathPointIdx - 3] = newHandle[0];
                                    pathPoints[pathPointIdx - 2] = newHandle[1];

                                }
                                else
                                    pathPoints[pathPointSymIdx] = GetSymmetryPoint(pathPoints[pathPointIdx - 1], e.Location);
                            }
                        }

                        selTempPath[0] = new GraphicsPath(pathPoints.ToArray(), pathByte.ToArray());

                        if (isMirror) selRefPath[0] = GetMirrorPath((GraphicsPath)selTempPath[0].Clone(), baseLine);

                        pictureBox1.Refresh();
                    }
                    break;
                case Mode.POINTSEL: //# When holding a point and moving
                    if (!isClicked || pathPointIdx == -1) return;

                    xOffset = e.Location.X - oldPosition.X;
                    yOffset = e.Location.Y - oldPosition.Y;//Todo Offset Move칠 때 어긋 남 ...

                    SetBezierPoint(e.Location, xOffset, yOffset);

                    if (!isMirror) selTempPath[0] = new GraphicsPath(pathPoints.ToArray(), vecList[selPathIndex[0]].path.PathTypes);
                    else
                    {
                        selTempPath[0] = new GraphicsPath(pathPoints.ToArray(), oriList[selPathIndex[0]].path.PathTypes);
                        selRefPath[0] = GetMirrorPath((GraphicsPath)selTempPath[0].Clone(), baseLine);
                    }
                    oldPosition = e.Location;
                    pictureBox1.Refresh();
                    break;
                case Mode.ROTATE:
                    PointF center = new PointF(selRects.Left + selRects.Width / 2, selRects.Top + selRects.Height / 2);
                    float angle = (float)GetAngleFromLine(Point.Round(center), e.Location) - (float)GetAngleFromLine(Point.Round(center), oldPosition);
                    RotateData(angle, center);
                    oldPosition = e.Location;
                    pictureBox1.Refresh();
                    break;
            }
        }

        public void pictureBox1_MouseUp(object sender, MouseEventArgs e)
        {
            if (bitCanvas.mDrawObject != CanvasType.VECTOR) return;
            if (e.Button != MouseButtons.Left) return;

            if (drawing)
            {
                isClicked = false;
                end = e.Location;
                List<VectorData> selList;
                if (isMirror) selList = oriList;
                else selList = vecList;

                switch (mode)
                {
                    //# draw
                    case Mode.LINE:
                    case Mode.CURVE:
                    case Mode.FREEDRAW:
                    case Mode.RECT:
                    case Mode.SQUARE:
                    case Mode.ELLIPSE:
                    case Mode.CIRCLE:
                        if(mode == Mode.FREEDRAW)
                        {
                            if(currentData.path.PointCount < 4)
                            {
                                pictureBox1.Refresh();
                                currentData.path = null;
                                drawing = false;
                                return;
                            }
                            List<VECTOR> tempW = new List<VECTOR>();
                            VECTOR tempPt = new VECTOR();

                            for (int i = 0; i < currentData.path.PointCount; i++)
                            {
                                tempPt.X = currentData.path.PathPoints[i].X;
                                tempPt.Y = currentData.path.PathPoints[i].Y;
                                tempW.Add(tempPt);
                            }
                            
                            //Curve 수 조절을 Error Parameter를 통해 할 수 있음 1로 갈수록 증가 
                            CubicBezier[] curves = CurveFit.Fit(tempW, 8);
                            Point point = new Point();
                            for (int i = 0; i < curves.Length; i++)
                            {
                                if(i == 0)
                                {
                                    point.X = (int)curves[i].p0.X;
                                    point.Y = (int)curves[i].p0.Y;
                                    changePt.Add(point);
                                    curvedByte.Add(0);
                                }
                                point.X = (int)curves[i].p1.X;
                                point.Y = (int)curves[i].p1.Y;
                                changePt.Add(point);
                                curvedByte.Add(3);

                                point.X = (int)curves[i].p2.X;
                                point.Y = (int)curves[i].p2.Y;
                                changePt.Add(point);
                                curvedByte.Add(3);

                                point.X = (int)curves[i].p3.X;
                                point.Y = (int)curves[i].p3.Y;
                                changePt.Add(point);
                                curvedByte.Add(3);
                            }
                            currentData.path = new GraphicsPath(changePt.ToArray(), curvedByte.ToArray());
                        }

                        //# 클릭으로 데이터가 생성되지 않도록 막음(Free Draw 제외)
                        if ((start.X != end.X && start.Y != end.Y) || mode == Mode.FREEDRAW)
                        {
                            currentData.isString = false;
                            CreateData();
                        }
                        pictureBox1.Refresh();
                        currentData.path = null;
                        drawing = false;
                        break;
                    case Mode.SPRAY:
                        if (tcl_spray.SelectedTab.Text == "Spray")
                        {
                            if (cmb_method.Text != "Curve")
                            {
                                currentData.path.Flatten();
                                SprayData(currentData.path);
                                pictureBox1.Refresh();
                                currentData.path = null;
                                drawing = false;
                            }
                        }
                        break;
                    case Mode.ZOOM:
                        for (int i = 0; i < selPathIndex.Count; i++)
                        {
                            ModifyData(ModifyMode.PATH, selPathIndex[i], (GraphicsPath)selTempPath[i].Clone());
                        }
                        pictureBox1.Refresh();
                        drawing = false;
                        break;
                    case Mode.SELLASSO:
                        Region temp = new Region(currentData.path);
                        isLasso = false;
                        for (int i = 0; i < selList.Count; i++)
                        {
                            for (int j = 0; j < selList[i].path.PointCount; j++)
                            {
                                if (temp.IsVisible(selList[i].path.PathData.Points[j]))
                                {
                                    isLasso = true;
                                    AddSelList(i);
                                    break;
                                }
                            }
                        }
                        RestoreSelMode();
                        break;
                    case Mode.SELECT:
                        if (selRegion != null)
                        {
                            SelectData(selRegion.GetBounds());
                            selRegion = null;
                        }

                        if (selPathIndex.Count > 0)
                        {
                            xOffset = e.Location.X - oldPosition.X;
                            yOffset = e.Location.Y - oldPosition.Y;
                            if (((Keyboard.GetKeyStates(Key.LeftAlt) & KeyStates.Down) > 0))
                            {
                                for (int i = 0; i < selPathIndex.Count; i++)
                                {
                                    currentData = GetModifyData(ref selList, ModifyMode.PATH, selPathIndex[i], (GraphicsPath)selTempPath[i].Clone());
                                    CreateData();
                                }

                                selPathIndex.Clear();
                                for (int i = 0; i < selTempPath.Count; i++)
                                {
                                    selPathIndex.Add(selList.Count - 1 - i);
                                    selTempPath[i] = (GraphicsPath)selList[selPathIndex[i]].path.Clone();
                                    if (isMirror)
                                    {
                                        selRefPath[i] = (GraphicsPath)refList[selPathIndex[i]].path.Clone();
                                    }
                                }
                            }
                            else
                            {
                                if (!isRectLasso) MoveData(e.Location.X - start.X, e.Location.Y - start.Y, true);
                            }
                        }
                        UpdateDisplay();
                        pictureBox1.Refresh();
                        isRectLasso = false;
                        drawing = false;
                        break;
                    case Mode.PENTOOL:
                        connPath = new GraphicsPath(pathPoints.ToArray(), pathByte.ToArray());
                        connPathList.Add((GraphicsPath)connPath.Clone());
                        isFirst = true;
                        pictureBox1.Refresh();
                        drawing = false;
                        break;
                    case Mode.POINTCHANGE:
                        ModifyData(ModifyMode.PATH, selPathIndex[0], selTempPath[0]);
                        isFirst = true;
                        pictureBox1.Refresh();
                        drawing = false;
                        pathPointIdx = -1;
                        pathPointSymIdx = -1;
                        thrBezirIdx = -1;
                        thrBezirSymIdx = -1;
                        break;
                    case Mode.POINTADD:
                        drawing = false;
                        break;
                    case Mode.POINTDEL:
                        break;
                    case Mode.POINTSEL:
                        if (!(selPathIndex.Count > 0)) return;

                        ModifyData(ModifyMode.PATH, selPathIndex[0], selTempPath[0]);
                        if (start != e.Location)
                        {
                            if (!isMirror)
                            {
                                SetVectorList(selPathIndex[0], vecList[selPathIndex[0]]);

                                doVector.Add(vecList.ToList());
                                doIdx = doVector.Count - 1;
                            }
                            else
                            {
                                oriDoVector.Add(oriList.ToList());
                                refDoVector.Add(refList.ToList());
                                refDoIdx = oriDoVector.Count - 1;
                            }
                        }
                        pictureBox1.Refresh();
                        currentData.path = null;
                        isThrBezir = false;
                        drawing = false;
                        pathPointIdx = -1;
                        pathPointSymIdx = -1;
                        thrBezirIdx = -1;
                        thrBezirSymIdx = -1;
                        break;
                }
                Commit();
            }
        }

        public void pictureBox1_Paint(Graphics gr)
        {
            if (bitCanvas.mDrawObject != CanvasType.VECTOR) return;

            if (!drawing) return;

            if (tsb_antialias.Checked)
            {
                gr.SmoothingMode = SmoothingMode.AntiAlias;
            }
            else
            {
                gr.SmoothingMode = SmoothingMode.None;
            }
            DrawDataList(gr);

            //# draw baseLine
            if (isMirror)
            {
                //# Show current location
                if (isHori)
                {
                    gr.DrawLine(new Pen(Color.Black, 0.25F), new Point(baseLine.X, 0), new Point(baseLine.X, pictureBox1.Height));
                }
                else
                {
                    gr.DrawLine(new Pen(Color.Black, 0.25F), new Point(0, baseLine.Y), new Point(pictureBox1.Width, baseLine.Y));
                }
            }

            switch (mode)
            {
                case Mode.TEXT:
                case Mode.PATHTEXT:
                    if (currentData.path != null)
                    {
                        gr.DrawPath(new Pen(Color.DeepSkyBlue), currentData.path);
                        DrawTextOnPath(gr, new SolidBrush(Color.Black), currentData.font, currentData.str, currentData.path);
                    }
                    return;
                case Mode.REVHORI:
                case Mode.REVVERT:
                    for (int i = 0; i < selTempPath.Count; i++)
                    {
                        gr.DrawPath(Pens.DeepSkyBlue, selTempPath[i]);
                    }

                    if (isHori)
                    {
                        gr.DrawLine(new Pen(Color.Black, 0.25F), new Point(baseLine.X, 0), new Point(baseLine.X, pictureBox1.Height));
                    }
                    else
                    {
                        gr.DrawLine(new Pen(Color.Black, 0.25F), new Point(0, baseLine.Y), new Point(pictureBox1.Width, baseLine.Y));
                    }
                    DrawBounds(selRectPoints, gr);
                    break;
                case Mode.REPEAT:
                    if (currentDataList != null)
                    {
                        for (int i = 0; i < currentDataList.Count; i++)
                        {
                            gr.FillPath(currentDataList[i].brush, currentDataList[i].path);
                            gr.DrawPath(currentDataList[i].pen, currentDataList[i].path);
                        }
                    }
                    break;
                case Mode.ZOOM:
                    for (int i = 0; i < selTempPath.Count; i++)
                    {
                        gr.DrawPath(Pens.DeepSkyBlue, selTempPath[i]); //todo 
                        if (isMirror)
                        {
                            gr.DrawPath(Pens.DeepSkyBlue, selRefPath[i]);
                        }
                    }
                    DrawBounds(selRectPoints, gr);
                    break;
                case Mode.SELLASSO:
                    //# Draw a lasso shape
                    if (isLasso)
                        gr.DrawPath(new Pen(Color.Gray), currentData.path);
                    break;
                case Mode.SELECT:
                    if (selRegion != null)
                    {
                        Pen selRegionPen = new Pen(Color.Black);
                        selRegionPen.DashStyle = DashStyle.Dash;
                        selRegionPen.DashPattern = new float[] { 4, 2 };
                        gr.DrawPath(selRegionPen, selRegion);
                    }
                    for (int i = 0; i < selTempPath.Count; i++)
                    {
                        gr.DrawPath(Pens.DeepSkyBlue, selTempPath[i]); //todo 
                        if (isMirror)
                        {
                            gr.DrawPath(Pens.DeepSkyBlue, selRefPath[i]);
                        }
                        else
                        {
                            if (vecList[selPathIndex[i]].isString && isClicked)
                            {
                                DrawTextOnPath(gr, new SolidBrush(Color.DeepSkyBlue), (Font)vecList[selPathIndex[i]].font, vecList[selPathIndex[i]].str, selTempPath[i]);
                            }
                        }
                    }
                    DrawBounds(selRectPoints, gr);
                    break;
                case Mode.PENTOOL:
                    if (connPath == null) return;

                    if (pathPoints.Count == 1)
                    {
                        gr.DrawRectangle(Pens.DeepSkyBlue, pathPoints[0].X - 3, pathPoints[0].Y - 3, 6, 6);
                    }
                    else
                    {
                        if (isFirst == false)
                        {
                            PointF symPt = GetSymmetryPoint(pathPoints[pathPoints.Count - 1], end);
                            gr.DrawLine(Pens.Red, symPt, end);
                            gr.DrawRectangle(Pens.Red, end.X - handleSize, end.Y - handleSize, handleSize * 2, handleSize * 2);
                            gr.DrawRectangle(Pens.Red, symPt.X - handleSize, symPt.Y - handleSize, handleSize * 2, handleSize * 2);
                            gr.DrawRectangle(Pens.DeepSkyBlue, pathPoints[pathPoints.Count - 1].X - handleSize, pathPoints[pathPoints.Count - 1].Y - handleSize, handleSize * 2, handleSize * 2);
                        }
                        gr.DrawPath(currentData.pen, connPath);
                        gr.FillPath(currentData.brush, connPath);
                        if (isMirror)
                        {
                            gr.DrawPath(currentData.pen, GetMirrorPath(connPath, baseLine));
                            gr.FillPath(currentData.brush, GetMirrorPath(connPath, baseLine));
                        }
                    }
                    break;
                case Mode.POINTCHANGE:
                case Mode.POINTSEL:
                    if (selPathIndex.Count == 0) return;

                    for (int i = 0; i < selPathIndex.Count; i++)
                    {
                        int bezierCnt = 0;
                        for (int k = 0; k < selTempPath[i].PointCount; k++)
                        {
                            switch (pathTypes[k])
                            {
                                case PathTypes.N:
                                    break;
                                case PathTypes.S:
                                case PathTypes.CE:
                                case PathTypes.LE:
                                case PathTypes.CL:
                                case PathTypes.CC:
                                    gr.DrawRectangle(Pens.DeepSkyBlue, selTempPath[i].PathPoints[k].X - handleSize, selTempPath[i].PathPoints[k].Y - handleSize, handleSize * 2, handleSize * 2);
                                    break;
                                case PathTypes.C1:
                                    if (pathTypes[k + 1] == PathTypes.N)
                                    {
                                        gr.DrawRectangle(Pens.Red, thrBezirData[bezierCnt].X - handleSize, thrBezirData[bezierCnt].Y - handleSize, handleSize * 2, handleSize * 2);
                                        gr.DrawLine(Pens.Red, thrBezirData[bezierCnt], selTempPath[i].PathPoints[k - 1]);
                                        bezierCnt++;
                                    }
                                    else
                                    {
                                        gr.DrawRectangle(Pens.Red, selTempPath[i].PathPoints[k].X - handleSize, selTempPath[i].PathPoints[k].Y - handleSize, handleSize * 2, handleSize * 2);
                                        gr.DrawLine(Pens.Red, selTempPath[i].PathPoints[k], selTempPath[i].PathPoints[k - 1]);
                                    }
                                    break;
                                case PathTypes.C2:
                                    if (pathTypes[k - 1] == PathTypes.N)
                                    {
                                        gr.DrawRectangle(Pens.Red, thrBezirData[bezierCnt].X - handleSize, thrBezirData[bezierCnt].Y - handleSize, handleSize * 2, handleSize * 2);
                                        gr.DrawLine(Pens.Red, thrBezirData[bezierCnt], selTempPath[i].PathPoints[k + 1]);
                                        bezierCnt++;
                                    }
                                    else
                                    {
                                        gr.DrawRectangle(Pens.Red, selTempPath[i].PathPoints[k].X - handleSize, selTempPath[i].PathPoints[k].Y - handleSize, handleSize * 2, handleSize * 2);
                                        gr.DrawLine(Pens.Red, selTempPath[i].PathPoints[k], selTempPath[i].PathPoints[k + 1]);
                                    }
                                    break;
                            }
                        }
                    }
                    gr.DrawPath(Pens.DeepSkyBlue, selTempPath[0]);
                    if (isMirror) gr.DrawPath(Pens.DeepSkyBlue, selRefPath[0]);
                    break;
                case Mode.POINTADD:
                case Mode.POINTDEL:
                    if (pathTypes == null) return;

                    for (int i = 0; i < selPathIndex.Count; i++)
                    {
                        for (int k = 0; k < vecList[selPathIndex[i]].path.PointCount; k++)
                        {
                            switch (pathTypes[k])
                            {
                                case PathTypes.S:
                                case PathTypes.CE:
                                case PathTypes.LE:
                                    gr.DrawRectangle(Pens.DeepSkyBlue, vecList[selPathIndex[i]].path.PathPoints[k].X - handleSize, vecList[selPathIndex[i]].path.PathPoints[k].Y - handleSize, handleSize * 2, handleSize * 2);
                                    break;
                            }
                        }
                    }
                    break;
                case Mode.KNIFE:
                    if (isMirror)
                    {
                        DrawSelPoint(oriList[selPathIndex[0]].path, gr);
                    }
                    else
                    {
                        DrawSelPoint(vecList[selPathIndex[0]].path, gr);
                    }
                    break;
                case Mode.ROTATE:
                    for (int i = 0; i < selTempPath.Count; i++)
                    {
                        gr.DrawPath(Pens.DeepSkyBlue, selTempPath[i]);
                        if (isMirror)
                        {
                            gr.DrawPath(Pens.DeepSkyBlue, selRefPath[i]);
                        }
                        else
                        {
                            if (vecList[selPathIndex[i]].isString && isClicked)
                            {
                                DrawTextOnPath(gr, new SolidBrush(Color.DeepSkyBlue), (Font)vecList[selPathIndex[i]].font, vecList[selPathIndex[i]].str, selTempPath[i]);
                            }
                        }
                    }
                    break;
                case Mode.SPRAY:
                    if (currentData.path != null)
                    {
                        gr.DrawPath(Pens.Gray, currentData.path);
                        if (isMirror)
                        {
                            gr.DrawPath(Pens.Gray, GetMirrorVector(currentData, baseLine).path);
                        }
                    }
                    break;
                default:
                    if (currentData.path != null)
                    {
                        gr.FillPath(currentData.brush, currentData.path);
                        gr.DrawPath(currentData.pen, currentData.path);
                        if (isMirror)
                        {
                            gr.FillPath(currentData.brush, GetMirrorVector(currentData, baseLine).path);
                            gr.DrawPath(currentData.pen, GetMirrorVector(currentData, baseLine).path);
                        }
                    }
                    DrawBounds(selRectPoints, gr);
                    break;
            }

            //# update size panel
            if (selPathIndex.Count != 0)
            {
                nud_x.Value = (int)selRects.X;
                nud_y.Value = (int)selRects.Y;
                nud_w.Value = (int)selRects.Width;
                nud_h.Value = (int)selRects.Height;
                nud_x.Refresh();
                nud_y.Refresh();
                nud_w.Refresh();
                nud_h.Refresh();
            }
        }

        //# 기섭 summary 작성
        public int GetRefRectPointIdx(int idx)
        {
            switch (idx)
            {
                case 0:
                    if (isHori) idx = 2;
                    else idx = 6;
                    break;
                case 1:
                    if (isHori) idx = 1;
                    else idx = 5;
                    break;
                case 2:
                    if (isHori) idx = 0;
                    else idx = 4;
                    break;
                case 3:
                    if (isHori) idx = 7;
                    else idx = 3;
                    break;
                case 4:
                    if (isHori) idx = 6;
                    else idx = 2;
                    break;
                case 5:
                    if (isHori) idx = 5;
                    else idx = 1;
                    break;
                case 6:
                    if (isHori) idx = 4;
                    else idx = 0;
                    break;
                case 7:
                    if (isHori) idx = 3;
                    else idx = 7;
                    break;
            }
            return idx;
        }

        public PointF GetRefRectStPoint(PointF point, bool hori)
        {
            if (hori)
            {
                point.X += (baseLine.X - point.X) * 2;
            }
            else
            {
                point.Y += (baseLine.Y - point.Y) * 2;
            }
            return point;
        }

        //=======================================================================
        //# Text (문자 & 패스 문자)
        //=======================================================================
        /// <summary>
        /// Text를 Typing 할 때마다 발생하는 Event
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void rtb_text_TextChanged(object sender, EventArgs e)
        {
            //# Receive input string
            currentData.str = richTextBox.Text;
            currentData.str = currentData.str.Replace("\r", "\r\n");

            if (!currentData.isPathText)
            {
                if (selPathIndex.Count == 0)
                {
                    currentData.path = (GraphicsPath)GetBaseLine(currentData.str, currentData.font, start).Clone();
                }
                else
                {
                    currentData.path = (GraphicsPath)GetBaseLine(currentData.str, currentData.font, vecList[selPathIndex[0]].path.PathPoints[0]).Clone();
                }
            }

            currentData.str = currentData.str.Replace("\n", "");
            drawing = true;
            pictureBox1.Refresh();
        }

        /// <summary>
        /// 새 Text를 입력할 때 마다 그 String의 Baseline을 구하기 위한 함수(Line)
        /// </summary>
        /// <param name="str">길이를 구할 문자열</param>
        /// <param name="font">길이의 기준이 되는 글꼴</param>
        /// <param name="start">Baseline의 시작점이 되는 위치</param>
        /// <returns></returns>
        private GraphicsPath GetBaseLine(string str, Font font, PointF start)
        {
            //# sjpark - 새로운 문자를 입력할 때 추가되는 Line을 구하는 함수
            Graphics gr = pictureBox1.CreateGraphics();
            GraphicsPath newPath = new GraphicsPath();

            string[] strlist = str.Split(new char[1] { '\r' }, StringSplitOptions.None);
            for (int i = 0; i < strlist.Length; i++)
            {
                GraphicsPath linePath = new GraphicsPath();
                float charWidth = GetCharacterWidths(gr, strlist[i], font).ToArray().Sum() + 5.0F;
                float charHeight = gr.MeasureString(strlist[0], font).Height;
                PointF lineStart = new PointF(start.X, (int)(start.Y + charHeight * i));
                PointF lineEnd = new PointF(lineStart.X + charWidth, lineStart.Y);
                linePath.AddLine(lineStart, lineEnd);
                linePath.CloseFigure();
                newPath.AddPath(linePath, false);
            }
            return newPath;
        }

        /// <summary>
        /// 지정된 Font를 사용하여 그릴 때 지정된 문자열의 넓이를 구하는 함수
        /// </summary>
        /// <param name="graphics"></param>
        /// <param name="text"></param>
        /// <param name="font"></param>
        /// <returns></returns>
        private IEnumerable<float> GetCharacterWidths(Graphics graphics, string text, Font font)
        {
            // The length of a space. Necessary because a space measured using StringFormat.GenericTypographic has no width.
            // We can't use StringFormat.GenericDefault for the characters themselves, as it adds unwanted spacing.
            var spaceLength = graphics.MeasureString(" ", font, Point.Empty, StringFormat.GenericTypographic).Width;

            return text.Select(c => c == ' ' ? spaceLength : graphics.MeasureString(c.ToString(), font, Point.Empty, StringFormat.GenericTypographic).Width);
        }

        /// <summary>
        /// Font를 ComboBox에 그려줌
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void cmb_fontName_DrawItem(object sender, DrawItemEventArgs e)
        {
            FontFamily family = FontFamily.Families[e.Index];
            Font myFont = new Font(family, 10);
            Rectangle rect = new Rectangle(2, e.Bounds.Top + 2, e.Bounds.Height, e.Bounds.Height - 4);
            e.Graphics.FillRectangle(new SolidBrush(Color.White), rect);
            e.Graphics.DrawString(FontFamily.Families[e.Index].Name, myFont, Brushes.Black,
                new RectangleF(e.Bounds.X + rect.Width, e.Bounds.Y, e.Bounds.Width, e.Bounds.Height));
            e.DrawFocusRectangle();
        }

        /// <summary>
        /// Font 변경
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        public void ChangeFont(object sender, EventArgs e)
        {
            //# current font 변경
            if (cmb_fontName.SelectedIndex < 0 || cmb_fontSize.SelectedIndex < 0) return;
            FontFamily family = FontFamily.Families[cmb_fontName.SelectedIndex];
            string fontSize = cmb_fontSize.Items[cmb_fontSize.SelectedIndex].ToString().Replace(" pt", "");

            FontStyle fontStyle = FontStyle.Regular;
            switch (cmb_fontStyle.SelectedIndex)
            {
                case 0:
                    fontStyle = FontStyle.Regular;
                    break;
                case 1:
                    fontStyle = FontStyle.Bold;
                    break;
                case 2:
                    fontStyle = FontStyle.Italic;
                    break;
                case 3:
                    fontStyle = FontStyle.Bold | FontStyle.Italic;
                    break;
            }
            currentData.font = new Font(family, Convert.ToSingle(fontSize), fontStyle);

            if (selPathIndex.Count == 0)
            {
                //# current string 변경
                pictureBox1.Refresh();
                richTextBox.Focus();
                richTextBox.Select(richTextBox.Text.Length, 0);
                if (!currentData.isPathText && currentData.path != null)
                {
                    currentData.path = (GraphicsPath)GetBaseLine(currentData.str, currentData.font, Point.Round(currentData.path.PathPoints[0])).Clone();
                }
            }
            else
            {
                //# modify
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    ModifyData(ModifyMode.FONT, selPathIndex[i], (Font)currentData.font.Clone());
                }
            }

            //# drawing
            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
        }

        //=======================================================================
        //# Spray (뿌리기)
        //=======================================================================
        /// <summary>
        /// TextBox에 입력된 데이터가 유효한지 확인
        /// </summary>
        /// <param name="box">sender</param>
        /// <param name="start">시작 범위</param>
        /// <param name="end">끝 범위</param>
        /// <returns>유효값 반환</returns>
        public int CheckValue(TextBox box, int start, int end)
        {
            if (!int.TryParse(box.Text, out int number))
            {
                MessageBox.Show("Please enter a number!");
                box.Focus();
                return -1;
            }
            else
            {
                if (number <= start || number > end)
                {
                    MessageBox.Show(String.Format("Out of range! {0} ~ {1}", start, end));
                    box.Focus();
                    return -1;
                }
            }
            return number;
        }

        /// <summary>
        /// 외곽선 뿌리기 진행
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btn_run_Click(object sender, EventArgs e)
        {
            if (selPathIndex.Count == 0) return;

            //# sjpark - 예외처리
            int cnt = CheckValue(txt_outline_cnt, 0, 100);
            int scale = CheckValue(txt_ratio, 0, 10000);

            if (cnt < 0 || scale < 0) return;

            List<VectorData> selList;
            if (!isMirror) selList = vecList;
            else selList = oriList;

            List<int> group = new List<int>();
            for (int j = 0; j < cnt; j++)
            {
                GraphicsPath tempPath = new GraphicsPath();
                System.Windows.Media.Matrix zoomMat = new System.Windows.Media.Matrix();

                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    tempPath.AddRectangle(selList[selPathIndex[i]].path.GetBounds());
                }

                PointF cPoint = new PointF(tempPath.GetBounds().Right - tempPath.GetBounds().Width / 2, tempPath.GetBounds().Bottom - tempPath.GetBounds().Height / 2);

                zoomMat.ScaleAt(Convert.ToSingle(scale) / 100, Convert.ToSingle(scale) / 100, cPoint.X, cPoint.Y);
                Matrix dmat = new Matrix((float)zoomMat.M11, (float)zoomMat.M12, (float)zoomMat.M21, (float)zoomMat.M22, (float)zoomMat.OffsetX, (float)zoomMat.OffsetY);

                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    if (selList[selPathIndex[i]].isString) continue;
                    currentData.path = (GraphicsPath)selList[selPathIndex[i]].path.Clone();
                    currentData.path.Transform(dmat);

                    currentData = GetModifyData(ref selList, ModifyMode.PATH, selPathIndex[i], (GraphicsPath)currentData.path.Clone());
                    CreateData();

                    selPathIndex.RemoveAt(i);
                    selPathIndex.Insert(i, selList.Count - 1);
                    group.Add(selList.Count - 1);
                }
            }
            groupList.Add(group.ToList());

            Commit();
            ClearSelList();
            RestoreSelMode();
        }

        //# 기섭 summary 작성
        public void SetVectorList(VectorData data, bool isView = true, string group = "", bool isLock = false)
        {
            //# 보완예정
        }

        public void SetVectorList(int idx, VectorData data)
        {
            //# 보완예정
        }

        public Bitmap CreateVectorListBitmap(VectorData data)
        {
            GraphicsPath gp = (GraphicsPath)data.path.Clone();
            Matrix trans = new Matrix();
            Matrix scale = new Matrix();

            if (gp.GetBounds().Width >= gp.GetBounds().Height)
            {
                scale.Scale(20 / gp.GetBounds().Width, 20 / gp.GetBounds().Width);
            }
            else
            {
                scale.Scale(20 / gp.GetBounds().Height, 20 / gp.GetBounds().Height);
            }
            gp.Transform(scale);
            trans.Translate(-gp.GetBounds().X, -gp.GetBounds().Y);
            gp.Transform(trans);

            Bitmap bmp = new Bitmap((int)gp.GetBounds().Width + 2, (int)gp.GetBounds().Height + 2);

            Graphics gr = Graphics.FromImage(bmp);
            if (tsb_antialias.Checked)
            {
                gr.SmoothingMode = SmoothingMode.AntiAlias;
            }
            else
            {
                gr.SmoothingMode = SmoothingMode.None;
            }

            if (!data.isString) gr.DrawPath(data.pen, gp);
            else
            {
                Font font = new Font(data.font.FontFamily, 8);

                gr.DrawString(data.str, font, data.brush, new PointF(0, 0));
            }

            gr.FillPath(data.brush, gp);

            return bmp;
        }

        /// <summary>
        /// InputSize를 통해 개체 생성하는 함수
        /// </summary>
        /// <param name="width"></param>
        /// <param name="height"></param>
        public void CreateObject(float width, float height)
        {
            currentData.path = new GraphicsPath();

            switch (mode)
            {
                case Mode.LINE:
                    Matrix mat = new Matrix();
                    mat.RotateAt(360 - height, start);
                    currentData.path.AddLine(start, new PointF(start.X + width, start.Y));
                    currentData.path.Transform(mat);
                    break;
                case Mode.RECT:
                case Mode.SQUARE:
                    currentData.path.AddRectangle(new RectangleF(start, new SizeF(width, height)));
                    break;
                case Mode.CIRCLE:
                case Mode.ELLIPSE:
                    currentData.path.AddEllipse(new RectangleF(start, new SizeF(width, height)));
                    break;
            }

            CreateData();

            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
        }

        /// <summary>
        /// GradientBrush 및 TextureBrush가 적용된 Data의 Path의 Matrix가 변화할 때 Brush도 변화시켜주기 위한 함수
        /// </summary>
        /// <param name="brush"></param>
        /// <param name="bound"></param>
        /// <returns></returns>
        public Brush CreateNewBrush(Brush brush, RectangleF bound)
        {
            Brush newBrush = brush;
            string brushClass = brush.ToString();
            if (brushClass == "System.Drawing.TextureBrush")
            {
                newBrush = new TextureBrush(((TextureBrush)brush).Image);
                ((TextureBrush)newBrush).Transform = (((TextureBrush)brush).Transform);
            }
            else if (brushClass == "System.Drawing.Drawing2D.LinearGradientBrush")
            {
                //# 반전 모드에 따라 Start / end 지점이 바뀐다.
                PointF ptStart = new PointF(bound.X, bound.Y);
                PointF ptEnd = new PointF(bound.X + bound.Width, bound.Y + bound.Height);
                Color colStart = ((LinearGradientBrush)brush).LinearColors[0];
                Color colEnd = ((LinearGradientBrush)brush).LinearColors[1];

                newBrush = new LinearGradientBrush(ptStart, ptEnd, colStart, colEnd);
                ((LinearGradientBrush)newBrush).Transform = (((LinearGradientBrush)brush).Transform);
            }
            return newBrush;
        }

        //=======================================================================
        //#  Draw Event
        //======================================================================
        /// <summary>
        /// vecList에 저장되어있는 Data를 그리는 함수
        /// </summary>
        /// <param name="gr"></param>
        public void DrawDataList(Graphics gr)
        {
            for (int i = 0; i < vecList.Count; i++)
            {
                //if ((bool)bitCanvas.mVectorList.dgv_vector[1, i].Value == false) continue;

                if (vecList[i].isString)
                {
                    //# sjpark - Hide existing data while editing text
                    if (i == nTextIndex)
                    {
                        continue;
                    }

                    DrawTextOnPath(gr, (Brush)vecList[i].brush.Clone(), (Font)vecList[i].font, vecList[i].str, (GraphicsPath)vecList[i].path.Clone());
                }
                else
                {
                    gr.FillPath(vecList[i].brush, vecList[i].path);
                    gr.DrawPath(vecList[i].pen, vecList[i].path);
                }
            }

            if (isMirror)
            {
                for (int i = 0; i < oriList.Count; i++)
                {
                    gr.FillPath(oriList[i].brush, oriList[i].path);
                    gr.FillPath(refList[i].brush, refList[i].path);
                    gr.DrawPath(oriList[i].pen, oriList[i].path);
                    gr.DrawPath(refList[i].pen, refList[i].path);
                }
            }
        }

        /// <summary>
        /// Path의 시작점과 끝점을 지정하여 DrawTextOnSegment 함수 호출
        /// </summary>
        /// <param name="gr"></param>
        /// <param name="brush"></param>
        /// <param name="font"></param>
        /// <param name="txt"></param>
        /// <param name="path">BaseLine</param>
        public void DrawTextOnPath(Graphics gr, Brush brush, Font font, string txt, GraphicsPath path)
        {
            if (!(path.PointCount > 0)) return;
            if (txt == null) return;
            // Make a copy so we don't mess up the original.
            path = (GraphicsPath)path.Clone();

            bool isLine = false;
            int oldcnt = path.PointCount;
            // Flatten the path into segments.
            path.Flatten();
            int newcnt = path.PointCount;
            if (oldcnt == newcnt)
            {
                isLine = true;
            }

            // Draw characters.
            int start_ch = 0;
            PointF start_point = path.PathPoints[0], end_point = path.PathPoints[0];
            for (int i = 0; i < path.PointCount; i++)
            {
                if (i == path.PointCount - 1)
                {
                    if (path.PathTypes[i] == 129)
                    {
                        if (isLine)
                        {
                            start_point = path.PathPoints[i];
                        }
                        end_point = path.PathPoints[0];
                    }
                }
                else
                {
                    end_point = path.PathPoints[i + 1];
                    if (path.PathTypes[i] == 129)
                    {
                        if (!isLine)
                        {
                            start_point = end_point;
                        }
                        else
                        {
                            start_point = path.PathPoints[i + 1];
                        }
                    }
                    else
                    {
                        if (isLine)
                        {
                            start_point = path.PathPoints[i];
                        }
                    }

                }
                DrawTextOnSegment(gr, brush, font, txt, ref start_ch, ref start_point, end_point);
                if (start_ch >= txt.Length) break;
            }
        }

        public void DrawSelPoint(GraphicsPath path, Graphics gr)
        {
            if (path == null || path.PointCount == 0) return;
            gr.DrawPath(new Pen(pathColor), path);
            for (int i = 0; i < path.PointCount - 1; i++)
            {
                gr.DrawRectangle(new Pen(pathColor), new Rectangle((int)path.PathPoints[i].X - 5, (int)path.PathPoints[i].Y - 5, 10, 10));
                if (path.PathTypes[i + 1] == 3)
                {
                    i += 2;
                }
            }
            gr.DrawRectangle(new Pen(pathColor), new Rectangle((int)path.PathPoints[path.PointCount - 1].X - 5, (int)path.PathPoints[path.PointCount - 1].Y - 5, 10, 10));
        }

        /// <summary>
        /// Path의 시작점과 끝점을 받아서 Text를 그려줌
        /// </summary>
        /// <param name="gr"></param>
        /// <param name="brush"></param>
        /// <param name="font"></param>
        /// <param name="txt"></param>
        /// <param name="first_ch"></param>
        /// <param name="start_point"></param>
        /// <param name="end_point"></param>
        public void DrawTextOnSegment(Graphics gr, Brush brush, Font font, string txt, ref int first_ch, ref PointF start_point, PointF end_point)
        {
            if (txt.Length == 0) return;

            float dx = end_point.X - start_point.X;
            float dy = end_point.Y - start_point.Y;
            float dist = (float)Math.Sqrt(dx * dx + dy * dy);
            dx /= dist;
            dy /= dist;

            // See how many characters will fit.
            int last_ch = first_ch;
            while (last_ch < txt.Length)
            {
                string test_string = txt.Substring(first_ch, last_ch - first_ch + 1);
                if (GetCharacterWidths(gr, test_string, font).ToArray().Sum() > dist)
                {
                    // This is one too many characters.
                    last_ch--;
                    break;
                }
                last_ch++;
            }
            if (last_ch < first_ch) return;
            if (last_ch >= txt.Length) last_ch = txt.Length - 1;
            string chars_that_fit = txt.Substring(first_ch, last_ch - first_ch + 1);

            // Update first_ch and start_point.
            first_ch = last_ch + 1;
            var characterWidths = GetCharacterWidths(gr, chars_that_fit, font).ToArray();
            var textLength = characterWidths.Sum();

            // Rotate and translate to position the characters.
            GraphicsState state = gr.Save();
            gr.TranslateTransform(0, -font.Height, MatrixOrder.Append);
            float angle = (float)(180 * Math.Atan2(dy, dx) / Math.PI);
            gr.RotateTransform(angle, MatrixOrder.Append);
            gr.TranslateTransform(start_point.X, start_point.Y, MatrixOrder.Append);

            if (drawing)
            {
                gr.DrawString(chars_that_fit, font, brush, 0, 0);
            }

            // Restore the saved state.
            gr.Restore(state);

            start_point = new PointF(start_point.X + dx * textLength, start_point.Y + dy * textLength);
        }

        /// <summary>
        /// 경로의 Bound 영역과 Handle을 그려주는 함수
        /// </summary>
        /// <param name="points"></param>
        /// <param name="gr"></param>
        public void DrawBounds(PointF[] points, Graphics gr)
        {
            if (selPathIndex.Count == 0) return;
            GetBounds();

            byte[] types = new byte[8];
            for (int i = 0; i < points.Length; i++)
            {
                if (i == 0) types[i] = 0;
                else if (i == points.Length - 1) types[i] = 129;
                else types[i] = 1;

                gr.DrawRectangle(new Pen(pathColor), new Rectangle((int)points[i].X - 5, (int)points[i].Y - 5, 10, 10));
            }
            gr.DrawRectangle(new Pen(pathColor), Rectangle.Round(selRects));
        }

        /// <summary>
        /// 경로의 Bound 영역과 Handle을 구하는 함수
        /// </summary>
        private void GetBounds()
        {
            if (selPathIndex.Count == 0) return;

            GraphicsPath totalPath = new GraphicsPath();
            GraphicsPath totalRefPath = new GraphicsPath();
            for (int i = 0; i < selPathIndex.Count; i++)
            {
                if (!isMirror)
                {
                    if (vecList[selPathIndex[i]].isString && !vecList[selPathIndex[i]].isPathText)
                    {
                        RectangleF tectRect = vecList[selPathIndex[i]].path.GetBounds();
                        totalPath.AddRectangle(new RectangleF(tectRect.X, tectRect.Y - vecList[selPathIndex[i]].font.Height, tectRect.Width, tectRect.Height + vecList[selPathIndex[i]].font.Height));
                    }
                    totalPath.AddRectangle(vecList[selPathIndex[i]].path.GetBounds());
                }
                else
                {
                    totalRefPath.AddRectangle(refList[selPathIndex[i]].path.GetBounds());
                    totalPath.AddRectangle(oriList[selPathIndex[i]].path.GetBounds());
                }
            }
            selRects = Rectangle.Round(totalPath.GetBounds());
            selRefRects = Rectangle.Round(totalRefPath.GetBounds());

            selRectPoints[0] = new PointF(selRects.X, selRects.Y);                                           //# LT
            selRectPoints[1] = new PointF(selRects.X + selRects.Width / 2, selRects.Y);                      //# CT
            selRectPoints[2] = new PointF(selRects.X + selRects.Width, selRects.Y);                          //# RT
            selRectPoints[3] = new PointF(selRects.X + selRects.Width, selRects.Y + selRects.Height / 2);    //# RM
            selRectPoints[4] = new PointF(selRects.X + selRects.Width, selRects.Y + selRects.Height);        //# RB
            selRectPoints[5] = new PointF(selRects.X + selRects.Width / 2, selRects.Y + selRects.Height);    //# CB
            selRectPoints[6] = new PointF(selRects.X, selRects.Y + selRects.Height);                         //# LB
            selRectPoints[7] = new PointF(selRects.X, selRects.Y + selRects.Height / 2);                     //# LC

            selRectRect[0] = new RectangleF(selRects.X - 5, selRects.Y - 5, 10, 10);
            selRectRect[1] = new RectangleF(selRects.X + selRects.Width / 2 - 5, selRects.Y - 5, 10, 10);
            selRectRect[2] = new RectangleF(selRects.X + selRects.Width - 5, selRects.Y - 5, 10, 10);
            selRectRect[3] = new RectangleF(selRects.X + selRects.Width - 5, selRects.Y + selRects.Height / 2 - 5, 10, 10);
            selRectRect[4] = new RectangleF(selRects.X + selRects.Width - 5, selRects.Y + selRects.Height - 5, 10, 10);
            selRectRect[5] = new RectangleF(selRects.X + selRects.Width / 2 - 5, selRects.Y + selRects.Height - 5, 10, 10);
            selRectRect[6] = new RectangleF(selRects.X - 5, selRects.Y + selRects.Height - 5, 10, 10);
            selRectRect[7] = new RectangleF(selRects.X - 5, selRects.Y + selRects.Height / 2 - 5, 10, 10);
        }

        /// <summary>
        /// 이동 방향을 구하는 함수, 수평 / 수직 이동을 구현하기 위함
        /// </summary>
        /// <param name="ptStart">시작점</param>
        /// <param name="ptEnd">끝점</param>
        /// <returns></returns>
        public Direction GetDirection(PointF ptStart, PointF ptEnd)
        {
            //# 마우스 이동 방향
            Direction dir = Direction.NONE;
            //# shift key down
            if ((Keyboard.GetKeyStates(Key.LeftShift) & KeyStates.Down) > 0)
            {
                double angle = 360 - GetAngleFromLine(ptStart, ptEnd);
                //# y축만 이동할 때
                if ((angle >= 45 && angle < 135) || (angle >= 225 && angle < 315))
                {
                    dir = Direction.Y;
                }
                //# y축만 이동할 때
                else
                {
                    dir = Direction.X;
                }
            }
            return dir;
        }

        //# 기섭 summary 작성
        public List<GraphicsPath> GetZoomPath(PointF mouse, List<VectorData> data, List<GraphicsPath> zoomGP, PointF zoomPt, RectangleF rect, int zoomIndex)
        {//와드
            double scaleX; double scaleY;

            zoomRect = GetRect(zoomPt.X, zoomPt.Y, mouse.X, mouse.Y);
            zoomGP = new List<GraphicsPath>();

            scaleX = (Convert.ToSingle(zoomRect.Width) / Convert.ToSingle(rect.Width));
            scaleY = (Convert.ToSingle(zoomRect.Height) / Convert.ToSingle(rect.Height));
            for (int i = 0; i < selTempPath.Count; i++)
            {
                zoomGP.Add((GraphicsPath)data[selPathIndex[i]].path.Clone());
                switch (zoomIndex)
                {
                    case 0:
                        if ((mouse.X >= zoomPt.X) && (mouse.Y < zoomPt.Y))
                            zoomGP[i].Transform(new Matrix(-1, 0, 0, 1, 2 * (rect.X + rect.Width), 0));
                        else if ((mouse.X < zoomPt.X) && (mouse.Y >= zoomPt.Y))
                            zoomGP[i].Transform(new Matrix(1, 0, 0, -1, 0, 2 * (rect.Y + rect.Height)));
                        else if ((mouse.X >= zoomPt.X) && (mouse.Y >= zoomPt.Y))
                            zoomGP[i].Transform(new Matrix(-1, 0, 0, -1, 2 * (rect.X + rect.Width), 2 * (rect.Y + rect.Height)));
                        break;
                    case 1:
                        if (mouse.Y >= zoomPt.Y)
                        {
                            zoomGP[i].Transform(new Matrix(1, 0, 0, -1, 0, 2 * (rect.Y + rect.Height)));
                        }
                        scaleX = 1.0f;
                        break;
                    case 2:
                        if ((mouse.X < zoomPt.X) && (mouse.Y < zoomPt.Y))
                            zoomGP[i].Transform(new Matrix(-1, 0, 0, 1, 2 * rect.X, 0));
                        else if ((mouse.X >= zoomPt.X) && (mouse.Y >= zoomPt.Y))
                            zoomGP[i].Transform(new Matrix(1, 0, 0, -1, 0, 2 * (rect.Y + rect.Height)));
                        else if ((mouse.X < zoomPt.X) && (mouse.Y >= zoomPt.Y))
                            zoomGP[i].Transform(new Matrix(-1, 0, 0, -1, 2 * rect.X, 2 * (rect.Y + rect.Height)));
                        break;
                    case 3:
                        if (mouse.X < zoomPt.X)
                        {
                            zoomGP[i].Transform(new Matrix(-1, 0, 0, 1, 2 * rect.X, 0));
                        }
                        scaleY = 1.0f;
                        break;
                    case 4:
                        if ((mouse.X < zoomPt.X) && (mouse.Y >= zoomPt.Y))
                            zoomGP[i].Transform(new Matrix(-1, 0, 0, 1, 2 * rect.X, 0));
                        else if ((mouse.X >= zoomPt.X) && (mouse.Y < zoomPt.Y))
                            zoomGP[i].Transform(new Matrix(1, 0, 0, -1, 0, 2 * rect.Y));
                        else if ((mouse.X < zoomPt.X) && (mouse.Y < zoomPt.Y))
                            zoomGP[i].Transform(new Matrix(-1, 0, 0, -1, 2 * rect.X, 2 * rect.Y));
                        break;
                    case 5:
                        if (mouse.Y < zoomPt.Y)
                        {
                            zoomGP[i].Transform(new Matrix(1, 0, 0, -1, 0, 2 * rect.Y));
                        }
                        scaleX = 1.0f;
                        break;
                    case 6:
                        if ((mouse.X >= zoomPt.X) && (mouse.Y >= zoomPt.Y))
                            zoomGP[i].Transform(new Matrix(-1, 0, 0, 1, 2 * (rect.X + rect.Width), 0));
                        else if ((mouse.X < zoomPt.X) && (mouse.Y < zoomPt.Y))
                            zoomGP[i].Transform(new Matrix(1, 0, 0, -1, 0, 2 * (rect.Y)));
                        else if ((mouse.X >= zoomPt.X) && (mouse.Y < zoomPt.Y))
                            zoomGP[i].Transform(new Matrix(-1, 0, 0, -1, 2 * (rect.X + rect.Width), 2 * rect.Y));
                        break;
                    case 7:
                        if (mouse.X >= zoomPt.X)
                        {
                            zoomGP[i].Transform(new Matrix(-1, 0, 0, 1, 2 * (rect.X + rect.Width), 0));
                        }
                        scaleY = 1.0f;
                        break;
                }
            }


            System.Windows.Media.Matrix zoomMat = new System.Windows.Media.Matrix();
            if ((Keyboard.GetKeyStates(Key.LeftShift) & KeyStates.Down) > 0)
                zoomMat.ScaleAt(scaleX, scaleX, zoomPt.X, zoomPt.Y);
            else
                zoomMat.ScaleAt(scaleX, scaleY, zoomPt.X, zoomPt.Y);

            Matrix dmat = new Matrix((float)zoomMat.M11, (float)zoomMat.M12, (float)zoomMat.M21, (float)zoomMat.M22, (float)zoomMat.OffsetX, (float)zoomMat.OffsetY);

            for (int i = 0; i < zoomGP.Count; i++)
            {
                zoomGP[i].Transform(dmat);
            }

            return zoomGP;
        }

        /// <summary>
        /// Path 위의 점을 선택하기 위한 범위를 반환하는 함수
        /// </summary>
        /// <param name="pt1">첫 번째 점</param>
        /// <param name="pt2">두 번째 점</param>
        /// <param name="offset">유효 범위를 지정하기 위한 Offset</param>
        /// <returns>점과 점 사이의 영역 Return</returns>
        public Region GetPointToRegion(Point pt1, Point pt2, int offset = 10)
        {
            Region result;
            Region temp;

            GraphicsPath gp = new GraphicsPath();
            gp.AddEllipse(pt1.X - offset, pt1.Y - offset, offset * 2, offset * 2);
            result = new Region(gp);
            gp.Reset();

            gp.AddEllipse(pt2.X - offset, pt2.Y - offset, offset * 2, offset * 2);
            temp = new Region(gp);
            result.Union(temp);
            gp.Reset();

            List<Point> poly = new List<Point>();
            if ((Math.Abs(pt2.X - pt1.X) > Math.Abs(pt2.Y - pt1.Y)) || pt2.Y - pt1.Y == 0) //오른쪽 아래 방향 (아래)
            {
                poly.Add(new Point(pt1.X, pt1.Y - offset));
                poly.Add(new Point(pt2.X, pt2.Y - offset));
                poly.Add(new Point(pt2.X, pt2.Y + offset));
                poly.Add(new Point(pt1.X, pt1.Y + offset));
            }
            else if ((Math.Abs(pt2.X - pt1.X) <= Math.Abs(pt2.Y - pt1.Y)) || pt2.X - pt1.X == 0) //오른쪽 위 방향 (위)
            {
                poly.Add(new Point(pt1.X - offset, pt1.Y));
                poly.Add(new Point(pt2.X - offset, pt2.Y));
                poly.Add(new Point(pt2.X + offset, pt2.Y));
                poly.Add(new Point(pt1.X + offset, pt1.Y));
            }

            gp.AddPolygon(poly.ToArray());
            temp = new Region(gp);
            result.Union(temp);

            return result;
        }

        /// <summary>
        /// Point에 따른 직사각형을 구하기 위한 함수
        /// </summary>
        /// <param name="startX"></param>
        /// <param name="startY"></param>
        /// <param name="endX"></param>
        /// <param name="endY"></param>
        /// <returns></returns>
        public RectangleF GetSquare(float startX, float startY, float endX, float endY)
        {
            RectangleF rect;
            if (endX - startX > 0 && endY - startY > 0)
            {
                rect = new RectangleF(startX, startY, Math.Abs(endY - startY), Math.Abs(endY - startY));
            }
            else if (endX - startX > 0 && endY - startY < 0)
            {
                rect = new RectangleF(startX, endY, Math.Abs(endY - startY), Math.Abs(endY - startY));
            }
            else if (endX - startX < 0 && endY - startY > 0)
            {
                rect = new RectangleF(endX, startY, Math.Abs(endX - startX), Math.Abs(endX - startX));
            }
            else
            {
                if (endX >= endY) rect = new RectangleF(endX, startY - (startX - endX), Math.Abs(endX - startX), Math.Abs(endX - startX));
                else rect = new RectangleF(startX - (startY - endY), endY, Math.Abs(endY - startY), Math.Abs(endY - startY));
            }

            return rect;
        }

        /// <summary>
        /// Point에 따른 정사각형을 구하기 위한 함수
        /// </summary>
        /// <param name="startX"></param>
        /// <param name="startY"></param>
        /// <param name="endX"></param>
        /// <param name="endY"></param>
        /// <returns></returns>
        public RectangleF GetRect(float startX, float startY, float endX, float endY)
        {
            RectangleF rect;

            if (endX - startX > 0 && endY - startY > 0)
            {
                rect = new RectangleF(startX, startY, Math.Abs(endX - startX), Math.Abs(endY - startY));
            }
            else if (endX - startX > 0 && endY - startY < 0)
            {
                rect = new RectangleF(startX, endY, Math.Abs(endX - startX), Math.Abs(startY - endY));
            }
            else if (endX - startX < 0 && endY - startY > 0)
            {
                rect = new RectangleF(endX, startY, Math.Abs(startX - endX), Math.Abs(endY - startY));
            }
            else
            {
                rect = new RectangleF(endX, endY, Math.Abs(startX - endX), Math.Abs(startY - endY));
            }
            return rect;
        }

        //t값을 알았을 때 x, y 좌표값을 Return 하는 함수
        public float GetBezierPosition(float t, float p0, float p1, float p2, float p3)
        {
            return (1 - t) * (1 - t) * (1 - t) * p0 + 3 * (1 - t) * (1 - t) * t * p1 + 3 * (1 - t) * t * t * p2 + t * t * t * p3;
        }

        public List<PointF> GetTangentLine(float t, PointF p0, PointF p1, PointF p2, PointF p3)
        {
            List<PointF> se = new List<PointF>();//직선을 연결할 Start, End Point List

            PointF start = new PointF(GetBezierPosition((float)(t - 0.01), p0.X, p1.X, p2.X, p3.X), GetBezierPosition((float)(t - 0.01), p0.Y, p1.Y, p2.Y, p3.Y));
            PointF end = new PointF(GetBezierPosition((float)(t + 0.01), p0.X, p1.X, p2.X, p3.X), GetBezierPosition((float)(t + 0.01), p0.Y, p1.Y, p2.Y, p3.Y));

            se.Add(start);
            se.Add(end);

            return se.ToList();
        }

        //pm = mousePosition, p0 = Start, p1 = Control1, p2 = Control2, p3 = End
        //t에 대한 3차 방정식의 실근을 구해서 List에 담고
        //t를 y방정식에 대입해서 y값을 구하고
        //mouse point와 제일 가까운 x,y 쌍을 Return
        public PointF GetAddedBezierPoint(Point pm, PointF p0, PointF p1, PointF p2, PointF p3)
        {
            List<double> t;//(0~1)사이 값
            double xValue, yValue = 0.0f; //Bezier Curve 위의 내가 추가할 점
            double tValue; //그 점에서의 t의 값
            xValue = pm.X;
            //t에 대한 3차 방정식에 xValue를 넣어서 실근 t를 구하고 listT에 넣는다
            t = GetCubicEquationValue(pm.X, Convert.ToDouble(p0.X), Convert.ToDouble(p1.X), Convert.ToDouble(p2.X), Convert.ToDouble(p3.X));

            //실근 중에서 MouseY랑 가장 가까운 
            float tempValue_y;
            for (int i = 0; i < t.Count; i++)
            {
                tempValue_y = GetBezierPosition((float)t[i], p0.Y, p1.Y, p2.Y, p3.Y);
                if (i == 0)
                {
                    yValue = tempValue_y;
                    tValue = t[i];
                    t_value = (float)tValue;
                }
                else
                {
                    if (Math.Abs(tempValue_y - pm.Y) < Math.Abs(yValue - pm.Y))
                    {
                        yValue = tempValue_y;
                        tValue = t[i];
                        t_value = (float)tValue;
                    }
                }
            }
            return new PointF((float)xValue, (float)yValue);
        }

        public List<double> GetCubicEquationValue(int pm, double p0, double p1, double p2, double p3)
        {
            List<double> root = new List<double>();
            double a, b, c, d;
            //float h = p3 - 3 * p2 + 3 * p1 - p0;
            a = p3 - (3 * p2) + (3 * p1) - p0;
            b = (3 * p2) - (6 * p1) + (3 * p0);
            c = (3 * p1) - (3 * p0);
            d = p0 - pm;

            b /= a;
            c /= a;
            d /= a;

            double disc, q, r, dum1, s, t, term1, r13;
            q = (3.0 * c - (b * b)) / 9.0;
            r = -(27.0 * d) + b * (9.0 * c - 2.0 * (b * b));
            r /= 54.0;
            disc = q * q * q + r * r;
            term1 = (b / 3.0);

            double[] x_real = new double[3];
            double[] x_imag = new double[3];

            x_imag[0] = 0;
            if (disc > 0)   // One root real, two are complex
            {
                s = r + sqrt(disc);
                s = s < 0 ? -cbrt(-s) : cbrt(s);
                t = r - sqrt(disc);
                t = t < 0 ? -cbrt(-t) : cbrt(t);
                x_real[0] = -term1 + s + t;
                term1 += (s + t) / 2.0;
                x_real[2] = x_real[1] = -term1;
                term1 = sqrt(3.0) * (-t + s) / 2;
                x_imag[1] = term1;
                x_imag[2] = -term1;
            }
            // The remaining options are all real
            else if (disc == 0)  // All roots real, at least two are equal.
            {
                x_imag[2] = x_imag[1] = 0;
                r13 = r < 0 ? -cbrt(-r) : cbrt(r);
                x_real[0] = -term1 + 2.0 * r13;
                x_real[2] = x_real[1] = -(r13 + term1);
            }
            // Only option left is that all roots are real and unequal (to get here, q < 0)
            else
            {
                x_imag[2] = x_imag[1] = 0;
                q = -q;
                dum1 = q * q * q;
                dum1 = Math.Acos(r / sqrt(dum1));
                r13 = 2.0 * sqrt(q);
                x_real[0] = -term1 + r13 * Math.Cos(dum1 / 3.0);
                x_real[1] = -term1 + r13 * Math.Cos((dum1 + 2.0 * Math.PI) / 3.0);
                x_real[2] = -term1 + r13 * Math.Cos((dum1 + 4.0 * Math.PI) / 3.0);
            }

            for (int i = 0; i < x_real.Length; i++)
            {
                if (x_real[i] >= 0 && x_real[i] <= 1.0 && x_imag[i] == 0)
                {
                    root.Add((float)x_real[i]);
                }
            }

            return root.ToList();
        }

        //Bezier Curve를 감싸는 최소 영역
        public RectangleF GetBoundsOfBezier(float ax, float ay, float bx, float by, float cx, float cy, float dx, float dy)
        {
            float px, py, qx, qy, rx, ry, sx, sy, tx, ty,
                  tobx, toby, tocx, tocy, todx, tody, toqx, toqy,
                  torx, tory, totx, toty;
            float x, y, minx, miny, maxx, maxy;

            minx = miny = float.PositiveInfinity;
            maxx = maxy = float.NegativeInfinity;

            tobx = bx - ax; toby = by - ay;  // directions
            tocx = cx - bx; tocy = cy - by;
            todx = dx - cx; tody = dy - cy;
            float step = 1 / 40f;      // precision
            for (float d = 0; d < 1.001; d += step)
            {
                px = ax + d * tobx; py = ay + d * toby;
                qx = bx + d * tocx; qy = by + d * tocy;
                rx = cx + d * todx; ry = cy + d * tody;
                toqx = qx - px; toqy = qy - py;
                torx = rx - qx; tory = ry - qy;

                sx = px + d * toqx; sy = py + d * toqy;
                tx = qx + d * torx; ty = qy + d * tory;
                totx = tx - sx; toty = ty - sy;

                x = sx + d * totx; y = sy + d * toty;
                minx = Math.Min(minx, x); miny = Math.Min(miny, y);
                maxx = Math.Max(maxx, x); maxy = Math.Max(maxy, y);
            }
            return new RectangleF(minx, miny, maxx - minx, maxy - miny);
        }

        public double sqrt(double a) { return Math.Sqrt(a); }
        public double cbrt(double a) { return Math.Pow(a, 0.333f); }
        /// <summary>
        /// 대칭점 구하는 함수
        /// </summary>
        /// <param name="p1"></param>
        /// <param name="p2"></param>
        /// <param name="reverse"></param>
        /// <returns></returns>
        private PointF GetSymmetryPoint(PointF p1, PointF p2, bool reverse = false)
        {
            float x, y;
            if (!reverse)
            {
                x = 2 * p1.X - p2.X;
                y = 2 * p1.Y - p2.Y;
            }
            else
            {
                x = 2 * p2.X - p1.X;
                y = 2 * p2.Y - p1.Y;
            }

            PointF result = new PointF(x, y);
            return result;
        }

        /// <summary>
        /// Line을 그릴 때 각도를 계산하는 함수
        /// </summary>
        /// <param name="start"></param>
        /// <param name="end"></param>
        /// <returns></returns>
        public double GetAngleFromLine(PointF start, PointF end)
        {
            double result = Math.Round(Math.Atan2(end.Y - start.Y, end.X - start.X) * (180.0 / Math.PI), 2);

            if (result < 0)
            {
                result = 360 + result;
            }

            return result;
        }

        //# 기섭 summary 작성
        private PointF GetIntersectPoint(float centerX, float centerY, float r, float p1_x, float p1_y, float p2_x, float p2_y)
        {
            PointF inter1 = new PointF();
            PointF inter2 = new PointF();

            float a, b1, c, d, m, n;

            if (p1_x != p2_x)
            {
                m = (p2_y - p1_y) / (p2_x - p1_x);
                n = (p1_y * p2_x - p1_x * p2_y) / (p2_x - p1_x);

                a = m * m + 1;
                b1 = (m * n - m * centerY - centerX);
                c = (centerX * centerX + centerY * centerY - r * r + n * n - 2 * n * centerY);
                d = b1 * b1 - a * c;

                inter1.X = (float)(-(b1 + Math.Sqrt(d)) / a);
                inter1.Y = m * inter1.X + n;

                inter2.X = (float)(-(b1 - Math.Sqrt(d)) / a);
                inter2.Y = m * inter2.X + n;
            }
            else//수직일 때 점이 움직임.. 원인을 찾아야한당 요기당..
            {
                inter1.X = p1_x;
                inter1.Y = (float)(centerY + Math.Sqrt(r * r - (p1_x - centerX) * (p1_x - centerX)));

                inter2.X = p1_x;
                inter2.Y = (float)(centerY - Math.Sqrt(r * r - (p1_x - centerX) * (p1_x - centerX)));
                //Console.WriteLine("1 = " + inter1 + ")( 2 = " + inter2);
            }


            float d1, d2;
            d1 = (float)Math.Sqrt(Math.Pow(inter1.X - pathPoints[pathPointIdx].X, 2) +
                                  Math.Pow(inter1.Y - pathPoints[pathPointIdx].Y, 2));
            d2 = (float)Math.Sqrt(Math.Pow(inter2.X - pathPoints[pathPointIdx].X, 2) +
                                  Math.Pow(inter2.Y - pathPoints[pathPointIdx].Y, 2));
            if (d1 > d2) return inter1;
            else return inter2;
        }

        /// <summary>
        /// 1, 2, 3, 4, 분면에 그려지는 사각형에 따라 Arc Data Set
        /// </summary>
        /// <param name="startX"></param>
        /// <param name="startY"></param>
        /// <param name="endX"></param>
        /// <param name="endY"></param>
        /// <param name="arc"></param>
        public void SetArcPath(int startX, int startY, int endX, int endY, GraphicsPath arc)
        {
            Rectangle rect;
            if (startX != endX && startY != endY)
            {
                if (endX - startX > 0 && endY - startY > 0)
                {
                    rect = new Rectangle(startX, startY - (endY - startY), Math.Abs(endX - startX) * 2, Math.Abs(endY - startY) * 2);
                    arc.AddArc(rect, 90f, 90f);
                }
                else if (endX - startX > 0 && endY - startY < 0)
                {
                    rect = new Rectangle(startX, endY, Math.Abs(endX - startX) * 2, Math.Abs(startY - endY) * 2);
                    arc.AddArc(rect, 180f, 90f);
                }
                else if (endX - startX < 0 && endY - startY > 0)
                {
                    rect = new Rectangle(endX - (startX - endX), startY - (endY - startY), Math.Abs(startX - endX) * 2, Math.Abs(endY - startY) * 2);
                    arc.AddArc(rect, 0f, 90f);
                }
                else
                {
                    rect = new Rectangle(endX - (startX - endX), endY, Math.Abs(startX - endX) * 2, Math.Abs(startY - endY) * 2);
                    arc.AddArc(rect, 270f, 90f);
                }
            }
        }

        public bool GetIntersectPoint2D(PointF p1, PointF p2, PointF p3, PointF p4, ref PointF inter)
        {
            float d = (p1.X - p2.X) * (p3.Y - p4.Y) - (p1.Y - p2.Y) * (p3.X - p4.X);

            if (d == 0) return false;

            float pre = (p1.X * p2.Y - p1.Y * p2.X), post = (p3.X * p4.Y - p3.Y * p4.X);
            float x = (pre * (p3.X - p4.X) - (p1.X - p2.X) * post) / d;
            float y = (pre * (p3.Y - p4.Y) - (p1.Y - p2.Y) * post) / d;

            inter = new PointF(x, y);

            return true;
        }

        public PointF GetIntersectPoint2D(PointF p1, PointF p2, PointF p3, PointF p4)
        {
            PointF result = new Point(0, 0);
            float d = (p1.X - p2.X) * (p3.Y - p4.Y) - (p1.Y - p2.Y) * (p3.X - p4.X);

            if (d == 0) return result;

            float pre = (p1.X * p2.Y - p1.Y * p2.X), post = (p3.X * p4.Y - p3.Y * p4.X);
            float x = (pre * (p3.X - p4.X) - (p1.X - p2.X) * post) / d;
            float y = (pre * (p3.Y - p4.Y) - (p1.Y - p2.Y) * post) / d;

            result = new PointF(x, y);

            return result;
        }

        public PointF[] GetthrBezirPoint(PointF s, PointF c, PointF e)
        {
            PointF[] controlPt = new PointF[2];
            //c1 = Control Point
            float bX = s.X + (2.0f / 3.0f) * (c.X - s.X);
            float bY = s.Y + (2.0f / 3.0f) * (c.Y - s.Y);
            float cX = e.X + (2.0f / 3.0f) * (c.X - e.X);
            float cY = e.Y + (2.0f / 3.0f) * (c.Y - e.Y);

            controlPt[0] = new PointF(bX, bY);
            controlPt[1] = new PointF(cX, cY);

            return controlPt;
        }
        public int GetThrBezierIdx(int ptIdx)
        {
            int idx = -1;

            for (int i = 0; i < pathTypes.Count; i++)
            {
                if (pathTypes[i] == PathTypes.N) idx++;
                if (i == ptIdx) break;
            }
            return idx;
        }
        public void SetthrBezirAndLinePath()
        {
            for (int i = 0; i < pathTypes.Count - 1; i++)
            {
                if (pathTypes[i] == PathTypes.N && pathTypes[i + 1] == PathTypes.N)
                {
                    pathTypes.RemoveRange(i, 2);
                    pathByte.RemoveRange(i, 2);
                    pathPoints.RemoveRange(i, 2);

                    if (pathByte[i] == 131)
                    {
                        if (pathByte[i - 1] == 1)
                        {
                            if (pathPoints[i] == pathPoints[0])//첫점하고 끝점이 같은 점이면 지우고 직선으로 닫음
                            {
                                pathByte.RemoveAt(i);
                                pathTypes.RemoveAt(i);
                                pathPoints.RemoveAt(i);

                                pathByte[i - 1] = 129;
                                pathTypes[i - 1] = PathTypes.CL;
                            }
                            else
                            {
                                pathByte[i] = 129;
                                pathTypes[i] = PathTypes.CL;
                            }
                        }
                        else
                        {
                            pathByte.RemoveAt(i);
                            pathTypes.RemoveAt(i);
                            pathPoints.RemoveAt(i);
                            pathByte[i - 1] = 163;
                            pathTypes[i - 1] = PathTypes.CL;
                            if (pathByte[i - 2] == 1)
                            {
                                pathByte[i - 1] = 129;
                                pathTypes[i - 1] = PathTypes.CL;
                            }
                        }
                    }
                    else if (pathByte[i] == 163)
                    {
                        pathByte[i] = 129;
                        pathTypes[i] = PathTypes.CL;
                    }
                    else
                    {
                        pathByte[i] = 1;
                        if (i == pathTypes.Count - 1 && pathByte[i - 1] != 3) pathTypes[i] = PathTypes.LE;
                    }
                }
            }
            thrBezirData.Clear();
            for (int i = 0; i < pathTypes.Count; i++)
            {
                if (pathTypes[i] == PathTypes.N)
                {
                    if (pathTypes[i - 1] == PathTypes.C1)
                    {
                        thrBezirData.Add(GetIntersectPoint2D(pathPoints[i - 2], pathPoints[i - 1], pathPoints[i], pathPoints[i + 1]));
                    }
                    else if (pathTypes[i + 1] == PathTypes.C2)
                    {
                        thrBezirData.Add(GetIntersectPoint2D(pathPoints[i - 1], pathPoints[i], pathPoints[i + 1], pathPoints[i + 2]));
                    }
                }
            }
        }
        /// <summary>
        /// thrbeizer Point를 선택했을 때 자신을 기준으로
        /// 앞 뒤에 Control Point가 있는지 확인해서 있으면
        /// pathPointSymidx를 설정해주어 같이 이동할 수 있도록
        /// </summary>
        public void SetPathPointSymIdx()
        {
            int cnt = pathTypes.Count;

            if (pathTypes[pathPointIdx + 1] == PathTypes.C2)
            {
                if (pathTypes[(pathPointIdx + 3) % cnt] == PathTypes.C1)
                {
                    if (pathTypes[(pathPointIdx + 4) % cnt] == PathTypes.N)
                        thrBezirSymIdx = thrBezirIdx + 1;
                    else
                        pathPointSymIdx = (pathPointIdx + 3) % cnt;
                }
                else if (pathTypes[1] == PathTypes.C1)
                {
                    if (pathTypes[2] == PathTypes.N)
                        thrBezirSymIdx = 0;//thridx
                    else
                        pathPointSymIdx = 1;
                }
            }
            else
            {
                if (pathPointIdx == 2)
                {
                    if (pathTypes[cnt - 3] == PathTypes.N) //맞은편이 3 Bezier
                        if (pathTypes[2] == PathTypes.N)
                            thrBezirSymIdx = thrBezirData.Count - 1;
                        else
                            pathPointSymIdx = cnt - 2;
                    else if (pathTypes[cnt - 2] == PathTypes.C2)//맞은편이 4Bezier
                    {
                        pathPointSymIdx = cnt - 2;
                    }
                }
                else
                {
                    if (pathTypes[pathPointIdx - 3] == PathTypes.C2)
                    {
                        if (pathTypes[pathPointIdx - 4] == PathTypes.N)
                            thrBezirSymIdx = thrBezirIdx - 1;
                        else
                            pathPointSymIdx = pathPointIdx - 3;
                    }
                }
            }
        }
        public void AntiClockWiseCheck(int xOffset, int yOffset)
        {
            //선택한 점의 반시계 반향에 있는 점 검사 ^_^
            if (pathPointIdx == 0)//연결돼어 있는 곡선일 때 첫번째 점 예외 처리
            {
                //닫힌 곡선이 아닐 때는 반대로 순환하면서 확인하면 안됨
                if (pathTypes[pathTypes.Count - 1] != PathTypes.CC) return;

                //마지막점이 131이면 같이 움직여줘야 함 붙어있는 곡선이라서
                if (pathByte[pathByte.Count - 1] == 131) pathPoints[pathPoints.Count - 1] = pathPoints[0];

                if (pathTypes[pathTypes.Count - 3] == PathTypes.N)
                {
                    thrBezirIdx = GetThrBezierIdx(pathTypes.Count - 3);
                    thrBezirData[thrBezirIdx] = new PointF(thrBezirData[thrBezirIdx].X + xOffset, thrBezirData[thrBezirIdx].Y + yOffset);

                    PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPoints.Count - 4], thrBezirData[thrBezirIdx], pathPoints[pathPoints.Count - 1]);
                    pathPoints[pathPoints.Count - 3] = newHandle[0];
                    pathPoints[pathPoints.Count - 2] = newHandle[1];
                }
                else if (pathTypes[pathTypes.Count - 2] == PathTypes.N)
                {
                    thrBezirIdx = GetThrBezierIdx(pathTypes.Count - 2);

                    PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPoints.Count - 4], thrBezirData[thrBezirIdx], pathPoints[pathPoints.Count - 1]);
                    pathPoints[pathPoints.Count - 3] = newHandle[0];
                    pathPoints[pathPoints.Count - 2] = newHandle[1];
                }
                else
                {
                    pathPoints[pathPoints.Count - 2] = new PointF(pathPoints[pathPoints.Count - 2].X + xOffset, pathPoints[pathPoints.Count - 2].Y + yOffset);
                }
            }
            else//일반적인 곡점일 때
            {
                if (pathTypes[pathPointIdx - 1] == PathTypes.S) return;

                if (pathTypes[pathPointIdx - 2] == PathTypes.N)
                {
                    thrBezirIdx = GetThrBezierIdx(pathPointIdx - 2);
                    thrBezirData[thrBezirIdx] = new PointF(thrBezirData[thrBezirIdx].X + xOffset, thrBezirData[thrBezirIdx].Y + yOffset);

                    PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPointIdx - 3], thrBezirData[thrBezirIdx], pathPoints[pathPointIdx]);
                    pathPoints[pathPointIdx - 2] = newHandle[0];
                    pathPoints[pathPointIdx - 1] = newHandle[1];
                }
                else if (pathTypes[pathPointIdx - 1] == PathTypes.N)
                {
                    thrBezirIdx = GetThrBezierIdx(pathPointIdx - 1);

                    PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPointIdx - 3], thrBezirData[thrBezirIdx], pathPoints[pathPointIdx]);
                    pathPoints[pathPointIdx - 2] = newHandle[0];
                    pathPoints[pathPointIdx - 1] = newHandle[1];
                }
                else
                {
                    pathPoints[pathPointIdx - 1] = new PointF(pathPoints[pathPointIdx - 1].X + xOffset, pathPoints[pathPointIdx - 1].Y + yOffset);
                }
            }
        }
        //선택한 점의 시계 방향에 있는 점 검사 ^_^
        public void ClockWiseCheck(int xOffset, int yOffset)
        {
            //다음 점 속성이 S, E, CL 이면 곡점이 아님
            if (pathTypes[pathPointIdx + 1] == PathTypes.S || pathTypes[pathPointIdx + 1] == PathTypes.LE || pathTypes[pathPointIdx + 1] == PathTypes.CL) return;

            if (pathTypes[pathPointIdx + 2] == PathTypes.N)
            {
                thrBezirIdx = GetThrBezierIdx(pathPointIdx + 2);

                thrBezirData[thrBezirIdx] = new PointF(thrBezirData[thrBezirIdx].X + xOffset, thrBezirData[thrBezirIdx].Y + yOffset);

                PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPointIdx], thrBezirData[thrBezirIdx], pathPoints[pathPointIdx + 3]);
                pathPoints[pathPointIdx + 1] = newHandle[0];
                pathPoints[pathPointIdx + 2] = newHandle[1];
            }
            else if (pathTypes[pathPointIdx + 1] == PathTypes.N)
            {
                thrBezirIdx = GetThrBezierIdx(pathPointIdx + 1);

                PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPointIdx], thrBezirData[thrBezirIdx], pathPoints[pathPointIdx + 3]);
                pathPoints[pathPointIdx + 1] = newHandle[0];
                pathPoints[pathPointIdx + 2] = newHandle[1];
            }
            else
            {
                pathPoints[pathPointIdx + 1] = new PointF(pathPoints[pathPointIdx + 1].X + xOffset, pathPoints[pathPointIdx + 1].Y + yOffset);
            }
        }

        public void SetBezierPoint(Point location, int xOffset = 0, int yOffset = 0)
        {
            if (isThrBezir)//선택한 점이 ThrBezier Point 이면
            {
                thrBezirData[thrBezirIdx] = location;
                //선택한 점의 C1, C2를 이동//
                if (pathTypes[pathPointIdx - 1] == PathTypes.C1)
                {
                    PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPointIdx - 2], thrBezirData[thrBezirIdx], pathPoints[pathPointIdx + 1]);
                    pathPoints[pathPointIdx - 1] = newHandle[0];
                    pathPoints[pathPointIdx] = newHandle[1];
                }
                else
                {
                    PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPointIdx - 1], thrBezirData[thrBezirIdx], pathPoints[pathPointIdx + 2]);
                    pathPoints[pathPointIdx] = newHandle[0];
                    pathPoints[pathPointIdx + 1] = newHandle[1];
                }
                //
                //대칭 점에 있는 점을 이동
                float r;
                if (pathPointSymIdx != -1)
                {
                    if (pathTypes[pathPointSymIdx] == PathTypes.C1)
                        r = (float)Math.Sqrt(Math.Pow(pathPoints[pathPointSymIdx].X - pathPoints[pathPointSymIdx - 1].X, 2) +
                                             Math.Pow(pathPoints[pathPointSymIdx].Y - pathPoints[pathPointSymIdx - 1].Y, 2));
                    else
                        r = (float)Math.Sqrt(Math.Pow(pathPoints[pathPointSymIdx].X - pathPoints[pathPointSymIdx + 1].X, 2) +
                                             Math.Pow(pathPoints[pathPointSymIdx].Y - pathPoints[pathPointSymIdx + 1].Y, 2));
                    try
                    {
                        if (pathTypes[pathPointSymIdx] == PathTypes.C1)
                            pathPoints[pathPointSymIdx] = GetIntersectPoint(pathPoints[pathPointSymIdx - 1].X, pathPoints[pathPointSymIdx - 1].Y, r,
                                                                            pathPoints[pathPointSymIdx - 1].X, pathPoints[pathPointSymIdx - 1].Y,
                                                                            thrBezirData[thrBezirIdx].X, thrBezirData[thrBezirIdx].Y);
                        else if (pathTypes[pathPointSymIdx] == PathTypes.C2)
                            pathPoints[pathPointSymIdx] = GetIntersectPoint(pathPoints[pathPointSymIdx + 1].X, pathPoints[pathPointSymIdx + 1].Y, r,
                                                                            pathPoints[pathPointSymIdx + 1].X, pathPoints[pathPointSymIdx + 1].Y,
                                                                            thrBezirData[thrBezirIdx].X, thrBezirData[thrBezirIdx].Y);
                    }
                    catch (Exception ex)
                    {
                        Console.WriteLine(ex);
                        return;
                    }
                }
                else if (thrBezirSymIdx != -1) // 대칭에 있는 것도 3 베지어일 때 
                {
                    {
                        if (thrBezirIdx < thrBezirSymIdx)
                        {
                            if (thrBezirSymIdx == thrBezirData.Count - 1)
                                r = (float)Math.Sqrt(Math.Pow(thrBezirData[thrBezirSymIdx].X - pathPoints[0].X, 2) +
                                                     Math.Pow(thrBezirData[thrBezirSymIdx].Y - pathPoints[0].Y, 2));
                            else
                                r = (float)Math.Sqrt(Math.Pow(thrBezirData[thrBezirSymIdx].X - pathPoints[pathPointIdx + 2].X, 2) +
                                                     Math.Pow(thrBezirData[thrBezirSymIdx].Y - pathPoints[pathPointIdx + 2].Y, 2));

                        }
                        else
                        {
                            if (thrBezirSymIdx == 0)
                                r = (float)Math.Sqrt(Math.Pow(thrBezirData[thrBezirSymIdx].X - pathPoints[0].X, 2) +
                                                     Math.Pow(thrBezirData[thrBezirSymIdx].Y - pathPoints[0].Y, 2));
                            else
                                r = (float)Math.Sqrt(Math.Pow(thrBezirData[thrBezirSymIdx].X - pathPoints[pathPointIdx - 2].X, 2) +
                                                     Math.Pow(thrBezirData[thrBezirSymIdx].Y - pathPoints[pathPointIdx - 2].Y, 2));
                        }

                        try
                        {
                            if (thrBezirIdx < thrBezirSymIdx)
                            {
                                if (thrBezirSymIdx == thrBezirData.Count - 1)
                                {
                                    thrBezirData[thrBezirSymIdx] = GetIntersectPoint(pathPoints[0].X, pathPoints[0].Y, r,
                                                                                     pathPoints[0].X, pathPoints[0].Y,
                                                                                     thrBezirData[thrBezirIdx].X, thrBezirData[thrBezirIdx].Y);

                                    PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPoints.Count - 4], thrBezirData[thrBezirSymIdx], pathPoints[pathPoints.Count - 1]);
                                    pathPoints[pathPoints.Count - 3] = newHandle[0];
                                    pathPoints[pathPoints.Count - 2] = newHandle[1];
                                }
                                else
                                {
                                    thrBezirData[thrBezirSymIdx] = GetIntersectPoint(pathPoints[pathPointIdx + 2].X, pathPoints[pathPointIdx + 2].Y, r,
                                                                                     pathPoints[pathPointIdx + 2].X, pathPoints[pathPointIdx + 2].Y,
                                                                                     thrBezirData[thrBezirIdx].X, thrBezirData[thrBezirIdx].Y); ;

                                    PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPointIdx + 2], thrBezirData[thrBezirSymIdx], pathPoints[pathPointIdx + 5]);
                                    pathPoints[pathPointIdx + 3] = newHandle[0];
                                    pathPoints[pathPointIdx + 4] = newHandle[1];
                                }
                            }
                            else
                            {
                                if (thrBezirSymIdx == 0)
                                {
                                    thrBezirData[thrBezirSymIdx] = GetIntersectPoint(pathPoints[0].X, pathPoints[0].Y, r,
                                                                                     pathPoints[0].X, pathPoints[0].Y,
                                                                                     thrBezirData[thrBezirIdx].X, thrBezirData[thrBezirIdx].Y); ;

                                    PointF[] newHandle = GetthrBezirPoint(pathPoints[0], thrBezirData[thrBezirSymIdx], pathPoints[3]);
                                    pathPoints[1] = newHandle[0];
                                    pathPoints[2] = newHandle[1];
                                }
                                else
                                {
                                    thrBezirData[thrBezirSymIdx] = GetIntersectPoint(pathPoints[pathPointIdx - 2].X, pathPoints[pathPointIdx - 2].Y, r,
                                                                                     pathPoints[pathPointIdx - 2].X, pathPoints[pathPointIdx - 2].Y,
                                                                                     thrBezirData[thrBezirIdx].X, thrBezirData[thrBezirIdx].Y); ;

                                    PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPointIdx - 5], thrBezirData[thrBezirSymIdx], pathPoints[pathPointIdx - 2]);
                                    pathPoints[pathPointIdx - 4] = newHandle[0];
                                    pathPoints[pathPointIdx - 3] = newHandle[1];
                                }
                            }
                        }
                        catch (Exception ex)
                        {
                            Console.WriteLine(ex);
                            return;
                        }
                    }
                }
            }
            else//S or C1 or C2
            {
                pathPoints[pathPointIdx] = location;
                if (pathPointIdx != pathPoints.Count - 1)
                {
                    switch (pathTypes[pathPointIdx])
                    {
                        case PathTypes.S:
                            AntiClockWiseCheck(xOffset, yOffset);
                            ClockWiseCheck(xOffset, yOffset);
                            break;
                        case PathTypes.C1:
                            if (pathPointSymIdx != -1)
                            {
                                float r = (float)Math.Sqrt(Math.Pow(pathPoints[pathPointIdx - 1].X - pathPoints[pathPointSymIdx].X, 2) +
                                                           Math.Pow(pathPoints[pathPointIdx - 1].Y - pathPoints[pathPointSymIdx].Y, 2));
                                try
                                {
                                    pathPoints[pathPointSymIdx] = GetIntersectPoint(pathPoints[pathPointIdx - 1].X, pathPoints[pathPointIdx - 1].Y, r,
                                                                                    pathPoints[pathPointIdx - 1].X, pathPoints[pathPointIdx - 1].Y,
                                                                                    pathPoints[pathPointIdx].X, pathPoints[pathPointIdx].Y);
                                }
                                catch (Exception ex)
                                {
                                    Console.WriteLine(ex);
                                    return;
                                }
                            }
                            else if (thrBezirIdx != -1)
                            {
                                float r = (float)Math.Sqrt(Math.Pow(pathPoints[pathPointIdx - 1].X - thrBezirData[thrBezirIdx].X, 2) +
                                                           Math.Pow(pathPoints[pathPointIdx - 1].Y - thrBezirData[thrBezirIdx].Y, 2));
                                try
                                {
                                    thrBezirData[thrBezirIdx] = GetIntersectPoint(pathPoints[pathPointIdx - 1].X, pathPoints[pathPointIdx - 1].Y, r,
                                                                                  pathPoints[pathPointIdx - 1].X, pathPoints[pathPointIdx - 1].Y,
                                                                                  pathPoints[pathPointIdx].X, pathPoints[pathPointIdx].Y); ;

                                    if (pathPointIdx == 1)
                                    {
                                        PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPoints.Count - 4], thrBezirData[thrBezirIdx], pathPoints[pathPoints.Count - 1]);
                                        pathPoints[pathPoints.Count - 3] = newHandle[0];
                                        pathPoints[pathPoints.Count - 2] = newHandle[1];
                                    }
                                    else
                                    {
                                        PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPointIdx - 4], thrBezirData[thrBezirIdx], pathPoints[pathPointIdx - 1]);
                                        pathPoints[pathPointIdx - 3] = newHandle[0];
                                        pathPoints[pathPointIdx - 2] = newHandle[1];
                                    }
                                }
                                catch (Exception ex)
                                {
                                    Console.WriteLine(ex);
                                    return;
                                }
                            }
                            break;
                        case PathTypes.C2:
                            if (pathPointSymIdx != -1)
                            {

                                float r = (float)Math.Sqrt(Math.Pow(pathPoints[pathPointIdx + 1].X - pathPoints[pathPointSymIdx].X, 2) +
                                                           Math.Pow(pathPoints[pathPointIdx + 1].Y - pathPoints[pathPointSymIdx].Y, 2));
                                try
                                {
                                    pathPoints[pathPointSymIdx] = GetIntersectPoint(pathPoints[pathPointIdx + 1].X, pathPoints[pathPointIdx + 1].Y, r,
                                                                                    pathPoints[pathPointIdx + 1].X, pathPoints[pathPointIdx + 1].Y,
                                                                                    pathPoints[pathPointIdx].X, pathPoints[pathPointIdx].Y);
                                }
                                catch (Exception ex)
                                {
                                    Console.WriteLine(ex);
                                    return;
                                }
                            }
                            else if (thrBezirIdx != -1)
                            {
                                float r = (float)Math.Sqrt(Math.Pow(pathPoints[pathPointIdx + 1].X - thrBezirData[thrBezirIdx].X, 2) +
                                                           Math.Pow(pathPoints[pathPointIdx + 1].Y - thrBezirData[thrBezirIdx].Y, 2));
                                try
                                {
                                    thrBezirData[thrBezirIdx] = GetIntersectPoint(pathPoints[pathPointIdx + 1].X, pathPoints[pathPointIdx + 1].Y, r,
                                                                                  pathPoints[pathPointIdx + 1].X, pathPoints[pathPointIdx + 1].Y,
                                                                                  pathPoints[pathPointIdx].X, pathPoints[pathPointIdx].Y); ;

                                    if (pathPointIdx == pathPoints.Count - 2)
                                    {
                                        PointF[] newHandle = GetthrBezirPoint(pathPoints[0], thrBezirData[thrBezirIdx], pathPoints[3]);
                                        pathPoints[1] = newHandle[0];
                                        pathPoints[2] = newHandle[1];
                                    }
                                    else
                                    {
                                        PointF[] newHandle = GetthrBezirPoint(pathPoints[pathPointIdx + 1], thrBezirData[thrBezirIdx], pathPoints[pathPointIdx + 4]);
                                        pathPoints[pathPointIdx + 2] = newHandle[0];
                                        pathPoints[pathPointIdx + 3] = newHandle[1];
                                    }
                                }
                                catch (Exception ex)
                                {
                                    Console.WriteLine(ex);
                                    return;
                                }
                            }
                            break;
                        case PathTypes.CE:
                            if (pathTypes[pathPointIdx - 1] == PathTypes.C2)//선택한 이전 점이 C2이면 같이 움직임
                                pathPoints[pathPointIdx - 1] = new PointF(pathPoints[pathPointIdx - 1].X + xOffset, pathPoints[pathPointIdx - 1].Y + yOffset);
                            break;
                    }
                }
                //닫히지 않은 곡선의 마지막 점에 대한 예외 처리
                else if (pathTypes[pathPointIdx] == PathTypes.CE && pathTypes[pathPointIdx - 1] == PathTypes.C2)
                    pathPoints[pathPointIdx - 1] = new PointF(pathPoints[pathPointIdx - 1].X + xOffset, pathPoints[pathPointIdx - 1].Y + yOffset);
            }
        }
        public void SetPathTypes(GraphicsPath path)
        {
            int typeThreeCnt = 0;
            pathTypes = new List<PathTypes>();
            for (int k = 0; k < path.PointCount; k++)
            {
                Console.WriteLine(path.PathTypes[k]);
                if (path.PathTypes[k] == 0)//0이면 Start
                    pathTypes.Add(PathTypes.S);
                else if (path.PathTypes[k] == 1)//1인데
                {
                    if (k == path.PointCount - 1)//마지막 Point  
                        pathTypes.Add(PathTypes.LE);
                    else//아니면 Start
                        pathTypes.Add(PathTypes.S);
                }
                else if (path.PathTypes[k] == 3)//3인데
                {
                    typeThreeCnt++;
                    if (typeThreeCnt % 3 == 1) // 첫번째 3이면 Control P1
                    {
                        pathTypes.Add(PathTypes.C1);
                    }
                    else if (typeThreeCnt % 3 == 2) //두번째 3이면 Control P2
                    {
                        pathTypes.Add(PathTypes.C2);
                    }
                    else // 나머지가 0일 때는
                    {
                        if (k == path.PointCount - 1) //끝 Point면 End고
                            pathTypes.Add(PathTypes.CE);
                        else //아니면 Start임
                            pathTypes.Add(PathTypes.S);
                    }
                }
                else if (path.PathTypes[k] == 129)
                {
                    pathTypes.Add(PathTypes.CL);
                }
                else if (path.PathTypes[k] == 131)
                {
                    pathTypes.Add(PathTypes.CC);
                }// 139 등 계속 추가하자 걸릴 때 
                else if (path.PathTypes[k] == 129)
                {
                    pathTypes.Add(PathTypes.CL);
                }
                else
                {
                    pathTypes.Add(PathTypes.CE);
                }
            }
        }
        /// <summary>
        /// Line을 그리는 중에 Shift Key가 눌리면 수평 / 수직 Line으로 그리도록 함
        /// </summary>
        private void SetIntuitiveLine()
        {
            //# Line or Rect, Circle Shifkey Down
            GraphicsPath line = new GraphicsPath();
            if (mode == Mode.LINE || mode == Mode.SPRAY)
            {
                double angle = 360 - GetAngleFromLine(start, end);
                if ((angle >= 45 && angle < 135) || (angle >= 225 && angle < 315))
                {
                    line.AddLine(start, new Point(start.X, end.Y));
                }
                else
                {
                    line.AddLine(start, new Point(end.X, start.Y));
                }
                currentData.path = line;
            }
        }

        //# 기섭 summary 작성
        private void Swap<VectorData>(IList<VectorData> vector, int i1, int i2)
        {
            VectorData temp = vector[i1];
            vector[i1] = vector[i2];
            vector[i2] = temp;

            //# 보완예정

            for (int i = 0; i < groupList.Count; i++)
            {
                if (groupList[i].Contains(i1))
                {
                    groupList[i].Remove(i1);
                    groupList[i].Add(i2);
                }
                else if (groupList[i].Contains(i2))
                {
                    groupList[i].Remove(i2);
                    groupList[i].Add(i1);
                }
            }
        }

        private void tsm_forward_Click(object sender, EventArgs e)
        {
            if (selPathIndex.Count == 0) return;

            selPathIndex.Sort((x1, x2) => x2.CompareTo(x1));//오름
            if (!isMirror)
            {
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    if (selPathIndex[i] == vecList.Count - 1)
                    {
                        continue;
                    }
                    else
                    {
                        if (!selPathIndex.Contains(selPathIndex[i] + 1))
                        {
                            Swap(vecList, selPathIndex[i], selPathIndex[i] + 1);
                            selPathIndex[i] = selPathIndex[i] + 1;
                        }
                    }
                }
            }
            else
            {
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    if (selPathIndex[i] == oriList.Count - 1)
                    {
                        continue;
                    }
                    else
                    {
                        if (!selPathIndex.Contains(selPathIndex[i] + 1))
                        {
                            Swap(oriList, selPathIndex[i], selPathIndex[i] + 1);
                            selPathIndex[i] = selPathIndex[i] + 1;
                        }
                    }
                }
            }

            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
            Commit();
            //RestoreSelMode();
        }

        private void tsm_backward_Click(object sender, EventArgs e)
        {
            if (selPathIndex.Count == 0) return;

            selPathIndex.Sort((x1, x2) => x1.CompareTo(x2));
            if (!isMirror)
            {
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    if (selPathIndex[i] == 0)
                    {
                        continue;
                    }
                    else
                    {
                        if (!selPathIndex.Contains(selPathIndex[i] - 1))
                        {
                            Swap(vecList, selPathIndex[i], selPathIndex[i] - 1);
                            selPathIndex[i] = selPathIndex[i] - 1;
                        }
                    }
                }
            }
            else
            {
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    if (selPathIndex[i] == 0)
                    {
                        continue;
                    }
                    else
                    {
                        if (!selPathIndex.Contains(selPathIndex[i] - 1))
                        {
                            Swap(oriList, selPathIndex[i], selPathIndex[i] - 1);
                            selPathIndex[i] = selPathIndex[i] - 1;
                        }
                    }
                }
            }

            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
            Commit();
            //RestoreSelMode();
        }

        private void tsm_front_Click(object sender, EventArgs e)
        {
            if (selPathIndex.Count == 0) return;

            selPathIndex.Sort((x1, x2) => x1.CompareTo(x2));//오름
            if (!isMirror)
            {
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    vecList.Add(vecList[selPathIndex[i]]);
                }
                selPathIndex.Sort((x1, x2) => x2.CompareTo(x1));//내림
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    vecList.RemoveAt(selPathIndex[i]);
                }
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    selPathIndex[i] = vecList.Count - 1 - i;
                }

            }
            else
            {
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    oriList.Add(oriList[selPathIndex[i]]);
                    refList.Add(refList[selPathIndex[i]]);
                }
                selPathIndex.Sort((x1, x2) => x2.CompareTo(x1));
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    oriList.RemoveAt(selPathIndex[i]);
                    refList.RemoveAt(selPathIndex[i]);
                }
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    selPathIndex[i] = oriList.Count - 1 - i;
                }
            }

            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
            Commit();
            //RestoreSelMode();
        }

        private void tsm_back_Click(object sender, EventArgs e)
        {
            if (selPathIndex.Count == 0) return;

            selPathIndex.Sort((x1, x2) => x2.CompareTo(x1));
            if (!isMirror)
            {
                int idx = 0; //Insert 보정값
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    vecList.Insert(0, vecList[selPathIndex[i] + idx]);
                    idx++;
                }
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    vecList.RemoveAt(selPathIndex[i] + selPathIndex.Count);
                }
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    selPathIndex[i] = i;
                }
            }
            else
            {
                int idx = 0; //Insert 보정값
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    oriList.Insert(0, oriList[selPathIndex[i] + idx]);
                    refList.Insert(0, refList[selPathIndex[i] + idx]);
                    idx++;
                }
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    oriList.RemoveAt(selPathIndex[i] + selPathIndex.Count);
                    refList.RemoveAt(selPathIndex[i] + selPathIndex.Count);
                }
                for (int i = 0; i < selPathIndex.Count; i++)
                {
                    selPathIndex[i] = i;
                }
            }

            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
            Commit();
            //RestoreSelMode();
        }

        //=======================================================================
        //#  Undo / Redo Function
        //=======================================================================
        private void Undo()
        {
            mode = Mode.UNDO;
            drawing = true;

            if (!isMirror)
            {
                doIdx--;
                if (doIdx == -1)
                {
                    vecList.Clear();
                }
                else if (doIdx < -1)
                {
                    doIdx = -1;
                    return;
                }
                else
                {
                    vecList = doVector[doIdx].ToList();
                    for (int i = 0; i < vecList.Count; i++)
                    {
                        SetVectorList(vecList[i]);
                    }
                }
            }
            else
            {
                refDoIdx--;
                if (refDoIdx == -1)
                {
                    oriList.Clear();
                    refList.Clear();
                }
                else if (refDoIdx < -1)
                {
                    refDoIdx = -1;
                    return;
                }
                else
                {
                    oriList = oriDoVector[refDoIdx].ToList();
                    refList = refDoVector[refDoIdx].ToList();
                }
                selRefPath.Clear();
            }

            selPathIndex.Clear();
            selTempPath.Clear();
            pictureBox1.Refresh();
            drawing = false;
        }

        private void Redo()
        {
            mode = Mode.REDO;
            drawing = true;

            if (!isMirror)
            {
                doIdx++;
                if (doIdx == doVector.Count)
                {
                    doIdx = doVector.Count - 1;
                    return;
                }
                else
                {
                    vecList = doVector[doIdx].ToList();
                    for (int i = 0; i < vecList.Count; i++)
                    {
                        SetVectorList(vecList[i]);
                    }
                }
            }
            else
            {
                refDoIdx++;
                if (refDoIdx == oriDoVector.Count)
                {
                    refDoIdx = oriDoVector.Count - 1;
                    return;
                }
                else
                {
                    oriList = oriDoVector[refDoIdx].ToList();
                    refList = refDoVector[refDoIdx].ToList();
                }
                selRefPath.Clear();
            }

            selPathIndex.Clear();
            selTempPath.Clear();
            pictureBox1.Refresh();
            drawing = false;
        }

        //=======================================================================
        //#  Pen Setting
        //=======================================================================
        /// <summary>
        /// ComboBox index에 해당하는 LineCap 반환
        /// </summary>
        /// <param name="idx"></param>
        /// <returns></returns>
        public LineCap GetLineCap(int idx)
        {
            switch (idx)
            {
                case 0:
                    return LineCap.Flat;
                case 1:
                    return LineCap.Round;
                case 2:
                    return LineCap.Square;
                case 3:
                    return LineCap.Triangle;
                case 4:
                    return LineCap.DiamondAnchor;
                case 5:
                    return LineCap.ArrowAnchor;
                default:
                    return LineCap.Flat;
            }
        }

        /// <summary>
        /// Pen 변경
        /// </summary>
        public void ChangePen()
        {
            currentData.pen = new Pen(bitCanvas.mPalette.color, float.Parse(tscb_width.Text));

            if (cmb_dashStyle.SelectedIndex > 0)
            {
                currentData.pen.DashStyle = (DashStyle)cmb_dashStyle.SelectedIndex;

                List<float> pattern = new List<float>();
                switch (cmb_dashStyle.SelectedIndex)
                {
                    case 0: //# solid
                        txt_i1.Enabled = false; txt_i2.Enabled = false; txt_i3.Enabled = false;
                        txt_l1.Enabled = false; txt_l2.Enabled = false; txt_l3.Enabled = false;
                        break;
                    case 1: //# dot
                        txt_i1.Enabled = true; txt_i2.Enabled = false; txt_i3.Enabled = false;
                        txt_l1.Enabled = true; txt_l2.Enabled = false; txt_l3.Enabled = false;
                        pattern.Add(float.Parse(txt_l1.Text)); pattern.Add(float.Parse(txt_i1.Text));
                        break;
                    case 2: //# dash
                        txt_i1.Enabled = true; txt_i2.Enabled = false; txt_i3.Enabled = false;
                        txt_l1.Enabled = true; txt_l2.Enabled = false; txt_l3.Enabled = false;
                        pattern.Add(float.Parse(txt_l1.Text)); pattern.Add(float.Parse(txt_i1.Text));
                        break;
                    case 3: //# dashdot
                        txt_i1.Enabled = true; txt_i2.Enabled = true; txt_i3.Enabled = false;
                        txt_l1.Enabled = true; txt_l2.Enabled = true; txt_l3.Enabled = false;
                        pattern.Add(float.Parse(txt_l1.Text)); pattern.Add(float.Parse(txt_i1.Text));
                        pattern.Add(float.Parse(txt_l2.Text)); pattern.Add(float.Parse(txt_i2.Text));
                        break;
                    case 4: //# dashdotdot
                        txt_i1.Enabled = true; txt_i2.Enabled = true; txt_i3.Enabled = true;
                        txt_l1.Enabled = true; txt_l2.Enabled = true; txt_l3.Enabled = true;
                        pattern.Add(float.Parse(txt_l1.Text)); pattern.Add(float.Parse(txt_i1.Text));
                        pattern.Add(float.Parse(txt_l2.Text)); pattern.Add(float.Parse(txt_i2.Text));
                        pattern.Add(float.Parse(txt_l3.Text)); pattern.Add(float.Parse(txt_i3.Text));
                        break;
                    default:
                        break;

                }
                currentData.pen.DashPattern = pattern.ToArray();
            }

            currentData.pen.StartCap = GetLineCap(cmb_startCap.SelectedIndex);
            currentData.pen.EndCap = GetLineCap(cmb_endCap.SelectedIndex);
            if (cmb_lineJoin.SelectedIndex == 0)
            {
                currentData.pen.LineJoin = LineJoin.Miter;
            }
            else
            {
                currentData.pen.LineJoin = LineJoin.Round;
            }

            if (!(selPathIndex.Count > 0)) return;

            for (int i = 0; i < selPathIndex.Count; i++)
            {
                ModifyData(ModifyMode.PEN, selPathIndex[i], (Pen)currentData.pen.Clone());
            }

            RestoreSelMode();
        }

        public void cmb_pen_SelectedIndexChange(object sender, EventArgs e)
        {
            ChangePen();
        }

        private void Dash_KeyDown(object sender, KeyEventArgs e)
        {
            if (!float.TryParse(((TextBox)sender).Text, out float result)) return;
            if (e.KeyCode == Keys.Enter)
            {
                ChangePen();
            }
        }

        //=======================================================================
        //#  Brush
        //=======================================================================
        /// <summary>
        /// 단색 Brush 적용
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btn_solid_apply_Click(object sender, EventArgs e)
        {
            if (selPathIndex.Count == 0) return;

            currentData.brush = new SolidBrush(bitCanvas.mPalette.color);

            for (int i = 0; i < selPathIndex.Count; i++)
            {
                ModifyData(ModifyMode.BRUSH, selPathIndex[i], (Brush)currentData.brush.Clone());
            }

            Commit();
            pictureBox1.Refresh();
        }

        /// <summary>
        /// Gradient 시작색 설정
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void col_start_Click(object sender, EventArgs e)
        {
            col_start.BackColor = bitCanvas.mPalette.color;
        }

        /// <summary>
        /// Gradient 끝색 설정
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void col_end_Click(object sender, EventArgs e)
        {
            col_end.BackColor = bitCanvas.mPalette.color;
        }

        /// <summary>
        /// Gradient Brush 적용
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btn_gradient_apply_Click(object sender, EventArgs e)
        {
            //# 거울 모드일 때 작동 X
            if (isMirror) return;
            Color startColor = col_start.BackColor;
            Color endColor = col_end.BackColor;

            for (int i = 0; i < selPathIndex.Count; i++)
            {
                RectangleF bound = vecList[selPathIndex[i]].path.GetBounds();
                currentData.brush = new LinearGradientBrush(bound, startColor, endColor, Convert.ToSingle(nud_gradient_angle.Value));
                ModifyData(ModifyMode.BRUSH, selPathIndex[i], (Brush)CreateNewBrush(currentData.brush, selTempPath[i].GetBounds()));
            }

            Commit();
            pictureBox1.Refresh();
        }

        /// <summary>
        /// Texture Brush 적용
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btn_texture_apply_Click(object sender, EventArgs e)
        {
            //# 거울 모드일 때 작동 X
            if (isMirror) return;

            Bitmap textureImg = new Bitmap(patternImg, new Size(Convert.ToInt32(txt_width.Text.ToString()), Convert.ToInt32(txt_height.Text.ToString())));
            TextureBrush br = new TextureBrush(textureImg);
            br.RotateTransform(Convert.ToSingle(nud_texture_angle.Value));
            currentData.brush = br;
            for (int i = 0; i < selPathIndex.Count; i++)
            {
                ModifyData(ModifyMode.BRUSH, selPathIndex[i], (Brush)currentData.brush.Clone());
            }

            Commit();
            pictureBox1.Refresh();
        }

        /// <summary>
        /// Texture Image 불러오기
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void btn_file_Click(object sender, EventArgs e)
        {
            OpenFileDialog fileDialog = new OpenFileDialog();
            fileDialog.Filter = "Image Files(*.BMP;*.JPG;*.GIF)|*.BMP;*.JPG;*.GIF|All files (*.*)|*.*";
            fileDialog.ShowDialog();

            try
            {
                patternImg = (Bitmap)Image.FromFile(fileDialog.FileName);
                img_pattern.BackgroundImage = patternImg;
            }
            catch
            {
                MessageBox.Show("파일을 선택해주세요.");
                return;
            }

            txt_width.Enabled = true;
            txt_height.Enabled = true;
            txt_width.Text = patternImg.Width.ToString();
            txt_height.Text = patternImg.Height.ToString();
        }

        //=======================================================================
        //#  File
        //=======================================================================
        private void tsb_save_Click(object sender, EventArgs e)
        {
            /* Set SaveFileDialog */
            SaveFileDialog saveFileDialog = new SaveFileDialog();
            saveFileDialog.Filter = "Bitmap Image|*.bmp|JPeg Image|*.jpg|Png Image|*.png|TexMaster|*.txm|Scalable Vector Graphics|*.svg";
            saveFileDialog.Title = "Save an Image File";
            saveFileDialog.ShowDialog();

            Image canvasImg = pictureBox1.Image;
            Graphics grImg = Graphics.FromImage(canvasImg);

            if (saveFileDialog.FilterIndex != 3)
                grImg.Clear(Color.White);   //# 배경색
            else
                grImg.Clear(Color.Transparent);

            DrawDataList(grImg);

            /* Save Image */
            if (saveFileDialog.FileName != "")
            {
                FileStream fs = (FileStream)saveFileDialog.OpenFile();
                switch (saveFileDialog.FilterIndex)
                {
                    case 1: //# bmp
                        canvasImg.Save(fs, ImageFormat.Bmp);
                        break;

                    case 2: //# jpeg
                        canvasImg.Save(fs, ImageFormat.Jpeg);
                        break;

                    case 3: //# png
                        canvasImg.Save(fs, ImageFormat.Png);
                        break;
                    case 4: //# txm
                        BinaryFormatter formatter = new BinaryFormatter();
                        List<FileData> totalData = new List<FileData>();

                        for (int i = 0; i < vecList.Count; i++)
                        {
                            FileData eachData = new FileData();

                            eachData.xmlPath = GetSvgPath(vecList[i].path, vecList[i].path.PathTypes[vecList[i].path.PointCount - 1] == 129);

                            eachData.isString = vecList[i].isString;

                            string brushClass = vecList[i].brush.ToString();
                            if (brushClass == "System.Drawing.TextureBrush")
                            {
                                eachData.brushType = 2;
                                eachData.pattern = ((TextureBrush)vecList[i].brush).Image;
                            }
                            else if (brushClass == "System.Drawing.Drawing2D.LinearGradientBrush")
                            {
                                eachData.brushType = 1;
                                eachData.sColor = ((LinearGradientBrush)vecList[i].brush).LinearColors[0].ToArgb();
                                eachData.eColor = ((LinearGradientBrush)vecList[i].brush).LinearColors[1].ToArgb();
                            }
                            else
                            {
                                eachData.brushType = 0;
                                eachData.color = ((SolidBrush)vecList[i].brush).Color.ToArgb();
                            }

                            eachData.penWidth = vecList[i].pen.Width;
                            eachData.dash = vecList[i].pen.DashStyle;
                            eachData.sCap = vecList[i].pen.StartCap;
                            eachData.eCap = vecList[i].pen.EndCap;
                            eachData.lCap = vecList[i].pen.LineJoin;

                            eachData.pColor = vecList[i].pen.Color.ToArgb();

                            if (eachData.isString)
                            {
                                eachData.isPathText = vecList[i].isPathText;
                                eachData.str = vecList[i].str;
                                eachData.fontName = vecList[i].font.FontFamily.Name;
                                eachData.fontSize = vecList[i].font.Size;
                                eachData.fontStyle = vecList[i].font.Style;
                            }

                            totalData.Add(eachData);
                        }
                        try
                        {
                            formatter.Serialize(fs, totalData);
                        }
                        catch
                        {
                            throw;
                        }
                        finally
                        {
                            fs.Close();
                        }
                        break;
                    case 5: //# svg
                        fs.Close();
                        SvgDocument document = new SvgDocument
                        {
                            Width = pictureBox1.Width,
                            Height = pictureBox1.Height
                        };

                        document.ViewBox = new SvgViewBox(0, 0, pictureBox1.Width, pictureBox1.Height);

                        for (int i = 0; i < vecList.Count; i++)
                        {
                            if (vecList[i].isString)
                            {
                                Thread.CurrentThread.CurrentCulture = System.Globalization.CultureInfo.GetCultureInfoByIetfLanguageTag("en-US");
                                Thread.CurrentThread.CurrentUICulture = System.Globalization.CultureInfo.GetCultureInfoByIetfLanguageTag("en-US");

                                SvgText text = new SvgText();
                                text.Transforms = new SvgTransformCollection();
                                text.Transforms.Add(new SvgTranslate(vecList[i].path.PathPoints[0].X, vecList[i].path.PathPoints[0].Y));

                                string[] splitStr = vecList[i].str.Split(new char[1] { '\n' });

                                for (int j = 0; j < splitStr.Length; j++)
                                {
                                    SvgTextSpan txtSpan = new SvgTextSpan();
                                    txtSpan.X.Add(new SvgUnit(0));
                                    txtSpan.Y.Add(new SvgUnit(vecList[i].font.Height * j));
                                    SvgContentNode newContent = new SvgContentNode();
                                    newContent.Content = splitStr[j];
                                    txtSpan.Nodes.Add(newContent);
                                    txtSpan.FontFamily = vecList[i].font.Name;
                                    txtSpan.FontSize = vecList[i].font.Size;
                                    text.Children.Add(txtSpan);
                                }

                                document.Children.Add(text);

                                Thread.CurrentThread.CurrentCulture = System.Globalization.CultureInfo.GetCultureInfoByIetfLanguageTag("ko-KR");
                                Thread.CurrentThread.CurrentUICulture = System.Globalization.CultureInfo.GetCultureInfoByIetfLanguageTag("ko-KR");
                            }
                            else
                            {
                                GraphicsPath _path = new GraphicsPath();
                                SvgPathBuilder builder = new SvgPathBuilder();
                                SvgPathSegmentList list = builder.ConvertFromString(GetSvgPath(vecList[i].path, vecList[i].path.PathTypes[vecList[i].path.PointCount - 1] == 129)) as SvgPathSegmentList;

                                foreach (SvgPathSegment segment in list)
                                {
                                    segment.AddToPath(_path);
                                }

                                SvgColourServer brushColor = new SvgColourServer();
                                if (vecList[i].brush.ToString() == "System.Drawing.SolidBrush")
                                {
                                    brushColor = new SvgColourServer(((SolidBrush)vecList[i].brush).Color);
                                }

                                document.Children.Add(new SvgPath
                                {
                                    PathData = list,
                                    Fill = brushColor,
                                    Stroke = new SvgColourServer(vecList[i].pen.Color),
                                    StrokeWidth = vecList[i].pen.Width
                                });
                            }
                        }

                        var svgXml = document.GetXML();

                        File.WriteAllText(saveFileDialog.FileName, svgXml);
                        break;
                }

                fs.Close();
            }
        }

        private void tsb_load_Click(object sender, EventArgs e)
        {
            OpenFileDialog openFile = new OpenFileDialog();
            openFile.Filter = "Bitmap Image|*.bmp|JPeg Image|*.jpg|Png Image|*.png|TexMaster|*.txm|Scalable Vector Graphics|*.svg";
            openFile.ShowDialog();
            if (openFile.FileName == "") return;
            Console.WriteLine(File.ReadAllText(openFile.FileName));
            vecList.Clear();

            string ext = openFile.SafeFileName.Split('.')[1];

            if (ext == "svg")
            {
                XmlTextReader reader = null;

                try
                {
                    Thread.CurrentThread.CurrentCulture = System.Globalization.CultureInfo.GetCultureInfoByIetfLanguageTag("en-US");
                    Thread.CurrentThread.CurrentUICulture = System.Globalization.CultureInfo.GetCultureInfoByIetfLanguageTag("en-US");

                    // Load the reader with the data file and ignore all white space nodes.
                    reader = new XmlTextReader(openFile.FileName);
                    reader.WhitespaceHandling = WhitespaceHandling.None;

                    //# text
                    int textLineCnt = 0;
                    PointF startPoint = new PointF();   //# Text가 뿌려질 첫번째 점의 위치
                    string fontFamily = "굴림";
                    float fontSize = 20F;
                    GraphicsPath strPath = new GraphicsPath();
                    string strText = "";
                    // Parse the file and display each of the nodes.
                    while (reader.Read())
                    {
                        if (reader.NodeType == XmlNodeType.Element)
                        {
                            VectorData newVecData = new VectorData();
                            GraphicsPath path = new GraphicsPath();
                            Color fill = Color.Transparent;
                            Color stroke = Color.Transparent;
                            float strokeWidth = 1.0F;

                            switch (reader.Name)
                            {
                                case "svg":
                                    int canvasWidth = Convert.ToInt32(reader.GetAttribute("width").Replace("px", ""));
                                    int canvasHeight = Convert.ToInt32(reader.GetAttribute("height").Replace("px", ""));

                                    pictureBox1.Size = new Size(canvasWidth, canvasHeight);
                                    continue;
                                case "line":
                                    float x1 = Convert.ToSingle(reader.GetAttribute("x1"));
                                    float y1 = Convert.ToSingle(reader.GetAttribute("y1"));
                                    float x2 = Convert.ToSingle(reader.GetAttribute("x2"));
                                    float y2 = Convert.ToSingle(reader.GetAttribute("y2"));

                                    path.AddLine(new PointF(x1, y1), new PointF(x2, y2));
                                    break;
                                case "path":
                                    path = GetGraphicsPath(reader.GetAttribute("d"));
                                    break;
                                case "rect":
                                    float x = Convert.ToSingle(reader.GetAttribute("x"));
                                    float y = Convert.ToSingle(reader.GetAttribute("y"));
                                    float width = Convert.ToSingle(reader.GetAttribute("width"));
                                    float height = Convert.ToSingle(reader.GetAttribute("height"));

                                    path.AddRectangle(new RectangleF(x, y, width, height));
                                    break;
                                case "ellipse":
                                    float cx = Convert.ToSingle(reader.GetAttribute("cx"));
                                    float cy = Convert.ToSingle(reader.GetAttribute("cy"));
                                    float rx = Convert.ToSingle(reader.GetAttribute("rx"));
                                    float ry = Convert.ToSingle(reader.GetAttribute("ry"));

                                    path.AddEllipse(new RectangleF(cx - rx, cy - ry, rx * 2, ry * 2));
                                    break;
                                case "polyline":
                                case "polygon":
                                    List<PointF> pathPoints = new List<PointF>();
                                    string strPoints = reader.GetAttribute("points").Replace("\r\n", "");
                                    strPoints = strPoints.Replace("\t", "");
                                    string[] strPair = strPoints.Split(new char[1] { ' ' }, StringSplitOptions.RemoveEmptyEntries);
                                    for (int i = 0; i < strPair.Length; i++)
                                    {
                                        strPair[i] = strPair[i].Trim();
                                        string[] strPoint = strPair[i].Split(new char[1] { ',' }, StringSplitOptions.RemoveEmptyEntries);
                                        pathPoints.Add(new PointF(Convert.ToSingle(strPoint[0]), Convert.ToSingle(strPoint[1])));
                                    }

                                    for (int i = 0; i < pathPoints.Count - 1; i++)
                                    {
                                        path.AddLine(pathPoints[i], pathPoints[i + 1]);
                                    }

                                    if (reader.Name == "polygon")
                                    {
                                        path.CloseFigure();
                                    }
                                    break;
                                default:
                                    if (reader.Name == "text")
                                    {
                                        string transform = reader.GetAttribute("transform").Replace("matrix(", "");
                                        transform = transform.Replace(")", "");
                                        string[] strSplit = transform.Split(new char[1] { ' ' });

                                        textLineCnt = 0;
                                        startPoint = new PointF(Convert.ToSingle(strSplit[4]), Convert.ToSingle(strSplit[5]));
                                        strPath = new GraphicsPath();
                                        strText = "";
                                        fontFamily = "굴림";

                                        if (float.TryParse(reader.GetAttribute("font-size"), out float size))
                                        {
                                            fontSize = size;
                                        }

                                        foreach (FontFamily family in FontFamily.Families)
                                        {
                                            if (reader.GetAttribute("font-family") != null)
                                            {
                                                if (family.Name == reader.GetAttribute("font-family").Replace("\'", ""))
                                                {
                                                    fontFamily = reader.GetAttribute("font-family").Replace("\'", "");
                                                }
                                            }
                                        }
                                    }
                                    else if (reader.Name == "tspan")
                                    {
                                        fontFamily = "굴림";

                                        if (float.TryParse(reader.GetAttribute("font-size"), out float size))
                                        {
                                            fontSize = size;
                                        }

                                        foreach (FontFamily family in FontFamily.Families)
                                        {
                                            if (family.Name == reader.GetAttribute("font-family").Replace("\'", ""))
                                            {
                                                fontFamily = reader.GetAttribute("font-family").Replace("\'", "");
                                            }
                                        }
                                    }
                                    continue;
                            }

                            //# Svg created by Texmaster
                            if (reader.GetAttribute("style") != null)
                            {
                                string[] str = reader.GetAttribute("style").Split(new char[2] { ':', ';' }, StringSplitOptions.RemoveEmptyEntries);
                                if (str[0] == "fill")
                                {
                                    if (str[1] != "none")
                                    {
                                        if (str[1][0] == '#')
                                        {
                                            fill = ColorTranslator.FromHtml(str[1]);
                                            /*
                                            if (reader.GetAttribute("fill-opacity") != null)
                                                fill = Color.FromArgb((int)(255 * Convert.ToSingle(reader.GetAttribute("fill-opacity"))),
                                                    fill.R, fill.G, fill.B);
                                            */
                                        }
                                        else
                                        {
                                            fill = Color.FromName(str[1]);
                                        }
                                    }
                                }

                                if (str[2] == "stroke")
                                {
                                    if (str[3] != "none")
                                    {
                                        if (str[3][0] == '#')
                                        {
                                            stroke = ColorTranslator.FromHtml(str[3]);
                                            /*
                                            if (reader.GetAttribute("fill-opacity") != null)
                                                stroke = Color.FromArgb((int)(255 * Convert.ToSingle(reader.GetAttribute("fill-opacity"))),
                                                    stroke.R, stroke.G, stroke.B);
                                            */
                                        }
                                        else
                                        {
                                            stroke = Color.FromName(str[3]);
                                        }
                                    }
                                }
                            }
                            //# Svg created by Illustrator
                            else
                            {
                                if (reader.GetAttribute("fill") != "none")
                                {
                                    if (reader.GetAttribute("fill") != null)
                                        fill = ColorTranslator.FromHtml(reader.GetAttribute("fill"));
                                    else
                                        fill = Color.Black;

                                    if (reader.GetAttribute("opacity") != null)
                                        fill = Color.FromArgb((int)(255 * Convert.ToSingle(reader.GetAttribute("opacity"))), fill.R, fill.G, fill.B);
                                }

                                if (reader.GetAttribute("stroke") != "none")
                                {
                                    if (reader.GetAttribute("stroke") != null)
                                    {
                                        stroke = ColorTranslator.FromHtml(reader.GetAttribute("stroke"));
                                    }

                                    if (reader.GetAttribute("opacity") != null)
                                        stroke = Color.FromArgb((int)(255 * Convert.ToSingle(reader.GetAttribute("opacity"))), stroke.R, stroke.G, stroke.B);
                                }

                            }

                            if (float.TryParse(reader.GetAttribute("stroke-width"), out float result))
                            {
                                strokeWidth = result;
                            }

                            newVecData.isString = false;
                            newVecData.pen = new Pen(stroke, strokeWidth);
                            newVecData.brush = new SolidBrush(fill);
                            newVecData.path = path;
                            vecList.Add(newVecData);
                            SetPathTypes(newVecData.path);
                            pathTypesList.Add(pathTypes);
                            SetVectorList(newVecData);

                        }
                        else if (reader.NodeType == XmlNodeType.EndElement)
                        {
                            switch (reader.Name)
                            {
                                case "text":
                                    VectorData newVecData = new VectorData();
                                    newVecData.isString = true;
                                    newVecData.isPathText = false;
                                    newVecData.pen = new Pen(Color.Transparent, 1.0F);
                                    newVecData.brush = new SolidBrush(Color.Black);
                                    newVecData.path = strPath;
                                    newVecData.font = new Font(fontFamily, fontSize);
                                    if (strText[strText.Length - 1] == '\n')
                                    {
                                        strText = strText.Substring(0, strText.Length - 1);
                                    }
                                    newVecData.str = strText;
                                    vecList.Add(newVecData);
                                    SetPathTypes(newVecData.path);
                                    pathTypesList.Add(pathTypes);
                                    SetVectorList(newVecData);
                                    break;
                                case "tspan":
                                    strText += "\n";
                                    break;
                                default:
                                    break;
                            }
                        }
                        else if (reader.NodeType == XmlNodeType.Text)
                        {
                            GraphicsPath linePath = new GraphicsPath();
                            Graphics gr = pictureBox1.CreateGraphics();
                            strText += reader.Value;
                            textLineCnt++;
                            float charWidth = GetCharacterWidths(gr, reader.Value, new Font(fontFamily, fontSize)).ToArray().Sum() + 5.0F;
                            float charHeight = gr.MeasureString(reader.Value, new Font(fontFamily, fontSize)).Height;
                            PointF linePoint = new PointF(startPoint.X, (int)(startPoint.Y + charHeight * textLineCnt));
                            linePath.AddLine(linePoint, new PointF(linePoint.X + charWidth, linePoint.Y));
                            linePath.CloseFigure();
                            strPath.AddPath(linePath, false);
                        }
                    }

                }
                finally
                {
                    if (reader != null)
                        reader.Close();

                    Thread.CurrentThread.CurrentCulture = System.Globalization.CultureInfo.GetCultureInfoByIetfLanguageTag("ko-KR");
                    Thread.CurrentThread.CurrentUICulture = System.Globalization.CultureInfo.GetCultureInfoByIetfLanguageTag("ko-KR");
                }

            }
            else if (ext == "txm")
            {
                List<FileData> totalData = new List<FileData>();

                FileStream fs = new FileStream(openFile.FileName, FileMode.Open);
                BinaryFormatter formatter = new BinaryFormatter();

                vecList = new List<VectorData>();

                try
                {
                    totalData = (List<FileData>)formatter.Deserialize(fs);

                    for (int i = 0; i < totalData.Count; i++)
                    {
                        FileData eachData = totalData[i];
                        VectorData data = new VectorData();

                        data.path = GetGraphicsPath(eachData.xmlPath);

                        data.isString = eachData.isString;

                        switch (eachData.brushType)
                        {
                            case 0: //# solid
                                data.brush = new SolidBrush(Color.FromArgb(eachData.color));
                                break;
                            case 1: //# gradient
                                RectangleF bound = data.path.GetBounds();
                                //# todo - rotate
                                data.brush = new LinearGradientBrush(bound, Color.FromArgb(eachData.sColor), Color.FromArgb(eachData.eColor), 0F);
                                break;
                            case 2: //# texture
                                data.brush = new TextureBrush(eachData.pattern);
                                break;
                        }

                        data.pen = new Pen(Color.FromArgb(eachData.pColor), eachData.penWidth);
                        data.pen.DashStyle = eachData.dash;
                        data.pen.StartCap = eachData.sCap;
                        data.pen.EndCap = eachData.eCap;
                        data.pen.LineJoin = eachData.lCap;

                        if (data.isString)
                        {
                            data.str = eachData.str;
                            data.font = new Font(eachData.fontName, eachData.fontSize, eachData.fontStyle);
                            data.isPathText = eachData.isPathText;
                        }

                        vecList.Add(data);
                        SetPathTypes(data.path);
                        pathTypesList.Add(pathTypes);
                        SetVectorList(data);
                    }

                }
                catch
                {
                    throw;
                }
                finally
                {
                    fs.Close();
                }
            }

            drawing = true;
            pictureBox1.Refresh();
            drawing = false;

        }

        /// <summary>
        /// GraphicsPath를 Svg의 String Path 형식으로 변경
        /// </summary>
        /// <param name="savePath"></param>
        /// <param name="close"></param>
        /// <returns></returns>
        private string GetSvgPath(GraphicsPath savePath, bool close)
        {
            string svg = "";
            if (savePath.PathTypes.Length == 0)
            {
                MessageBox.Show("Do not Exist Vector Data!");
            }

            int bezierCnt = 0;
            for (int i = 0; i < savePath.PathTypes.Length; i++)
            {
                switch (savePath.PathTypes[i])
                {
                    case 0:
                        svg = svg + "M " + savePath.PathPoints[i].X.ToString();
                        svg = svg + "," + savePath.PathPoints[i].Y.ToString();
                        svg = svg + " ";
                        break;
                    case 1:
                        svg = svg + "L " + savePath.PathPoints[i].X.ToString();
                        svg = svg + "," + savePath.PathPoints[i].Y.ToString();
                        svg = svg + " ";
                        break;
                    case 3:
                        if (bezierCnt % 3 == 0)
                        {
                            svg = svg + "C " + savePath.PathPoints[i].X.ToString();
                            svg = svg + "," + savePath.PathPoints[i].Y.ToString();
                            svg = svg + " ";
                        }
                        else
                        {
                            svg = svg + savePath.PathPoints[i].X.ToString();
                            svg = svg + "," + savePath.PathPoints[i].Y.ToString();
                            svg = svg + " ";
                        }
                        bezierCnt++;
                        break;
                    case 129:
                        if (savePath.PathTypes[i - 1] == 1)
                        {
                            svg = svg + "L " + savePath.PathPoints[i].X.ToString();
                            svg = svg + "," + savePath.PathPoints[i].Y.ToString();
                        }
                        else if (savePath.PathTypes[i - 1] == 3)
                        {
                            svg = svg + savePath.PathPoints[i].X.ToString();
                            svg = svg + savePath.PathPoints[i].Y.ToString();
                            bezierCnt++;
                        }
                        if (close)
                        {
                            svg = svg + " Z";
                        }
                        break;
                    case 131:
                    case 161:
                    case 163:
                        svg = svg + savePath.PathPoints[i].X.ToString();
                        svg = svg + "," + savePath.PathPoints[i].Y.ToString();
                        if (close)
                        {
                            svg = svg + "Z";
                        }
                        if (savePath.PathTypes[i - 1] == 3) bezierCnt++;
                        break;
                    default:
                        break;
                }
            }
            return svg;

        }

        /// <summary>
        /// Svg의 String Path 형식을 GraphicsPath로 변경
        /// </summary>
        /// <param name="svgString"></param>
        /// <returns></returns>
        private GraphicsPath GetGraphicsPath(string svgString)
        {
            GraphicsPath _path = new GraphicsPath();
            SvgPathBuilder builder = new SvgPathBuilder();
            SvgPathSegmentList list = builder.ConvertFromString(svgString) as SvgPathSegmentList;

            foreach (SvgPathSegment segment in list)
            {
                segment.AddToPath(_path);
            }

            return _path;
        }

        private void tsd_arrange_Click(object sender, EventArgs e)
        {
            cms_arrange.Show(new Point(MousePosition.X - cms_arrange.Width, MousePosition.Y));
        }

        //=======================================================================
        //#  PathFinder (Foundation)
        //=======================================================================
        /// <summary>
        /// 선택된 Data들의 ClipType에 따른 결합된 path 적용
        /// </summary>
        /// <param name="type"></param>
        private void PathFinder(ClipType type)
        {
            Clipper c = new Clipper();
            PolyTree polyTree = new PolyTree();

            for (int i = 0; i < selPathIndex.Count; i++)
            {
                GraphicsPath path = new GraphicsPath();
                if (!isMirror)
                {
                    path = (GraphicsPath)vecList[selPathIndex[i]].path.Clone();
                }
                else
                {
                    path = (GraphicsPath)oriList[selPathIndex[i]].path.Clone();
                }
                Matrix mat = new Matrix();
                path.Flatten(mat, 0.01F);
                Polygon poly1 = new Polygon();
                for (int j = 0; j < path.PointCount; j++)
                {
                    poly1.Add(new IntPoint((long)path.PathPoints[j].X, (long)path.PathPoints[j].Y));
                }

                if (i == 0)
                {
                    c.AddPolygon(poly1, PolyType.ptSubject);
                }
                else
                {
                    c.AddPolygon(poly1, PolyType.ptClip);
                }

            }

            c.Execute(type, polyTree, PolyFillType.pftNonZero, PolyFillType.pftNonZero);

            GraphicsPath result = new GraphicsPath();
            for (int i = 0; i < polyTree.ChildCount; i++)
            {
                GraphicsPath temp = new GraphicsPath();
                PolyNode nodeList = polyTree.Childs[i];
                Point start = new Point();
                Point end = new Point();
                for (int j = 0; j < nodeList.Childs.Count; j++)
                {
                    PolyNode node = nodeList.Childs[j];
                    if (node.IsHole)
                    {
                        for (int k = 0; k < node.Contour.Count; k++)
                        {
                            start = new Point((int)node.Contour[k].X, (int)node.Contour[k].Y);
                            if (k == node.Contour.Count - 1)
                            {
                                end = new Point((int)node.Contour[0].X, (int)node.Contour[0].Y);
                            }
                            else
                            {
                                end = new Point((int)node.Contour[k + 1].X, (int)node.Contour[k + 1].Y);
                            }
                            temp.AddLine(start, end);
                        }
                        result.AddPath(temp, false);
                    }
                }
                temp = new GraphicsPath();
                for (int j = 0; j < nodeList.Contour.Count; j++)
                {
                    start = new Point((int)nodeList.Contour[j].X, (int)nodeList.Contour[j].Y);
                    if (j == nodeList.Contour.Count - 1)
                    {
                        end = new Point((int)nodeList.Contour[0].X, (int)nodeList.Contour[0].Y);
                    }
                    else
                    {
                        end = new Point((int)nodeList.Contour[j + 1].X, (int)nodeList.Contour[j + 1].Y);
                    }
                    temp.AddLine(start, end);
                }
                result.AddPath(temp, false);
            }

            if (!isMirror)
                currentData = GetModifyData(ref vecList, ModifyMode.PATH, selPathIndex[0], result);
            else
                currentData = GetModifyData(ref oriList, ModifyMode.PATH, selPathIndex[0], result);

            selPathIndex.Sort((x1, x2) => x2.CompareTo(x1));
            for (int i = 0; i < selPathIndex.Count; i++)
            {
                DeleteData(selPathIndex[i]);
            }

            selPathIndex.Sort((x1, x2) => x1.CompareTo(x2));
            CreateData(selPathIndex[0]);

            Commit();
            ClearSelList();
            RestoreSelMode();
        }

        private void btn_union_Click(object sender, EventArgs e)
        {
            PathFinder(ClipType.ctUnion);
        }

        private void btn_overlap_Click(object sender, EventArgs e)
        {
            PathFinder(ClipType.ctIntersection);
        }

        private void btn_different_Click(object sender, EventArgs e)
        {
            PathFinder(ClipType.ctDifference);
        }

        private void btn_xor_Click(object sender, EventArgs e)
        {
            PathFinder(ClipType.ctXor);
        }

        //=======================================================================
        //#  크기 변경
        //=======================================================================
        private void nud_size_KeyDown(object sender, KeyEventArgs e)
        {
            if (e.KeyCode == Keys.Enter)
            {
                Console.WriteLine(selRects.X + ", " + selRects.Y + ", " + selRects.Width + ", " + selRects.Height);
                ChangeFigureSize(sender);
                drawing = true;
                pictureBox1.Refresh();
                drawing = false;
            }
        }

        private void nud_size_Leave(object sender, EventArgs e)
        {
            ChangeFigureSize(sender);
            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
        }

        private void nud_size_ValueChanged(object sender, EventArgs e)
        {
            ChangeFigureSize(sender);
            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
        }

        /// <summary>
        /// 선택된 객체의 크기를 변경
        /// </summary>
        /// <param name="sender"></param>
        private void ChangeFigureSize(object sender)
        {
            if (selPathIndex.Count == 0) return;

            NumericUpDown num = (NumericUpDown)sender;

            Matrix mat = new Matrix();
            System.Windows.Media.Matrix zoomMat = new System.Windows.Media.Matrix();

            List<VectorData> selList;
            if (!isMirror) selList = vecList;
            else selList = oriList;

            switch (num.Name)
            {
                case "nud_x":
                    MoveData(Convert.ToInt32((int)nud_x.Value - selRects.X), 0, true);
                    break;
                case "nud_y":
                    MoveData(0, Convert.ToInt32((int)nud_y.Value - selRects.Y), true);
                    break;
                case "nud_w":
                    if (tsb_constrain.Checked)
                    {
                        zoomMat.ScaleAt((double)nud_w.Value / selRects.Width, (double)nud_w.Value / selRects.Width, selRects.X + (selRects.Width / 2), selRects.Y + (selRects.Height / 2));
                    }
                    else
                    {
                        zoomMat.ScaleAt((double)nud_w.Value / selRects.Width, 1.0f, selRects.X + (selRects.Width / 2), selRects.Y + (selRects.Height / 2));
                    }
                    mat = new Matrix((float)zoomMat.M11, (float)zoomMat.M12, (float)zoomMat.M21, (float)zoomMat.M22, (float)zoomMat.OffsetX, (float)zoomMat.OffsetY);
                    for (int i = 0; i < selPathIndex.Count; i++)
                    {
                        selList[selPathIndex[i]].path.Transform(mat);
                        selTempPath[i].Transform(mat);
                        ModifyData(ModifyMode.PATH, selPathIndex[i], (GraphicsPath)selList[selPathIndex[i]].path.Clone());
                        if (isMirror)
                        {
                            selRefPath[i] = (GraphicsPath)refList[selPathIndex[i]].path.Clone();
                        }
                    }
                    break;
                case "nud_h":
                    if (tsb_constrain.Checked)
                    {
                        zoomMat.ScaleAt((double)nud_h.Value / selRects.Height, (double)nud_h.Value / selRects.Height, selRects.X + (selRects.Width / 2), selRects.Y + (selRects.Height / 2));
                    }
                    else
                    {
                        zoomMat.ScaleAt(1.0f, (double)nud_h.Value / selRects.Height, selRects.X + (selRects.Width / 2), selRects.Y + (selRects.Height / 2));
                    }
                    mat = new Matrix((float)zoomMat.M11, (float)zoomMat.M12, (float)zoomMat.M21, (float)zoomMat.M22, (float)zoomMat.OffsetX, (float)zoomMat.OffsetY);
                    for (int i = 0; i < selPathIndex.Count; i++)
                    {
                        selList[selPathIndex[i]].path.Transform(mat);
                        selTempPath[i].Transform(mat);
                        ModifyData(ModifyMode.PATH, selPathIndex[i], (GraphicsPath)selList[selPathIndex[i]].path.Clone());
                        if (isMirror)
                        {
                            selRefPath[i] = (GraphicsPath)refList[selPathIndex[i]].path.Clone();
                        }
                    }
                    break;
            }
        }

        //=======================================================================
        //# flag 변경
        //=======================================================================
        private void tsb_lock_Click(object sender, EventArgs e)
        {
            if (tsb_lock.Checked)
            {
                tsb_lock.Checked = false;
                //moveBaseLine = true;
            }
            else
            {
                tsb_lock.Checked = true;
                moveBaseLine = false;
            }
        }

        private void tsb_fill_Click(object sender, EventArgs e)
        {
            if (tsb_fill.Checked)
            {
                tsb_fill.Checked = false;
                currentData.brush = new SolidBrush(Color.Transparent);
            }
            else
            {
                tsb_fill.Checked = true;
                currentData.brush = new SolidBrush(currentData.pen.Color);
            }
        }

        private void tsb_antialias_Click(object sender, EventArgs e)
        {
            if (tsb_antialias.Checked)
            {
                tsb_antialias.Checked = false;
            }
            else
            {
                tsb_antialias.Checked = true;
            }

            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
        }

        private void tsb_constrain_Click(object sender, EventArgs e)
        {
            if (tsb_constrain.Checked)
            {
                tsb_constrain.Checked = false;
            }
            else
            {
                tsb_constrain.Checked = true;
            }
        }

        /// <summary>
        /// 선색 면색 변경
        /// </summary>
        /// <param name="sender"></param>
        /// <param name="e"></param>
        private void tsb_change_Click(object sender, EventArgs e)
        {
            List<VectorData> selList;
            if (!isMirror) selList = vecList;
            else selList = oriList;
            for (int i = 0; i < selPathIndex.Count; i++)
            {
                Pen newPen = (Pen)selList[selPathIndex[i]].pen.Clone();
                Brush newBrush = (Brush)selList[selPathIndex[i]].brush.Clone();
                Color newPenColor = ((SolidBrush)newBrush).Color;
                Color newBrushColor = ((Pen)newPen).Color;
                newPen.Color = newPenColor;
                ((SolidBrush)newBrush).Color = newBrushColor;

                ModifyData(ModifyMode.PEN, selPathIndex[i], newPen);
                ModifyData(ModifyMode.BRUSH, selPathIndex[i], newBrush);
            }

            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
        }

        /// <summary>
        /// picturBox refresh
        /// </summary>
        public void VectorCanvasRefresh()
        {
            drawing = true;
            pictureBox1.Refresh();
            drawing = false;
        }
    }

    /*
    An Algorithm for Automatically Fitting Digitized Curves
    by Philip J. Schneider
    from "Graphics Gems", Academic Press, 1990
    */
    public static class FitCurves
    {
        /*  Fit the Bezier curves */
        private const int MAXPOINTS = 10000;
        public static List<WinPoint> FitCurve(WinPoint[] d, double error)
        {
            WinVector tHat1, tHat2;    /*  Unit tangent vectors at endpoints */

            tHat1 = ComputeLeftTangent(d, 0);
            tHat2 = ComputeRightTangent(d, d.Length - 1);
            List<WinPoint> result = new List<WinPoint>();
            FitCubic(d, 0, d.Length - 1, tHat1, tHat2, error, result);
            return result;
        }

        private static void FitCubic(WinPoint[] d, int first, int last, WinVector tHat1, WinVector tHat2, double error, List<WinPoint> result)
        {
            WinPoint[] bezCurve;        /*  Control points of fitted Bezier curve*/
            double[] u;                 /*  Parameter values for point  */
            double[] uPrime;            /*  Improved parameter values */
            double maxError;            /*  Maximum fitting error    */
            int splitPoint;             /*  Point to split point set at  */
            int nPts;                   /*  Number of points in subset  */
            double iterationError;      /*  Error below which you try iterating  */
            int maxIterations = 4;      /*  Max times to try iterating  */
            WinVector tHatCenter;       /*  Unit tangent vector at splitPoint */
            int i;

            iterationError = error * error;
            nPts = last - first + 1;

            /*  Use heuristic if region only has two points in it */
            if (nPts == 2)
            {
                double dist = (d[first] - d[last]).Length / 3.0;

                bezCurve = new WinPoint[4];
                bezCurve[0] = d[first];
                bezCurve[3] = d[last];
                bezCurve[1] = (tHat1 * dist) + bezCurve[0];
                bezCurve[2] = (tHat2 * dist) + bezCurve[3];

                result.Add(bezCurve[1]);
                result.Add(bezCurve[2]);
                result.Add(bezCurve[3]);
                return;
            }

            /*  Parameterize points, and attempt to fit curve */
            u = ChordLengthParameterize(d, first, last);
            bezCurve = GenerateBezier(d, first, last, u, tHat1, tHat2);

            /*  Find max deviation of points to fitted curve */
            maxError = ComputeMaxError(d, first, last, bezCurve, u, out splitPoint);
            if (maxError < error)
            {
                result.Add(bezCurve[1]);
                result.Add(bezCurve[2]);
                result.Add(bezCurve[3]);
                return;
            }

            /*  If error not too large, try some reparameterization  */
            /*  and iteration */
            if (maxError < iterationError)
            {
                for (i = 0; i < maxIterations; i++)
                {
                    uPrime = Reparameterize(d, first, last, u, bezCurve);
                    bezCurve = GenerateBezier(d, first, last, uPrime, tHat1, tHat2);
                    maxError = ComputeMaxError(d, first, last, bezCurve, uPrime, out splitPoint);
                    if (maxError < error)
                    {
                        result.Add(bezCurve[1]);
                        result.Add(bezCurve[2]);
                        result.Add(bezCurve[3]);
                        return;
                    }
                    u = uPrime;
                }
            }

            /*  Fitting failed -- split at max error point and fit recursively */
            tHatCenter = ComputeCenterTangent(d, splitPoint);
            FitCubic(d, first, splitPoint, tHat1, tHatCenter, error, result);
            tHatCenter.Negate();
            FitCubic(d, splitPoint, last, tHatCenter, tHat2, error, result);
        }

        static WinPoint[] GenerateBezier(WinPoint[] d, int first, int last, double[] uPrime, WinVector tHat1, WinVector tHat2)
        {
            int i;
            WinVector[,] A = new WinVector[MAXPOINTS, 2];/*  Precomputed rhs for eqn    */

            int nPts;                                    /*  Number of pts in sub-curve */
            double[,] C = new double[2, 2];              /*  Matrix C     */
            double[] X = new double[2];                  /*  Matrix X         */
            double det_C0_C1, det_C0_X, det_X_C1;        /*  Determinants of matrices */
            double alpha_l, alpha_r;                     /*  Alpha values, left and right */

            WinVector tmp;                               /*  Utility variable     */
            WinPoint[] bezCurve = new WinPoint[4];       /*  RETURN bezier curve ctl pts  */
            nPts = last - first + 1;

            /* Compute the A's  */
            for (i = 0; i < nPts; i++)
            {
                WinVector v1, v2;
                v1 = tHat1;
                v2 = tHat2;
                v1 *= B1(uPrime[i]);
                v2 *= B2(uPrime[i]);
                A[i, 0] = v1;
                A[i, 1] = v2;
            }

            /* Create the C and X matrices  */
            C[0, 0] = 0.0;
            C[0, 1] = 0.0;
            C[1, 0] = 0.0;
            C[1, 1] = 0.0;
            X[0] = 0.0;
            X[1] = 0.0;

            for (i = 0; i < nPts; i++)
            {
                C[0, 0] += V2Dot(A[i, 0], A[i, 0]);
                C[0, 1] += V2Dot(A[i, 0], A[i, 1]);
                /*  C[1][0] += V2Dot(&A[i][0], &A[i][9]);*/
                C[1, 0] = C[0, 1];
                C[1, 1] += V2Dot(A[i, 1], A[i, 1]);

                tmp = ((WinVector)d[first + i] -
                       (
                       ((WinVector)d[first] * B0(uPrime[i])) +
                        (
                        ((WinVector)d[first] * B1(uPrime[i])) +
                         (
                         ((WinVector)d[last] * B2(uPrime[i])) +
                          ((WinVector)d[last] * B3(uPrime[i]))))));

                X[0] += V2Dot(A[i, 0], tmp);
                X[1] += V2Dot(A[i, 1], tmp);
            }

            /* Compute the determinants of C and X  */
            det_C0_C1 = C[0, 0] * C[1, 1] - C[1, 0] * C[0, 1];
            det_C0_X = C[0, 0] * X[1] - C[1, 0] * X[0];
            det_X_C1 = X[0] * C[1, 1] - X[1] * C[0, 1];

            /* Finally, derive alpha values */
            alpha_l = (det_C0_C1 == 0) ? 0.0 : det_X_C1 / det_C0_C1;
            alpha_r = (det_C0_C1 == 0) ? 0.0 : det_C0_X / det_C0_C1;

            /* If alpha negative, use the Wu/Barsky heuristic (see text) */
            /* (if alpha is 0, you get coincident control points that lead to
             * divide by zero in any subsequent NewtonRaphsonRootFind() call. */
            double segLength = (d[first] - d[last]).Length;
            double epsilon = 1.0e-6 * segLength;
            if (alpha_l < epsilon || alpha_r < epsilon)
            {
                /* fall back on standard (probably inaccurate) formula, and subdivide further if needed. */
                double dist = segLength / 3.0;
                bezCurve[0] = d[first];
                bezCurve[3] = d[last];
                bezCurve[1] = (tHat1 * dist) + bezCurve[0];
                bezCurve[2] = (tHat2 * dist) + bezCurve[3];
                return (bezCurve);
            }

            /*  First and last control points of the Bezier curve are */
            /*  positioned exactly at the first and last data points */
            /*  Control points 1 and 2 are positioned an alpha distance out */
            /*  on the tangent vectors, left and right, respectively */
            bezCurve[0] = d[first];
            bezCurve[3] = d[last];
            bezCurve[1] = (tHat1 * alpha_l) + bezCurve[0];
            bezCurve[2] = (tHat2 * alpha_r) + bezCurve[3];
            return (bezCurve);
        }

        /*
         *  Reparameterize:
         *  Given set of points and their parameterization, try to find a better parameterization.
        */
        static double[] Reparameterize(WinPoint[] d, int first, int last, double[] u, WinPoint[] bezCurve)
        {
            int nPts = last - first + 1;
            int i;
            double[] uPrime = new double[nPts];      /*  New parameter values    */

            for (i = first; i <= last; i++)
            {
                uPrime[i - first] = NewtonRaphsonRootFind(bezCurve, d[i], u[i - first]);
            }
            return uPrime;
        }



        /*
         *  NewtonRaphsonRootFind :
         *  Use Newton-Raphson iteration to find better root.
        */
        static double NewtonRaphsonRootFind(WinPoint[] Q, WinPoint P, double u)
        {
            double numerator, denominator;
            WinPoint[] Q1 = new WinPoint[3], Q2 = new WinPoint[2];   /*  Q' and Q''          */
            WinPoint Q_u, Q1_u, Q2_u;                                /*  u evaluated at Q, Q', & Q''  */
            double uPrime;                                           /*  Improved u          */
            int i;

            /* Compute Q(u) */
            Q_u = BezierII(3, Q, u);

            /* Generate control vertices for Q' */
            for (i = 0; i <= 2; i++)
            {
                Q1[i].X = (Q[i + 1].X - Q[i].X) * 3.0;
                Q1[i].Y = (Q[i + 1].Y - Q[i].Y) * 3.0;
            }

            /* Generate control vertices for Q'' */
            for (i = 0; i <= 1; i++)
            {
                Q2[i].X = (Q1[i + 1].X - Q1[i].X) * 2.0;
                Q2[i].Y = (Q1[i + 1].Y - Q1[i].Y) * 2.0;
            }

            /* Compute Q'(u) and Q''(u) */
            Q1_u = BezierII(2, Q1, u);
            Q2_u = BezierII(1, Q2, u);

            /* Compute f(u)/f'(u) */
            numerator = (Q_u.X - P.X) * (Q1_u.X) + (Q_u.Y - P.Y) * (Q1_u.Y);
            denominator = (Q1_u.X) * (Q1_u.X) + (Q1_u.Y) * (Q1_u.Y) + (Q_u.X - P.X) * (Q2_u.X) + (Q_u.Y - P.Y) * (Q2_u.Y);

            if (denominator == 0.0f) return u;

            /* u = u - f(u)/f'(u) */
            uPrime = u - (numerator / denominator);
            return (uPrime);
        }

        /*
         *  Bezier :
         *  Evaluate a Bezier curve at a particular parameter value
        */
        static WinPoint BezierII(int degree, WinPoint[] V, double t)
        {
            int i, j;
            WinPoint Q;                     /*  Point on curve at parameter t    */
            WinPoint[] Vtemp;               /*  Local copy of control points     */

            /* Copy array   */
            Vtemp = new WinPoint[degree + 1];
            for (i = 0; i <= degree; i++)
            {
                Vtemp[i] = V[i];
            }

            /* Triangle computation */
            for (i = 1; i <= degree; i++)
            {
                for (j = 0; j <= degree - i; j++)
                {
                    Vtemp[j].X = (1.0 - t) * Vtemp[j].X + t * Vtemp[j + 1].X;
                    Vtemp[j].Y = (1.0 - t) * Vtemp[j].Y + t * Vtemp[j + 1].Y;
                }
            }

            Q = Vtemp[0];
            return Q;
        }

        /*
         *  B0, B1, B2, B3 :
         *  Bezier multipliers
        */
        static double B0(double u)
        {
            double tmp = 1.0 - u;
            return (tmp * tmp * tmp);
        }


        static double B1(double u)
        {
            double tmp = 1.0 - u;
            return (3 * u * (tmp * tmp));
        }

        static double B2(double u)
        {
            double tmp = 1.0 - u;
            return (3 * u * u * tmp);
        }

        static double B3(double u)
        {
            return (u * u * u);
        }

        /*
         * ComputeLeftTangent, ComputeRightTangent, ComputeCenterTangent :
         * Approximate unit tangents at endpoints and "center" of digitized curve
        */
        static WinVector ComputeLeftTangent(WinPoint[] d, int end)
        {
            WinVector tHat1;
            tHat1 = d[end + 1] - d[end];
            tHat1.Normalize();
            return tHat1;
        }

        static WinVector ComputeRightTangent(WinPoint[] d, int end)
        {
            WinVector tHat2;
            tHat2 = d[end - 1] - d[end];
            tHat2.Normalize();
            return tHat2;
        }

        static WinVector ComputeCenterTangent(WinPoint[] d, int center)
        {
            WinVector V1, V2, tHatCenter = new WinVector();

            V1 = d[center - 1] - d[center];
            V2 = d[center] - d[center + 1];
            tHatCenter.X = (V1.X + V2.X) / 2.0;
            tHatCenter.Y = (V1.Y + V2.Y) / 2.0;
            tHatCenter.Normalize();
            return tHatCenter;
        }


        /*
         *  ChordLengthParameterize :
         *  Assign parameter values to digitized points 
         *  using relative distances between points.
        */
        static double[] ChordLengthParameterize(WinPoint[] d, int first, int last)
        {
            int i;
            double[] u = new double[last - first + 1];           /*  Parameterization        */

            u[0] = 0.0;
            for (i = first + 1; i <= last; i++)
            {
                u[i - first] = u[i - first - 1] + (d[i - 1] - d[i]).Length;
            }

            for (i = first + 1; i <= last; i++)
            {
                u[i - first] = u[i - first] / u[last - first];
            }

            return u;
        }

        /*
         *  ComputeMaxError :
         *  Find the maximum squared distance of digitized points
         *  to fitted curve.
        */
        static double ComputeMaxError(WinPoint[] d, int first, int last, WinPoint[] bezCurve, double[] u, out int splitPoint)
        {
            int i;
            double maxDist;        /*  Maximum error       */
            double dist;           /*  Current error       */
            WinPoint P;            /*  Point on curve      */
            WinVector v;           /*  Vector from point to curve  */

            splitPoint = (last - first + 1) / 2;
            maxDist = 0.0;
            for (i = first + 1; i < last; i++)
            {
                P = BezierII(3, bezCurve, u[i - first]);
                v = P - d[i];
                dist = v.LengthSquared;
                if (dist >= maxDist)
                {
                    maxDist = dist;
                    splitPoint = i;
                }
            }
            return maxDist;
        }

        private static double V2Dot(WinVector a, WinVector b)
        {
            return ((a.X * b.X) + (a.Y * b.Y));
        }
    }
}
