

/**
 *   @file  srio_status_decode.c
 *
 *   @brief
 *      Utilities to decode SRIO status register values.
 *
 */

/* Standard library and XDC Include Files. */
#include <stdio.h>
#include <xdc/runtime/System.h>

/* CSL SRIO Functional Layer */
#include <ti/csl/csl_srioAuxPhyLayer.h>

/* Benchmarking Include Files. */
#include "srio_config.h"

CSL_SrioHandle      hSrio_Direct = (CSL_SrioHandle)CSL_SRIO_CONFIG_REGS;

/**
 *  @b Description
 *  @n
 *      The function is used to display a register's name, address and value.
 *
 *  @retval
 *      Control code text string translation.
 */

void clearSrioStatusErrors (void)
{
	/* Clear Errors */
	CSL_SRIO_SetErrorDetectCSR (hSrio_Direct,0x0);
}

/**
 *  @b Description
 *  @n
 *      The function is used to decode the Logical/Transport Layer Error Detect CSR
 *
 *  @retval
 *      Success - 0
 */
int32_t displaySrioLTLEDCSRErrorStatus (void)
{
	Uint8				bitPos;
	Uint32				errorStatus, errorMask;
	char* indentText;

	indentText = "  ";

	CSL_SRIO_GetErrorDetectCSR (hSrio_Direct,&errorStatus);
	clearSrioStatusErrors ();
	printf ("Debug: %sSRIO status: 0x%08X.\n", indentText, errorStatus);
    for (bitPos = 0; bitPos < 32; bitPos++)
    {
		errorMask = 0x1 << bitPos;
		switch (errorStatus & errorMask)
		{
			case 0x80000000:
				System_printf ("Debug:   %sIO error response\n", indentText);
				break;
			case 0x40000000:
				System_printf ("Debug:   %sMessage error response\n", indentText);
				break;
			case 0x20000000:
				System_printf ("Debug:   %sGSM error response\n", indentText);
				break;
			case 0x10000000:
				System_printf ("Debug:   %sMessage Format Error\n", indentText);
				break;
			case 0x08000000:
				System_printf ("Debug:   %sIllegal transaction decode\n", indentText);
				break;
			case 0x04000000:
				System_printf ("Debug:   %sIllegal transaction target error\n", indentText);
				break;
			case 0x02000000:
				System_printf ("Debug:   %sMessage Request Time-out\n", indentText);
				break;
			case 0x01000000:
				System_printf ("Debug:   %sPacket Response Time-out\n", indentText);
				break;
			case 0x00800000:
				System_printf ("Debug:   %sUnsolicited Response\n", indentText);
				break;
			case 0x00400000:
				System_printf ("Debug:   %sUnsupported Transaction\n", indentText);
				break;
			case 0x00000080:
				System_printf ("Debug:   %sRX CPPI Security Violation\n", indentText);
				break;
			case 0x00000040:
				System_printf ("Debug:   %sRX I/O DMA Access Error\n", indentText);
				break;
		}
    }
	return 0;
}

/**
 *  @b Description
 *  @n
 *      The function is used to display the Port Response Timeout CSR value
 *
 *  @retval
 *      Success - 0
 */
int32_t displayPortResponseTimeout (void)
{
	Uint32	timeout;
	CSL_SRIO_GetPortResponseTimeoutCSR (hSrio_Direct,&timeout);
	System_printf ("Debug:   Port response timeout: 0x%x\n", timeout);
	return 0;
}

/**
 *  @b Description
 *  @n
 *      The function is used to display the Port General CSR decodes
 *
 *  @retval
 *      Success - 0
 */
int32_t displayPortGeneralCSR (void)
{
    Uint8 hostDev;
    Uint8 masterEn;
    Uint8 disc;
	CSL_SRIO_GetPortGeneralCSR (hSrio_Direct,&hostDev,&masterEn,&disc);
	if (hostDev == 1)
		System_printf ("Debug:   SRIO is a Host Device.\n");
	else
		System_printf ("Debug:   SRIO is an Agent or Slave device.\n");

	if (masterEn == 1)
		System_printf ("Debug:   SRIO processing element can issue requests.\n");
	else
		System_printf ("Debug:   SRIO processing element cannot issue requests.\n");

	if (disc == 1)
		System_printf ("Debug:   SRIO device has been discovered by another processing element.\n");
	else
		System_printf ("Debug:   SRIO device has not been previously discovered.\n");

	return 0;

}


