/*
 *  Copyright 2007 by Texas Instruments Incorporated.
 *  All rights reserved. Property of Texas Instruments Incorporated.
 *  Restricted rights to use, duplicate or disclose this code are
 *  granted through contract.
 *
 *  @(#) TCP/IP_Network_Developers_Kit 1.93.00.09 08-16-2007 (ndk-c09)
 */
//--------------------------------------------------------------------------
// IP Stack Server Demonstration Program
//--------------------------------------------------------------------------
// newservers.c
//
// This module demonstrates the use of the server daemon added in NDK 1.7.
//
// It provides all the functionality of:
//     echosrv() : echosrv.c
//     datasrv() : datasrv.c
//     nullsrv() : nullsrv.c
//     oobsrv()  : oobsrv.c
//
// The original source files are still provided to illustrate a traditional
// server. This file contains only the daemon service task functions
// required when using the server daemon.
//
// Author: Michael A. Denio
// Copyright 2003 by Texas Instruments Inc.
//-------------------------------------------------------------------------
#include <csl_edma2.h>
#include <csl_intcAux.h>
#include "fa_server.h"
#include "com.h"
#include "command.h"
#include "flash.h"
#include "environment.h"
#include "port.h"
#include "machineControl.h"
#include "timer.h"
#include "master.h"
#include "tablet_counter.h"
#include "system_setup.h"
#include "capture.h"
#include "tablet_control.h"
#include "HCB_System_Process.h"
#include "rs485master.h"
#include "tablet_control.h"

#include "Environment.h"

/** 
 *   @file  echosrv.c
 *
 *   @brief   
 *      This program implements a TCP and UDP echo server, which echos back any
 *      input it receives.
 *
 *  \par
 *  NOTE:
 *      (C) Copyright 2008, Texas Instruments, Inc.
 *
 *  \par
 */

//configure TCP data buffer 

#define ECHO_BUFFER_MAX_SIZE		4 * 1024 * 1024
#pragma DATA_SECTION(EchoBuffer, ".sdram");
#pragma DATA_ALIGN(EchoBuffer, 1024);
unsigned char EchoBuffer[ECHO_BUFFER_MAX_SIZE];

#define GLOABL_TCP_BUFFER_SIZE		256 * 1024
#pragma DATA_ALIGN(GlobalTCPRxDataBuffer, 8);
#pragma DATA_ALIGN(GlobalTCPTxDataBuffer, 8);
unsigned char GlobalTCPRxDataBuffer[GLOABL_TCP_BUFFER_SIZE];
unsigned char GlobalTCPTxDataBuffer[GLOABL_TCP_BUFFER_SIZE];

unsigned char GlobalSysResultBuffer[1024];
unsigned char GlobalAppResultBuffer[1024];

#pragma DATA_SECTION(CPBDataBuffer, ".sdram");
#pragma DATA_ALIGN(CPBDataBuffer, 8);
unsigned char CPBDataBuffer[640*480*5];

#pragma DATA_ALIGN (ResponseDataAddr, 8)
unsigned int ResponseDataAddr[1];
extern TBlackoutInfo BlackoutInfo;
unsigned char GlobalTCPBuffer[1024];
//configure TCP socket
struct   sockaddr_in sin1;
SOCKET   stcp = INVALID_SOCKET;
SOCKET   sudp = INVALID_SOCKET;
SOCKET   sresult = INVALID_SOCKET;
SOCKET   stcpactive = INVALID_SOCKET;
SOCKET   stcpbusy;
fd_set ibits, obits, xbits;
int snd_count = 0;
int rcv_count = 0;
int      size,result;
HANDLE   hBuffer;
char     *pBuf;
char     *hBuf;
struct   timeval timeout;    
TTCPDataHeader TCPRXHeader, TCPTXHeader;
int remain_small_size;
int remain_large_size;
int remain_small_offset;
int remain_large_offset;
int laser_shutter_control;
extern int PortTestOn;
TCPControlInfo TcpPacketInfo;
TCPReceiveData TcpReceiveDataInfo;
extern PortStatusInfo portInfo;
extern int ThreeDSensorStatus[2];

short int read_THREED_CAMERA_SENSOR;
int checkDistributorSig;

void FA_TCPServer()
{
	fdOpenSession( TaskSelf() );
	
	TcpPacketInfo.Status == SERVER_STATUS_IDLE;
	TcpPacketInfo.rx_control == SERVER_STATUS_RECEIVE;
	memset(&TcpPacketInfo, 0 ,sizeof(TCPControlInfo));
	memset(&TCPRXHeader, 0 ,sizeof(TTCPDataHeader));
	memset(&TCPTXHeader, 0 ,sizeof(TTCPDataHeader));

	// Create the main TCP listen socket
	sresult = FA_connect();	
	if(sresult == INVALID_SOCKET)
		FA_disconnect();

	// Configure our timeout to be 15 seconds
	timeout.tv_sec  = 2;
	timeout.tv_usec = 0;

	TcpPacketInfo.Status = SERVER_STATUS_RECEIVE;
	
	// Run until task is destroyed by the system
	for(;;)
	{
		FD_ZERO(&ibits);
		FD_ZERO(&obits);
		FD_ZERO(&xbits);

		FD_SET(stcp, &ibits);
		TcpPacketInfo.Status = SERVER_STATUS_RECEIVE;

		if(TcpPacketInfo.Status == SERVER_STATUS_RECEIVE || TcpPacketInfo.Status == SERVER_STATUS_IDLE)
		{
			ReceiveTCPPacket(pBuf);
		}
		
	    if(TcpPacketInfo.Status == SERVER_STATUS_PROCESSING)
		{
			Tcp_Processing(TcpPacketInfo);
		}
		if(TcpPacketInfo.Status == SERVER_STATUS_SEND)
		{
			SendTCPPacket(TcpPacketInfo);
		}

		TcpPacketInfo.Status = SERVER_STATUS_IDLE;

	}
}

SOCKET FA_connect()
{
	struct   timeval timeout;    
	int I = 1;
	timeout.tv_sec  = 2;
	timeout.tv_usec = 0;

    stcp = socket(AF_INET, SOCK_STREAMNC, 0);
	setsockopt(stcp, SOL_SOCKET, SO_SNDTIMEO, &timeout, sizeof(timeout));
	setsockopt(stcp, SOL_SOCKET, SO_RCVTIMEO, &timeout, sizeof(timeout));

//	setsockopt(stcp, IPPROTO_TCP, TCP_NOPUSH, &I, 4 );
    if( stcp == INVALID_SOCKET )
        return INVALID_SOCKET;

    // Set Port = 7, leaving IP address = Any
    bzero( &sin1, sizeof(struct sockaddr_in) );
    sin1.sin_family = AF_INET;
    sin1.sin_len    = sizeof( sin1 );
    sin1.sin_port   = htons(1002);

    // Bind socket
    if ( bind( stcp, (PSA) &sin1, sizeof(sin1) ) < 0 )
        return INVALID_SOCKET;

    // Start listening
    if ( listen( stcp, 1) < 0 )
        return INVALID_SOCKET;


	return stcp;
}


void FA_disconnect()
{
    if( stcp != INVALID_SOCKET )
	{	
        fdClose( stcp );
		stcp = INVALID_SOCKET;
	}
	TaskBlock( TaskSelf() );
}

void ReceiveTCPPacket(unsigned char *buf)
{
	unsigned int dataBufferMaxSize;
	
	if( stcp == INVALID_SOCKET ) 
		FA_disconnect();

    // Wait for socket activity
    if( stcpactive == INVALID_SOCKET )
    {
        // Wait without timeout
        result = fdSelect( 4, &ibits, &obits, &xbits, 0 );
    }
    else
    {
        // Wait for set timeout - abort active connection on no activity
        FD_SET(stcpactive, &ibits);
        result = fdSelect( 4, &ibits, &obits, &xbits, 0 );
        if( result <= 0 )
        {
            fdClose( stcpactive );
            stcpactive = INVALID_SOCKET;
        }
    }

    if( result < 0 ){
	    FA_disconnect();
	}	
    // Check for a new TCP connection
    if( FD_ISSET(stcp, &ibits) )
    {
        // We have a new connection. Assign it so sbusy at
        // first...
//	  fdClose( stcpactive );
	  stcpactive = INVALID_SOCKET;
        size = sizeof( sin1 );
        stcpbusy = accept( stcp, (PSA)&sin1, &size );

        // If the active socket is free use it, else print out
        // a busy message
        if( stcpactive == INVALID_SOCKET )
            stcpactive = stcpbusy;
        else
            fdClose( stcpbusy );
    }

    // Check for new data on active TCP connection
    if( stcpactive != INVALID_SOCKET && FD_ISSET(stcpactive, &ibits) )
    {
        // There is data available on the active connection
        rcv_count = (int)recvnc( stcpactive, (void **)&buf, 0, &hBuffer );

		if (rcv_count <= 0 ){
			fdClose( stcpactive );
			stcpactive = INVALID_SOCKET;
			return;
		}

		if(buf[0] == 0xa5 && buf[1] == 0xa5 && rcv_count == 16 && TcpPacketInfo.Command==0)
		{
			ReceiveTCPHeader(buf);
	
			TcpPacketInfo.Command = TCPRXHeader.Command;
			TcpPacketInfo.SmallDataAddress = (unsigned char *)GlobalTCPRxDataBuffer;
			TcpPacketInfo.LargeDataAddress = GetRXLargeDataAddress(&dataBufferMaxSize);
			TcpPacketInfo.SmallDataSize = TCPRXHeader.SmallDataLength;
			TcpPacketInfo.LargeDataSize = TCPRXHeader.LargeDataLength;
			remain_small_size = TCPRXHeader.SmallDataLength;
			remain_large_size = TCPRXHeader.LargeDataLength;
			remain_small_offset = 0;
			remain_large_offset = 0;
			if(TCPRXHeader.SmallDataLength > 0  || TCPRXHeader.LargeDataLength > 0)
			{

				TcpPacketInfo.Status = SERVER_STATUS_RECEIVE;
				if(TCPRXHeader.LargeDataLength > 0 && TCPRXHeader.SmallDataLength <= 0)
				{
					TcpPacketInfo.rx_control= SERVER_CONTROL_RECEIVE_LARGE_DATA;
					TcpReceiveDataInfo.LargeDataSize= TCPRXHeader.LargeDataLength;
					TcpReceiveDataInfo.SmallDataSize= TCPRXHeader.SmallDataLength;
				}

				else if(TCPRXHeader.SmallDataLength > 0 && TCPRXHeader.LargeDataLength <= 0)
				{
					TcpPacketInfo.rx_control= SERVER_CONTROL_RECEIVE_SMALL_DATA;
					TcpReceiveDataInfo.LargeDataSize= TCPRXHeader.LargeDataLength;
					TcpReceiveDataInfo.SmallDataSize= TCPRXHeader.SmallDataLength;
				}
				else
				{
					TcpPacketInfo.rx_control= SERVER_CONTROL_RECEIVE_ALL_DATA;
					TcpReceiveDataInfo.LargeDataSize= TCPRXHeader.LargeDataLength;
					TcpReceiveDataInfo.SmallDataSize= TCPRXHeader.SmallDataLength;					
				}
				recvncfree( hBuffer );	
				return;
			}
			if(TCPRXHeader.RequestKind== CMD_TYPE_APP_RECEIVE_RESULT)
			{
				TcpPacketInfo.Status = SERVER_STATUS_PROCESSING;
				TcpPacketInfo.tx_control= SERVER_CONTROL_SEND_HEADER;
			}
			else
			{
				TcpPacketInfo.Status = SERVER_STATUS_SEND;
				TcpPacketInfo.tx_control= SERVER_CONTROL_SEND_HEADER;
			}

			recvncfree( hBuffer );	
			return ;	
		}
		if(TcpPacketInfo.rx_control == SERVER_CONTROL_RECEIVE_SMALL_DATA)
		{
		
			ReceiveSmallData(buf, rcv_count, remain_small_offset);
			remain_small_size -= rcv_count;
			remain_small_offset += rcv_count;
			if(remain_small_size > 0)
			{
				TcpPacketInfo.Status = SERVER_STATUS_RECEIVE;
				TcpPacketInfo.rx_control = SERVER_CONTROL_RECEIVE_SMALL_DATA;
				recvncfree( hBuffer );
				return; 
			}
			
			if(TCPRXHeader.RequestKind== CMD_TYPE_APP_RECEIVE_RESULT)
			{
				TcpPacketInfo.Status = SERVER_STATUS_PROCESSING;
			}
			else
			{
				TcpPacketInfo.Status = SERVER_STATUS_SEND;
			}

		}
		else if(TcpPacketInfo.rx_control == SERVER_CONTROL_RECEIVE_LARGE_DATA)
		{
			ReceiveLagreData(buf, rcv_count, remain_large_offset);
			remain_large_size -= rcv_count;
			remain_large_offset += rcv_count;
			if(remain_large_size > 0)
			{
				TcpPacketInfo.Status = SERVER_STATUS_RECEIVE;
				TcpPacketInfo.rx_control = SERVER_CONTROL_RECEIVE_LARGE_DATA;
				recvncfree( hBuffer );
				return; 
			}
			if(TCPRXHeader.RequestKind== CMD_TYPE_APP_RECEIVE_RESULT)
			{
				TcpPacketInfo.Status = SERVER_STATUS_PROCESSING;
			}
			else
			{
				TcpPacketInfo.Status = SERVER_STATUS_SEND;
			}

		}
		else if(TcpPacketInfo.rx_control == SERVER_CONTROL_RECEIVE_ALL_DATA){
			if(remain_small_size > 0)
			{
				ReceiveSmallData(buf, rcv_count, remain_small_offset);
				remain_small_size -= rcv_count;
				remain_small_offset += rcv_count;
				TcpPacketInfo.Status = SERVER_STATUS_RECEIVE;
				TcpPacketInfo.rx_control = SERVER_CONTROL_RECEIVE_ALL_DATA;
				recvncfree( hBuffer );
				return; 
			}
			if(remain_large_size > 0)
			{
				ReceiveLagreData(buf, rcv_count, remain_large_offset);
				remain_large_size -= rcv_count;
				remain_large_offset += rcv_count;
				if(remain_large_size <=0)
				{
					TcpPacketInfo.Status = SERVER_STATUS_SEND;
				}
				else
				{
					TcpPacketInfo.Status = SERVER_STATUS_RECEIVE;
					TcpPacketInfo.rx_control = SERVER_CONTROL_RECEIVE_ALL_DATA;
				}
				recvncfree( hBuffer );
				return;  
			}		
			if(TCPRXHeader.RequestKind== CMD_TYPE_APP_RECEIVE_RESULT)
			{
				TcpPacketInfo.Status = SERVER_STATUS_PROCESSING;
			}
			else
			{
				TcpPacketInfo.Status = SERVER_STATUS_SEND;
			}
		}
    	recvncfree( hBuffer );
    	TCP_TransferComplete();		
	}	
}

void SendTCPPacket(TCPControlInfo send)
{
	if(TCPRXHeader.Header == TCP_PACKET_HEADER)
		SendTCPHeader();
	if(TCPTXHeader.SmallDataLength > 0)
	{
		if(TCPRXHeader.RequestKind == CMD_TYPE_APP_REQUEST)
		{
			ResponseDataAddr[0] = 1;

			SendSmallData((unsigned char *)ResponseDataAddr,TCPTXHeader.SmallDataLength );
		}
		else if(TCPRXHeader.RequestKind == CMD_TYPE_APP_POLLING_RESULT)
		{
			ResponseDataAddr[0] = 1;
			SendSmallData((unsigned char *)ResponseDataAddr,TCPTXHeader.SmallDataLength );
		}
		else if(TCPRXHeader.RequestKind == CMD_TYPE_APP_RECEIVE_RESULT)
		{

			SendSmallData(TcpPacketInfo.ResultSmallDataAddress,TCPTXHeader.SmallDataLength);
		}
	}
	if(TCPTXHeader.LargeDataLength > 0)
	{
		SendLagreData(TcpPacketInfo.ResultLargeDataAddress,TCPTXHeader.LargeDataLength);
	}
	
	memset(&TCPRXHeader, 0 ,sizeof(TTCPDataHeader));
	memset(&TCPTXHeader, 0 ,sizeof(TTCPDataHeader));
	memset(&TcpPacketInfo, 0 ,sizeof(TCPControlInfo));
	TcpPacketInfo.Status = SERVER_STATUS_IDLE;
}

void ReceiveTCPHeader(unsigned char *buf)
{
	memcpy(&TCPRXHeader,buf,rcv_count);
}

void ReceiveSmallData(unsigned char *buf, int size, int offset)
{
	memcpy(TcpPacketInfo.SmallDataAddress+offset,buf,size);
}

void ReceiveLagreData(unsigned char *buf, int size, int offset)
{
	memcpy(TcpPacketInfo.LargeDataAddress+offset,buf,size);
	TcpPacketInfo.rx_control = SERVER_CONTROL_RECEIVE_COMPLETE;
}

void SendTCPHeader()
{
	

	if(TCPRXHeader.RequestKind != CMD_TYPE_APP_RECEIVE_RESULT)
	{
		TCPTXHeader.Header 			= TCP_PACKET_HEADER;
		TCPTXHeader.Command	 		= TCPRXHeader.Command;
		TCPTXHeader.RequestKind 	= TCPRXHeader.RequestKind;
		TCPTXHeader.reserved		= 0x1234;
		TCPTXHeader.SmallDataLength	= sizeof(unsigned int);
		TCPTXHeader.LargeDataLength	= 0;
	}
	else
	{
		TCPTXHeader.Header 			= TCP_PACKET_HEADER;
		TCPTXHeader.Command	 		= TCPRXHeader.Command;
		TCPTXHeader.RequestKind 	= TCPRXHeader.RequestKind;
		TCPTXHeader.SmallDataLength	= (TcpPacketInfo.ResultSmallDataSize+ 3) / 4 * 4;
		TCPTXHeader.LargeDataLength	= (TcpPacketInfo.ResultLargeDataSize+ 3) / 4 * 4;
	}
		
	snd_count = send( stcpactive, &TCPTXHeader, sizeof(TTCPDataHeader), 0 );

	if( snd_count < 0 )
    {
        fdClose( stcpactive );
        stcpactive = INVALID_SOCKET;
    }
}

void SendSmallData(unsigned char *buf, int size)
{
	snd_count = send( stcpactive, buf, size, 0 );
	if( snd_count < 0 )
    {
        fdClose( stcpactive );
        stcpactive = INVALID_SOCKET;
    }
	TcpPacketInfo.tx_control == SERVER_CONTROL_SEND_COMPLETE;	
}

void SendLagreData(unsigned char *buf, int size)
{
	snd_count = send( stcpactive, buf, size, 0 );
	if( snd_count < 0 )
    {
        fdClose( stcpactive );
        stcpactive = INVALID_SOCKET;
    }
	TcpPacketInfo.tx_control == SERVER_CONTROL_SEND_COMPLETE;

}

void TCP_TransferComplete(void)
{

}

unsigned char *GetRXLargeDataAddress(unsigned int *bufferMaxSize)
{
	unsigned short command = TCPRXHeader.Command;
	if (command == CMD_ECHO)
	{
		*bufferMaxSize = ECHO_BUFFER_MAX_SIZE;
		return (unsigned char *)EchoBuffer;		
	}
	else if (command == CMD_DOWNLOAD)
	{
		*bufferMaxSize = DOWNLOAD_MAX_SIZE;
		return (unsigned char *)ProgramBuffer;		
	}
	else
	{
		*bufferMaxSize = 0;
		return NULL;
	}
}

void Tcp_Processing()		// called from main loop
{
	unsigned short command = TcpPacketInfo.Command;
	TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)GlobalTCPTxDataBuffer;
	TcpPacketInfo.ResultSmallDataSize = 0;
	TcpPacketInfo.ResultLargeDataAddress = NULL;
	TcpPacketInfo.ResultLargeDataSize = 0;
	TcpPacketInfo.LargeDataSize = TcpReceiveDataInfo.LargeDataSize;
	TcpPacketInfo.SmallDataSize = TcpReceiveDataInfo.SmallDataSize;

	memset(&TcpReceiveDataInfo, 0 , sizeof(TCPReceiveData));

	if (command == CMD_INSPECTION_RESULT)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *) &InspectionResultTable;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TInspectionResultTable);

		USBCommTimeout = 0;
	}
	else if (command == CMD_STATUS)
	{
		int statusKind;
		statusKind = *(int *)TcpPacketInfo.SmallDataAddress;;
		if (statusKind == HCB_STATUS_KIND_MACHINE)		// machine status
		{
			TcpPacketInfo.ResultSmallDataAddress = (unsigned char *) &MachineStatus;
			TcpPacketInfo.ResultSmallDataSize = sizeof(TMachineStatus);
		}
		else if (statusKind == HCB_STATUS_KIND_ALARM)	// alarm status
		{
			TcpPacketInfo.ResultSmallDataAddress = (unsigned char *) &MachineAlarmState;
			TcpPacketInfo.ResultSmallDataSize = sizeof(TMachineAlarmState);
		}
		else if (statusKind == HCB_STATUS_KIND_WARNING)	// warning status
		{
			TcpPacketInfo.ResultSmallDataAddress = (unsigned char *) &MachineWarningState;
			TcpPacketInfo.ResultSmallDataSize = sizeof(TMachineWarningState);
		}
	}
	else if (command == CMD_MONITORING_DATA)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *) &MonitoringInfo;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TMonitoringInfo);
	}
	else if (command == CMD_PORT_READ)
	{
		memcpy(GlobalTCPBuffer, InputPortValue, sizeof(InputPortValue));
		memcpy(GlobalTCPBuffer + sizeof(InputPortValue), OutputPortValue, sizeof(OutputPortValue));
		memcpy(GlobalTCPBuffer + sizeof(InputPortValue) + sizeof(OutputPortValue), EncoderValue, sizeof(EncoderValue));
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)GlobalTCPBuffer;
		TcpPacketInfo.ResultSmallDataSize = sizeof(InputPortValue)+ sizeof(OutputPortValue)+sizeof(EncoderValue);
	}
	else if (command == CMD_TABLET_INSPECT_INFO)
	{
		int requestNum;

		requestNum = *(int *)TcpPacketInfo.SmallDataAddress;;

		if (requestNum == 1)
		{
/*
			if(!SimulationInspectionSW)
			{
				TabletInspectInfo.GoodTabletCount = TabletInspectInfo.CounterSensorTabletCount;
			}
*/
			TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&TabletInspectInfo;
			TcpPacketInfo.ResultSmallDataSize = sizeof(TTabletInspectInfo);
		}
	}
	else if (command == CMD_TABLET_PROCESS_INFO)
	{
		int requestNum;

		requestNum = *(int *)TcpPacketInfo.SmallDataAddress;;

		if (requestNum == 1)
		{
			TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&TabletInfo;
			TcpPacketInfo.ResultSmallDataSize = sizeof(TTabletInfo);
		}
	}
	else if (command == CMD_VIBRATOR_INFO)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&VibratorInfo;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TVibratorInfo);
	}
	else if (command == CMD_MACHINE_CONTROL)
	{
		MC_DecodeMachineControl(TcpPacketInfo.SmallDataAddress);
	}
	else if(command == CMD_MACHINE_POWER_OFF)
	{
		BlackoutInfo.BlackoutStep =	0;
		BlackoutInfo.BlackoutOccured = 1;
	}
	else if (command == CMD_READ_PORT_STATUS)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *) &PortStatus;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TPortStatus);
	}
	else if (command == CMD_READ_3D_SENSOR_STATUS)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *) &ThreeDSensorStatus;
		TcpPacketInfo.ResultSmallDataSize = sizeof(int) * 2;
	}
	else if (command == CMD_IDLE_PROCESS_INFO)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *) &IdleProcessData;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TIdleProcessData);
	}
	else if (command == CMD_MACHINE_CAPTURE)
	{
		TMachineCaptureSetupData *machineCaputreSetupData;		
		machineCaputreSetupData = (TMachineCaptureSetupData *) TcpPacketInfo.SmallDataAddress;
		TabletControl_Capture_Start(machineCaputreSetupData->SuctionDiskSpeed, machineCaputreSetupData->StudiedTabletSensorLength, machineCaputreSetupData->IlluminatorMask,machineCaputreSetupData->TabletSideThick, machineCaputreSetupData->CaptureOption);
	}
	else if (command == CMD_PORT_TEST)
	{
		int mode;
		int portIndex;
		int value;
		TOPTestData *opTestData;
		mode = *((int *)TcpPacketInfo.SmallDataAddress);
		if (mode == 0)
		{			
			portIndex = *((int *)TcpPacketInfo.SmallDataAddress + 1);
			value = *((int *)TcpPacketInfo.SmallDataAddress + 2);
			PORT_ModifyPhysicalOutPort(portIndex, value);
		}
		else
		{
			opTestData = (TOPTestData *)((int *)TcpPacketInfo.SmallDataAddress + 1);
			if (opTestData->TestCount == 0)
			{
				MC_OPTest_Stop();
			}
			else
			{
				MC_OPTest_Start(opTestData->TestCount, opTestData->OPList, 
                                opTestData->IsPhysicalPort,opTestData->DelayIntervalList, 
                                opTestData->OnIntervalList,opTestData->OffIntervalList);
			}
		}
	}	
	else if (command == CMD_CHECK_SENSOR_INTERVAL)
	{
		int func;
		float tabletLength;
		int suctionDiskSpeed;
		int studiedTabletSensorLength;
		func = *(int *) TcpPacketInfo.SmallDataAddress;
		if (func == 0)
		{
			TabletControl_Stop();
		}
		else if (func == 1)
		{
			tabletLength = *((float *)TcpPacketInfo.SmallDataAddress + 1);
			suctionDiskSpeed = *((int *)TcpPacketInfo.SmallDataAddress + 2);
			studiedTabletSensorLength = *((int *)TcpPacketInfo.SmallDataAddress + 3);
			TabletControl_CheckInerval_Start(suctionDiskSpeed, studiedTabletSensorLength);
		}
	}
	else if (command == CMD_TABLET_TEST)
	{
		int func;
		func = *(int *) TcpPacketInfo.SmallDataAddress;
		if (func == 0)
		{
			TabletControl_Stop();
		}
		else if (func == 1)
		{
			TabletControl_CheckInerval_Start();
		}
	}
	else if (command == CMD_STUDY_READY)
	{
		int suctionDiskSpeed;
		int studiedTabletSensor1Length;
		int studiedTabletSensor2Length;
		int studiedSensorIntervalA;
		int studiedSensorIntervalB;
		int studyOption;
		int TabletSideThick;

		suctionDiskSpeed = *((int *)TcpPacketInfo.SmallDataAddress);
		studiedTabletSensor1Length = *((int *)TcpPacketInfo.SmallDataAddress + 1);
		studiedTabletSensor2Length = *((int *)TcpPacketInfo.SmallDataAddress + 2);
		studiedSensorIntervalA = *((int *)TcpPacketInfo.SmallDataAddress + 3);
		studiedSensorIntervalB = *((int *)TcpPacketInfo.SmallDataAddress + 4);
		studyOption = *((int *)TcpPacketInfo.SmallDataAddress + 5);
		TabletSideThick = *((int *)TcpPacketInfo.SmallDataAddress + 6);
		TabletControl_Study_Start(suctionDiskSpeed, studiedTabletSensor1Length, studiedTabletSensor2Length, 
		                            studiedSensorIntervalA, studiedSensorIntervalB, TabletSideThick ,studyOption);
	}
	else if (command == CMD_STUDY_END)
	{
		TabletControl_Stop();
	}
	else if (command == CMD_TABLET_COUNTER_READY)
	{
		int suctionDiskSpeed;
		int studiedTabletSensor1Length;
		int studiedTabletSensor2Length;
		int studiedSensorIntervalA;
		int studiedSensorIntervalB;
		char defectGenEnabled, defectGenKind, uninsGenEnabled, uninsGenKind;
		unsigned int defectGenMask, uninsGenMask;
		int defectGenMaskCount, defectGenRatio, uninsGenMaskCount, uninsGenRatio;
		suctionDiskSpeed = *((int *) TcpPacketInfo.SmallDataAddress);
		studiedTabletSensor1Length = *((int *) TcpPacketInfo.SmallDataAddress + 1);
		studiedTabletSensor2Length = *((int *) TcpPacketInfo.SmallDataAddress + 2);
		studiedSensorIntervalA = *((int *) TcpPacketInfo.SmallDataAddress + 3);
		studiedSensorIntervalB = *((int *) TcpPacketInfo.SmallDataAddress + 4);
		defectGenEnabled = *((char *) TcpPacketInfo.SmallDataAddress + 20);
		defectGenKind = *((char *) TcpPacketInfo.SmallDataAddress + 21);
		uninsGenEnabled = *((char *) TcpPacketInfo.SmallDataAddress + 22);
		uninsGenKind = *((char *) TcpPacketInfo.SmallDataAddress + 23);
		defectGenMask = *((unsigned int *) TcpPacketInfo.SmallDataAddress + 6);
		defectGenMaskCount = *((int *) TcpPacketInfo.SmallDataAddress + 7);
		defectGenRatio = *((int *) TcpPacketInfo.SmallDataAddress + 8);
		uninsGenMask = *((unsigned int *) TcpPacketInfo.SmallDataAddress + 9);
		uninsGenMaskCount = *((int *) TcpPacketInfo.SmallDataAddress + 10);
		uninsGenRatio = *((int *) TcpPacketInfo.SmallDataAddress + 11);
		TabletControl_Tablet_Counter_Start(suctionDiskSpeed, studiedTabletSensor1Length, studiedTabletSensor2Length, studiedTabletSensor2Length,
			studiedSensorIntervalA, studiedSensorIntervalB, defectGenEnabled, defectGenKind, uninsGenEnabled, uninsGenKind, 
			defectGenMask, defectGenMaskCount, defectGenRatio, uninsGenMask, uninsGenMaskCount, uninsGenRatio);
	}
	else if (command == CMD_INSPECTION_READY)
	{
		THCBInspectionOption *hcbInspectionOption;
		TTabletInspectInfo *tabletInspectInfo;
		hcbInspectionOption = (THCBInspectionOption *) TcpPacketInfo.SmallDataAddress;
		tabletInspectInfo = (TTabletInspectInfo *)((unsigned char *) TcpPacketInfo.SmallDataAddress + sizeof(THCBInspectionOption));
		SimulationInspectionSW = 0;
		TabletControl_Inspection_Start(hcbInspectionOption, tabletInspectInfo);
	}
	else if (command == CMD_INSPECTION_END)
	{
		TabletControl_Stop();
	}
	else if (command == CMD_MACHINE_CAPTURE_STOP)
	{
		TabletControl_Stop();
	}
	else if (command == CMD_INSPECTION_RESUME)
	{
		TabletControl_ContinueInspection(TcpPacketInfo.SmallDataAddress);
	}
	else if (command == CMD_UNMANED_OPERATION)
	{
		MC_UnmanedOperation((TUnmanedOperationData *)TcpPacketInfo.SmallDataAddress);
	}
	else if (command == CMD_READ_UNMANED_OPERATION_INFO)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&UnmanedOperationData;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TUnmanedOperationData);
	}
	else if (command == CMD_TABLET_COUNTER)
	{
		int func;
		func = *((int *)TcpPacketInfo.SmallDataAddress);
		if (func == 0)
		{
			TabletCounter_Reset();
			SimpleTabletCounter_Reset();
		}
		else if (func == 1)
		{
			TabletCounter_Ready(1);
		}
		else if (func == 2)
		{
			TabletCounter_Sleep();
		}
		else if (func == 5)
		{
			TabletCounter_Ready(0);
		}
		else if (func == 3)
		{			
			*((unsigned int *)GlobalTCPBuffer) = TabletCounter.Counter;
			TcpPacketInfo.ResultSmallDataSize = sizeof(unsigned int);
		}
		else if (func == 4)
		{			
			TcpPacketInfo.ResultLargeDataAddress = (unsigned char *)&TabletCounter;
			TcpPacketInfo.ResultLargeDataSize = sizeof(TTabletCounter);
		}
		else if (func == 6)
		{			
			TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&SimpleTabletCounter;
			TcpPacketInfo.ResultSmallDataSize = sizeof(TSimpleTabletCounter);
		}
	}
	else if (command == CMD_TABLET_CAPTURE)
	{
		TCaptureCommandParam_HCB *captureCommandParam;
		captureCommandParam = (TCaptureCommandParam_HCB *)TcpPacketInfo.SmallDataAddress;
		if (captureCommandParam->Action == 0)
		{
			HCB_Capture_Stop();
		}
		else if (captureCommandParam->Action == 1)
		{
			HCB_Capture_Ready(captureCommandParam->FrameTime, captureCommandParam->LEDMask, captureCommandParam->LEDDelay, captureCommandParam->LEDOnTime, captureCommandParam->ShutterDelay);
		}		
		else if (captureCommandParam->Action == 2)
		{
			HCB_Capture_Ready_Multi(captureCommandParam->FrameTime, captureCommandParam->LEDMask, captureCommandParam->LEDDelay, captureCommandParam->LEDOnTime, captureCommandParam->ShutterDelay);
		}		
	}
	
	else if (command == CMD_TABLET_TPB_CAPTURE)
	{
		TCaptureCommandParam_HCB *captureCommandParam;
		captureCommandParam = (TCaptureCommandParam_HCB *)TcpPacketInfo.SmallDataAddress;
		if (captureCommandParam->Action == 0)
		{
			HCB_TPB_Capture_Stop();
		}
		else if (captureCommandParam->Action == 1)
		{
			HCB_TPB_Capture_Start(captureCommandParam->FrameTime, captureCommandParam->LEDMask, captureCommandParam->LEDDelay, captureCommandParam->LEDOnTime, captureCommandParam->ShutterDelay);
		}				
	}
	
	else if (command == CMD_TERMINATION_INFO)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&TerminationInfo;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TTerminationInfo);
	}
	else if (command == CMD_INVALIDATE_TERMINATION_INFO)
	{
		InvalidateTerminationInfo();
	}
	else if (command == CMD_CAMERA_RESET)
	{
		int action;
		int hcbCamIndex;
		action = *(int *)TcpPacketInfo.SmallDataAddress;
		hcbCamIndex = *((int *)TcpPacketInfo.SmallDataAddress + 1);
        /*
		if (hcbCamIndex == 0)
		{
			if (action == 0)
			{
				PORT_ModifyOutPort(OP_3D_CAMERA_1_RESET, 0);
			}
			else
			{
				PORT_ModifyOutPort(OP_3D_CAMERA_1_RESET, 1);
			}
		}
		else
		{
			if (action == 0)
			{
				PORT_ModifyOutPort(OP_3D_CAMERA_2_RESET, 0);
			}
			else
			{
				PORT_ModifyOutPort(OP_3D_CAMERA_2_RESET, 1);
			}
		}*/
	}
	else if (command == CMD_LED_CHANGE)
	{
		int ledMask = *(unsigned int *)TcpPacketInfo.SmallDataAddress;
		HCB_Capture_LEDChange(ledMask);
	}
	else if (command == CMD_SERVO_MOTOR)
	{	
		int motorIndex;
        int moveSpeed;
        int accTime;
        int decTime;
		int dir;
		unsigned int motorAction;
		motorIndex = *((int *)TcpPacketInfo.SmallDataAddress);
		motorAction = *((unsigned int *)TcpPacketInfo.SmallDataAddress + 1);	
		if (motorAction == 1)
		{
			moveSpeed = *((int *)TcpPacketInfo.SmallDataAddress + 2);
			accTime = *((int *)TcpPacketInfo.SmallDataAddress + 3);
			dir = *((int *)TcpPacketInfo.SmallDataAddress + 4);
			Servo_Motor_Run_Param(motorIndex, accTime, dir, moveSpeed);
		}
		else if (motorAction == 2)
		{
			decTime = *((int *)TcpPacketInfo.SmallDataAddress + 2);
			Servo_Motor_Stop_Param(motorIndex, decTime);
		}
		else if (motorAction == 3)
		{
			Servo_Motor_ServoOn(motorIndex);
		}
		else if (motorAction == 4)
		{
			Servo_Motor_ServoOff(motorIndex);
		}
	}
	else if (command == CMD_ALARM_RESET)
	{
		MC_Alarm_Reset();
	}
	else if (command == CMD_BUZZER_ON)
	{
		MC_BuzzerControl_SuspendBuzzerOn();
	}
	else if (command == CMD_BUZZER_OFF)
	{
		MC_BuzzerControl_Stop();
	}
	else if (command == CMD_SET_MACHINE_MODE)
	{
		int machineControlMode;
		machineControlMode = *(int *)TcpPacketInfo.SmallDataAddress;
		MC_SetMachineControlMode(machineControlMode);
	}
	else if (command == CMD_PORT_SETUP)
	{
		PORT_Setup(TcpPacketInfo.SmallDataAddress);
	}
	else if (command == CMD_PORT_AO_WRITE)
	{
		unsigned short aoIndex = *(unsigned short *)TcpPacketInfo.SmallDataAddress;
		unsigned short aoValue = *((unsigned short *)TcpPacketInfo.SmallDataAddress + 1);
		PORT_AO_Write(aoIndex, aoValue);
	}
	else if (command == CMD_MACHINE_PARAM_SETUP)
	{
		memcpy(&MachineSetupData, TcpPacketInfo.SmallDataAddress, sizeof(TMachineSetupData));
		WriteSystemSetup();
		ApplySystemSetup();
	}
	else if (command == CMD_USER_MACHINE_ENV_SETUP)
	{
		memcpy(&UserMachineEnv, TcpPacketInfo.SmallDataAddress, sizeof(TUserMachineEnv));
		WriteUserEnv();
	}
	else if (command == CMD_DOWNLOAD)
	{
		CSL_IntcGlobalEnableState 	state;
		unsigned int pID;
		int time;
		CSL_intcGlobalDisable(&state);
		pID = *(unsigned int *)TcpPacketInfo.SmallDataAddress;
		time = *((int *)TcpPacketInfo.SmallDataAddress + 1);
		WriteToFlash(pID, TcpPacketInfo.LargeDataAddress, TcpPacketInfo.LargeDataSize, time);
		CSL_intcGlobalEnable(&state);
	}
	else if (command == CMD_RESET)
	{
		Mcbsp_Start = 0;
		CSL_mcbspClose(hMcbsp1);
		System_Reset_Trigger_Time = Board1MSTmr;
		System_Reset_SW = TRUE;		
	}
	else if (command == CMD_FLASH_INFO)
	{
		TcpPacketInfo.ResultLargeDataAddress = (unsigned char *) GetFileSystem();
		TcpPacketInfo.ResultLargeDataSize = sizeof(TFFSystem);
	}
	else if (command == CMD_FLASH_FORMAT)
	{
		FlashFormat();
	}
	else if (command == CMD_MCBSP_TEST)
	{
		int func;
		int dataSize;
		int frequency;
		int spbMask;
		func = *((int *)TcpPacketInfo.SmallDataAddress);
		if (func == 0)
		{
			Master_McBSPTest_Stop();
		}
		else if (func == 1)
		{
			dataSize = *((int *)TcpPacketInfo.SmallDataAddress + 1);
			frequency = *((int *)TcpPacketInfo.SmallDataAddress + 2);
			spbMask = *((int *)TcpPacketInfo.SmallDataAddress + 3);
			Master_McBSPTest_Start(dataSize, frequency, spbMask);
		}
	}
	else if (command == CMD_MCBSP_TEST_RESULT)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *) &McBSPTestResultData;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TMcBSPTestResultData);
	}
	else if (command == CMD_MCBSP_DEBUG_INFO)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *) &McbspDebugInfo;
		TcpPacketInfo.ResultSmallDataSize = sizeof(McbspDebugInfo);
	}
	else if (command == CMD_VERSION_INFO)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&ProgramVersion;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TProgramVersion);
	}
	else if (command == CMD_ECHO)
	{
		TcpPacketInfo.ResultSmallDataAddress = TcpPacketInfo.SmallDataAddress;
		TcpPacketInfo.ResultSmallDataSize = TcpPacketInfo.SmallDataSize;
		TcpPacketInfo.ResultLargeDataAddress = TcpPacketInfo.LargeDataAddress;
		TcpPacketInfo.ResultLargeDataSize = TcpPacketInfo.LargeDataSize;
	}
	else if (command == CMD_RS485_TEST)
	{
		RS485_Test(TcpPacketInfo.SmallDataAddress, TcpPacketInfo.SmallDataSize);
	}
	else if (command == CMD_RS485_TEST_STATUS)
	{
		*(int *)GlobalTCPBuffer = RS485MasterTestData.Status;
		*((int *)GlobalTCPBuffer + 1) = RS485MasterTestData.RXLength;
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)GlobalTCPBuffer;
		TcpPacketInfo.ResultSmallDataSize = sizeof(int) * 2;
	}
	else if (command == CMD_RS485_TEST_RX_DATA)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *) RS485MasterTestData.RXData;
		TcpPacketInfo.ResultSmallDataSize = RX485_MASTER_RX_BUFFER_SIZE;
	}
	else if (command == CMD_RS485_DEBUG_INFO)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *) &RS485DebugInfo;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TRS485DebugInfo);
	}
	else if (command == CMD_READ_ERROR_LIST)
	{
		int maxCount;
		int getCount;
		maxCount = *(int *) TcpPacketInfo.SmallDataAddress;
		if (maxCount > 16) maxCount = 16;
		getCount = ErrorQueue_Get(GlobalTCPBuffer + sizeof(int), maxCount);
		ErrorQueue_Remove(getCount);
		*(int *)GlobalTCPBuffer = getCount;
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)GlobalTCPBuffer;
		TcpPacketInfo.ResultSmallDataSize = getCount * sizeof(TErrorUnit) + sizeof(int);
	}
	else if (command == CMD_READ_PROGRAM_HEADER)
	{
		Mcbsp_Start = 1;
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&HCBProgramHeader;
		TcpPacketInfo.ResultSmallDataSize = sizeof(THCBProgramHeader);
	}
	else if (command == CMD_READ_DB_HEADER)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&HCBDBHeader;
		TcpPacketInfo.ResultSmallDataSize = sizeof(THCBDBHeader);
	}
	else if (command == CMD_READ_MS_HEADER)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&HCBMachineSpecificHeader;
		TcpPacketInfo.ResultSmallDataSize = sizeof(THCBMachineSpecificHeader);
	}
	else if (command == CMD_READ_COMPATIBILITY)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&HCBCompatibility;
		TcpPacketInfo.ResultSmallDataSize = sizeof(THCBCompatibility);
	}
	else if (command == CMD_WRITE_MS_HEADER)
	{
		THCBMachineSpecificHeader *hcbMachineSpecificHeader = (THCBMachineSpecificHeader *) TcpPacketInfo.SmallDataAddress;
		WriteHCBMSHeader(hcbMachineSpecificHeader);
	}
	else if (command == CMD_READ_MACHINE_INFO)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&MachineSetupData;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TMachineSetupData);
	}
	else if (command == CMD_READ_USER_MACHINE_ENV)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&UserMachineEnv;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TUserMachineEnv);
	}
	else if (command == CMD_READ_PORT_MAP_INFO)
	{
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&PortMapInfo;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TPortMapInfo);
	}
	else if (command == CMD_SAFETY_RESET)
	{
		PORT_ModifyOutPort(OP_SAFETY_CIRCUIT_RESET, 1);
		SafetyResetFlag = 1;
	}
	else if (command == CMD_SAFETY_REVERT) // operator m
	{
		PORT_ModifyOutPort(OP_SAFETY_CIRCUIT_SELECT, 0);
		PORT_ModifyOutPort(OP_SAFETY_CIRCUIT_RESET, 1);
		SafetyResetFlag = 1;
	}
	else if (command == CMD_SAFETY_MAINTAENACE) // main m
	{
		PORT_ModifyOutPort(OP_SAFETY_CIRCUIT_SELECT, 1);
		PORT_ModifyOutPort(OP_SAFETY_CIRCUIT_RESET, 1);
		SafetyResetFlag = 1;
	}
	else if (command == CMD_LONG_WORK)
	{
		Sleep(2000);
	}
	else if (command == CMD_PIO05_PORT_ONOFF)
	{
		PortStatusInfo* portInfo_tcp = (PortStatusInfo*)TcpPacketInfo.SmallDataAddress;
		memcpy(&portInfo,portInfo_tcp, sizeof(PortStatusInfo));
	}
	else if(command == CMD_LASER_SHUTTER_ONOFF)
	{
		laser_shutter_control = *(int *)TcpPacketInfo.SmallDataAddress;
		if(laser_shutter_control == 1)
		{
			PORT_ModifyOutPort(OP_LASER_SHUTTER, 1);
		}
		else
		{
			PORT_ModifyOutPort(OP_LASER_SHUTTER, 0);
		}
	}
/*
	else if(command ==CMD_OUTPORT_ON)
	{
		int outportNum;
		outportNum = *(int *) TcpPacketInfo.SmallDataAddress;		
	}
	else if(command ==CMD_OUTPORT_OFF)
	{
		int outportNum;
		outportNum = *(int *) TcpPacketInfo.SmallDataAddress;		
	}
*/	else if (command == CMD_PC_FLAG)
	{
		TPcUiFlag * UIFLAG;
		UIFLAG = (TPcUiFlag*) TcpPacketInfo.SmallDataAddress;
		
		/*if(UIFLAG->InsReady)
			PORT_ChangeOutPort(OP_LINK_READY, 1);
		else
			PORT_ChangeOutPort(OP_LINK_READY, 0);

		if(UIFLAG->InsActive)
			PORT_ChangeOutPort(OP_LINK_ACTION, 1);
		else
			PORT_ChangeOutPort(OP_LINK_ACTION, 0);*/

		/*if(UIFLAG->Alarm)
			PORT_ModifyOutPort(OP_LINK_ALARM, 0);
		else
			PORT_ModifyOutPort(OP_LINK_ALARM, 1);*/

		/*if(UIFLAG->EMR)
			PORT_ChangeOutPort(OP_LINK_EMR, 0);
		else
			PORT_ChangeOutPort(OP_LINK_EMR, 1);

		if(UIFLAG->APPActive)
			PORT_ChangeOutPort(OP_LINK_APPACTION, 1);
		else
			PORT_ChangeOutPort(OP_LINK_APPACTION, 0);*/
	}
	else if (command == CMD_UVLASER_STATUS_SETUP)
	{
		TUVLaserStatus *UVLaser;
		UVLaser = (TUVLaserStatus*) TcpPacketInfo.SmallDataAddress;
		
		if(UVLaser->SELMAStatus.Ready)
			PORT_ModifyOutPort(OP_LINK_READY, 1);
		else
			PORT_ModifyOutPort(OP_LINK_READY, 0);

		if(UVLaser->SELMAStatus.Active)
			PORT_ModifyOutPort(OP_LINK_ACTION, 1);
		else
			PORT_ModifyOutPort(OP_LINK_ACTION, 0);

		if(UVLaser->SELMAStatus.EMR)
			PORT_ModifyOutPort(OP_LINK_EMR, 1);
		else
			PORT_ModifyOutPort(OP_LINK_EMR, 0);

		if(UVLaser->SELMAStatus.APPActive)
			PORT_ModifyOutPort(OP_LINK_APPACTION, 1);
		else
			PORT_ModifyOutPort(OP_LINK_APPACTION, 0);


	}
	else if (command == CMD_ACROCIA_INSPECTION_INFO_RESET)
	{
		TabletInspectInfo.MisejectActiveCount = 0;
	}
	else if (command == CMD_INSPECTION_WAITSTOP)
	{
		int WaitStopType;
		WaitStopType = *(int *) TcpPacketInfo.SmallDataAddress;
		MC_MainControl_WaitStop(WaitStopType);
	}
	else if (command == CMD_ACROSIA_TO_HCB_ALARM_UPDATE)
	{
		int MesAlarmType = *((int *) TcpPacketInfo.SmallDataAddress);

		if(MesAlarmType == 1)
		{
			MachineAlarmState.mes_status = 1;
		}
		else if(MesAlarmType == 2)
		{
			MachineAlarmState.upstream_status = 1;
		}
		else if(MesAlarmType == 3)
		{
			MachineAlarmState.downstream_status = 1;
		}
		else if(MesAlarmType == 4)
		{
			MachineAlarmState.Qjp_object_status = 1;
		}
		else if(MesAlarmType == 10)
		{
			MachineAlarmState.downstream_status = 0;
		}
		else if(MesAlarmType == 20)
		{
			MachineAlarmState.downstream_status =0;
		}
		else if(MesAlarmType == 30)
		{
			MachineAlarmState.downstream_status = 0;
		}
		else if(MesAlarmType == 40)
		{
			MachineAlarmState.Qjp_object_status = 0;
		}
		
	}
	else if (command == CMD_VIRTUAL_INSPECTION_START)
	{
		SimulationInspectionPeriod = *((int *)TcpPacketInfo.SmallDataAddress);
		SimulationInspectionSW = 1;
	}	
	else if (command == CMD_LASER_SWITCH_ON)
	{
		LaserTriggerSW = 1;
	}
	else if (command == CMD_LASER_TESTMODE_ON)
	{
		LaserTestMode = *((int *)TcpPacketInfo.SmallDataAddress);
	}	
	else if (command == CMD_VIRTUAL_INSPECTION_REFRESH)
	{
		SimulationInspectionPeriod = *((int *)TcpPacketInfo.SmallDataAddress);
	}
	else if (command == CMD_VIRTUAL_INSPECTION_END)
	{
		SimultationStopWaiting = 1;
		SimulationInspectionSW = 0;
		SimulationInspectionPeriod = 0;
	}
	else if (command == CMD_READ_THREEDSENSOR_MEM)
	{
		int temp_size_data = sizeof(read_THREED_CAMERA_SENSOR);
		read_THREED_CAMERA_SENSOR = ~THREED_CAMERA_SENSOR;
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&read_THREED_CAMERA_SENSOR;
		TcpPacketInfo.ResultSmallDataSize = sizeof(short int);
	}
	else if (command == CMD_READ_DISTRIBUTOR_SIGNAL)
	{	
		 checkDistributorSig = MC_DistributorCheckBeforeInspection();
		
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&checkDistributorSig ;
		TcpPacketInfo.ResultSmallDataSize = sizeof(int);		
	}
	else if (command == CMD_EJECTOR_TEST_SETUP)
	{	
		TEjectorTestSetupData EjectorTestSetupData;
		
		memcpy(&EjectorTestSetupData, TcpPacketInfo.SmallDataAddress, sizeof(TEjectorTestSetupData));
		
		ApplySuctionDiskSpeed(EjectorTestSetupData.SuctionDiskSpeed, EjectorTestSetupData.StudiedSensorIntervalA, 
		EjectorTestSetupData.StudiedSensorIntervalB, EjectorTestSetupData.CaptureMode);	

		CalculateExpectedSensorLength(TABLET_CONTROL_FUNC_INSPECTION, EjectorTestSetupData.SuctionDiskSpeed, EjectorTestSetupData.StudiedTabletSensor1Length, EjectorTestSetupData.StudiedTabletSensor2Length, EjectorTestSetupData.StudiedTabletSensor3Length);

		//InitTabletTable(TabletInspectInfo.TotalTabletCount + 1);
		InitTabletTable(1);
	
		TabletControlData.Func = TABLET_CONTROL_FUNC_EJECTOR_TEST; 
		
	}	
	else if (command == CMD_DISTRIBUTOR_MONITOR)
	{			
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&DistributorMonitor;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TDistributorMonitor);		
	}
	else if (command == CMD_ENCODER_MONITOR)
	{			
		TcpPacketInfo.ResultSmallDataAddress = (unsigned char *)&EncoderMonitor;
		TcpPacketInfo.ResultSmallDataSize = sizeof(TEncoderMonitor);		
	}	
	if(command != 0)
		TcpPacketInfo.Status = SERVER_STATUS_SEND;	
}
