

#ifndef __SRIO_H__
#define __SRIO_H__

#ifdef __cplusplus
    extern "C" {
#endif // __cplusplus 
/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include "cslr.h"
#include "cslr_srio.h"

/****************************************************************************
 *                             Local constants                              *
 ****************************************************************************/

 /** @name SRIO message types
 *  Definitions for the Srio Message types
 */
//@{
#define REQ_NREAD               0x24  ///< NREAD message type ftype=2
#define REQ_NWRITE              0x54  ///< NWRITE message type ftype=5
#define REQ_NWRITE_R            0x55  ///< NWRITE_R message type fypte=5
#define REQ_SWRITE              0x60  ///< SWRITE message type ftype=6
#define REQ_MAINTENANCE_R       0x80  ///< Maintenance read message type ftype=8 ttype = 0
#define REQ_MAINTENANCE_W       0x81  ///< Maintenance write message type ftype=8 ttype = 1
#define REQ_DOORBELL            0xA0  ///< Doorbell message type ftype=10
//@}

 /** @name SRIO descriptor flags
 *  Definitions for the bitmasks of the Srio Descriptor Flags
 */
//@{
#define SRIO_DSC_FLAG_SOP               		0x80000000  ///< Start of packet flag
#define SRIO_DSC_FLAG_EOP               		0x40000000  ///< End of packet flag
#define SRIO_DSC_FLAG_OWNERSHIP         	0x20000000  ///< Ownership flag
#define SRIO_DSC_FLAG_EOQ               		0x10000000  ///< End of queue flag
#define SRIO_DSC_FLAG_TDOWNCMPLT        0x08000000  ///< Teardown complete
//@}

 /** @name Speed parameter
 *  Definition of the speed settings for the port
 */
//@{
#define SPEED1_25G  0 ///< 1.25 Gbits/s 
#define SPEED2_50G  1 ///< 2.50 Gbits/s 
#define SPEED3_125G 2 ///< 3.125 Gbits/s (default)
//@}

 /** @name Port mode parameter
 *  Definition of the portmode parameter
 */
//@{
#define PMODE4X  0 ///< 4X mode 
#define PMODE1X  1 ///< 1X mode 
//@}

 /** @name Loopback parameter
 *  Definition of the loopback parameter
 */
//@{
#define LOOPBACK_NO  0 ///< No dsp srio digital loopback
#define LOOPBACK_YES 1 ///< Use dsp srio digital loopback (for tests)
//@}

#define DSP0BASEADD 0x30 ///< Dsp 1 Srio ID


#define DSP1BASEADD 0x41//0x41 ///< Dsp 2 Srio ID
#define DSP2BASEADD 0x42 ///< Dsp 2 Srio ID
#define DSP3BASEADD 0x43 ///< Dsp 2 Srio ID
#define DSP4BASEADD 0x44 ///< Dsp 2 Srio ID
#define DSP5BASEADD 0x45 ///< Dsp 2 Srio ID
#define DSP6BASEADD 0x46 ///< Dsp 2 Srio ID
#define DSP7BASEADD 0x47 ///< Dsp 2 Srio ID
#define DSP8BASEADD 0x48 ///< Dsp 2 Srio ID

#define SRIO_PORT_0		0	//port0
#define SRIO_PORT_1		1	// Port1 (1x mode only) 
#define SRIO_PORT_2		2	// Port2 (1x mode only) 
#define SRIO_PORT_3		3	// Port3 (1x mode only) 

#define RATE CSL_SRIO_SERDES_RATE_FULL ///< Full rate
#define ICRR0 0x00000000
#define ICRR1 0x11111111
#define SELECTED_LSU 1 

/****************************************************************************
 *                             Public functions                             *
 ****************************************************************************/

/****************************************************************************
 *   NAME : Srio_Init
 ************************************************************************//**
 *
 *   This function initializes the specified SRIO port of the DSP and gives it an ID.
 *   Most of the parameters are given from the setup structure pSetup,
 *   but the additionnals single parameters (portmode, portspeed, etc)
 *   will override defaults values given in pSetup.
 *
 *   WARNING: Parameters must match hardware srio evm powerup configuration that
 *   is related to the port to be used. These evm parameters cannot be changed by
 *   this function, but are needed to configure the dsp correctly for the evm and
 *   the onboard srio switch. If user want to change srio powerup evm configuration,
 *   the configuration must be changed by DSP2 only using BSL functions (this is
 *   possible only for Variant11 boards and later).
 *
 *   @param [in] pSetup
 *      Pointer to CSL SRIO setup structure   
 *
 *   @param [in] blken
 *      Block enable 
 *
 *   @param [in] port
 *      Port to initialize  (only port 0 and 1 are possible on Variant11 boards and later,
 *      on Variant10 boards, dsp 1 can use port 0 only and dsp 2 can use 0,1,2,3)
 * 
 *   @param [in] ID
 *      DSP srio base adress ID (default dsp 1: 0x40 or dsp 2: 0x50)

 *   @param [in] portmode
 *      Port operation mode (0: 4X mode (default) or 1: 1X mode)
 *   
 *   @param [in] portspeed
 *      Port operation speed (0: 1.25,  1: 2.5, 2:3.125 (default) Gbits/s)
 *   
 *   @param [in] loopback
 *      Dsp internal srio loopback mode (1: Yes,  0: No (0 for normal operation))
 *   
 *   @ingroup EVM_SRIO
 *
 ***************************************************************************/        
void Srio_Init(CSL_SrioHwSetup *pSetup,
    Int32 blken, 
    Int32 port,
    Int32 ID,
    Int32 portmode,
    Int32 portspeed,
    Int32 loopback);

/****************************************************************************
 *   NAME : Srio_LSUSetup
 ************************************************************************//**
 *
 *   This function sets up the Load Store Unit.
 *   
 *   @ingroup EVM_SRIO
 *
 ***************************************************************************/    
void Srio_LSUSetup();


/****************************************************************************
 *   NAME : Srio_SetupInterrupt
 ************************************************************************//**
 *
 *   This function sets up Srio interrupts
 *   
 *   @ingroup EVM_SRIO
 *
 ***************************************************************************/ 
void Srio_SetupInterrupt();

/****************************************************************************
 *   NAME : Srio_CloseInterrupt
 ************************************************************************//**
 *
 *   This function disables the SRIO interrupts
 *   
 *   @ingroup EVM_SRIO
 *
 ***************************************************************************/ 
void Srio_CloseInterrupt();

/****************************************************************************
 *   NAME : Srio_Event20Isr
 ************************************************************************//**
 *
 *   This function is the event 20 interrupt service routine, checking for errors.
 *   
 *   @ingroup EVM_SRIO
 *
 ***************************************************************************/ 
void Srio_Event20Isr();

/****************************************************************************
 *   NAME : Srio_Event21Isr
 ************************************************************************//**
 *
 *   This function is the event 21 interrupt service routine, which happens 
 *     upon a complete transaction. 
 *   
 *   @ingroup EVM_SRIO
 *
 ***************************************************************************/ 
void IsrSrioTest();


/****************************************************************************
 *   NAME : Srio_Write
 ************************************************************************//**
 *
 *   This function writes a array of data in a connected Srio device at the 
 *     specified address
 *
 *   @param [in] src
 *      Address of the data to be sent
 *
 *   @param [in] dst
 *      Destination address in the target's memory space 
 *
 *   @param [in] len
 *      Data buffer length
 * 
 *   @param [in] type
 *      Transaction packet type
 * 
 *   @param [in] srcport
 *      Port used for write on DSP
 * 
 *   @param [in] dstID
 *      Destination ID of the target
 *   
 *   @ingroup EVM_SRIO
 *
 ***************************************************************************/     
Int32 Srio_ReadWrite(Uint32 src, Uint32 dst, Int32 len, Int32 type, Int32 srcport, Int32 dstID);


/****************************************************************************
 *   NAME : Srio_DoorBell
 ************************************************************************//**
 *
 *   This function generates a doorbell interruption to Srio device at the 
 *     specified address
 *
 *   @param [in] uiDoorBellInfo
 *      Doorbell info to send
 * 
 *   @param [in] srcport
 *      Port used for write on DSP
 * 
 *   @param [in] dstID
 *      Destination ID of the target
 *   
 *   @ingroup EVM_SRIO
 *
 ***************************************************************************/     
Int32 Srio_DoorBell(Uint32 uiDoorBellInfo, Int32 srcport, Int32 dstID);

/****************************************************************************
 *   NAME : Srio_MaintenanceWrite
 ************************************************************************//**
 *
 *   This function writes a CSR or a CAR register in the destination SRIO
 *           registers
 *
 *   @param [in] i32Port
 *      Port used for write
 *
 *   @param [in] i32ID
 *      Destination ID of the target
 *
 *   @param [in] i32HopCount
 *      Hop Count to reach target 
 * 
 *   @param [in] i32Reg
 *      Register to be written
 * 
 *   @param [in] i32Val
 *      Value to be written
 *   
 *   @ingroup EVM_SRIO
 *
 ***************************************************************************/ 
Int32 Srio_MaintenanceWrite(Int32 i32Port, Int32 i32ID, Int32 i32HopCount, 
                            Int32 i32DestReg, Int32 i32Val);

/****************************************************************************
 *   NAME : Srio_MaintenanceRead
 ************************************************************************//**
 *
 *   This function reads a CSR or a CAR register in the destination SRIO
 *           registers
 *
 *   @param [in] i32Port
 *      Port used for write
 *
 *   @param [in] i32ID
 *      Destination ID of the target
 *
 *   @param [in] i32HopCount
 *      Hop Count to reach target 
 * 
 *   @param [in] i32Reg
 *      Register to be read
 * 
 *   @param [out] i32Val
 *      Value to read
 *   
 *   @ingroup EVM_SRIO
 *
 ***************************************************************************/ 
Int32 Srio_MaintenanceRead(Int32 i32Port, Int32 i32ID, Int32 i32HopCount, 
                           Int32 i32Reg, Int32 *i32Val);
                                        
/****************************************************************************
 *   NAME : Srio_LockPort
 ************************************************************************//**
 *
 *   This function executes a port lockdown on the selected port. 
 *
 *   @param [in] u32Port
 *      Port used for write
 *   
 *   @ingroup EVM_SRIO
 *
 ***************************************************************************/
void Srio_LockPort(Uint32 u32Port);

/****************************************************************************
 *   NAME : Srio_UnlockPort
 ************************************************************************//**
 *
 *   This function disables the port lockdown on the selected port. 
 *
 *   @param [in] u32Port
 *      Port used for write
 *   
 *   @ingroup EVM_SRIO
 *
 ***************************************************************************/
void Srio_UnlockPort(Uint32 u32Port);

#ifdef __cplusplus
}
#endif //__cplusplus
#endif //__SRIO_H__

