/****************************************************************************
 *
 *
 *         **  **     **  ******  ********  ********  ********  **    **
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ************************************************************************//**
 *
 * @file       evm_cpld.h
 * 
 * @brief      EVM Utility functions for writing/reading registers on the CPLD
 *  
 *
 * (c) 2008, Lyrtech RD Inc. All rights reserved.              
 *
 * @ingroup     EVM_CPLD
 *
 ****************************************************************************
 *                                                                                          
 * "$Revision: 1.8 $"
 * "$Date: 2009/03/13 16:10:10 $"
 *
 ***************************************************************************/

#ifndef __EVM_CPLD_H__
#define __EVM_CPLD_H__


#ifdef __cplusplus
    extern "C" {
#endif // __cplusplus 


/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <tistdtypes.h>


/****************************************************************************
 *                             Local constants                              *
 ****************************************************************************/

 
// 
#define EVM_CPLD_I2C_ADDRESS       (0x70)   ///<  Default CPLD I2C address

/** @name EVM_CPLD_error_codes
 *  EVM CPLD functions error status codes.
 */
//@{
#define EVM_CPLD_OK                0        ///<  CPLD No Error Code
#define EVM_CPLD_WRITEREG_FAIL     0x1000   ///<  CPLD Write Failure Error
#define EVM_CPLD_READREG_FAIL      0x1002   ///<  CPLD Read Failure Error
//@}


/** @name EVM_CPLD_regs
 *  CPLD register definitions
 */
//@{
    
/** @name EVM_CPLD_Register_0
 *  Reg 0 : CPLD and PCB version
 */
//@{
#define EVM_CPLD_PCB_VERSION                (0x00)  ///< CPLD Command register
                                        
#define EVM_CPLD_PCB_VERSION_MASK           (0xE0)  ///< PCB Version Mask
#define EVM_CPLD_VERSION_MASK               (0x1F)  ///< CPLD Version Mask
//@}     

/** @name EVM_CPLD_Register_1
 *  Reg 1 : Bicolor LEDs  
 */
//@{                                             
#define EVM_CPLD_BICOLOR_LED                (0x01)  ///< CPLD Bicolor LED register
                                        
#define EVM_CPLD_BICOLOR_LED1_MASK          (0x03)  ///< Bicolor LED 1 Mask
#define EVM_CPLD_BICOLOR_LED1_NONE          (0x00)  ///< Bicolor LED 1 No led lit
#define EVM_CPLD_BICOLOR_LED1_GREEN         (0x01)  ///< Bicolor LED 1 Green led lit
#define EVM_CPLD_BICOLOR_LED1_RED           (0x02)  ///< Bicolor LED 1 Red led lit
#define EVM_CPLD_BICOLOR_LED1_YELLOW        (0x03)  ///< Bicolor LED 1 Yellow led lit
                                        
#define EVM_CPLD_BICOLOR_LED2_MASK          (0x0C)  ///< Bicolor LED 2 Mask
#define EVM_CPLD_BICOLOR_LED2_NONE          (0x00)  ///< Bicolor LED 2 No led lit
#define EVM_CPLD_BICOLOR_LED2_GREEN         (0x04)  ///< Bicolor LED 2 Green led lit
#define EVM_CPLD_BICOLOR_LED2_RED           (0x08)  ///< Bicolor LED 2 Red led lit
#define EVM_CPLD_BICOLOR_LED2_YELLOW        (0x0C)  ///< Bicolor LED 2 Yellow led lit

#define EVM_CPLD_BICOLOR_OVERRIDE_MASK      (0x10)  ///< Bicolor Override Mask
#define EVM_CPLD_BICOLOR_OVERRIDE_JTAGMODE  (0x00)  ///< Bicolor Override reflect JTAG Mode
#define EVM_CPLD_BICOLOR_OVERRIDE_DSP       (0x10)  ///< Bicolor Override controlled by DSP

#define EVM_CPLD_BICOLOR_PGLED_MASK         (0xC0)  ///< Power Good Mask
#define EVM_CPLD_BICOLOR_PGLED_OFF          (0x00)  ///< Power Good Led OFF: Power problem
#define EVM_CPLD_BICOLOR_PGLED_BLINK        (0x40)  ///< Power Good: power ok, waiting for dsp boot
#define EVM_CPLD_BICOLOR_PGLED_ON           (0x80)  ///< Power Good: power ok, all dsp boot ok
//@} 

/** @name EVM_CPLD_Register_2
 *  Reg 2 : Jtag Emulation Mode Register 
 */
//@{  
#define EVM_CPLD_JTAG_EMU_MODE              (0x02)  ///< CPLD Jtag Emulation Mode Register

#define EVM_CPLD_JTAG_EMU_MODE_MASK         (0x07)  ///< Jtag Emulation Mask
#define EVM_CPLD_JTAG_EMU_MODE_ALLOFF       (0x00)  ///< All emulators disconnected
#define EVM_CPLD_JTAG_EMU_MODE_USB_C1       (0x01)  ///< USB emulator on dsp1 only
#define EVM_CPLD_JTAG_EMU_MODE_JTAG_C1      (0x02)  ///< JTAG emulator on dsp1 only
#define EVM_CPLD_JTAG_EMU_MODE_JTAG_C1C2    (0x03)  ///< JTAG emulator DSP1 to DSP2 daisy chain
#define EVM_CPLD_JTAG_EMU_MODE_USB_C2       (0x04)  ///< USB emulator on dsp2 only
#define EVM_CPLD_JTAG_EMU_MODE_JTAG_C2      (0x05)  ///< JTAG emulator on dsp2 only
#define EVM_CPLD_JTAG_EMU_MODE_JTAG_C1_C2   (0x06)  ///< JTAG independant emulator on dsp1 and dsp2
#define EVM_CPLD_JTAG_EMU_MODE_USB_C1C2     (0x07)  ///< USB emulator DSP1 to DSP2 daisy chain
//@} 

/** @name EVM_CPLD_Register_3
 *  Reg 3 : Global Reset Register
 */
//@{ 
#define EVM_CPLD_GLOBAL_RESET                 (0x03)  ///< CPLD Global Reset Register
#define EVM_CPLD_GLOBAL_RESET_MASK            (0x01)  ///< Global Reset Mask
#define EVM_CPLD_GLOBAL_RESET_HWRESET         (0x01)  ///< Hardware Reset
#define EVM_CPLD_GLOBAL_RESET_RELEASE         (0x00)  ///< Reset release
                                        
#define EVM_CPLD_GLOBAL_RESET_C1RSTST_MASK    (0x04)  ///< DSP 1 Reset status mask
#define EVM_CPLD_GLOBAL_RESET_C1RSTST_ACT     (0x00)  ///< DSP 1 Reset status active
#define EVM_CPLD_GLOBAL_RESET_C1RSTST_REL     (0x04)  ///< DSP 1 Reset status inactive

#define EVM_CPLD_GLOBAL_RESET_C2RSTST_MASK    (0x08)  ///< DSP 2 Reset status mask
#define EVM_CPLD_GLOBAL_RESET_C2RSTST_ACT     (0x00)  ///< DSP 2 Reset status active
#define EVM_CPLD_GLOBAL_RESET_C2RSTST_REL     (0x08)  ///< DSP 2 Reset status inactive

#define EVM_CPLD_GLOBAL_RESET_SRIO_RST_MASK   (0x10)  ///< SRIO Switch reset mask
#define EVM_CPLD_GLOBAL_RESET_SRIO_RST_ACT    (0x10)  ///< SRIO Switch is in reset        
#define EVM_CPLD_GLOBAL_RESET_SRIO_RST_REL    (0x00)  ///< SRIO Switch is not in reset  
                                                                             
#define EVM_CPLD_GLOBAL_RESET_SRIO_SWRST_MASK (0x20)  ///< SRIO Software Switch reset mask
#define EVM_CPLD_GLOBAL_RESET_SRIO_SWRST_REL  (0x20)  ///< SRIO Software Switch is not in reset        
#define EVM_CPLD_GLOBAL_RESET_SRIO_SWRST_ACT  (0x00)  ///< SRIO Software Switch is in reset  
//@} 

/** @name EVM_CPLD_Register_4
 *  Reg 4 : Switches Register
 */
//@{
#define EVM_CPLD_SWITCHES_RAW               (0x04)  ///< CPLD Switches Raw Register

#define EVM_CPLD_SWITCH1_RAW_MASK           (0x01)  ///< Switch 1 Raw Value Mask
#define EVM_CPLD_SWITCH1_RAW_PRESSED        (0x00)  ///< Switch 1 Raw Pressed
#define EVM_CPLD_SWITCH1_RAW_RELEASED       (0x01)  ///< Switch 1 Raw Released
                                        
#define EVM_CPLD_SWITCH2_RAW_MASK           (0x02)  ///< Switch 2 Raw Value Mask
#define EVM_CPLD_SWITCH2_RAW_PRESSED        (0x00)  ///< Switch 2 Raw Pressed
#define EVM_CPLD_SWITCH2_RAW_RELEASED       (0x02)  ///< Switch 2 Raw Released
                                        
#define EVM_CPLD_SWITCH3_RAW_MASK           (0x04)  ///< Switch 3 Raw Value Mask
#define EVM_CPLD_SWITCH3_RAW_PRESSED        (0x00)  ///< Switch 2 Raw Pressed
#define EVM_CPLD_SWITCH3_RAW_RELEASED       (0x04)  ///< Switch 2 Raw Released
//@}

/** @name EVM_CPLD_Register_5
 *  Reg 5 : Dip Switches Register
 */
//@{
#define EVM_CPLD_DIP_SWITCHES               (0x05)  ///< Dip Switches Register

#define EVM_CPLD_DIP_SWITCH1_MASK           (0x01)  ///< Dip Switch 1 Value Mask
#define EVM_CPLD_DIP_SWITCH1_PRESSED        (0x00)  ///< Dip Switch 1 Pressed
#define EVM_CPLD_DIP_SWITCH1_RELEASED       (0x01)  ///< Dip Switch 1 Released
                                        
#define EVM_CPLD_DIP_SWITCH2_MASK           (0x02)  ///< Dip Switch 2 Value Mask
#define EVM_CPLD_DIP_SWITCH2_PRESSED        (0x00)  ///< Dip Switch 2 Pressed
#define EVM_CPLD_DIP_SWITCH2_RELEASED       (0x02)  ///< Dip Switch 2 Released
                                        
#define EVM_CPLD_DIP_SWITCH3_MASK           (0x04)  ///< Dip Switch 3 Value Mask
#define EVM_CPLD_DIP_SWITCH3_PRESSED        (0x00)  ///< Dip Switch 3 Pressed
#define EVM_CPLD_DIP_SWITCH3_RELEASED       (0x04)  ///< Dip Switch 3 Released
                                        
#define EVM_CPLD_DIP_SWITCH4_MASK           (0x08)  ///< Dip Switch 4 Value Mask
#define EVM_CPLD_DIP_SWITCH4_PRESSED        (0x00)  ///< Dip Switch 4 Pressed
#define EVM_CPLD_DIP_SWITCH4_RELEASED       (0x08)  ///< Dip Switch 4 Released
                                        
#define EVM_CPLD_DIP_SWITCH5_MASK           (0x10)  ///< Dip Switch 5 Value Mask
#define EVM_CPLD_DIP_SWITCH5_PRESSED        (0x00)  ///< Dip Switch 5 Pressed
#define EVM_CPLD_DIP_SWITCH5_RELEASED       (0x10)  ///< Dip Switch 5 Released
                                        
#define EVM_CPLD_DIP_SWITCH6_MASK           (0x20)  ///< Dip Switch 6 Value Mask
#define EVM_CPLD_DIP_SWITCH6_PRESSED        (0x00)  ///< Dip Switch 6 Pressed
#define EVM_CPLD_DIP_SWITCH6_RELEASED       (0x20)  ///< Dip Switch 6 Released
                                        
#define EVM_CPLD_DIP_SWITCH7_MASK           (0x40)  ///< Dip Switch 7 Value Mask
#define EVM_CPLD_DIP_SWITCH7_PRESSED        (0x00)  ///< Dip Switch 7 Pressed
#define EVM_CPLD_DIP_SWITCH7_RELEASED       (0x40)  ///< Dip Switch 7 Released
                                        
#define EVM_CPLD_DIP_SWITCH8_MASK           (0x80)  ///< Dip Switch 8 Value Mask
#define EVM_CPLD_DIP_SWITCH8_PRESSED        (0x00)  ///< Dip Switch 8 Pressed
#define EVM_CPLD_DIP_SWITCH8_RELEASED       (0x80)  ///< Dip Switch 8 Released
//@}

/** @name EVM_CPLD_Register_6
 *  Reg 6 : Miscellenous info/Ctrl
 */
//@{
#define EVM_CPLD_MISCEL_INFOCTL             (0x06)  // Miscellenous info register

#define EVM_CPLD_MISCEL_INFOCTL_MODEL_MASK  (0x01)  ///< Board Model Mask
#define EVM_CPLD_MISCEL_INFOCTL_MODEL_LOW   (0x00)  ///< Board Model Low
#define EVM_CPLD_MISCEL_INFOCTL_MODEL_HIGH  (0x01)  ///< Board Model High
//@}

/** @name EVM_CPLD_Register_7
 *  Reg 7 : Jumper Status register
 */
//@{
#define EVM_CPLD_JUMPER_STATS               (0x07)  // Jumper Status register

#define EVM_CPLD_BOOT1_JUMPER_MASK          (0x01)  ///< Boot 1 Jumper Mask
#define EVM_CPLD_BOOT1_JUMPER_LOW           (0x00)  ///< Boot 1 Jumper Low
#define EVM_CPLD_BOOT1_JUMPER_HIGH          (0x01)  ///< Boot 1 Jumper High
                                        
#define EVM_CPLD_BOOT2_JUMPER_MASK          (0x02)  ///< Boot 2 Jumper Mask
#define EVM_CPLD_BOOT2_JUMPER_LOW           (0x00)  ///< Boot 2 Jumper Low
#define EVM_CPLD_BOOT2_JUMPER_HIGH          (0x02)  ///< Boot 2 Jumper High
//@}

/** @name EVM_CPLD_Register_9
 *  Reg 9 : Power Status Low Register
 */
//@{
#define EVM_CPLD_POWER_Status_LOW                 (0x09)  ///< Power Status Low Register
                                    
#define EVM_CPLD_POWER1_EVM1V8_STATUS_MASK        (0x01)  ///< Power 1 EVM 1.8V Status Mask                                          
#define EVM_CPLD_POWER1_EVM1V8_STATUS_NOTOK       (0x00)  ///< Power 1 EVM 1.8V Not Ok                                               
#define EVM_CPLD_POWER1_EVM1V8_STATUS_OK          (0x01)  ///< Power 1 EVM 1.8V OK                                                   
                                                                                                                                 
#define EVM_CPLD_POWER2_SYS1V8_STATUS_MASK        (0x02)  ///< Power 2 1.8V Sys Status Mask                                          
#define EVM_CPLD_POWER2_SYS1V8_STATUS_NOTOK       (0x00)  ///< Power 2 1.8V Sys Not Ok                                               
#define EVM_CPLD_POWER2_SYS1V8_STATUS_OK          (0x02)  ///< Power 2 1.8V Sys OK                                                   
                                                                                                                                 
#define EVM_CPLD_POWER3_EVM3V3_STATUS_MASK        (0x04)  ///< Power 3 EVM 3.3V Status Mask                                          
#define EVM_CPLD_POWER3_EVM3V3_STATUS_NOTOK       (0x00)  ///< Power 3 EVM 3.3V Not Ok                                               
#define EVM_CPLD_POWER3_EVM3V3_STATUS_OK          (0x04)  ///< Power 3 EVM 3.3V OK                                                   
                                                                                                                                 
#define EVM_CPLD_POWER4_EVM1V1_STATUS_MASK        (0x08)  ///< Power 4 EVM 1.1V Status Mask                                          
#define EVM_CPLD_POWER4_EVM1V1_STATUS_NOTOK       (0x00)  ///< Power 4 EVM 1.1V Not Ok                                               
#define EVM_CPLD_POWER4_EVM1V1_STATUS_OK          (0x08)  ///< Power 4 EVM 1.1V OK                                                   
                                                                                                                                                
#define EVM_CPLD_POWER5_COREC1SYS_STATUS_MASK     (0x10)  ///< Power 5 Core C1 Sys Status Mask                                       
#define EVM_CPLD_POWER5_COREC1SYS_STATUS_NOTOK    (0x00)  ///< Power 5 Core C1 Sys Not Ok                                            
#define EVM_CPLD_POWER5_COREC1SYS_STATUS_OK       (0x10)  ///< Power 5 Core C1 Sys OK                                                
                                                                                                                              
#define EVM_CPLD_POWER6_COREC23V3SYS_STATUS_MASK  (0x20)  ///< Power 6 Core C2 3.3V Sys Status Mask                            
#define EVM_CPLD_POWER6_COREC23V3SYS_STATUS_NOTOK (0x00)  ///< Power 6 Core C2 3.3V Sys Not Ok                                 
#define EVM_CPLD_POWER6_COREC23V3SYS_STATUS_OK    (0x20)  ///< Power 6 Core C2 3.3V Sys OK                                     
                                                                                                                              
#define EVM_CPLD_POWER7_VSC82341V2_STATUS_MASK    (0x40)  ///< Power 7 Vsc8234 1.2V Status Mask                                  
#define EVM_CPLD_POWER7_VSC82341V2_STATUS_NOTOK   (0x00)  ///< Power 7 Vsc8234 1.2V Not Ok                                           
#define EVM_CPLD_POWER7_VSC82341V2_STATUS_OK      (0x40)  ///< Power 7 Vsc8234 1.2V OK                                               
                                                                                                                              
#define EVM_CPLD_POWER8_C2DVDD3V3_STATUS_MASK     (0x80)  ///< Power 8 C2 Dvdd 3.3V Status Mask                                                                         
#define EVM_CPLD_POWER8_C2DVDD3V3_STATUS_NOTOK    (0x00)  ///< Power 8 C2 Dvdd 3.3V Not Ok                                                                              
#define EVM_CPLD_POWER8_C2DVDD3V3_STATUS_OK       (0x80)  ///< Power 8 C2 Dvdd 3.3V OK         
//@}

/** @name EVM_CPLD_Register_10
 *  Reg 10 : Power Status High Register
 */
//@{                                   
#define EVM_CPLD_POWER_Status_HIGH                 (0x0A)  ///< Power Status Low Register                       
                                                                     /*****/                                  
#define EVM_CPLD_POWER9_C2DVDD1V2_STATUS_MASK      (0x01)  ///< Power 9 C2 Dvdd 1.2V Status Mask                  
#define EVM_CPLD_POWER9_C2DVDD1V2_STATUS_NOTOK     (0x00)  ///< Power 9 C2 Dvdd 1.2V Not Ok                       
#define EVM_CPLD_POWER9_C2DVDD1V2_STATUS_OK        (0x01)  ///< Power 9 C2 Dvdd 1.2V OK                           
                                                                                                              
#define EVM_CPLD_POWER10_C2DVDD1V8_STATUS_MASK     (0x02)  ///< Power 10 C2 Dvdd 1.8V Status Mask                 
#define EVM_CPLD_POWER10_C2DVDD1V8_STATUS_NOTOK    (0x00)  ///< Power 10 C2 Dvdd 1.8V Not Ok                      
#define EVM_CPLD_POWER10_C2DVDD1V8_STATUS_OK       (0x02)  ///< Power 10 C2 Dvdd 1.8V OK                          
                                                                                                              
#define EVM_CPLD_POWER11_C1DVDD1V8_STATUS_MASK     (0x04)  ///< Power 11 C1 Dvdd 1.8V Status Mask                 
#define EVM_CPLD_POWER11_C1DVDD1V8_STATUS_NOTOK    (0x00)  ///< Power 11 C1 Dvdd 1.8V Not Ok         
#define EVM_CPLD_POWER11_C1DVDD1V8_STATUS_OK       (0x04)  ///< Power 11 C1 Dvdd 1.8V OK             
                                                                      
#define EVM_CPLD_POWER12_C1DVDD1V2_STATUS_MASK     (0x08)  ///< Power 12 C1 Dvdd 1.2V Status Mask    
#define EVM_CPLD_POWER12_C1DVDD1V2_STATUS_NOTOK    (0x00)  ///< Power 12 C1 Dvdd 1.2V Not Ok         
#define EVM_CPLD_POWER12_C1DVDD1V2_STATUS_OK       (0x08)  ///< Power 12 C1 Dvdd 1.2V OK             
                                                                     
#define EVM_CPLD_POWER13_C1DVDD3V3_STATUS_MASK     (0x10)  ///< Power 13 C1 Dvdd 3.3V Status Mask    
#define EVM_CPLD_POWER13_C1DVDD3V3_STATUS_NOTOK    (0x00)  ///< Power 13 C1 Dvdd 3.3V Not Ok         
#define EVM_CPLD_POWER13_C1DVDD3V3_STATUS_OK       (0x10)  ///< Power 13 C1 Dvdd 3.3V OK    

#define EVM_CPLD_POWER_PROGRESS_MASK               (0x40)  ///< Power sequenece progress Mask    
#define EVM_CPLD_POWER_PROGRESS_NOTDONE            (0x00)  ///< Power sequenced not completed or problem         
#define EVM_CPLD_POWER_PROGRESS_DONE               (0x40)  ///< Power sequence completed    
//@}

/** @name EVM_CPLD_Register_11
 *  Reg 11 : GPIO DSP 1 Low Register  
 */
//@{                                                
#define EVM_CPLD_GPIO_DSP1_LOW        (0x0B)  ///< GPIO DSP 1 Low Register   
                                                                          
#define EVM_CPLD_DSP1_GPIO0_MASK      (0x01)  ///< DSP 1 GPIO 0 Mask        
#define EVM_CPLD_DSP1_GPIO0_LOW       (0x00)  ///< DSP 1 GPIO 0 Low            
#define EVM_CPLD_DSP1_GPIO0_HIGH      (0x01)  ///< DSP 1 GPIO 0 High               
                                                                          
#define EVM_CPLD_DSP1_GPIO1_MASK      (0x02)  ///< DSP 1 GPIO 1 Mask    
#define EVM_CPLD_DSP1_GPIO1_LOW       (0x00)  ///< DSP 1 GPIO 1 Low     
#define EVM_CPLD_DSP1_GPIO1_HIGH      (0x02)  ///< DSP 1 GPIO 1 High    
                                                                        
#define EVM_CPLD_DSP1_GPIO2_MASK      (0x04)  ///< DSP 1 GPIO 2 Mask    
#define EVM_CPLD_DSP1_GPIO2_LOW       (0x00)  ///< DSP 1 GPIO 2 Low     
#define EVM_CPLD_DSP1_GPIO2_HIGH      (0x04)  ///< DSP 1 GPIO 2 High    
                                                                        
#define EVM_CPLD_DSP1_GPIO3_MASK      (0x08)  ///< DSP 1 GPIO 3 Mask     
#define EVM_CPLD_DSP1_GPIO3_LOW       (0x00)  ///< DSP 1 GPIO 3 Low      
#define EVM_CPLD_DSP1_GPIO3_HIGH      (0x08)  ///< DSP 1 GPIO 3 High     
                                                                        
#define EVM_CPLD_DSP1_GPIO4_MASK      (0x10)  ///< DSP 1 GPIO 4 Mask    
#define EVM_CPLD_DSP1_GPIO4_LOW       (0x00)  ///< DSP 1 GPIO 4 Low     
#define EVM_CPLD_DSP1_GPIO4_HIGH      (0x10)  ///< DSP 1 GPIO 4 High    
                                                                        
#define EVM_CPLD_DSP1_GPIO5_MASK      (0x20)  ///< DSP 1 GPIO 5 Mask   
#define EVM_CPLD_DSP1_GPIO5_LOW       (0x00)  ///< DSP 1 GPIO 5 Low    
#define EVM_CPLD_DSP1_GPIO5_HIGH      (0x20)  ///< DSP 1 GPIO 5 High   
                                                                        
#define EVM_CPLD_DSP1_GPIO6_MASK      (0x40)  ///< DSP 1 GPIO 6 Mask     
#define EVM_CPLD_DSP1_GPIO6_LOW       (0x00)  ///< DSP 1 GPIO 6 Low      
#define EVM_CPLD_DSP1_GPIO6_HIGH      (0x40)  ///< DSP 1 GPIO 6 High     
                                                                        
#define EVM_CPLD_DSP1_GPIO7_MASK      (0x80)  ///< DSP 1 GPIO 7 Mask
#define EVM_CPLD_DSP1_GPIO7_LOW       (0x00)  ///< DSP 1 GPIO 7 Low 
#define EVM_CPLD_DSP1_GPIO7_HIGH      (0x80)  ///< DSP 1 GPIO 7 High
//@}

/** @name EVM_CPLD_Register_12
 *  Reg 12 : GPIO DSP 1 High Register  
 */
//@{                                        
#define EVM_CPLD_GPIO_DSP1_HIGH       (0x0C)  ///< GPIO DSP 1 High Register   
                                                                          
#define EVM_CPLD_DSP1_GPIO8_MASK      (0x01)  ///< DSP 1 GPIO 8 Mask        
#define EVM_CPLD_DSP1_GPIO8_LOW       (0x00)  ///< DSP 1 GPIO 8 Low            
#define EVM_CPLD_DSP1_GPIO8_HIGH      (0x01)  ///< DSP 1 GPIO 8 High               
                                                                          
#define EVM_CPLD_DSP1_GPIO9_MASK      (0x02)  ///< DSP 1 GPIO 9 Mask    
#define EVM_CPLD_DSP1_GPIO9_LOW       (0x00)  ///< DSP 1 GPIO 9 Low     
#define EVM_CPLD_DSP1_GPIO9_HIGH      (0x02)  ///< DSP 1 GPIO 9 High    
                                                                        
#define EVM_CPLD_DSP1_GPIO10_MASK     (0x04)  ///< DSP 1 GPIO 10 Mask    
#define EVM_CPLD_DSP1_GPIO10_LOW      (0x00)  ///< DSP 1 GPIO 10 Low     
#define EVM_CPLD_DSP1_GPIO10_HIGH     (0x04)  ///< DSP 1 GPIO 10 High    
                                                                         
#define EVM_CPLD_DSP1_GPIO11_MASK     (0x08)  ///< DSP 1 GPIO 11 Mask     
#define EVM_CPLD_DSP1_GPIO11_LOW      (0x00)  ///< DSP 1 GPIO 11 Low      
#define EVM_CPLD_DSP1_GPIO11_HIGH     (0x08)  ///< DSP 1 GPIO 11 High     
                                                                        
#define EVM_CPLD_DSP1_GPIO12_MASK     (0x10)  ///< DSP 1 GPIO 12 Mask    
#define EVM_CPLD_DSP1_GPIO12_LOW      (0x00)  ///< DSP 1 GPIO 12 Low     
#define EVM_CPLD_DSP1_GPIO12_HIGH     (0x10)  ///< DSP 1 GPIO 12 High    
                                                                        
#define EVM_CPLD_DSP1_GPIO13_MASK     (0x20)  ///< DSP 1 GPIO 13 Mask   
#define EVM_CPLD_DSP1_GPIO13_LOW      (0x00)  ///< DSP 1 GPIO 13 Low    
#define EVM_CPLD_DSP1_GPIO13_HIGH     (0x20)  ///< DSP 1 GPIO 13 High   
                                                                        
#define EVM_CPLD_DSP1_GPIO14_MASK     (0x40)  ///< DSP 1 GPIO 14 Mask     
#define EVM_CPLD_DSP1_GPIO14_LOW      (0x00)  ///< DSP 1 GPIO 14 Low      
#define EVM_CPLD_DSP1_GPIO14_HIGH     (0x40)  ///< DSP 1 GPIO 14 High     
                                                                        
#define EVM_CPLD_DSP1_GPIO15_MASK     (0x80)  ///< DSP 1 GPIO 15 Mask
#define EVM_CPLD_DSP1_GPIO15_LOW      (0x00)  ///< DSP 1 GPIO 15 Low 
#define EVM_CPLD_DSP1_GPIO15_HIGH     (0x80)  ///< DSP 1 GPIO 15 High       
//@}

/** @name EVM_CPLD_Register_13
 *  Reg 13 : GPIO DSP 2 Low Register  
 */
//@{                                       
#define EVM_CPLD_GPIO_DSP2_LOW        (0x0D)  ///< GPIO DSP 2 Low Register         
                                                                            
#define EVM_CPLD_DSP2_GPIO0_MASK      (0x01)  ///< DSP 2 GPIO 0 Mask           
#define EVM_CPLD_DSP2_GPIO0_LOW       (0x00)  ///< DSP 2 GPIO 0 Low           
#define EVM_CPLD_DSP2_GPIO0_HIGH      (0x01)  ///< DSP 2 GPIO 0 High          
                                                                            
#define EVM_CPLD_DSP2_GPIO1_MASK      (0x02)  ///< DSP 2 GPIO 1 Mask          
#define EVM_CPLD_DSP2_GPIO1_LOW       (0x00)  ///< DSP 2 GPIO 1 Low           
#define EVM_CPLD_DSP2_GPIO1_HIGH      (0x02)  ///< DSP 2 GPIO 1 High          
                                                                            
#define EVM_CPLD_DSP2_GPIO2_MASK      (0x04)  ///< DSP 2 GPIO 2 Mask          
#define EVM_CPLD_DSP2_GPIO2_LOW       (0x00)  ///< DSP 2 GPIO 2 Low           
#define EVM_CPLD_DSP2_GPIO2_HIGH      (0x04)  ///< DSP 2 GPIO 2 High          
                                                                            
#define EVM_CPLD_DSP2_GPIO3_MASK      (0x08)  ///< DSP 2 GPIO 3 Mask          
#define EVM_CPLD_DSP2_GPIO3_LOW       (0x00)  ///< DSP 2 GPIO 3 Low           
#define EVM_CPLD_DSP2_GPIO3_HIGH      (0x08)  ///< DSP 2 GPIO 3 High          
                                                                            
#define EVM_CPLD_DSP2_GPIO4_MASK      (0x10)  ///< DSP 2 GPIO 4 Mask          
#define EVM_CPLD_DSP2_GPIO4_LOW       (0x00)  ///< DSP 2 GPIO 4 Low           
#define EVM_CPLD_DSP2_GPIO4_HIGH      (0x10)  ///< DSP 2 GPIO 4 High          
                                                                            
#define EVM_CPLD_DSP2_GPIO5_MASK      (0x20)  ///< DSP 2 GPIO 5 Mask          
#define EVM_CPLD_DSP2_GPIO5_LOW       (0x00)  ///< DSP 2 GPIO 5 Low           
#define EVM_CPLD_DSP2_GPIO5_HIGH      (0x20)  ///< DSP 2 GPIO 5 High          
                                                                            
#define EVM_CPLD_DSP2_GPIO6_MASK      (0x40)  ///< DSP 2 GPIO 6 Mask          
#define EVM_CPLD_DSP2_GPIO6_LOW       (0x00)  ///< DSP 2 GPIO 6 Low           
#define EVM_CPLD_DSP2_GPIO6_HIGH      (0x40)  ///< DSP 2 GPIO 6 High          
                                                                            
#define EVM_CPLD_DSP2_GPIO7_MASK      (0x80)  ///< DSP 2 GPIO 7 Mask          
#define EVM_CPLD_DSP2_GPIO7_LOW       (0x00)  ///< DSP 2 GPIO 7 Low           
#define EVM_CPLD_DSP2_GPIO7_HIGH      (0x80)  ///< DSP 2 GPIO 7 High          
//@}

/** @name EVM_CPLD_Register_14
 *  Reg 14 : GPIO DSP 2 Hich Register  
 */
//@{                                      
#define EVM_CPLD_GPIO_DSP2_HIGH       (0x0E)  ///< GPIO DSP 2 High Register       
                                                                            
#define EVM_CPLD_DSP2_GPIO8_MASK      (0x01)  ///< DSP 2 GPIO 8 Mask           
#define EVM_CPLD_DSP2_GPIO8_LOW       (0x00)  ///< DSP 2 GPIO 8 Low           
#define EVM_CPLD_DSP2_GPIO8_HIGH      (0x01)  ///< DSP 2 GPIO 8 High          
                                                                            
#define EVM_CPLD_DSP2_GPIO9_MASK      (0x02)  ///< DSP 2 GPIO 9 Mask          
#define EVM_CPLD_DSP2_GPIO9_LOW       (0x00)  ///< DSP 2 GPIO 9 Low           
#define EVM_CPLD_DSP2_GPIO9_HIGH      (0x02)  ///< DSP 2 GPIO 9 High          
                                                                            
#define EVM_CPLD_DSP2_GPIO10_MASK     (0x04)  ///< DSP 2 GPIO 10 Mask         
#define EVM_CPLD_DSP2_GPIO10_LOW      (0x00)  ///< DSP 2 GPIO 10 Low          
#define EVM_CPLD_DSP2_GPIO10_HIGH     (0x04)  ///< DSP 2 GPIO 10 High         
                                                                            
#define EVM_CPLD_DSP2_GPIO11_MASK     (0x08)  ///< DSP 2 GPIO 11 Mask         
#define EVM_CPLD_DSP2_GPIO11_LOW      (0x00)  ///< DSP 2 GPIO 11 Low          
#define EVM_CPLD_DSP2_GPIO11_HIGH     (0x08)  ///< DSP 2 GPIO 11 High         
                                                                            
#define EVM_CPLD_DSP2_GPIO12_MASK     (0x10)  ///< DSP 2 GPIO 12 Mask         
#define EVM_CPLD_DSP2_GPIO12_LOW      (0x00)  ///< DSP 2 GPIO 12 Low          
#define EVM_CPLD_DSP2_GPIO12_HIGH     (0x10)  ///< DSP 2 GPIO 12 High         
                                                                            
#define EVM_CPLD_DSP2_GPIO13_MASK     (0x20)  ///< DSP 2 GPIO 13 Mask         
#define EVM_CPLD_DSP2_GPIO13_LOW      (0x00)  ///< DSP 2 GPIO 13 Low          
#define EVM_CPLD_DSP2_GPIO13_HIGH     (0x20)  ///< DSP 2 GPIO 13 High         
                                                                            
#define EVM_CPLD_DSP2_GPIO14_MASK     (0x40)  ///< DSP 2 GPIO 14 Mask         
#define EVM_CPLD_DSP2_GPIO14_LOW      (0x00)  ///< DSP 2 GPIO 14 Low          
#define EVM_CPLD_DSP2_GPIO14_HIGH     (0x40)  ///< DSP 2 GPIO 14 High         
                                                                            
#define EVM_CPLD_DSP2_GPIO15_MASK     (0x80)  ///< DSP 2 GPIO 15 Mask         
#define EVM_CPLD_DSP2_GPIO15_LOW      (0x00)  ///< DSP 2 GPIO 15 Low          
#define EVM_CPLD_DSP2_GPIO15_HIGH     (0x80)  ///< DSP 2 GPIO 15 High      
//@}


/** @name EEPROM Program Register
 *  Reg 15 : EEPROM Program Register 
 */
//@{                                      
#define EVM_CPLD_EEPROM_PROGRAM          (0x0F)  ///< CPLD EEPROM configuration program register      
                                                                             
#define EVM_CPLD_EEPROM_EEWRITE_MASK     (0x01)  ///< EEWRITE eeprom write control Mask           
#define EVM_CPLD_EEPROM_EEWRITE_ACT      (0x01)  ///< EEWRITE 0->1 transition write start          
#define EVM_CPLD_EEPROM_EEWRITE_REL      (0x00)  ///< EEWRITE no write           
                                                                             
#define EVM_CPLD_EEPROM_EEBUSY_MASK      (0x02)  ///< EEBUSY eeprom write in progress Mask           
#define EVM_CPLD_EEPROM_EEBUSY_ACT       (0x02)  ///< EEBUSY eeprom write in progress          
#define EVM_CPLD_EEPROM_EEBUSY_REL       (0x00)  ///< EEBUSY eeprom write completed           
                                                                             
#define EVM_CPLD_EEPROM_EEERROR_MASK     (0x04)  ///< EEERROR eeprom write error Mask           
#define EVM_CPLD_EEPROM_EEERROR_ACT      (0x04)  ///< EEERROR eeprom write error          
#define EVM_CPLD_EEPROM_EEERROR_REL      (0x00)  ///< EEERROR eeprom write no error           
//@}

/** @name SRIO Switch Configuration Register 1
 *  Reg 16 : SRIO Switch Configuration Register 1 
 */
//@{                                      
#define EVM_CPLD_SRIO_SWITCH_CONFIG_1         (0x10)  ///< SRIO Switch Configuration Register 1      
                                                                             
#define EVM_CPLD_SRIO_SWITCH_PWRDN_1_MASK     (0x01)  ///< Port 1 power down           
#define EVM_CPLD_SRIO_SWITCH_PWR_DOWN_1       (0x01)  ///< Port is powered down          
#define EVM_CPLD_SRIO_SWITCH_PWR_UP_1         (0x00)  ///< Port is powered up           
                                                                             
#define EVM_CPLD_SRIO_SWITCH_PWRDN_3_MASK     (0x02)  ///< Port 3 power down        
#define EVM_CPLD_SRIO_SWITCH_PWR_DOWN_3       (0x02)  ///< Port is powered down          
#define EVM_CPLD_SRIO_SWITCH_PWR_UP_3         (0x00)  ///< Port is powered up                
                                                                             
#define EVM_CPLD_SRIO_SWITCH_PWRDN_7_MASK     (0x04)  ///< Port 7 power down          
#define EVM_CPLD_SRIO_SWITCH_PWR_DOWN_7       (0x04)  ///< Port is powered down          
#define EVM_CPLD_SRIO_SWITCH_PWR_UP_7         (0x00)  ///< Port is powered up            

#define EVM_CPLD_SRIO_SWITCH_MODESEL_SP0_MASK (0x10)  ///< Port 0 mode selection (DSP 1)        
#define EVM_CPLD_SRIO_SWITCH_SP0_4X           (0x00)  ///< Port operating in 4x mode       
#define EVM_CPLD_SRIO_SWITCH_SP0_1X           (0x10)  ///< Port operating in 1x mode         
                                                                             
#define EVM_CPLD_SRIO_SWITCH_MODESEL_SP2_MASK (0x20)  ///< Port 2 mode selection (DSP 2)       
#define EVM_CPLD_SRIO_SWITCH_SP2_4X           (0x00)  ///< Port operating in 4x mode       
#define EVM_CPLD_SRIO_SWITCH_SP2_1X           (0x20)  ///< Port operating in 1x mode       
                                                                             
#define EVM_CPLD_SRIO_SWITCH_MODESEL_SP6_MASK (0x40)  ///< Port 6 mode selection (AMC connector)       
#define EVM_CPLD_SRIO_SWITCH_SP6_4X           (0x00)  ///< Port operating in 4x mode         
#define EVM_CPLD_SRIO_SWITCH_SP6_1X           (0x40)  ///< Port operating in 1x mode         
//@}

/** @name SRIO Switch Configuration Register 2
 *  Reg 17 : SRIO Switch Configuration Register 2 
 */
//@{                                      
#define EVM_CPLD_SRIO_SWITCH_CONFIG_2      (0x11)  ///< SRIO Switch Configuration Register 2      
                                                                               
#define EVM_CPLD_SRIO_SWITCH_TXSWAP_MASK   (0x01)  ///< TX pair swap          
#define EVM_CPLD_SRIO_SWITCH_TXSWAP_SWAP   (0x01)  ///< TX pairs are swapped         
#define EVM_CPLD_SRIO_SWITCH_TXSWAP_NOSWAP (0x00)  ///< TX pairs are not swapped         
                                                                            
#define EVM_CPLD_SRIO_SWITCH_RXSWAP_MASK   (0x02)  ///< RX pair swap         
#define EVM_CPLD_SRIO_SWITCH_RXSWAP_SWAP   (0x02)  ///< RX pairs are swapped         
#define EVM_CPLD_SRIO_SWITCH_RXSWAP_NOSWAP (0x00)  ///< RX pairs are not swapped                
                                                                            
#define EVM_CPLD_SRIO_SWITCH_SPEED_MASK    (0x0C)  ///< SRIO speed selection     
#define EVM_CPLD_SRIO_SWITCH_SPEED_1250    (0x00)  ///< 1.25Gbit/s
#define EVM_CPLD_SRIO_SWITCH_SPEED_2500    (0x04)  ///< 2.5Gbit/s
#define EVM_CPLD_SRIO_SWITCH_SPEED_3125    (0x08)  ///< 3.125Gbit/s (default)     
//@}

/** @name DSP Bootmode Options Configuration Register 1
 *  Reg 18 : DSP Bootmode Options Configuration Register 1 
 */
//@{                                      
#define EVM_CPLD_DSP_BOOT_CFG1               (0x12)  ///< DSP Bootmode Configuration Register 1      
                                                                               
#define EVM_CPLD_DSP_BOOT_DSP1_BTM1_CFG_MASK (0x0F)  ///< DSP1 Bootmode 1 options (i2c master)         
#define EVM_CPLD_DSP_BOOT_DSP2_BTM1_CFG_MASK (0xF0)  ///< DSP2 Bootmode 1 options (i2c master)
//@}

/** @name DSP Bootmode Options Configuration Register 2
 *  Reg 19 : DSP Bootmode Options Configuration Register 2 
 */
//@{                                      
#define EVM_CPLD_DSP_BOOT_CFG2               (0x13)  ///< DSP Bootmode Configuration Register 2      
                                                                               
#define EVM_CPLD_DSP_BOOT_DSP1_BTM4_CFG_MASK (0x0F)  ///< DSP1 Bootmode 4 options (hpi loading mode)         
//@}

/** @name DSP Bootmode Options Configuration Register 3
 *  Reg 20 : DSP Bootmode Options Configuration Register 3 
 */
//@{                                      
#define EVM_CPLD_DSP_BOOT_CFG3               (0x14)  ///< DSP Bootmode Configuration Register 3      
                                                                               
#define EVM_CPLD_DSP_BOOT_DSP1_BTM6_CFG_MASK (0x0F)  ///< DSP1 Bootmode 6 options (EMAC master)         
#define EVM_CPLD_DSP_BOOT_DSP2_BTM6_CFG_MASK (0xF0)  ///< DSP2 Bootmode 6 options (EMAC master)
//@}

/** @name DSP Bootmode Options Configuration Register 4
 *  Reg 21 : DSP Bootmode Options Configuration Register 4 
 */
//@{                                      
#define EVM_CPLD_DSP_BOOT_CFG4                  (0x15)  ///< DSP Bootmode Configuration Register 4      
                                                                               
#define EVM_CPLD_DSP_BOOT_DSP1_BTM1011_CFG_MASK (0x0F)  ///< DSP1 Bootmode 10 & 11 options (SRIO)         
#define EVM_CPLD_DSP_BOOT_DSP2_BTM1011_CFG_MASK (0xF0)  ///< DSP2 Bootmode 10 & 11 options (SRIO)
//@}
  

/****************************************************************************
 *                           Forward Declarations                           *
 ****************************************************************************/


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/

/****************************************************************************
 *   NAME : EVM_CPLD_writereg
 ************************************************************************//**
 *
 *   This function writes a value to a register of the CPLD device using the I2C bus.
 *
 *   @param [in] reg
 *      Register offset to write
 *  
 *   @param [in] val
 *      Value (byte) to write
 *
 *   @return
 *      !=EVM_CPLD_OK for failure, EVM_CPLD_OK for success
 *   
 *   @ingroup EVM_CPLD
 *
 ***************************************************************************/ 
Int32 EVM_CPLD_writereg(Uint8 reg, Uint8 val);
 
 
 /****************************************************************************
 *   NAME : EVM_CPLD_readreg
 ************************************************************************//**
 *
 *   This function reads the value of a register of the CPLD device using the I2C bus.
 *
 *   @param [in] reg
 *      Register offset to read
 *  
 *   @param [out] val
 *      Value (byte) read
 *
 *   @return
 *      !=EVM_CPLD_OK for failure, EVM_CPLD_OK for success
 *   
 *   @ingroup EVM_CPLD
 *
 ***************************************************************************/ 
Int32 EVM_CPLD_readreg(Uint8 reg, Uint8 *val );

 
  /****************************************************************************
 *   NAME : EVM_CPLD_readwritereg
 ************************************************************************//**
 *
 *   Read a CPLD register, change a bitfield and then writeback the register.
 *      
 *   Bitmask values and bitval are binary value with 1 set at the bit position
 *   to modify in the register.
 *   Pseudocode operation:
 *       1- Value1 = Read CPLD register regnum
 *       2- Value1 = Value1 & (~bitmask)
 *       3- Value1 = Value1 | (bitmask & bitval)
 *       4- Write ECP register regnum with Value1
 *
 *   @param [in] reg
 *      Register offset (8 bits).
 *
 *   @param [in] bitmask
 *     Bit mask value (8 bits) with bit to keep set as 1.
 *  
 *   @param [in] bitval
 *     Bit values (8 bits) to be set at mask position.
 *
 *   @return
 *      !=EVM_CPLD_OK for failure, EVM_CPLD_OK for success
 *   
 *   @ingroup EVM_CPLD
 *
 ***************************************************************************/   
Int32 EVM_CPLD_readwritereg( Uint8 reg, Uint8 bitmask, Uint8 bitval );


#ifdef __cplusplus
} 
#endif  // extern "C"

#endif //__EVM_CPLD_H__
   






































