/****************************************************************************
 *
 *
 *         **  **     **  ******  ********  ********  ********  **    **
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ************************************************************************//**
 *
 * @file       evm_gpio.h
 * 
 * @brief      EVM Utility functions for the control of the 
 *  
 *
 * (c) 2008, Lyrtech RD Inc. All rights reserved.              
 *
 * @ingroup     EVM_CPLD
 *
 ****************************************************************************
 *                                                                                          
 * "$Revision: 1.6 $"
 * "$Date: 2009/03/13 16:10:35 $"
 *
 ***************************************************************************/

#ifndef __EVM_GPIO_H__
#define __EVM_GPIO_H__

#ifdef __cplusplus
    extern "C" {
#endif // __cplusplus 


/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <tistdtypes.h>


/****************************************************************************
 *                             Local constants                              *
 ****************************************************************************/

/** @name GPIOs pins
 *  GPIOs of the DSP
 */
//@{   
#define EVM_GPIO_LENDIAN                0   ///< Endianness selection bootstrap
#define EVM_GPIO_BOOTMODE0              1   ///< Bootmode Pin 0
#define EVM_GPIO_BOOTMODE1              2   ///< Bootmode Pin 1
#define EVM_GPIO_BOOTMODE2              3   ///< Bootmode Pin 2
#define EVM_GPIO_BOOTMODE3              4   ///< Bootmode Pin 3
#define EVM_GPIO_DEVNUM0                5   ///< Device Number Pin 0
#define EVM_GPIO_DEVNUM1                6   ///< Device Number Pin 1
#define EVM_GPIO_DEVNUM2                7   ///< Device Number Pin 2
#define EVM_GPIO_DEVNUM3                8   ///< Device Number Pin 3
#define EVM_GPIO_RESERVED               9   ///< Reserved
#define EVM_GPIO_ID                     10  ///< Device ID (DSP 1 or 2)
#define EVM_GPIO_USERLED                11  ///< User LED
#define EVM_GPIO_MDIOSEL                12  ///< MDIO selection
#define EVM_GPIO_MDIOACK                13  ///< MDIO arbitration acknowledgement from CPLD 
#define EVM_GPIO_HPIINT_USER            14  ///< HPI Interrupt for DSP 2 and User GPIO for DSP 1
#define EVM_GPIO_SWITCHI2CEN_USER       15  ///< SRIO Switch eeprom control for DSP 2 and User GPIO for DSP 1
//@}

/** @name GPIOs errors
 *  Error codes for the GPIO module
 */
//@{ 
#define EVM_GPIO_NO_ERROR               0x0   ///< No error
#define EVM_GPIO_OPEN_ERROR             0x101 ///< Handle opening error               
#define EVM_GPIO_INIT_ERROR             0x102 ///< Module initialization error               
#define EVM_GPIO_GET_ERROR              0x103 ///< Pin read error               
#define EVM_GPIO_SET_ERROR              0x104 ///< Pin set error  
#define EVM_GPIO_CLEAR_ERROR            0x105 ///< Pin clear error                 
#define EVM_GPIO_ID_ERROR               0x106 ///< Device ID error        
#define EVM_GPIO_WRONG_ARGUMENT_ERROR   0x107 ///< Invalid argument given to function   
#define EVM_GPIO_CONFIG_ERROR           0x108 ///< Pin Config error
#define EVM_GPIO_NOT_IN_TESTMODE_ERROR  0x109 ///< Function not possible when not in test mode
//@}     
                                              
/** @name GPIO Default                       
 *  GPIO Default Configurations
 */
//@{ 
#define EVM_GPIO_DEFAULT_DIR    0xFFFFE7FF  ///<  GP11 UserLed and GP12 MDIOSel are outputs
#define EVM_GPIO_TSTMODE_DIR    0xFFFFE400  ///<  GP11 UserLed and GP12 MDIOSel, GPIO 00-09 are outputs
#define EVM_GPIO_DEFAULT_OUT    0x00000000  ///<  GP11 and GP12 default to LOW
//@}

/** @name GPIO Id Values
 *  Different IDs available for the DSP
 */
//@{
#define EVM_GPIO_DSP1   1    ///< Value of DSP1 ID 
#define EVM_GPIO_DSP2   0    ///< Value of DSP2 ID 
//@}

/** @name GPIO Mdio Status
 *  Mdio statuses 
 */
//@{
#define EVM_GPIO_MDIO_GRANTED       1    ///< MDIO bus is property of the DSP
#define EVM_GPIO_MDIO_UNAVAILABLE   0    ///< MDIO bus could not be granted
//@}

/** @name GPIO Led States
 *  Led States available
 */
//@{
#define EVM_GPIO_LED_ON     1    ///< LED is lit
#define EVM_GPIO_LED_OFF    0    ///< LED is not lit
//@}

/** @name  Hpi Host Interrupt GPIO
 *  GPIO
 */
//@{
#define EVM_GPIO_INT_TRUE     1    ///< Host is being interrupted
#define EVM_GPIO_INT_FALSE    0    ///< Host is not being interrupted
//@}

/** @name GPIO test mode states
 *  GPIO diagnostic test modes available states
 */
//@{
#define EVM_GPIO_TESTMODE_TRUE  1   ///< GPIO cpld loopback test mode
#define EVM_GPIO_TESTMODE_FALSE 0   ///< GPIO normal mode
//@}

/** @name GPIO Eeprom Control States
 *  SRIO Switch Eeprom control states available
 */
//@{
#define EVM_GPIO_EECTL_TRUE     1    ///< Eeprom is accessible by DSP
#define EVM_GPIO_EECTL_FALSE    0    ///< Eeprom is not accessible by DSP
//@}


/****************************************************************************
 *                           Forward Declarations                           *
 ****************************************************************************/


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/


/****************************************************************************
 *   NAME : EVM_GPIO_init
 ************************************************************************//**
 *
 *   This function will initialize all settings for the GPIO module. 
 *   It must be usually the first function called when using this module.
 *
 *   @return
 *      Error Code
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_init (void);


 /****************************************************************************
 *   NAME : EVM_GPIO_cpuid 
 ************************************************************************//**
 *
 *   This function will return which cpu role the current DSP is. Each dsp has 
 *   a different role on the evm and cannot do the same functions. This identification 
 *   allows the program to know the possible functions that can be done on a specific dsp. 
 *   See Figure 1: EVM simplified block diagram for dsp
 *
 *   @param [out] cpuid
 *      ID read on the board (EVM_GPIO_DSP1 or EVM_GPIO_DSP2)
 *  
 *   @return
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_cpuid (Int32 *cpuid);


 /****************************************************************************
 *   NAME : EVM_GPIO_askformdio
 ************************************************************************//**
 *
 *   This function will ask to control the mdio ethernet phy resource. 
 *   It will return the status if the resource was granted or not. 
 *   If the resource is held by the other dsp, the other dsp must release 
 *   it before it is available.
 *
 *   @param [out] granted
 *      Status if the resource was granted or not 
 *      (EVM_GPIO_MDIO_GRANTED or EVM_GPIO_MDIO_UNAVAILABLE)
 *  
 *   @return
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_askformdio (Int32 *granted);


/****************************************************************************
 *   NAME : EVM_GPIO_releasemdio
 ************************************************************************//**
 *
 *   This function will release the control of the mdio ethernet phy resource 
 *   if it is controlled by the current dsp. This function is used to release 
 *   control acquiered by using function EVM_GPIO_askformdio.
 *  
 *   @return
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_releasemdio (void);


 /****************************************************************************
 *   NAME : EVM_GPIO_getmdiostate
 ************************************************************************//**
 *
 *   This function will return the status if the current dsp is having exclusive 
 *   mdio resource or not (by using EVM_GPIO_askformdio function) .
 *
 *   @param [out] state
 *      Current mdio control state of this resource by the dsp. 
 *      (EVM_GPIO_MDIO_GRANTED or EVM_GPIO_MDIO_UNAVAILABLE)
 *  
 *   @return
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_getmdiostate (Int32 *state);


 /****************************************************************************
 *   NAME : EVM_GPIO_setdspledstate
 ************************************************************************//**
 *
 *   This function will control private DSP LED.
 *
 *   @param [in] ledstate
 *      Status of the led to be set. (EVM_GPIO_LED_ON, EVM_GPIO_LED_OFF)
 *  
 *   @return
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_setdspledstate (Int32 ledstate);


 /****************************************************************************
 *   NAME : EVM_GPIO_getdspledstate
 ************************************************************************//**
 *
 *   This function will return state of private DSP LED
 *
 *   @param [out] ledstate
 *      Status of the led. (EVM_GPIO_LED_ON, EVM_GPIO_LED_OFF)
 *  
 *   @return
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_getdspledstate (Int32 *ledstate);


 /****************************************************************************
 *   NAME : EVM_GPIO_setdccusergpio
 ************************************************************************//**
 *
 *   This function will control user gpio set as outputs 
 *   (GP14, GP15 for dsp1 and GP15 for dsp2) available on the DCC connector.
 *
 *   @param [in] iostate
 *      Each available bit position set to 1 will output a 1 on the pin, 
 *      set to 0 will output a 0 on the DCC pins.
 *  
 *   @return
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_setdccusergpio (Int32 iostate);


 /****************************************************************************
 *   NAME : EVM_GPIO_getdccusergpio
 ************************************************************************//**
 *
 *   This function will read user gpio set as inputs 
 *   (GP14, GP15 for dsp1 and GP15 for dsp2) available on the DCC connector.
 *
 *   @param [in] iostate
 *      Each available bit position set to 1 will output a 1 on the pin, 
 *      set to 0 will output a 0 on the DCC pins.
 *  
 *   @return
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_getdccusergpio (Int32 *iostate);


 /****************************************************************************
 *   NAME : setdccusergpiodir 
 ************************************************************************//**
 *
 *   This function will control user gpio direction as outputs/inputs 
 *   (GP14, GP15 for dsp1 and GP15 for dsp2) available on the DCC connector.
 *
 *   @param [in] iodir
 *      Each available bit position set to 1 will be input, set to 0 will 
 *      be output on the DCC pins.
 *  
 *   @return
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_setdccusergpiodir (Int32 iodir);


 /****************************************************************************
 *   NAME : EVM_GPIO_gethpiint 
 ************************************************************************//**
 *
 *   This function will return the state of the HPI interrupt from dsp1 while 
 *   dsp2 is using hpi to send data to dsp1. 
 *   This function is available for dsp2 only.
 *
 *   @param [out] intstate
 *      State of the HPI interrupt GPIO (EVM_GPIO_INT_TRUE, EVM_GPIO_INT_FALSE)
 *  
 *   @return
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_gethpiint(Int32 *intstate);  


 /****************************************************************************
 *   NAME : EVM_GPIO_settstmode 
 ************************************************************************//**
 *
 *   This function is used for diagnostic only. It is used to set the gpio in
 *   special test mode loopback with the cpld. Selects test mode only for
 *   diagnostic purpose, else use normal mode.
 *
 *   @param [in] tstmode
 *      Mode flag for GPIO test (EVM_GPIO_TESTMODE_TRUE, EVM_GPIO_TESTMODE_FALSE)
 *  
 *   @return
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_settstmode(Int32 tstmode);


 /****************************************************************************
 *   NAME : EVM_GPIO_settstio 
 ************************************************************************//**
 *
 *   This function is used for diagnostic only. It is used to change the gpio
 *   in special test mode loopback with the cpld. Before using this fonction,
 *   configures special test mode with EVM_GPIO_settstmode.
 *
 *   @param [in] ionumber
 *      ionumber to use from 0 to 9
 *  
 *   @param [in] iostate
 *      iostate to set (0 or 1)
 *  
 *   @return
 *   
 *   @ingroup EVM_GPIO
 *
 ***************************************************************************/ 
Int32 EVM_GPIO_settstio(Int32 ionumber, Int32 iostate);


#ifdef __cplusplus
}
#endif //__cplusplus
#endif //__EVM_GPIO_H__


   
