/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ************************************************************************//**
 *
 * @file       evm.h
 * 
 * @brief      Utility functions for EVM board configuration
 *  
 *
 * (c) 2008, Lyrtech RD Inc. All rights reserved.              
 *
 * @ingroup     EVM
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.29 $"
 * "$Date: 2009/09/16 16:39:02 $"
 *
 ***************************************************************************/

#ifndef __EVM_H__ 
#define __EVM_H__


#ifdef __cplusplus
    extern "C" {
#endif // __cplusplus 

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <tistdtypes.h>


/****************************************************************************
 *                             Global constants                             *
 ****************************************************************************/

#define EVM_BSLVERSION "V1.1.0.0"   ///< EVM BSL VERSION
 

/** @name EVM error codes
 *  Evm functions error status codes 
 */
//@{
#define EVM_INITOK               0      ///< No error
#define EVM_INCORRECT_VERSION    0x0501 ///< Incorrect binary bsl version not matching .h
#define EVM_INCORRECT_DSPID      0x0502 ///< This function cannot be run on this DSP
#define EVM_ERRORREAD_VERSION    0x0503 ///< Problem while reading evm versions
#define EVM_INCOMPATIBLE_VERSION 0x0504 ///< Board/firmware version is incompatible with this function
#define EVM_ERRORREAD_PROG       0x0505 ///< Error while reading eeprom programming register
#define EVM_EEPROMPROG_ERR       0x0506 ///< Eeprom configuration programming/verification error
#define EVM_EEPROMPROG_TO        0x0507 ///< Eeprom configuration programming timeout error
#define EVM_ERRORWRITE_EEREG1    0x0508 ///< Problem #1 writing eeprom configuration programming register
#define EVM_ERRORWRITE_EEREG2    0x0509 ///< Problem #2 writing eeprom configuration programming register
#define EVM_INCOMPATIBLE_SAVECFG 0x050A ///< Board/firmware version is incompatible and cannot save in eeprom config
#define EVM_ERRORWRITE_SRIORST1  0x050B ///< Problem cannot release srio reset
#define EVM_ERRORWRITE_SRIORST2  0x050C ///< Problem cannot activates srio reset
#define EVM_ERRORWRITE_CFG2      0x050D ///< Problem cannot write config register #2
#define EVM_ERRORWRITE_CFG1      0x050E ///< Problem cannot write config register #1
#define EVM_ERRORWRITE_BOOTCFG1  0x050F ///< Problem cannot write eeprom boot config register1
#define EVM_ERRORWRITE_BOOTCFG2  0x0510 ///< Problem cannot write eeprom boot config register2
#define EVM_ERRORWRITE_BOOTCFG3  0x0511 ///< Problem cannot write eeprom boot config register3
#define EVM_ERRORWRITE_BOOTCFG4  0x0512 ///< Problem cannot write eeprom boot config register4
#define EVM_TIMEOUT_ERROR        0x0513 ///< Timeout occured in DSP initialization
#define EVM_ERROR_PSC_INIT       0x0514 ///< Timeout occured at DSP PSC initialization
#define EVM_PLLMULT_ERROR        0x0515 ///< Unsupported pll1 multiplier was used
//@}

/** @brief Enum for SRIO port mode eeprom configuration */
typedef enum
{
    /** SRIO port operating in 4X port mode (default) */
    EVM_SRIO_PORT_MODE_4X = 0,

    /** SRIO port operating in 1X port mode */
    EVM_SRIO_PORT_MODE_1X = 1
} EVM_SrioPortMode;

/** @brief Enum for SRIO port lane mode eeprom configuration (global for all evm ports) */
typedef enum
{
    /** SRIO all ports lane in normal (not swapped) ABCD mode (default) */
    EVM_SRIO_LANE_MODE_ABCD = 0,

    /** SRIO all ports lane in swapped DCBA mode */
    EVM_SRIO_LANE_MODE_DCBA = 1
} EVM_SrioLaneMode;

/** @brief Enum for SRIO port speed eeprom configuration (global for all evm ports) */
typedef enum
{
    /** SRIO all ports speed operating at 1.25 Gbits/sec */
    EVM_SRIO_SPEED_MODE_1_25G = 0,

    /** SRIO all ports speed operating at 2.5 Gbits/sec */
    EVM_SRIO_SPEED_MODE_2_50G = 1,

    /** SRIO all ports speed operating at 3.125 Gbits/sec (default) */
    EVM_SRIO_SPEED_MODE_3_125G = 2
} EVM_SrioSpeedMode;

/** @brief Enum for DSP Boot mode eeprom optionnal parameter */
typedef enum
{
    /** DSP boot parameter option 0 */
    EVM_BOOTPARAM_OPT_00 = 0,

    /** DSP boot parameter option 1 */
    EVM_BOOTPARAM_OPT_01 = 1,

    /** DSP boot parameter option 2 */
    EVM_BOOTPARAM_OPT_02 = 2,

    /** DSP boot parameter option 3 */
    EVM_BOOTPARAM_OPT_03 = 3,

    /** DSP boot parameter option 4 */
    EVM_BOOTPARAM_OPT_04 = 4,

    /** DSP boot parameter option 5 */
    EVM_BOOTPARAM_OPT_05 = 5,

    /** DSP boot parameter option 6 */
    EVM_BOOTPARAM_OPT_06 = 6,

    /** DSP boot parameter option 7 */
    EVM_BOOTPARAM_OPT_07 = 7,

    /** DSP boot parameter option 8 */
    EVM_BOOTPARAM_OPT_08 = 8,

    /** DSP boot parameter option 9 */
    EVM_BOOTPARAM_OPT_09 = 9,

    /** DSP boot parameter option 10 */
    EVM_BOOTPARAM_OPT_10 = 10,

    /** DSP boot parameter option 11 */
    EVM_BOOTPARAM_OPT_11 = 11,

    /** DSP boot parameter option 12 */
    EVM_BOOTPARAM_OPT_12 = 12,

    /** DSP boot parameter option 13 */
    EVM_BOOTPARAM_OPT_13 = 13,

    /** DSP boot parameter option 14 */
    EVM_BOOTPARAM_OPT_14 = 14,

    /** DSP boot parameter option 15 */
    EVM_BOOTPARAM_OPT_15 = 15
}EVM_BootParam;

/** @brief Enum for DSP HPI Boot mode eeprom optionnal parameter */
typedef enum
{
    /** DSP hpi boot direct loading mode */
    EVM_HPIBOOTPARAM_DIRECTLOAD = 0,

    /** DSP hpi boot table loading mode */
    EVM_HPIBOOTPARAM_TABLEMODE = 2
}EVM_HpiBootParam;

// These are two possibles configuration for SRIO dsp bootmode

/** Default SRIO 4x fast 3.125Gbits/sec powerup evm configuration parameters (bootable) */
#define EVM_POWERUPCONFIGOPT_DEFAULT   { \
    EVM_SRIO_PORT_MODE_4X, \
    EVM_SRIO_PORT_MODE_4X, \
    EVM_SRIO_PORT_MODE_4X, \
    EVM_SRIO_LANE_MODE_ABCD, \
    EVM_SRIO_LANE_MODE_ABCD, \
    EVM_SRIO_SPEED_MODE_3_125G, \
    EVM_BOOTPARAM_OPT_00, \
    EVM_BOOTPARAM_OPT_00, \
    EVM_HPIBOOTPARAM_DIRECTLOAD, \
    EVM_BOOTPARAM_OPT_00, \
    EVM_BOOTPARAM_OPT_01, \
    EVM_BOOTPARAM_OPT_00, \
    EVM_BOOTPARAM_OPT_00 \
}

/** SRIO slow 1x 1.25Gbits/sec powerup evm configuration parameters (bootable) */
#define EVM_POWERUPCONFIGOPT_SLOW1X   { \
    EVM_SRIO_PORT_MODE_1X, \
    EVM_SRIO_PORT_MODE_1X, \
    EVM_SRIO_PORT_MODE_1X, \
    EVM_SRIO_LANE_MODE_ABCD, \
    EVM_SRIO_LANE_MODE_ABCD, \
    EVM_SRIO_SPEED_MODE_1_25G, \
    EVM_BOOTPARAM_OPT_00, \
    EVM_BOOTPARAM_OPT_00, \
    EVM_HPIBOOTPARAM_DIRECTLOAD, \
    EVM_BOOTPARAM_OPT_00, \
    EVM_BOOTPARAM_OPT_01, \
    EVM_BOOTPARAM_OPT_00, \
    EVM_BOOTPARAM_OPT_00 \
}

 /****************************************************************************
 * typedef struct : EVM_PowerupConfigOpt
 ************************************************************************//**
 *
 *    Configuration structure used for evm powerup options
 *   
 *    @ingroup EVM
 *
 ***************************************************************************/
typedef struct
{
    EVM_SrioPortMode SrioDsp1Mode;     ///< Dsp1 SRIO port mode 1x or 4x (default)
    EVM_SrioPortMode SrioDsp2Mode;     ///< Dsp2 SRIO port mode 1x or 4x (default)
    EVM_SrioPortMode SrioAMCMode;      ///< AMC connector SRIO port mode 1x or 4x (default)
    EVM_SrioLaneMode SrioTxLaneSwap;   ///< DSP1, DSP2 and AMC SRIO Transmit lanes swap DCBA or ABCD (default)
    EVM_SrioLaneMode SrioRxLaneSwap;   ///< DSP1, DSP2 and AMC SRIO Receive lanes swap DCBA or ABCD (default)
    EVM_SrioSpeedMode SrioSpeed;       ///< DSP1, DSP2 and AMC SRIO ports speed global (default 3.125 Gbits/sec)
    EVM_BootParam BootDsp1I2CMasterOpt;///< DSP1 bootmode 1 i2c master table address
                                       //   First eeprom boot parameter block address = BootDsp1I2CMasterOpt*0x80
    EVM_BootParam BootDsp2I2CMasterOpt;///< DSP2 bootmode 1 i2c master table address
                                       //   First eeprom boot parameter block address = BootDsp2I2CMasterOpt*0x80
    EVM_HpiBootParam BootDsp1HpiLoadOpt;///< DSP1 bootmode 4 hpi loading mode (direct or table)
    EVM_BootParam BootDsp1EmacMasterID;///< DSP1 bootmode 6 emac master device ID
    EVM_BootParam BootDsp2EmacMasterID;///< DSP2 bootmode 6 emac master device ID
    EVM_BootParam BootDsp1SrioID;      ///< DSP1 bootmode 10 & 11 srio device ID
                                       //   Device ID = BootDsp1SrioID +2
    EVM_BootParam BootDsp2SrioID;      ///< DSP2 bootmode 10 & 11 srio device ID
                                       //   Device ID = BootDsp2SrioID +2
}EVM_PowerupConfigOpt; 

/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/

 /****************************************************************************
 *   NAME : EVM_init(void)
 ************************************************************************//**
 *
 *   Init the evm configuration and resources for modules. This function
 *      must be the first function always use before using any other function
 *      of the BSL. It will initialize also chip CSL interrupt settings, so
 *      it must be called first before using any interruption through
 *      the CSL outside of the BSL to avoid unexpected behaviors.
 *  
 *   @return   Error code
 *   
 *   @ingroup EVM
 *
 ***************************************************************************/ 
Int32 EVM_init(void);


 /****************************************************************************
 *   NAME : EVM_bslversion(void)
 ************************************************************************//**
 *
 *   Get the evm bsl version. It can be used to display the binary library
 *      version of the bsl.
 *      See function EVM_checkbslversion.
 *  
 *   @return   Character string holding the bsl version
 *   
 *   @ingroup EVM
 *
 ***************************************************************************/ 
Int8 *EVM_bslversion(void);


 /****************************************************************************
 *   NAME : EVM_checkbslversion(Int8 *expectedversion)
 ************************************************************************//**
 *
 *   Verify that we have a correct binary bsl version matching .h definitions
 *      Call this function with the constant EVM_BSLVERSION as input
 *      parameters to get a version check status.
 *      EVM_BSLVERSION is in the evm.h file.
 *
 *   @param [in] expectedversion
 *      Character string holding the expected bsl version
 *  
 *   @return   Error code: EVM_INITOK if no problem is found, !EVM_INITOK if problem
 *   
 *   @ingroup EVM
 *
 ***************************************************************************/ 
Int32 EVM_checkbslversion(Int8 *expectedversion);


 /****************************************************************************
 *   NAME : EVM_preinit(Uint32 cachel1d_cfg, Uint32 cachel1p_cfg, Uint32 cachel2_cfg, Uint32 pll1_multv)
 ************************************************************************//**
 *
 *   Basic pre-initialization for DSP (replacing GEL file initialization).
 *   This function is not necessary when the evm is used from CCS
 *   with a JTAG debugger. In this situation, dsp default initialization
 *   would be provided by the GEL file.
 *   If this function is needed, it must be called as the first function of
 *   the BSL before the normal EVM_init call. It will replace GEL initialization
 *   that is normally done first, before evm_init is called in the user program.
 *   List of initializations performed:
 *    1- DSP Pll1 is set according to pll1_multv parameter (20 for 1000 MHz = 20*50)
 *    2- DSP Turn on all power domains
 *    3- DSP Set DDR2 for 333 MHz
 *    4- DSP Set EMIF for hpi (waitstates for 100 MHz EMIF)
 *    5- DSP configure L1D, L1P and L2 cache memory size according to parameters
 *       (values are direct values associated with dsp cache setup registers)
 *
 *    WARNINGS:
 *    - This function does not validate the parameters given.
 *    - Using this function while application is running from DDR2 may
 *      give unexpected results (possible program crash)
 *    - Memory for a specific internal memory section must not be
 *      configured as cache if the application program is using it for code/data ram.
 *    - This function is including embedded software loops, and have not been
 *      verified to run outside of an unitialized cpu pll (at 50 MHz) and without
 *      cpu cache activated, before calling this function.
 *
 *   Typical example for DSP running at 1000 MHz, cache L1D = L1P = 32K, L2 All ram:
 *      EVM_preinit(7,7,0,20);
 *
 *   @param [in] cachel1d_cfg
 *      L1D cache size dsp reg (7 = 32K , 0 = 0K)
 *
 *   @param [in] cachel1p_cfg
 *      L1P cache size dsp reg (7 = 32K , 0 = 0K)
 *
 *   @param [in] cachel2_cfg
 *      L2 cache size dsp reg  (7 = 3/4 RAM , 0 = ALL RAM)
 *
 *   @param [in] pll1_multv
 *      Core pll dsp multiplier reg +1 (20 for 1000 MHz = 20*50 MHz)
 *  
 *   @return   Error code: EVM_INITOK if no problem is found, !EVM_INITOK if problem
 *   
 *   @ingroup EVM
 *
 ***************************************************************************/ 
Int32 EVM_preinit(Uint32 cachel1d_cfg, Uint32 cachel1p_cfg, 
                  Uint32 cachel2_cfg, Uint32 pll1_multv);

                  
 /****************************************************************************
 *   NAME : EVM_setpowerupconfig(EVM_PowerupConfigOpt Config)
 ************************************************************************//**
 *
 *   Modifies the evm powerup configuration in eeprom memory. This configuration
 *      is used to override some evm factory default options. It can be used to
 *      change Serial Rapid IO (SRIO) parameters and dsp bootmode options
 *      parameters. To activate this override configuration after this change,
 *      the EVM must be powerup with the correct dip sw to override factory default
 *      eeprom configuration (SW5-1 OFF). To revert to factory default, the EVM
 *      must be powerup with SW5-1 ON. It should be noted that only DSP 2 can
 *      use this function to configure the EVM. If needed, usually it would be done
 *      one time to configure EVM for user requierements. This function is not
 *      available for Variant10 boards.
 *
 *   @param [in] Config
 *      Powerup configuration structure
 *  
 *   @return   Error code: EVM_INITOK if no problem is found, !EVM_INITOK if problem
 *   
 *   @ingroup EVM
 *
 ***************************************************************************/ 
Int32 EVM_setpowerupconfig(EVM_PowerupConfigOpt *Config);

                             
#ifdef __cplusplus
} 
#endif  // extern "C"                        
#endif // __EVM_H__
