/****************************************************************************
 *
 *
 *         **  **     **  ******  ********  ********  ********  **    **
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ************************************************************************//**
 *
 * @file       evm_eeprom.h
 * 
 * @brief      EVM Utility functions for writing/reading of the EVM's Eeproms
 *  
 *
 * (c) 2008, Lyrtech RD Inc. All rights reserved.              
 *
 * @ingroup     EVM_EEPROM
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.10 $"
 * "$Date: 2009/03/13 16:10:35 $"
 *
 ***************************************************************************/

#ifndef __EVM_EEPROM_H__
#define __EVM_EEPROM_H__

#ifdef __cplusplus
    extern "C" {
#endif // __cplusplus 


/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/
#include <tistdtypes.h>

/****************************************************************************
 *                             Local constants                              *
 ****************************************************************************/



/** @name EVM_EEPROM_error_codes
 *  EEPROM functions error status codes.
 */
//@{
#define EVM_EEPROM_NO_ERROR                 0x0   ///< No error
#define EVM_EEPROM_WRITE_ERROR            0x201   ///< Eeprom Write error
#define EVM_EEPROM_READ_ERROR             0x202   ///< Eeprom read error
#define EVM_EEPROM_VERIF_ERROR            0x203   ///< Eeprom program and verify mode error
#define EVM_EEPROM_BLOCK_TO_SMALL         0x204   ///< Block is to small for buffer
#define EVM_EEPROM_BLOCK_NOT_ACCCESSIBLE  0x205   ///< Block is not accessible to user
#define EVM_EEPROM_MEMALLOC_ERROR         0x206   ///< Memory allocation for verification buffer is impossible
#define EVM_EEPROM_BLOCK_INVALID          0x207   ///< Block is not a valid eeprom zone
#define EVM_EEPROM_WRITEBLOCK_TIMEOUT     0x208   ///< Eeprom write block failed after timeout
//@}

 /****************************************************************************
 * typedef enum : EVM_EEPROM_ZoneCode
 ************************************************************************//**
 *
 *    User data zones
 *   
 *    @ingroup EVM_EEPROM
 *
 ***************************************************************************/
typedef enum
{
    EVM_EEPROM_DSP_DEDICATED_ZONE       = 0x0,  ///< DSP Eeprom Dedicated Zone                                                                                                                            
    EVM_EEPROM_CPLD1_CONFIG_ZONE        = 0x1,  ///< CPLD Eeprom 1 Boot Configuration Zone                                                                                                                           
    EVM_EEPROM_DSP_USER_ZONE            = 0x4,  ///< DSP Eeprom User Zone                                       
    EVM_EEPROM_CPLD1_USER_ZONE          = 0x5   ///< CPLD Eeprom 1 User Zone                                                                            
}EVM_EEPROM_ZoneCode;                      

 /****************************************************************************
 * typedef enum : EVM_EEPROM_Mode
 ************************************************************************//**
 *
 *    Eeprom Programming modes
 *   
 *    @ingroup EVM_EEPROM
 *
 ***************************************************************************/
typedef enum
{
    EVM_EEPROM_PROGRAM          = 0x00,        ///< Program only
    EVM_EEPROM_PROGRAMANDVERIFY = 0x01         ///< Program and verify content 
}EVM_EEPROM_Mode; 

                                
/****************************************************************************
 *                           Forward Declarations                           *
 ****************************************************************************/


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/


/****************************************************************************
 *   NAME : EVM_EEPROM_init
 ************************************************************************//**
 *
 *   This function will initialize all settings for the eeprom module. 
 *   It must be usually the first function called when using this module.
 *
 *   @return
 *      
 *   
 *   @ingroup EVM_EEPROM
 *
 ***************************************************************************/ 
Int32 EVM_EEPROM_init (void);


 /****************************************************************************
 *   NAME : EVM_EEPROM_writeuserblock
 ************************************************************************//**
 *
 *   This function will write a block of data in the specified user data zone of 
 *   a eeprom identified by the zonecode. The block written has a specific len, 
 *   will be written at specified offset (byte offset) in eeprom memory zone, 
 *   and data source from buffer. The function can also verify after programming 
 *   as specified by the mode. This function is safe to write any user data in the 
 *   specified eeprom zone, cause it will not allow to override system configuration 
 *   information.
 *
 *   @param [in] zonecode
 *      Code enum to select the correct eeprom user data zone. 
 *
 *   @param [in] buffer
 *      Pointer on data buffer to be programmed
 *
 *   @param [in] offset
 *      Eeprom zone offset to begin programming.
 *
 *   @param [in] len
 *      Number of bytes to be programmed from the data buffer.
 *
 *   @param [in] mode
 *      Programming mode (program or program/verify)
 *  
 *   @param [out] VerifyBuffer
 *      Pointer on data buffer needed in program/verify mode. It must be
 *      as big as input buffer. It can be NULL if not used.
 *  
 *   @return
 *   
 *   @ingroup EVM_EEPROM
 *
 ***************************************************************************/ 
Int32 EVM_EEPROM_writeuserblock (EVM_EEPROM_ZoneCode zonecode,Int8 *buffer, 
                                 Int32 offset, Int32 len, EVM_EEPROM_Mode mode, 
                                 Int8 *Verifybuffer);
 
                            
/****************************************************************************
 *   NAME : EVM_EEPROM_readuserblock
 ************************************************************************//**
 *
 *   This function will read a block of data from the specified user data zone 
 *   of a eeprom identified by the zonecode. The block read has a specific len, 
 *   will be read from specified offset (byte offset) in eeprom memory zone, 
 *   and data destination to buffer. The provided buffer must be large enough 
 *   to hold the specified byte len.
 *
 *   @param [in] zonecode
 *      Code enum to select the correct eeprom user data zone.
 *
 *   @param [in] buffer
 *      Pointer on data buffer to store the read data
 *
 *   @param [in] offset
 *      Eeprom zone offset to begin reading.
 *
 *   @param [in] len
 *      Number of bytes to be read from the data buffer.
 *  
 *   @return
 *   
 *   @ingroup EVM_EEPROM
 *
 ***************************************************************************/                       
Int32 EVM_EEPROM_readuserblock (EVM_EEPROM_ZoneCode zonecode, Int8 *buffer, 
                                                    Int32 offset, Int32 len);
  
                                                    
/****************************************************************************
 *   NAME : EVM_EEPROM_getuserblockinfo
 ************************************************************************//**
 *
 *   This function will returns specified zone information structure. 
 *   This function may be used while using functions EVM_EEPROM_readuserblock 
 *   and EVM_EEPROM_writeuserblock to get information as an example of the size 
 *   of a specific eeprom user data zone.
 *
 *   @param [in] zonecode
 *      Zone code enum to select the correct eeprom user data zone.
 *
 *   @param [in] iSize
 *      Pointer to the size of the block
 *  
 *   @return
 *   
 *   @ingroup EVM_EEPROM
 *
 ***************************************************************************/ 
Int32 EVM_EEPROM_getuserblockinfo (EVM_EEPROM_ZoneCode zonecode, Int32 *iSize);


 /****************************************************************************
 *   NAME : EVM_EEPROM_writedspconfigblock
 ************************************************************************//**
 *
 *   This function will write a block of data in the dsp config data test zone of 
 *   dsp eeprom. User must not write data in this zone because it will be overwritten.
 *   The block written has a specific len, will be written at specified offset
 *   (byte offset) in eeprom memory zone, and data source from buffer. The function
 *   can also verify after programming as specified by the mode.
 *
 *   @param [in] buffer
 *      Pointer on data buffer to be programmed
 *
 *   @param [in] offset
 *      Eeprom zone offset to begin programming.
 *
 *   @param [in] len
 *      Number of bytes to be programmed from the data buffer.
 *
 *   @param [in] mode
 *      Programming mode (program or program/verify)
 *  
 *   @param [out] VerifyBuffer
 *      Pointer on data buffer needed in program/verify mode. It must be
 *      as big as input buffer. It can be NULL if not used.
 *  
 *   @return
 *   
 *   @ingroup EVM_EEPROM
 *
 ***************************************************************************/ 
Int32 EVM_EEPROM_writedspconfigblock (Int8 *buffer,Int32 offset, Int32 len,
                                      EVM_EEPROM_Mode mode, Int8 *Verifybuffer);


/****************************************************************************
 *   NAME : EVM_EEPROM_readdspconfigblock
 ************************************************************************//**
 *
 *   This function will read a block of data from the dsp data test zone 
 *   of dsp eeprom. This zone will be overriden for test purpose and is not
 *   user usable. The block read has a specific len, will be read from specified
 *   offset (byte offset) in eeprom memory zone, and data destination to buffer.
 *   The provided buffer must be large enough to hold the specified byte len.
 *
 *   @param [in] buffer
 *      Pointer on data buffer to store the read data
 *
 *   @param [in] offset
 *      Eeprom zone offset to begin reading.
 *
 *   @param [in] len
 *      Number of bytes to be read from the data buffer.
 *  
 *   @return
 *   
 *   @ingroup EVM_EEPROM
 *
 ***************************************************************************/                       
Int32 EVM_EEPROM_readdspconfigblock (Int8 *buffer,Int32 offset, Int32 len);


#ifdef __cplusplus
}
#endif //__cplusplus 
#endif //__EVM_EEPROM_H__
