/***************************************************************************
 *
 *
 *         **   **    **  ******  ********  ********  ********  **    **    
 *        **    **   **  **   ** ********  ********  ********  **    **
 *       **     *****   **   **    **     **        **        **    **
 *      **       **    ******     **     ****      **        ********
 *     **       **    **  **     **     **        **        **    **
 *    *******  **    **   **    **     ********  ********  **    **
 *   *******  **    **    **   **     ********  ********  **    **  
 *
 *            L Y R T E C H   S I G N A L   P R O C E S S I N G              
 *
 ***************************************************************************
 *                                                                          
 *  Project     : EVM
 *  File        : EVM_SERIALCOMM.h
 *  Description : Utility functions for sending and receiveing packets on the McBSP
 *
 *                      Copyright (c) Lyrtech inc. 2008                        
 *
 ***************************************************************************
 *                                                                                          
 * "$Revision: 1.13 $"
 * "$Date: 2009/02/27 19:38:53 $"
 *
 ***************************************************************************/

#ifndef __EVM_SERIALCOMM_H__
#define __EVM_SERIALCOMM_H__

#ifdef __cplusplus
    extern "C" {
#endif // __cplusplus 

/****************************************************************************
 *                                 Includes                                 *
 ****************************************************************************/


/****************************************************************************
 *                             Local constants                              *
 ****************************************************************************/

/** @name EVM serial comm error codes
 *  Serial comm functions error status codes.
 */
//@{
#define SERIALCOMM_PORT_0 0     ///< McBSP port 0
#define SERIALCOMM_PORT_1 1     ///< McBSP port 1 
//@}

/** @name EVM serial comm error codes
 *  Serial comm functions error status codes.
 */
//@{
#define SERIALCOMM_WRONG_INSTANCE         0x401      ///< Wrong instance of mcbsp selected  
#define SERIALCOMM_INIT_ERROR             0x403      ///< Initialization error
#define SERIALCOMM_TIMEOUT_ERROR          0x404      ///< Reception timeout error
#define SERIALCOMM_TX_MAXRETRIES_REACHED  0x405      ///< Maximum number of transmission retries reached
#define SERIALCOMM_RX_INVALID_PACKET      0x406      ///< The data packet is invalid/corrupted
#define SERIALCOMM_TX_INVALID_PACKET_SIZE 0x407      ///< The packet to sent is bigger than max possible size
#define SERIALCOMM_RX_USER_BUFFER_FULL    0x408      ///< The receiver user buffer is full
#define SERIALCOMM_RX_ACK_BUFFER_FULL     0x409      ///< The receiver ack buffer is full
#define SERIALCOMM_OK                     0          ///< No error
//@}


/****************************************************************************
 *                             Local variables                              *
 ****************************************************************************/

 
/****************************************************************************
 *                           Forward Declarations                           *
 ****************************************************************************/


/****************************************************************************
 *                             Private Functions                            *
 ****************************************************************************/


/****************************************************************************
 *                             Public Functions                             *
 ****************************************************************************/

 /****************************************************************************
 *   NAME : EVM_SERIALCOMM_init(Int32 iSerialPort)
 ************************************************************************//**
 *
 *   This function initializes a port of the McBSP for serial communication
 *   WARNING: actual implementation support to use this serial comm module
 *   on a single port at the time (on board port or external one), but not both
 *   at the same time.
 *   When configuring the McBSP, it will configure also 2 interrupts vectors
 *   needed for the serial communication:
 *       when using iSerialPort 0 (irq Vect_5 and Vect_6) for on board port,
 *    or when using iSerialPort 1 (irq Vect_7 and Vect_8) for external port.
 *
 *   @param [in] iSerialPort
 *      Port of the McBSP to be initialized
 *  
 *   @return
 *   
 *   @ingroup EVM_SERIALCOMM
 *
 ***************************************************************************/ 
Int32 EVM_SERIALCOMM_init(Int32 iSerialPort);


 /****************************************************************************
 *   NAME : EVM_SERIALCOMM_send(Int32 iSerialPort, Uint32 *pu32Data, Uint32 Size,
 *                              Uint32 MaxRetries)
 ************************************************************************//**
 *
 *   This function sets up a transmission and waits for its completion
 *
 *   @param [in] iSerialPort
 *      Port of the McBSP to be used
 *
 *   @param [in] pu32Data
 *      Data to be sent
 * 
 *   @param [in] Size
 *      Number of 32bits elements in the transmission
 * 
 *   @param [in] MaxRetries
 *      Number of time to attempt the sending operation
 * 
 *   @return
 *   
 *   @ingroup EVM_SERIALCOMM
 *
 ***************************************************************************/ 
Int32 EVM_SERIALCOMM_send(Int32 iSerialPort, Uint32 *pu32Data, Uint32 Size,
                          Uint32 MaxRetries);


 /****************************************************************************
 *   NAME : EVM_SERIALCOMM_receive(Int32 iSerialPort)
 ************************************************************************//**
 *
 *   This function sets up a data reception
 *
 *   @param [in] iSerialPort
 *      Port of the McBSP to be used
 * 
 *   @return
 *   
 *   @ingroup EVM_SERIALCOMM
 *
 ***************************************************************************/ 
void EVM_SERIALCOMM_receive(Int32 iSerialPort);


 /****************************************************************************
 *   NAME : EVM_SERIALCOMM_waitforreception(Int32 iSerialPort, Uint32 *pu32Data,
 *                                          Uint32 maxsize,Uint32 MaxRetries)
 ************************************************************************//**
 *
 *   This function waits for the reception of a complete transmission and 
 *      returns the data
 *
 *   @param [in] iSerialPort
 *      Port of the McBSP to be used
 *
 *   @param [in] pu32Data
 *      Data received
 * 
 *   @param [in] maxsize
 *      Maximum allowed size to receive in pu32Data buffer
 * 
 *   @param [in] MaxRetries
 *      Number of time to attempt the receiving operation
 * 
 *   @return
 *   
 *   @ingroup EVM_SERIALCOMM
 *
 ***************************************************************************/ 
Int32 EVM_SERIALCOMM_waitforreception(Int32 iSerialPort, Uint32 *pu32Data,
                                      Uint32 maxsize,Uint32 MaxRetries);


 /****************************************************************************
 *   NAME : EVM_SERIALCOMM_close(Int32 iSerialPort)
 ************************************************************************//**
 *
 *   This function deinitializes the McBSP and closes interrupt handles
 *
 *   @param [in] iSerialPort
 *      Port of the McBSP to be closed
 * 
 *   @return
 *   
 *   @ingroup EVM_SERIALCOMM
 *
 ***************************************************************************/ 
Int32 EVM_SERIALCOMM_close(Int32 iSerialPort);


#ifdef __cplusplus
}
#endif //__cplusplus 
#endif //__EVM_SERIALCOMM_H__

