/* ===============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006, 2007, 2008
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 * ===============================================================================
 */
/** ==============================================================================
 *   @file  csl_adtf2.h
 *
 *   @path  $(CSLPATH)\inc
 *
 *   @desc  Header file for functional layer of ADTF2 CSL
 *  - The different symbolic constants, enumerations, structure definitions
 *    and function prototype declarations
 *
 */
/* ===============================================================================
 *  Revision History
 *  ================
 *
 *  30-April-2008   File created
 *  16-July-2008 Update for Doxygen
 *
 * ===============================================================================
 */


 /** @defgroup CSL_ADTF2_API ADTF2
  *
  * @section Introduction
  *
  * @subsection xxx Overview
  * The ADTF2 is an optional module that can be implemented external to the a TI DSP subsystem that supports cTools trace.
  * This module can be used to capture the trace output from the DSP subsystem, format the trace data into ATB compliant
  * trace packets and then export these packets to an ATB interface.
  *
  * @subsection References
  *    -# TI DSP TO ATB TRACE WRAPPER REVISION 2 (ADTF2) FUNCTIONAL SPECIFICATION AND REQUIREMENTS (Version 0.97, 17 July 2007)
  */



#ifndef _CSL_ADTF2_H_
#define _CSL_ADTF2_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <soc.h>
#include <csl_types.h>
#include <csl_error.h>
#include <cslr_adtf2.h>

/**
@defgroup CSL_ADTF2_DATASTRUCT  ADTF2 Data Structures
@ingroup CSL_ADTF2_API
*/
/**
@defgroup CSL_ADTF2_FUNCTION  ADTF2 Functions
@ingroup CSL_ADTF2_API
*/
/**
@defgroup CSL_ADTF2_ENUM  ADTF2 Enumerated Data Types
@ingroup CSL_ADTF2_API
*/


/** @addtogroup CSL_ADTF2_ENUM
 @{ */
/**
 * This is the set of control commands that are passed to
 * @a CSL_adtf2HwControl(), with an optional argument type-casted to @a void*
 *
 * The arguments, if any, to be passed with each command are specified
 * next to that command.
 */
typedef enum {
     CSL_ADTF2_CMD_FLUSH_ADTF2FLUSH = 0,
     CSL_ADTF2_CMD_ENA_ADTF2 = 1,
	 CSL_ADTF2_CMD_DIS_ADTF2 = 2,
     CSL_ADTF2_CMD_RESETTRACE = 3,
	 CSL_ADTF2_CMD_CLEAR_OVERFLOW = 4,
	 CSL_ADTF2_CMD_ENA_TRIGGER = 5,
     CSL_ADTF2_CMD_DIS_TRIGGER = 6,
     CSL_ADTF2_CMD_ENA_MULTIPACKET = 7,
     CSL_ADTF2_CMD_ENA_SINGLEPACKET = 8,
     CSL_ADTF2_CMD_ENA_NOP_DISCARD = 9,
     CSL_ADTF2_CMD_DIS_NOP_DISCARD = 10
} CSL_Adtf2ControlCmd;


/**
 * This is the set of query commands to get the status of various
 * operations in ADTF2
 *
 * The arguments, if any, to be passed with each command are specified
 * next to that command. */
typedef enum {
    CSL_ADTF2_QUERY_INPUTWIDTH = 0,
    CSL_ADTF2_QUERY_ATID = 1,
    CSL_ADTF2_QUERY_ADTF2ENABLE = 2,
    CSL_ADTF2_QUERY_NOPDISCARD = 3,
    CSL_ADFT2_OVERFLOW = 4,
    CSL_ADFT2_TRIGGERSTATUS = 5,
    CSL_ADFT2_MULTIPACKET_STATUS = 6
} CSL_Adtf2HwStatusQuery;
/**
@} */


/** @addtogroup CSL_ADTF2_DATASTRUCT
 @{ */
/**
 * @brief This structure will have the base-address information for the peripheral
 *  instance
 */
typedef struct {
    /** Base-address of the Configuration registers of ADTF2.
     */
    CSL_Adtf2RegsOvly    regs;
} CSL_Adtf2BaseAddress;


/**
 * @brief ADTF2 specific parameters. Present implementation doesn't have
 *  any specific parameters.
 */
typedef struct {
    /** Bit mask to be used for module specific parameters.
     *  The below declaration is just a place-holder for future
     *  implementation.
     */
    CSL_BitMask16   flags;
} CSL_Adtf2Param;

/**
 * @brief ADTF2 specific context information. Present implementation doesn't
 *  have any Context information.
 */
typedef struct {
    /** Context information of ADTF2.
     *  The below declaration is just a place-holder for future
     *  implementation.
     */
    Uint16  contextInfo;
} CSL_Adtf2Context;

/**
 * @brief This structure/object holds the context of the instance of ADTF2
 * opened using @a CSL_adtf2Open() function.
 *
 * Pointer to this object is passed as ADTF2 Handle to all ADTF2 CSL APIs.
 * @a CSL_adtf2Open() function intializes this structure based on the parameters
 * passed */
typedef struct CSL_Adtf2Obj {
    /** Pointer to the register overlay structure of the ADTF2 */
    CSL_Adtf2RegsOvly    regs;

    /** Instance of ADTF2 being referred by this object  */
    CSL_InstNum         adtf2Num;
} CSL_Adtf2Obj;


/** @brief This is a pointer to @a CSL_Adtf2Obj and is passed as the first
 * parameter to all ADTF2 CSL APIs
 */
typedef struct CSL_Adtf2Obj *CSL_Adtf2Handle;
/**
@} */


/* ************************************************************************\
 * ADTF2 global function declarations
\* ************************************************************************/

/** @addtogroup CSL_ADTF2_FUNCTION
 @{ */
/** ============================================================================
 *   @n@b CSL_adtf2Init
 *
 *   @b Description
 *   @n This is the initialization function for the ADTF2. This function must
 *      be called before calling any other API from this CSL. This function is
 *		idem-potent. Currently, the function just returns status CSL_SOK,
 *      without doing anything.
 *
 *   @b Arguments
     @verbatim
            pContext   Context information for the instance. Should be NULL
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
          CSL_Status status;
          ...
          status = CSL_adtf2Init(NULL);
          ...

     @endverbatim
 *  ===========================================================================
 */
CSL_Status  CSL_adtf2Init (
    CSL_Adtf2Context    *pContext
);

/** ============================================================================
 *   @n@b CSL_adtf2Open
 *
 *   @b Description
 *   @n This function returns the handle to the ADTF2 controller
 *      instance. This handle is passed to all other CSL APIs.
 *
 *   @b Arguments
 *   @verbatim
 *          pAdtf2Obj     Pointer to adtf2 object.
 *
 *          adtf2Num      Instance of  DSP ADTF2 to be opened.
 *                      There are six instance of the adtf2 available.
 *
 *          pAdtf2Param   Module specific parameters.
 *
 *          pStatus     Status of the function call
 *    @endverbatim
 *
 *   <b> Return Value </b>  CSL_Adtf2Handle
 *   @n                         Valid adtf2 handle will be returned if
 *                              status value is equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_adtf2Init() must be called successfully in order before calling CSL_adtf2Open().
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK             Valid adtf2 handle is returned
 *   @li            CSL_ESYS_FAIL       The adtf2 instance is invalid
 *   @li            CSL_ESYS_INVPARAMS  Invalid parameter
 *
 *        2.    ADTF2 object structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. ADTF2 object structure
 *
 *   @b Example
 *   @verbatim
 *          CSL_Status         status;
 *          CSL_Adtf2Obj         adtf2Obj;
 *          CSL_Adtf2Handle      hAdtf2;
 *
 *          ...
 *
 *          hAdtf2 = CSL_adtf2Open(&adtf2Obj, CSL_ADTF2_0, NULL, &status);
 *
 *          ...
 *  @endverbatim
 * =============================================================================
 */
CSL_Adtf2Handle CSL_adtf2Open (
    CSL_Adtf2Obj      *hAdtf2Obj,
    CSL_InstNum      adtf2Num,
    CSL_Adtf2Param    *pAdtf2Param,
    CSL_Status       *pStatus
);

/** ===========================================================================
 *   @n@b CSL_adtf2Close
 *
 *   @b Description
 *   @n This function closes the specified instance of ADTF2.
 *
 *   @b Arguments
 *   @verbatim
            hAdtf2            Handle to the ADTF2

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Close successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_adtf2Init() and CSL_adtf2Open() must be called successfully
 *       in order before calling CSL_adtf2Close().
 *
 *   <b> Post Condition </b>
 *   @n  The ADTF2 CSL APIs can not be called until the ADTF2 CSL is reopened again using CSL_adtf2Open().
 *
 *   @b Modifies
 *   @n Obj structure values
 *
 *   @b Example
 *   @verbatim
            CSL_Adtf2Handle      hAdtf2;
            CSL_Status         status;
            ...

            status = CSL_adtf2Close(hAdtf2);
            ...

     @endverbatim
 * ===========================================================================
 */
CSL_Status  CSL_adtf2Close (
    CSL_Adtf2Handle    hAdtf2
);


/** ============================================================================
 *   @n@b CSL_adtf2HwControl
 *
 *   @b Description
 *   @n This function takes an input control command with an optional argument
 *      and accordingly controls the operation/configuration of ADTF2.
 *
 *   @b Arguments
 *   @verbatim
            hAdtf2       ADTF2 handle returned by successful 'open'

            cmd         The command to this API indicates the action to be
                        taken on ADTF2.
                        Control command, refer @a CSL_Adtf2ControlCmd for the
                        list of commands supported

            arg         An optional argument.
                        Optional argument as per the control command,
                        void * casted

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li         CSL_SOK               - Status info return successful.
 *   @li         CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li         CSL_ESYS_INVCMD       - Invalid command
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_adtf2Init() and CSL_adtf2Open() must be called successfully
 *       in order before calling CSL_adtf2HwControl().
 *
 *       Refer to @a CSL_Adtf2HwControlCmd for the argument type (@a void*) that
 *       needs to be passed with the control command
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of ADTF2.
 *
 *   @b Example
 *   @verbatim
      CSL_Status status;
      Uint32  arg;
      CSL_Adtf2Handle hAdtf2;
      ...
      // ADTF2 object defined and HwSetup structure defined and initialized
      ...

      // Init successfully done
      ...
      // Open successfully done
      ...

      arg = 1;
      status = CSL_adtf2HwControl(hAdtf2,
                                CSL_ADTF2_CMD_SET_ADTF2ENABLE,
                                &arg);

     @endverbatim
 * =============================================================================
 */
CSL_Status  CSL_adtf2HwControl (
    CSL_Adtf2Handle        hAdtf2,
    CSL_Adtf2ControlCmd    cmd,
    void*                 arg
);

/** ============================================================================
 *   @n@b CSL_adtf2GetHwStatus
 *
 *   @b Description
 *   @n Gets the status of different operations or some setup-parameters of
 *      ADTF2. The status is returned through the third parameter.
 *
 *   @b Arguments
 *   @verbatim
            hAdtf2     ADTF2 handle returned by successful 'open'

            query     The query to this API of ADTF2 which indicates the
                      status to be returned.

                      Query command, refer CSL_Adtf2HwStatusQuery for the list
                      of query commands supported

            response  Response from the query.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVQUERY   - Invalid query command
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_adtf2Init() and CSL_adtf2Open() must be called successfully
 *       in order before calling CSL_adtf2GetHwStatus().
 *       Refer to @a CSL_Adtf2HwStatusQuery for the argument to be passed along
 *       with the corresponding query command.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n Third parameter response
 *
 *   @b Example
 *   @verbatim
        CSL_Adtf2Handle hAdtf2;
        CSL_Status status;
        Uint32 response;
        ...
        status = CSL_adtf2GetHwStatus(hAdtf2,
                                     CSL_ADTF2_QUERY_ADTF2OWN_STATUS,
                                     &response);


     @endverbatim
 * =============================================================================
 */
CSL_Status  CSL_adtf2GetHwStatus (
    CSL_Adtf2Handle           hAdtf2,
    CSL_Adtf2HwStatusQuery    query,
    void                    *response
);


/** ============================================================================
 *   @n@b CSL_adtf2GetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_adtf2Open()
 *       function call. This function is open for re-implementing if the user
 *       wants to modify the base address of the peripheral object to point to
 *       a different location and there by allow CSL initiated write/reads into
 *       peripheral. MMR's go to an alternate location.
 *
 *   @b Arguments
 *   @verbatim
            adtf2Num        Specifies the instance of the ADTF2 to be opened.

            pAdtf2Param     Module specific parameters.

            pBaseAddress  Pointer to baseaddress structure containing base
                          address details.

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_OK            Open call is successful
 *   @li                    CSL_ESYS_FAIL     The instance number is invalid.
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base Address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. Base address structure is modified.
 *
 *   @b Example
 *   @verbatim
        CSL_Status          status;
        CSL_Adtf2BaseAddress  baseAddress;

       ...
      status = CSL_adtf2GetBaseAddress(CSL_ADTF2_0, NULL, &baseAddress);

    @endverbatim
 * ===========================================================================
 */
CSL_Status   CSL_adtf2GetBaseAddress (
        CSL_InstNum             adtf2Num,
        CSL_Adtf2Param           *pAdtf2Param,
        CSL_Adtf2BaseAddress     *pBaseAddress
);
/**
@} */


#ifdef __cplusplus
}
#endif

#endif

