/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006, 2008
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *   ===========================================================================
 */
/** ============================================================================
 *   @file  csl_chip.h
 *
 *   @path  $(CSLPATH)\inc
 *
 *   @desc  API header file for CHIP module CSL
 *
 * =============================================================================
 */
/** @defgroup CSL_CHIP_API CHIP 
 *
 *  @section Introduction 
 *  
 * @subsection xxx Overview
 * This module deals with all System On Chip (SOC) configurations. It consists
 * of Configuration Registers specific for the chip. Following are the Registers
 * associated with the CHIP module:
 *   - Addressing Mode Register - This register specifies the addressing mode
 * for the registers which can perform linear or circular addressing, also
 * contain sizes for circular addressing
 *   - Control Status Register - This register contains the control and status
 * bits. This register is used to control the mode of cache. This is also used
 * to enable or disable all the interrupts except reset and non maskable interrupt.
 *   - Interrupt Flag Register - This register contains the status of INT4-INT15
 * and NMI interrupt. Each corresponding bit in the IFR is set to 1 when that
 * interrupt occurs; otherwise, the bits are cleared to 0.
 *   - Interrupt Set Register - This register allows user to manually set the
 * maskable interrupts (INT4-INT15) in the interrupt flag register (IFR).
 * Writing a 1 to any of the bits in ISR causes the corresponding interrupt flag
 * to be set in IFR.
 *   - Interrupt Clear Register - This register allows user to manually clear
 * the maskable interrupts (INT15-INT4) in the interrupt flag register (IFR).
 * Writing a 1 to any of the bits in ICR causes the corresponding interrupt flag
 * to be cleared in IFR.
 *   - Interrupt Enable Register - This register enables and disables individual
 * interrupts and this not accessible in User mode.
 *   - Interrupt Service Table Pointer Register - This register is used to
 * locate the interrupt service routine (ISR).
 *   - Interrupt Return Pointer Register - This register contains the return
 * pointer that directs the CPU to the proper location to continue program
 * execution after processing a maskable interrupt.
 *   - Nonmaskable Interrupt (NMI) Return Pointer Register - This register
 * contains the return pointer that directs the CPU to the proper location to
 * continue program execution after processing of a non-maskable interrupt (NMI).
 *   - Exception Return Pointer Register - This register contains the return
 * pointer that directs the CPU to the proper location to continue program
 * execution after processing of a exception.
 *   - Time Stamp Counter Registers - The CPU contains a free running 64-bit
 * counter that advances each CPU clock after counting is enabled. The counter
 * is accessed using two 32-bit read-only control registers, Time Stamp Counter
 * Registers - Low (TSCL) and Time Stamp Counter Registers - High (TSCH). The
 * counter is enabled by writing to TSCL. The value written is ignored. Once
 * enabled, counting cannot be disabled under program control. Counting is
 * disabled in the following cases:
 *        - After exiting the reset state.
 *        - When the CPU is fully powered down.
 *   - SPLOOP Inner Loop Count Register - The SPLOOP or SPLOOPD instructions
 * use the SPLOOP inner loop count register (ILC), as the count of the number
 * of iterations left to perform. The ILC content is decremented at each stage
 * boundary until the ILC content reaches 0.
 *   - SPLOOP Reload Inner Loop Count Register - Predicated SPLOOP or SPLOOPD
 * instructions used in conjunction with a SPMASKR or SPKERNELR instruction use
 * the SPLOOP reload inner loop count register (RILC), as the iteration count
 * value to be written to the SPLOOP inner loop count register (ILC) in the
 * cycle before the reload operation begins.
 *   - E1 Phase Program Counter - This register contains the 32-bit address of
 * the fetch packet in the E1 pipeline phase.
 *   - DSP Core Number Register - This register provides an identifier to
 * shared resources in the system which identifies which CPU is accessing those
 * resources. The contents of this register are set to a specific value at reset.
 *   - Saturation Status Register - This register provides saturation flags for
 * each functional unit, making it possible for the program to distinguish
 * between saturations caused by different instructions in the same execute packet.
 *   - GMPY Polynomial.A Side Register - The GMPY instruction uses the 32-bit
 * polynomial in the GMPY polynomial-A side register (GPLYA), when the
 * instruction is executed on the M1 unit.
 *   - GMPY Polynomial.B Side Register - The GMPY instruction uses the 32-bit
 * polynomial in the GMPY polynomial-B side register (GPLYB), when the 
 * instruction is executed on the M2 unit.
 *   - Galois Field Polynomial Generator Function Register - This register
 * controls the field size and the Galois field polynomial generator of the
 * Galois field multiply hardware.
 *   - Task State Register - This register contains all of the status bits that
 * determine or indicate the current execution environment. TSR is saved in the
 * event of an interrupt or exception to the ITSR or NTSR, respectively.
 *   - Interrupt Task State Register - This register is used to store the
 * contents of the task state register (TSR) in the event of an interrupt.
 *   - NMI/Exception Task State Register - This register is used to store
 * the contents of the task state register (TSR) and the conditions under which
 * an an exception occurred in the event of a nonmaskable interrupt (NMI) or an
 * exception.
 *   - Exception Flag Register - This register contains bits that indicate
 * which exceptions have been detected. Clearing the EFR bits is done by
 * writing a 1 to the corresponding bit position in the exception clear
 * register (ECR).
 *   - Exception Clear Register - This register is used to clear individual
 * bits in the exception flag register (EFR). Writing a 1 to any bit in ECR
 * clears the corresponding bit in EFR.
 *   - Internal Exception Report Register - This register contains flags that
 * indicate the cause of the internal exception.
 *   - Restricted Entry Point Address Register - This register is used by the
 * SWENR instruction as the target of the change of control when an SWENR
 * instruction is issued. The contents of REP should be preinitialized by the
 * processor in Supervisor mode before any SWENR instruction is issued.
 */
 
 
/* =============================================================================
 *  Revision History
 *  ===============
 *  07-Mar-2006 ds  To fix bug PSG00001004, Added a dummy code chipReadNone2, 
 *                  before ILC read in CSL_chipReadReg () API
 *                  To fix bug PSG00001012, Added dummy code chipWriteNone2, 
 *                  before ILC write in CSL_chipWriteReg () API
 *  19-May-2006 NG  Updated the enum CSL_ChipReg ie., from AMR to CSL_CHIP_AMR
 *                  according to coding guidelines.
 *                  REP value changed from 32 to 15
 *  25-Jul-2008   Update for Doxygen
 * =============================================================================
 */

#ifndef _CSL_CHIP_H_
#define _CSL_CHIP_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <csl.h>
#include <cslr_chip.h>


/**
@defgroup CSL_CHIP_ENUM  CHIP Enumerated Data Types
@ingroup CSL_CHIP_API
*/

/**
@defgroup CSL_CHIP_FUNCTION  CHIP Functions
@ingroup CSL_CHIP_API
*/

/* CHIP global macro declarations */

/* CHIP global typedef declarations */

/** @addtogroup CSL_CHIP_ENUM
 @{ */
/** Enum for the CHIP registers */
typedef enum  {
    /** Addressing Mode Register */
    CSL_CHIP_AMR = 0,
    /** Control Status Register */
    CSL_CHIP_CSR = 1,
    /** Interrupt Flag Register */
    CSL_CHIP_IFR = 2,
    /** Interrupt Set Register */
    CSL_CHIP_ISR = 2,
    /** Interrupt Clear Register */
    CSL_CHIP_ICR = 3,
    /** Interrupt Enable Register */
    CSL_CHIP_IER = 4,
    /** Interrupt Service Table Pointer Register */
    CSL_CHIP_ISTP= 5,
    /** Interrupt Return Pointer Register */
    CSL_CHIP_IRP= 6,
    /** Nonmaskable Interrupt (NMI) Return Pointer Register */
    CSL_CHIP_NRP= 7,
    /** Exception Return Pointer Register */
    CSL_CHIP_ERP= 7,
    /** Time Stamp Counter Register - Low */
    CSL_CHIP_TSCL= 10,
    /** Time Stamp Counter Registers - High */
    CSL_CHIP_TSCH= 11,
    /** SPLOOP Inner Loop Count Register */
    CSL_CHIP_ILC= 13,
    /** SPLOOP Reload Inner Loop Count Register */
    CSL_CHIP_RILC= 14,
    /** Restricted Entry Point Address Register */
    CSL_CHIP_REP= 15,
    /** E1 Phase Program Counter */
    CSL_CHIP_PCE1= 16,
    /** DSP Core Number Register */
    CSL_CHIP_DNUM= 17,
    /** Saturation Status Register */
    CSL_CHIP_SSR= 21,
    /** GMPY Polynomial.A Side Register */
    CSL_CHIP_GPLYA= 22,
    /** GMPY Polynomial.B Side Register */
    CSL_CHIP_GPLYB= 23,
    /** Galois Field Polynomial Generator Function Register */
    CSL_CHIP_GFPGFR= 24,
    /** Task State Register */
    CSL_CHIP_TSR= 26,
    /** Interrupt Task State Register */
    CSL_CHIP_ITSR= 27,
    /** NMI/Exception Task State Register */
    CSL_CHIP_NTSR= 28,
    /** Exception Flag Register */
    CSL_CHIP_EFR= 29,
    /** Exception Clear Register */
    CSL_CHIP_ECR= 29,
    /** Internal Exception Report Register */
    CSL_CHIP_IERR= 31
} CSL_ChipReg;

/**
@}
*/

/** @addtogroup CSL_CHIP_FUNCTION
 @{ */
/* CHIP global function declarations   */

/** ============================================================================
 *   @n@b CSL_chipWriteReg
 *
 *   @b Description
 *   @n This API writes specified control register with the specified 
 *      value 'val'. 
 *
 *   @b Arguments
 *   @verbatim      
            reg      Specifies the control register to be written to
 
            val      Value to be written.
 
     @endverbatim
 *
 *
 *   <b> Return Value   Uint32
 *   @li                    Old programmed value
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The reg control register is written with the value passed
 *
 *   @b Modifies
 *   @n None
 *
 * <b> Usage Constraints: </b>
 *  @n Please refer to the C64x+ user guide for constraints while accessing 
 *     registers in different privilege levels.
 *
 *   @b Example
 *   @verbatim
 
        Uint32 regVal;
        regVal = CSL_chipWriteReg (CSL_CHIP_AMR, 56);


    @endverbatim
 * ===========================================================================
 */
Uint32  CSL_chipWriteReg (
    CSL_ChipReg    reg,
    CSL_Reg32      val
);

/** ============================================================================
 *   @n@b CSL_chipReadReg
 *
 *   @b Description
 *   @n This API reads the specified control register. 
 *
 *   @b Arguments
 *   @verbatim      
            reg      Specifies the control register to be read
 
     @endverbatim
 *
 *
 *   <b> Return Value   Uint32
 *   @li                    The control register value read
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *  <b> Usage Constraints: </b>
 *  @n Please refer to the C64x+ user guide for constraints while accessing 
 *     registers in different privilege levels.
 *
 *   @b Example
 *   @verbatim
 
        Uint32 regVal;
        regVal = CSL_chipReadReg (CSL_CHIP_AMR);


    @endverbatim
 * ===========================================================================
 */
Uint32  CSL_chipReadReg(
    CSL_ChipReg    reg
);
/**
@}
*/

#ifdef __cplusplus
}
#endif

#endif
