/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2008
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *  ============================================================================
 */

/** 
 *  @file    csl_ddr2.h
 *
 *  @brief   Header file for functional layer of CSL
 *
 *  @Path    $(CSLPATH)\inc
 *
 *  @desc
 *    - The different enumerations, structure definitions
 *      and function declarations
 *
 * Modification 1
 *    - Created on: 11th April,2005
 *    - Reason: created the sources
 *
 * @author Ramitha Mathew.
 * 
 * 
 */

/** 
 * @defgroup CSL_DDR2_API DDR2 EMIF
 *
 * @section Introduction
 *
 * @subsection xxx Overview
 * This page describes the Functions, Data Structures, Enumerations and Macros
 * within DDR2 module. This is a 32-bit DDR2 SDRAM interface. The 32-bit DDR2
 * Memory Controller bus is used to interface to DDR2 devices. The DDR2
 * external bus only interfaces to DDR2 devices; it does not share the bus with
 * any other types of peripherals. The DDR2 memory controller interfaces with
 * JESD79D-2A standard compliant DDR2 SDRAM devices. Memory types such as DDR1
 * SDRAM, SDR SDRAM, SBSRAM, and asynchronous memories are not supported. The
 * DDR2 memory controller SDRAM can be used for program and data storage.
 * The DDR2 memory controller supports the following features:
 *	 - JESD79D-2A standard compliant DDR2 SDRAM
 *	 - 256 Mbyte memory space
 *	 - Data bus width of 32 or 16 bits
 *	 - CAS latencies: 2, 3, 4, and 5
 *	 - Internal banks: 1, 2, 4, and 8
 *	 - Burst length: 8
 *	 - Burst type: sequential
 *	 - 1 CE signal
 *	 - Page sizes: 256, 512, 1024, and 2048
 *	 - SDRAM auto-initialization
 *	 - Self-refresh mode
 *	 - Prioritized refresh
 *	 - Programmable refresh rate and backlog counter
 *	 - Programmable timing parameters
 *	 - Little-endian and big endian transfers

 

 *
 *
 * @subsection References
 *    -# TMS320TCI648x DSP DDR2 Memory Controller User's Guide SPRU894B    
 *    (June 2007)
 *
 */

/*  ============================================================================
 *  Revision History
 *  ================
 *  30-May-2008 Removed CSL_DDR2_QUERY_IFRDY status query.
 *  28-Jul-2008 Update for Doxygen
 *  11-Nov-2008 Defined enums CSL_Ddr2Differential, CSL_Ddr2SdramDrStrength.
 *              Added fields ddr2Ddqs, ddr2Term0, ddr2Term1, ddrDisDll, sdramDrive0 to CSL_Ddr2Settings.  
 *              Removed CSL_DDR2_QUERY_SELF_REFRESH status query. Added CSL_DDR2_QUERY_IFRDY, 
 *              CSL_DDR2_QUERY_DDRSLRATE status query.
 *              Removed CSL_DDR2_CMD_SELF_REFRESH control command.
 *  ============================================================================
 */

#ifndef _CSL_DDR2_H_
#define _CSL_DDR2_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <cslr.h>
#include <soc.h>
#include <csl_error.h>
#include <csl_types.h>
#include <cslr_ddr2.h>

/**
@defgroup CSL_DDR2_SYMBOL  DDR2 Symbols Defined
@ingroup CSL_DDR2_API
*/
/**
@defgroup CSL_DDR2_DATASTRUCT  DDR2 Data Structures
@ingroup CSL_DDR2_API
*/
/**
@defgroup CSL_DDR2_FUNCTION  DDR2 Functions
@ingroup CSL_DDR2_API
*/
/**
@defgroup CSL_DDR2_ENUM  DDR2 Enumerated Data Types
@ingroup CSL_DDR2_API
*/


 
/** @addtogroup CSL_DDR2_ENUM
 @{ */
/******************************************************************************
 * DDR2 global typedef declarations
 ******************************************************************************
 */

/** @brief Enumeration for bit field narrow_mode of SDRAM Config Register */ 
typedef enum {
    /** DDR2 SDRAM data bus width is 32 bits  */
    CSL_DDR2_NORMAL_MODE      = 0,
    
    /** DDR2 SDRAM data bus width is 16 bits   */
    CSL_DDR2_NARROW_MODE     = 1
}CSL_Ddr2Mode; 

/** @brief Enumeration for bit field cl of SDRAM Config Register */ 
typedef enum {
    /** Cas Latency is 2  */
    CSL_DDR2_CAS_LATENCY_2    = 2,
    
    /** Cas Latency is 3   */
    CSL_DDR2_CAS_LATENCY_3    = 3,
    
    /** Cas Latency is 4  */
    CSL_DDR2_CAS_LATENCY_4    = 4,
    
    /** Cas Latency is 5   */
    CSL_DDR2_CAS_LATENCY_5    = 5
} CSL_Ddr2CasLatency; 

/** @brief Enumeration for bit field ibank of SDRAM Config Register */ 
typedef enum {
    /** DDR2 SDRAM has one internal bank  */
    CSL_DDR2_1_SDRAM_BANKS    = 0,
    
    /** DDR2 SDRAM has two internal banks   */
    CSL_DDR2_2_SDRAM_BANKS    = 1,
    
    /** DDR2 SDRAM has four internal bank  */
    CSL_DDR2_4_SDRAM_BANKS    = 2,
    
    /** DDR2 SDRAM has eight internal banks   */
    CSL_DDR2_8_SDRAM_BANKS    = 3
} CSL_Ddr2IntBank; 

/** @brief Enumeration for bit field pagesize of SDRAM Config Register */ 
typedef enum {
    /** 256-word pages requiring 8 column address bits  */
    CSL_DDR2_256WORD_8COL_ADDR      = 0,
    
    /** 512-word pages requiring 9 column address bits  */
    CSL_DDR2_512WORD_9COL_ADDR      = 1,
    
    /** 1024-word pages requiring 10 column address bits */
    CSL_DDR2_1024WORD_10COL_ADDR    = 2,
    
    /** 2048-word pages requiring 11 column address bits */
    CSL_DDR2_2048WORD_11COL_ADDR    = 3
} CSL_Ddr2PageSize; 

/** @brief Enumeration for bit field DDQS of SDRAM Config Register */ 
typedef enum {
    /** DDR2 SDRAM single ended differential DQS enable.  */
    CSL_DDR2_SINGLE_ENDED_DQS      = 0,
    
    /** DDR2 SDRAM differential DQS enable.  */
    CSL_DDR2_DIFFERENTIAL_DQS      = 1
} CSL_Ddr2Differential; 


/** @brief Enumeration for bit field sdram drive strength of SDRAM Config Register */ 
typedef enum {
    /** Disable termination  */
    CSL_DDR2_SDRAM_NORMAL           = 0,
    
    /** 75 Ohm termination  */
    CSL_DDR2_SDRAM_WEAK             = 1
} CSL_Ddr2SdramDrStrength; 

/**
@}
*/


/** @addtogroup CSL_DDR2_DATASTRUCT
 @{ */
/** 
 *  @brief Module specific context information.  
 */
typedef struct {
    /** Context information of DDR2 external memory interface CSL passed as an
     *  argument to CSL_ddr2Init().Present implementation of DDR2 CSL doesn't 
     *  have any context information; hence assigned NULL.
     *  The below declaration is just a place-holder for future implementation.
     */
    Uint16    contextInfo;
    
} CSL_Ddr2Context;

/** 
 * @brief This structure contains the base-address information for the DDR2
 *        instance
 */
typedef struct {
    /** Base-address of the configuration registers of the peripheral */
    CSL_Ddr2RegsOvly    regs;
} CSL_Ddr2BaseAddress;


/** 
 * @brief This Object contains the reference to the instance of DDR2 opened
 *  using the @a CSL_ddr2Open().
 *
 *  The pointer to this, is passed to all DDR2 CSL APIs.
 */
typedef struct CSL_Ddr2Obj {
    /** This is a pointer to the registers of the instance of DDR2
     *  referred to by this object
     */
    CSL_Ddr2RegsOvly    regs;
    
    /** This is the instance of DDR2 being referred to by this object  */
    CSL_InstNum         perNum;    
} CSL_Ddr2Obj;

/** 
 * @brief this is a pointer to @a CSL_Ddr2Obj and is passed as the first
 *  parameter to all DDR2 CSL APIs
 */
typedef struct CSL_Ddr2Obj *CSL_Ddr2Handle;

/** 
 * @brief This is module specific parameters.  Present implementation of DDR2 
 *        CSL doesn't have any module specific parameters.
 */
typedef struct {
    /** Bit mask to be used for module specific parameters. The below
     *  declaration is just a place-holder for future implementation. Passed as 
     *  an argument to CSL_ddr2Open().
     */
    CSL_BitMask16    flags;
} CSL_Ddr2Param;

/** 
 * @brief DDR2 SDRAM Timing1 structure.
 * All fields needed for DDR2 SDRAM Timing1 are present in this structure.
 */
typedef struct {
    /** Specifies TRFC value: Minimum number of DDR2 EMIF cycles from Refresh 
     *  or Load command to Refresh or Activate command, minus one 
     */
    Uint8    trfc;
    /** Specifies TRP value: Minimum number of DDR2 EMIF cycles from Pre-charge
     *  to Active or Refresh command, minus one  
     */
    Uint8    trp;
    /** Specifies TRCD value: Minimum number of DDR2 EMIF cycles from Active to 
     *  Read or Write command, minus one  
     */
    Uint8    trcd;
    /** Specifies TWR value: Minimum number of DDR2 EMIF cycles from last write 
     *  transfer to Pre-charge command, minus one 
     */
    Uint8    twr;
    /** Specifies TRAS value: Minimum number of DDR2 EMIF cycles from Activate 
     *  to Pre-charge command, minus one  
     */
    Uint8    tras;
    /** Specifies TRC value: Minimum number of DDR2 EMIF cycles from Activate 
     *  command to Activate command, minus one 
     */
    Uint8    trc;
    /** Specifies TRRD value: Minimum number of DDR2 EMIF cycles from Activate 
     *  command to Activate command for a differnt bank, minus one 
     */
    Uint8    trrd;
    /** Specifies the minimum number of DDR2 EMIF clock cycles from last DDR 
     *  Write to DDR Read,  minus one        
     */
    Uint8    twtr;
} CSL_Ddr2Timing1;

/** @brief DDR2 SDRAM Timing2 structure.
 *
 * All fields needed for DDR2 SDRAM Timing2 are present in this structure.
 */
typedef struct {    
    /** Specifies the minimum number of DDR2 EMIF clock cycles from ODT enable 
     *  to write data driven for DDR2 SDRAM.          
     */
    Uint8    todt;
    /** Specifies the minimum number of DDR2 EMIF clock cycles from Self-Refresh
     *  exit to any command other than a Read command, minus one.
     */
    Uint8    tsxnr;
    /** Specifies the minimum number of DDR2 EMIF clock cycles from Self-Refresh
     *  exit to a Read command for DDR SDRAM, minus one.
     */
    Uint8    tsxrd;
    /** Specifies the minimum number of DDR2 EMIF clock cycles from the last 
     *  Read command to a Pre-charge command for DDR2 SDRAM, minus one.
     */
    Uint8    trtp;
    /** Specifies the minimum number of DDR2 EMIF clock cycles between 
     *  pado_mcke_o changes, minus one.
     */
    Uint8    tcke;
} CSL_Ddr2Timing2;

/** @brief DDR2 SDRAM Settings structure.
 *
 * All fields needed for DDR2 SDRAM settings are present in this structure.
 */
typedef struct {
    /** CAS Latency */
    CSL_Ddr2CasLatency      casLatncy;
    /** Defines number of banks inside connected SDRAM devices */
    CSL_Ddr2IntBank         ibank;
    /** Defines the internal page size of connected SDRAM devices */
    CSL_Ddr2PageSize        pageSize;     
    /** SDRAM data bus width */
    CSL_Ddr2Mode            narrowMode;
    /** Enables SDRAM */
    Uint8                   sdramEn;
    /** Enables DDR */
    Uint8                   ddrEn;
    /** Enables DDR2 */
    Uint8                   ddr2En;
    /** DDR2 SDRAM differential DQS enable. */
    CSL_Ddr2Differential    ddr2Ddqs;
    /** DDR2 termination[0] resistor value. */
    Uint8                   ddr2Term0;
    /** DDR2 termination[1] resistor value. */
    Uint8                   ddr2Term1;
    /** Disables DDR DLL */
    Uint8                   ddrDisDll;
    /** DDR2 SDRAM drive strength. */
    CSL_Ddr2SdramDrStrength sdramDrive0;
} CSL_Ddr2Settings;

/** 
 * @brief This has all the fields required to configure DDR2 at Power Up
 *  (after a Hardware Reset) or a Soft Reset.  
 *
 *  This structure is used to setup or obtain existing setup of
 *  DDR2 using @a CSL_ddr2HwSetup() & @a CSL_ddr2GetHwSetup() functions
 *  respectively.
 */
typedef struct {
    /** Refresh Rate */
    Uint16               refreshRate;
    /** Structure for DDR2 SDRAM Timing1 */
    CSL_Ddr2Timing1    *timing1Param;
    /** Structure for DDR2 SDRAM Timing2 */
    CSL_Ddr2Timing2     *timing2Param;
    /** Structure for DDR2 SDRAM configuration */
    CSL_Ddr2Settings    *setParam;
    /** Read latency */
    Uint8                 readLatncy;    
} CSL_Ddr2HwSetup;

/** 
 * @brief DDR2 Module ID and Revision structure
 *
 *  This structure is used for querying the DDR2 module ID and revision 
 */
typedef struct {
    /** DDR2 EMIF Module ID */
    Uint16    modId;
    /** DDR2 EMIF Major Revision */
    Uint8     majRev;
    /** DDR2 EMIF Minor Revision */
    Uint8     minRev;
} CSL_Ddr2ModIdRev;   

/** 
 * @brief DDR2 config structure which is used in CSL_ddr2HwSetupRaw function
 */
typedef struct {
    /** SDRAM Config Register */
    volatile Uint32 SDCFG;
    /** SDRAM Refresh Control Register */
    volatile Uint32 SDRFC;    
    /** SDRAM Timing1 Register*/

    volatile Uint32 SDTIM1;
    /** SDRAM Timing2 Register */
    volatile Uint32 SDTIM2;    
    /** VBUSM Burst Priority Register */
    volatile Uint32 BPRIO;
	/**DDR2 Memory Controller Control Register (DMCCTL)*/
    volatile Uint32 DMCCTL;     
} CSL_Ddr2Config;
/**
@}
*/

/** @addtogroup CSL_DDR2_ENUM
 @{ */
/** 
 * @brief Enumeration for queries passed to @a CSL_ddr2GetHwStatus()
 *
 * This is used to get the status of different operations 
 */
typedef enum {
    /** Get the DDR2 EMIF module ID and revision numbers (response type: 
     *  @a (CSL_Ddr2ModIdRev*)) 
     */
    CSL_DDR2_QUERY_REV_ID,     
    /** Get the EMIF refresh rate information
     *  (response type: @a Uint16 *)  
     */
    CSL_DDR2_QUERY_REFRESH_RATE,
    /** Gets the the current endian of DDR2 emif from the SDRAM Status register
     *   (response type: @a Uint8*)   
     */
    CSL_DDR2_QUERY_ENDIAN,
    /** Reflects the value on the IFRDY_ready port (active high) that
     *  defines whether the DDR IFRDY is ready for normal operation.
     *   (response type: @a Uint8*)   
     */
    CSL_DDR2_QUERY_IFRDY,
    /** Reflects the value on the DDRSLRATE port status. Pulling the DDRSLRATE 
     * input pin high selects the normal slew rate. If pulled low, the slew rate 
     * is reduced by 33%.
     *   (response type: @a Uint8*)   
     */
    CSL_DDR2_QUERY_DDRSLRATE
} CSL_Ddr2HwStatusQuery;

/** 
 * @brief Enumeration for commands passed to @a CSL_ddr2HwControl()
 *
 * This is used to select the commands to control the operations
 * existing setup of DDR2. The arguments to be passed with each
 * enumeration if any are specified next to the enumeration.
 */
typedef enum {    
    /** Enters the Refresh rate value : argument @a (Uint16 *) */     
    CSL_DDR2_CMD_REFRESH_RATE=0,     
    /** Number of memory transfers after which the DDR2 EMIF momentarily 
     *  raises the priority of old commands in the VBUSM Command FIFO. : 
     *  argument @a (Uint8 *) 
     */
    CSL_DDR2_CMD_PRIO_RAISE        
} CSL_Ddr2HwControlCmd;
/**
@}
*/
 
/**
@addtogroup CSL_DDR2_SYMBOL
@{
*/
/** @brief The defaults of DDR2 SDRAM Timing1 Control structure */
#define CSL_DDR2_TIMING1_DEFAULTS  {\
    (Uint8)CSL_DDR2_SDTIM1_T_RFC_RESETVAL, \
    (Uint8)CSL_DDR2_SDTIM1_T_RP_RESETVAL,  \
    (Uint8)CSL_DDR2_SDTIM1_T_RCD_RESETVAL, \
    (Uint8)CSL_DDR2_SDTIM1_T_WR_RESETVAL, \
    (Uint8)CSL_DDR2_SDTIM1_T_RAS_RESETVAL, \
    (Uint8)CSL_DDR2_SDTIM1_T_RC_RESETVAL, \
    (Uint8)CSL_DDR2_SDTIM1_T_RRD_RESETVAL, \
    (Uint8)CSL_DDR2_SDTIM1_T_WTR_RESETVAL \
}

/** @brief The defaults of DDR2 SDRAM Timing2 Control structure */
#define CSL_DDR2_TIMING2_DEFAULTS  { \
    (Uint8)CSL_DDR2_SDTIM2_T_ODT_RESETVAL, \
    (Uint8)CSL_DDR2_SDTIM2_T_XSNR_RESETVAL, \
    (Uint8)CSL_DDR2_SDTIM2_T_XSRD_RESETVAL, \
    (Uint8)CSL_DDR2_SDTIM2_T_RTP_RESETVAL,  \
    (Uint8)CSL_DDR2_SDTIM2_T_CKE_RESETVAL   \
}

/** @brief The defaults of DDR2 SDRAM Config structure */
#define CSL_DDR2_SETTING_DEFAULTS  {  \
    (CSL_Ddr2CasLatency)CSL_DDR2_CAS_LATENCY_3, \
    (CSL_Ddr2IntBank)CSL_DDR2_4_SDRAM_BANKS, \
    (CSL_Ddr2PageSize)CSL_DDR2_256WORD_8COL_ADDR, \
    (CSL_Ddr2Mode)CSL_DDR2_NORMAL_MODE,  \
	0x0, 0x0, 0x0, \
    (CSL_Ddr2Differential)CSL_DDR2_SINGLE_ENDED_DQS, \
    0x1, \
    0X0, \
    0x0, \
    (CSL_Ddr2SdramDrStrength)CSL_DDR2_SDRAM_NORMAL \
}

/** @brief The default Config structure */
#define CSL_DDR2_CONFIG_DEFAULTS { \
    CSL_DDR2_SDCFG_RESETVAL,       \
    CSL_DDR2_SDRFC_RESETVAL,       \
    CSL_DDR2_SDTIM1_RESETVAL,       \
    CSL_DDR2_SDTIM2_RESETVAL,       \
    CSL_DDR2_BPRIO_RESETVAL, \
	CSL_DDR2_DMCCTL_RESETVAL\
}
/**
@}
*/

/** @addtogroup CSL_DDR2_FUNCTION
 @{ */
/******************************************************************************
 * DDR2 global function declarations
 ******************************************************************************
 */
/** ============================================================================
 *   @n@b   CSL_ddr2Init
 *
 *  @b Description
 *  @n This function is idempotent i.e. calling it many times is same as calling
 *     it once.  This function presently does nothing.
 *
 * <b> Usage Constraints: </b>
 *  This function should be called before using any of the CSL APIs
 *
 *   @b Arguments
     @verbatim
            pContext   Context information for the instance. Should be NULL
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 * @verbatim
   ...
   
   CSL_ddr2Init( NULL );
   ...    
   }
   @endverbatim
 *
 *
 * =============================================================================
 */
CSL_Status  CSL_ddr2Init (
    CSL_Ddr2Context    *pContext
);

/** ============================================================================
 *   @n@b   CSL_ddr2Open
 *
 *   @b Description
 *   @n This function returns the handle to the DDR2 instance. This
 *      handle is passed to all other CSL APIs.
 *
 *   @b Arguments
 *   @verbatim

            pDdr2Obj        Pointer to the object that holds reference to the
                            instance of DDR2 requested after the call
            
            ddr2Num         Instance of DDR2 to which a handle is requested
 
            pDdr2Param      Pointer to module specific parameters
 
            pStatus         pointer for returning status of the function call

     @endverbatim
 *
 *   <b> Return Value </b>  
 *      CSL_Ddr2Handle - Valid DDR2 instance handle will be returned if status
 *    @li                value is equal to CSL_SOK.
 * 
 *   <b> Pre Condition </b>
 *   @n  @a The DDR2 must be successfully initialized via CSL_ ddr2Init() 
 *          before calling this function.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b  Modifies
 *   @n  None
 *
 *   @b Example:
 *   @verbatim
         CSL_Status         status;
         CSL_Ddr2Obj        ddr2Obj;
         CSL_Ddr2Handle     hDdr2;
 
         hDdr2 = CSL_ddr2Open(&ddr2Obj,
                              CSL_DDR2,
                              NULL,
                              &status
                             );
     @endverbatim
 *
 * =============================================================================
 */
CSL_Ddr2Handle CSL_ddr2Open (
    CSL_Ddr2Obj      *pDdr2Obj,
    CSL_InstNum      ddr2Num,
    CSL_Ddr2Param    *pDdr2Param,
    CSL_Status       *pStatus
);

/** ============================================================================
 *   @n@b CSL_ddr2Close                                                       
 *                                                                             
 *   @b Description                                                            
 *   @n This function marks that CSL for the external memory interface instance 
 *      is CSL for the external memory interface instance need to be reopened 
 *      before external memory interface CSL API.                                        
 *                                                                             
 *   @b Arguments                                                              
 *   @verbatim                                                                 
            hDdr2         Pointer to the object that holds reference to the
                          instance of DDR2 requested after the call        
     @endverbatim                                                              
 *                                                                             
 *   <b> Return Value </b>  CSL_Status                                         
 *   @li                    CSL_SOK            - external memory interface is 
 *                                               close successfully                  
 *                                                                             
 *   @li                    CSL_ESYS_BADHANDLE - The handle passed is invalid  
 *                                                                             
 *   <b> Pre Condition </b>                                                    
 *   @n  Both @a CSL_ddr2Init() and @a CSL_ddr2Open() must be called 
 *     successfully in that order before @a CSL_ddr2Close() can be called.
 *                                                                             
 *   <b> Post Condition </b>                                                   
 *   @n  1. The external memory interface CSL APIs can not be called until the    
 *          external memory interface CSL is reopened again using CSL_ddr2Open()   
 *                                                                             
 *   @b Modifies                                                               
 *   @n  hDdr2 structure                                                                 
 *                                                                             
 *   @b Example                                                                
 *   @verbatim                                                                 
            CSL_ddr2Handle   hDdr2;                                              
                                                                               
            ...                                                                
                                                                               
            CSL_ddr2Close (hDdr2);                                               
                                                                               
            ...                                                                
     @endverbatim     
 *
 * =============================================================================
 */               
CSL_Status  CSL_ddr2Close (
    CSL_Ddr2Handle    hDdr2
);

/** ============================================================================
 *   @n@b CSL_ddr2GetBaseAddress
 *
 * @b Description
 * @n The get base address call will give the External memory interface current 
 *    current instance base address
 *
 *   @b Arguments
 *   @verbatim      
            ddr2Num         Specifies the instance of the DDR2 external memory
                            interface for which the base address is requested
 
            pDdr2Param      Module specific parameters.
 
            pBaseAddress    Pointer to the base address structure to return the
                            base address details.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_OK              Successful on getting the base 
 *                                              address of ddr2
 *   @li                    CSL_ESYS_FAIL       The external memory interface 
 *                                              instance is not available.
 *   @li                    CSL_ESYS_INVPARAMS  Invalid parameter 
 *
 *   <b> Pre Condition </b>
 *   @n  @a CSL_ddr2Init() and CSL_ddr2Open () must be called successfully.
 *
 *   <b> Post Condition </b>
 *   @n  Base address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. Base address structure.
 *
 *   @b Example
 *   @verbatim
        CSL_Status          status;
        CSL_Ddr2BaseAddress  baseAddress;

       ...
       status = CSL_ddr2GetBaseAddress(CSL_DDR2, NULL, &baseAddress);

    @endverbatim
 *  @return Returns the status of the get base address operation
 *
 * ============================================================================
 */
CSL_Status CSL_ddr2GetBaseAddress (
    CSL_InstNum            ddr2Num,
    CSL_Ddr2Param          *pDdr2Param,
    CSL_Ddr2BaseAddress    *pBaseAddress
);

/** ============================================================================
 *   @n@b CSL_ddr2HwSetupRaw
 *
 *   @b Description
 *   @n This function initializes the device registers with the register-values
 *      provided through the config data structure.  
 *
 *   @b Arguments
 *   @verbatim
            hDdr2         Handle to the DDR2 external memory interface instance
            
            config        Pointer to the config structure containing the
                          device register values
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Configuration successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Configuration structure
 *                                                pointer is not properly
 *                                                initialized
 *
 *   <b> Pre Condition </b>
 *   @n  @a CSL_ddr2Init() and @a CSL_ddr2Open () must be called successfully.        
 *       before calling this function.          
 *
 *   <b> Post Condition </b>
 *   @n  The registers of the specified DDR2 EMIF instance will be 
 *       setup according to the values passed through the config structure
 *
 *   @b Modifies
 *   @n Hardware registers of the DDR2 EMIF
 *    
 *   @b Example
 *   @verbatim
        CSL_Ddr2Handle           hDdr2;
        CSL_Ddr2Config           config = CSL_DDR2_CONFIG_DEFAULTS;
        CSL_Status               status;

        ...       
       
        status = CSL_ddr2HwSetupRaw (hDdr2, &config);
        ..
        
            
     @endverbatim
 * =============================================================================
 */
CSL_Status  CSL_ddr2HwSetupRaw (
    CSL_Ddr2Handle    hDdr2,
    CSL_Ddr2Config    *config
);

/** ============================================================================
 * @n@b   CSL_ddr2HwSetup
 *
 * @b Description
 * @n This function initializes the device registers with the appropriate values
 *  provided through the HwSetup data structure. 
 *
 *  @b Arguments
 *  @verbatim      
            hDdr2           Pointer to the peripheral data object of the
                            DDR2 external memory interface instance
 
            setup           Pointer to setup structure which contains the
                            information to program DDR2 to a useful state
 
    @endverbatim
 *
 *  <b> Return Value </b>  CSL_Status
 *  @li                    CSL_SOK               Hwsetup successful
 *  @li                    CSL_ESYS_BADHANDLE   The handle passed is invalid
 *  @li                    CSL_ESYS_INVPARAMS   The param passed is invalid
 *
 *  <b> Pre Condition </b>
 *  Both @a CSL_ddr2Init() and @a CSL_ddr2Open() must be called
 *  successfully in that order before this function can be called.  The user
 *  has to allocate space for & fill in the main setup structure appropriately
 *  before calling this function
 *
 *  <b> Post Condition </b>
 *  @n  DDR2 registers are configured according to the hardware setup parameters
 *
 *  @b Modifies
 *  @n DDR2 registers
 *
 *  @b Example:
 *  @verbatim
     CSL_Ddr2Handle hDdr2;     
     CSL_Ddr2Timing1 tim1 = CSL_DDR2_TIMING1_DEFAULTS;
     CSL_Ddr2Timing2 tim2 = CSL_DDR2_TIMING2_DEFAULTS;
     CSL_Ddr2Settings set = CSL_DDR2_SETTING_DEFAULTS;
     CSL_Ddr2HwSetup hwSetup ;
     
     hwSetup.refreshRate = (Uint16)0x753;
     hwSetup.timing1Param = &tim1;
     hwSetup.timing2Param = &tim2;
     hwSetup.setParam = &set;
     
     CSL_ddr2HwSetup(hDdr2, &hwSetup);
    @endverbatim
 *
 *
 * =============================================================================
 */
CSL_Status  CSL_ddr2HwSetup (
    CSL_Ddr2Handle     hDdr2,
    CSL_Ddr2HwSetup    *setup
);

/** ============================================================================
 * @n@b   CSL_ddr2GetHwSetup
 *
 * @b Description
 * @n This function gets the current setup of the DDR2.  The status is
 *    returned through @a CSL_Ddr2HwSetup.  The obtaining of status
 *    is the reverse operation of @a CSL_ddr2HwSetup() function.
 * 
 * @b Arguments                                                              
 * @verbatim                                                                 
            hDdr2        Handle to the external memory interface instance
			setup        Pointer to setup structure which contains the
                         information to program DDR2 to a useful state
   @endverbatim                                                              
 *                                                                             
 * <b> Return Value </b>  CSL_Status
 * @li                    CSL_SOK             - Hardware setup successful
 * @li                    CSL_ESYS_INVPARAMS - The param passed is invalid    
 * @li                    CSL_ESYS_BADHANDLE - Handle is not valid    
 *
 * <b> Pre Condition </b>
 *     Both @a CSL_ddr2Init() and @a CSL_ddr2Open() must be called successfully
 *     in order before calling @a CSL_ddr2GetHwSetup().
 *
 * <b> Post Condition </b>
 * @n  None  
 *
 * @b Modifies
 * @n Second parameter setup value
 *
 * @b Example:
 * @verbatim
      CSL_Ddr2Handle hDdr2;
      CSL_Status status;      
      CSL_Ddr2Timing1 tim1;
      CSL_Ddr2Timing2 tim2;
      CSL_Ddr2Settings set;
      CSL_Ddr2HwSetup hwSetup ;
     
     
     hwSetup.timing1Param = &tim1;
     hwSetup.timing2Param = &tim2;
     hwSetup.setParam = &set;
       ...
      status = CSL_ddr2GetHwSetup(hDdr2, &hwSetup);
   @endverbatim
 *
 *
 * =============================================================================
 */
CSL_Status  CSL_ddr2GetHwSetup (
    CSL_Ddr2Handle     hDdr2,
    CSL_Ddr2HwSetup    *setup
);

/** ============================================================================
 * @n@b   CSL_ddr2HwControl
 *
 * @b Description
 * @n Control operations for the DDR2.  For a particular control operation, the
 *  pointer to the corresponding data type needs to be passed as argument 
 *  HwControl function Call.  All the arguments (Structure elements included) 
 *  passed to  the HwControl function are inputs. For the list of commands 
 *  supported and argument type that can be @a void* casted & passed with a 
 *  particular command refer to @a CSL_Ddr2HwControlCmd.
 *
 *  @b Arguments
 *  @verbatim      
            hDdr2           Pointer to the peripheral data object of the
                            DDR2 external memory interface instance
 
            cmd             The command to this API indicates the action to be 
			                taken 
 
            arg             An optional argument @a void* casted
 
    @endverbatim
 *
 *  <b> Return Value </b>  CSL_Status
 *  @li                    CSL_SOK             -  Command successful
 *  @li                    CSL_ESYS_BADHANDLE -  The handle passed is invalid
 *  @li                    CSL_ESYS_INVCMD    -  The Command passed is invalid      
 *
 *  <b> Pre Condition </b>
 *  Both @a CSL_ddr2Init() and @a CSL_ddr2Open() must be called successfully
 *  in order before calling @a CSL_ddr2HwControl(). For the argument type that 
 *  can be @a void* casted & passed with a particular command refer to 
 *  @a CSL_Ddr2HwControlCmd.
 *
 *  <b> Post Condition </b>
 *  @n  DDR2 registers are configured according to the command passed.
 *
 * @b Modifies
 * @n DDR2 registers
 *
 *  @b Example:
 *  @verbatim
       CSL_Ddr2Handle hDdr2;
       CSL_Status status;
       CSL_Ddr2SelfRefresh command;
       
       command = CSL_DDR2_CMD_REFRESH_RATE;
       ...
       status = CSL_ddr2HwControl(hDdr2,
                                  CSL_DDR2_CMD_REFRESH_RATE,
                                  &command);
    @endverbatim
 *
 *
 * =============================================================================
 */
CSL_Status  CSL_ddr2HwControl (
    CSL_Ddr2Handle          hDdr2,
    CSL_Ddr2HwControlCmd    cmd,
    void                    *arg
);

/** ============================================================================
 * @n@b   CSL_ddr2GetHwStatus
 *
 * @b Description
 * @n This function is used to read the current device configuration, status 
 *  flags and the value present associated registers.  Following table details 
 *  the various status queries supported and the associated data structure to 
 *  record the response.  User should allocate memory for the said data type and
 *  pass its pointer as an unadorned void* argument to the status query call. 
 *  For details about the various status queries supported and the associated 
 *  data structure to record the response, refer to @a CSL_Ddr2HwStatusQuery
 *
 *   @b Arguments
 *   @verbatim      
            hDdr2           Pointer to the peripheral data object of the
                            DDR2 external memory interface instance
 
            query           The query to this API which indicates the status 
                            to be returned 
 
            response        Placeholder to return the status. @a void* casted
 
     @endverbatim
 *
 *  <b> Return Value </b>  CSL_Status
 *  @li                    CSL_SOK             - Hardware status call is 
 *                                              successful
 *  @li                    CSL_ESYS_BADHANDLE - Not a valid Handle  
 *  @li                    CSL_ESYS_INVQUERY  - Invalid Query
 *
 * <b> Pre Condition </b>
 *  Both @a CSL_ddr2Init() and @a CSL_ddr2Open() must be called successfully
 *  in order before callinbg @a CSL_ddr2GetHwStatus(). For the argument type 
 *  that can be @a void* casted & passed with a particular command refer to 
 *  @a CSL_Ddr2HwStatusQuery
 *
 * <b> Post Condition </b>
 * @n  None
 *
 * @b Modifies
 * @n Third parameter, response vlaue 
 *
 * @b Example:
 * @verbatim
      CSL_Ddr2Handle hDdr2;
      CSL_Status status;
      Uint16  response;
       ...
      status = CSL_ddr2GetHwStatus(hDdr2,
                                   CSL_DDR2_QUERY_REFRESH_RATE,
                                   &response);
   @endverbatim
 *
 *
 * =============================================================================
 */
CSL_Status  CSL_ddr2GetHwStatus (
    CSL_Ddr2Handle           hDdr2,
    CSL_Ddr2HwStatusQuery    query,
    void                     *response
);
/**
@}
*/


#ifdef __cplusplus
}
#endif

#endif

