/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2008
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 *  ============================================================================
 */

/** 
 * @file    csl_ddr2Aux.h
 *
 * @brief   Header file for functional layer of CSL
 *
 * @path    $(CSLPATH)\inc
 *
 * @desc
 *    - The defines inline function definitions 
 *
 *  Modification 1
 *    - Modified on: 11/4/2005
 *    - Reason: created the sources
 *
 *  @date 11th April, 2005
 *  @author Ramitha Mathew.
 *
 */

/*  ============================================================================
 *  Revision History
 *  ================
 *  30-May-2008 Removed CSL_ddr2GetIFRDY() as per UG.
 *  11-Nov-2008 Added CSL_ddr2GetIFRDY() and CSL_ddr2GetDdrslRate() 
 *              Removed CSL_ddr2GetSlfRfrsh() and CSL_ddr2Selfrefresh() as per UG
 *  ============================================================================
 */

#ifndef _CSL_DDR2AUX_H_
#define _CSL_DDR2AUX_H_

#include <csl_ddr2.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
 *  Status query functions of the DDR2 EMIF
 */
/** ============================================================================
 *   @n@b CSL_ddr2GetRevId
 *
 *   @b Description
 *   @n Gets the the current value of Rev ID value of DDR2 external memory
 *      interface
 *
 *   @b Arguments
 *   @verbatim
        hDdr2              Handle to the external memory interface instance
        
        status             Revision status structure  
 *   @endverbatim
 *
 *   <b> Return Value </b>
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n assigns the current value of the revision status of DDR2 external 
 *    @n memory interface to the rev status structure parameter passed as 
 *       parameter
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_Ddr2Handle    hDdr2;
        CSL_Ddr2ModIdRev  status;
        
        CSL_ddr2GetRevId(hDdr2, status);
        ...
     @endverbatim
 * ===========================================================================
 */
CSL_IDEF_INLINE
void CSL_ddr2GetRevId (
    CSL_Ddr2Handle      hDdr2,
    CSL_Ddr2ModIdRev    *status    
)
{
    status->modId = (Uint16)CSL_FEXT(hDdr2->regs->MIDR, DDR2_MIDR_MOD_ID);
    status->majRev = (Uint8)CSL_FEXT(hDdr2->regs->MIDR, DDR2_MIDR_MJ_REV);
    status->minRev = (Uint8)CSL_FEXT(hDdr2->regs->MIDR, DDR2_MIDR_MN_REV);
}

/** ============================================================================
 *  @n@b CSL_ddr2GetEndian
 *
 *  @b Description
 *  @n Gets the the current endian of DDR2 emif from the SDRAM Status register
 *
 *  @b Arguments
 *  @verbatim
       hDdr2              Handle to the external memory interface instance
       
       response           holds the endian of DDR2 after function call
 *  @endverbatim
 *
 *  <b> Return Value </b>
 *       None 
 *
 *  <b> Pre Condition </b>
 *  @n  None
 *
 *  <b> Post Condition </b>
 *  @n  None
 *
 *  @b Modifies
 *  @n None
 *
 *  @b Example
 *  @verbatim
        CSL_Ddr2Handle     hDdr2;
        Bool               *response;
        
        CSL_ddr2GetEndian(hDdr2, response);
        ...
    @endverbatim
 * ===========================================================================
 */
CSL_IDEF_INLINE
void CSL_ddr2GetEndian (
    CSL_Ddr2Handle    hDdr2,
    Uint8             *response 
)
{
    *(response) = (Uint8)CSL_FEXT(hDdr2->regs->DMCSTAT, DDR2_DMCSTAT_BE);    
}

/** ============================================================================
 *   @n@b CSL_ddr2GetIFRDY
 *
 *   @b Description
 *   @n Gets the the current IFRDY status of DDR2 emif from the SDRAM 
 *      Status register
 *
 *   @b Arguments
 *   @verbatim
        hDdr2              Handle to the external memory interface instance
        
        response           holds IFRDY status after function call
 *   @endverbatim
 *
 *   <b> Return Value </b>
 *  @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_Ddr2Handle     hDdr2;
        Bool               *response;
        
        CSL_ddr2GetIFRDY(hDdr2, response);
        ...
     @endverbatim
 * ============================================================================
 */

CSL_IDEF_INLINE
void CSL_ddr2GetIFRDY (
    CSL_Ddr2Handle    hDdr2,
    Uint8             *response 
)
{    
    *(response) = (Uint8)CSL_FEXT(hDdr2->regs->DMCSTAT, DDR2_DMCSTAT_IFRDY);
}

/** ============================================================================
 *   @n@b CSL_ddr2GetRefreshRate
 *
 *   @b Description
 *   @n Gets the refresh rate information of the DDR2 external memory interface 
 *
 *   @b Arguments
 *   @verbatim
        hDdr2        Handle to the DDR2 external memory interface instance
 *   @endverbatim
 *
 *   <b> Return Value </b> Uint16
                    
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_Ddr2Handle     hDdr2;
        Uint16   rate;
        rate = CSL_ddr2GetRefreshRate(hDdr2);
        ...
     @endverbatim
 * ============================================================================
 */
CSL_IDEF_INLINE
Uint16 CSL_ddr2GetRefreshRate (
    CSL_Ddr2Handle    hDdr2
)
{
    Uint16    rate;
    rate = (Uint16) CSL_FEXT(hDdr2->regs->SDRFC, DDR2_SDRFC_REFRESH_RATE);  
     
    return rate;
}

/** ============================================================================
 *   @n@b CSL_ddr2GetDdrslRate
 *
 *   @b Description
 *   @n Gets the the current DDR2 Slew Rate from the DDR2 memory controller control 
 *      register
 *
 *   @b Arguments
 *   @verbatim
        hDdr2              Handle to the external memory interface instance
        
        response           holds DDR2 Slew Rate status after function call
 *   @endverbatim
 *
 *   <b> Return Value </b>
 *  @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
        CSL_Ddr2Handle     hDdr2;
        Bool               *response;
        
        CSL_ddr2GetDdrslRate(hDdr2, response);
        ...
     @endverbatim
 * ============================================================================
 */
CSL_IDEF_INLINE
void CSL_ddr2GetDdrslRate (
    CSL_Ddr2Handle    hDdr2,
    Uint8             *response 
)
{    
    *(response) = (Uint8)CSL_FEXT(hDdr2->regs->DMCCTL, DDR2_DMCCTL_DDRSLRATE);
}

/** ============================================================================
 *   @n@b CSL_ddr2RefreshRate
 *
 *   @b Description
 *      This API sets the refresh rate value with the value of arg passed as 
 *      parameter
 *      
 *   @b Arguments
 *   @verbatim
        hDdr2        Handle to the DDR2 EMIF instance
        
        val          refresh rate value

     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_ddr2Init () and CSL_ddr2Open should be successfully called.
 *
 *   <b> Post Condition </b>
 *   @n  DDR2 refresh rate will be set with the argument passed
 *
 *   @b  Modifies
 *   @n  SDRAM Refresh control register
 *
 *   @b  Example
 *   @verbatim
        CSL_Ddr2Handle     hDdr2;
        void               *arg;
        (Uint16) *arg = 0x00010;
        ...
        CSL_ddr2RefreshRate(hDdr2, arg);
        ...
     @endverbatim
 * ===========================================================================
 */
CSL_IDEF_INLINE
void CSL_ddr2RefreshRate (
    CSL_Ddr2Handle    hDdr2,
    Uint16            refRate
)
{
    CSL_FINS(hDdr2->regs->SDRFC, DDR2_SDRFC_REFRESH_RATE, refRate);
}

/** ============================================================================
 *   @n@b CSL_ddr2PrioRaise
 *
 *   @b Description
 *      This API writes the arg passed into the pr_old_count field of VBUSM 
 *      Burst Priority register.
 *      
 *   @b Arguments
 *   @verbatim
        hDdr2        Handle to the DDR2 EMIF
        
        val          Value to raise priority value

     @endverbatim
 *
 *   <b> Return Value </b> 
 *       None 
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_ddr2Init () and CSL_ddr2Open should be successfully called.   
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b  Modifies
 *   @n  VBUSM Burst Priority register
 *
 *   @b  Example
 *   @verbatim
        CSL_Ddr2Handle     hDdr2;
        void               *arg;
        (Uint8) *arg = 0x00010;
        ...
        CSL_ddr2PrioRaise(hDdr2, arg); 
        ...
     @endverbatim
 * ===========================================================================
 */
CSL_IDEF_INLINE
void CSL_ddr2PrioRaise (
    CSL_Ddr2Handle    hDdr2,
    Uint8             val
)
{
    CSL_FINS(hDdr2->regs->BPRIO, DDR2_BPRIO_PRIO_RAISE, val);
}

#ifdef __cplusplus
}
#endif

#endif /* _CSL_DDR2AUX_H_ */

