/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2008
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *  ============================================================================
 */

/** @file csl_pllc.h
 *
 *  @brief PLLC functional layer API header file
 *
 *  Path: \(CSLPATH)\inc
 */

/* ============================================================================
 * Revision History
 * ================
 * 20-Mar-2008 Added PLLC for new soc
 * 23-Jul-2008 Updated for Doxygen 
 * ============================================================================
 */

/** @defgroup CSL_PLLC_API PLLC
 *
 * @section Introduction
 *
 * @subsection xxx Overview
 * The primary PLL controller (PLL1) generates the input clock to the C64x+ 
 * megamodule (including the CPU) as well as most of the system peripherals 
 * such as the multichannel buffered serial ports (McBSPs) and the external memory interface (EMIF).
 * The PLL1 controller features a software-programmable PLL multiplier controller (PLLM) and 
 * ten dividers (PREDIV, D2 through D9). The PLL1 controller uses the device input clock CLKIN1 
 * to generate a system reference clock (SYSREFCLK) and eight system clocks (SYSCLK2 through SYSCLK9).
 * The divider ratio bits of dividers D2, D4, D5, are fixed. The divider ratio bits of dividers 
 * D3, D6, D7 and D5 are programmable through the PLL controller divider registers PLLDIV3, PLLDIV6, PLLDIV7 and PLLDIV8 respectively.
 * The secondary PLL generates interface clocks for the DDR2 memory controller and is fixed.
 *
 *  @subsection References
 *    -# PLLC User's Guide SPRU806.pdf (December 2005)
 *
 */

 
#ifndef _CSL_PLLC_H_
#define _CSL_PLLC_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <cslr.h>
#include <soc.h>
#include <csl_error.h>
#include <csl_types.h>
#include <cslr_pllc.h>


/**
@defgroup CSL_PLLC_SYMBOL  PLLC Symbols Defined
@ingroup CSL_PLLC_API
*/
/**
@defgroup CSL_PLLC_DATASTRUCT  PLLC Data Structures
@ingroup CSL_PLLC_API
*/
/**
@defgroup CSL_PLLC_FUNCTION  PLLC Functions
@ingroup CSL_PLLC_API
*/
/**
@defgroup CSL_PLLC_ENUM PLLC Enumerated Data Types
@ingroup CSL_PLLC_API
*/

/**
@addtogroup CSL_PLLC_SYMBOL
@{
*/
/*****************************************************************************\
          PLLC global macro declarations
\*****************************************************************************/
/** \brief PLLC Delay */
#define CSL_PLLC_DELAY_2000			2000
/** \brief PLLC Valid Key for configure Reset Control and Reset Cfg registers */
#define CSL_PLLC_RSTCTRL_VALID_KEY	0x5A69

/** \brief PLLC Controller Status
 *  
 * Set when GO operation (divide-ratio change and
 *   clock alignment) is in progress
 */

#define CSL_PLLC_STATUS_GO CSL_FMKT (PLLC_PLLSTAT_GOSTAT, INPROG)

/** \brief PLLC Divider enable */
/** Enable predivider */
#define CSL_PLLC_DIVEN_POSTDIV    (1 << 0)
/** Enable divider D3 for SYSCLK3 */
#define CSL_PLLC_DIVEN_PLLDIV3   (1 << 1)
/** Enable divider D6 for SYSCLK6 */
#define CSL_PLLC_DIVEN_PLLDIV6   (1 << 2)
/** Enable divider D7 for SYSCLK7 */
#define CSL_PLLC_DIVEN_PLLDIV7   (1 << 3)
/** Enable divider D8 for SYSCLK8 */
#define CSL_PLLC_DIVEN_PLLDIV8   (1 << 4)

/** \brief PLLC Divider Select  */
/** Divider D3 for SYSCLK3 */
#define CSL_PLLC_DIVSEL_PLLDIV3     (1)
/** Divider D6 for SYSCLK6 */
#define CSL_PLLC_DIVSEL_PLLDIV6     (2)
/** Divider D7 for SYSCLK7 */
#define CSL_PLLC_DIVSEL_PLLDIV7     (3)
/** Divider D8 for SYSCLK8 */
#define CSL_PLLC_DIVSEL_PLLDIV8     (4)

/**
@} */

/**************************************************************************\
* PLLC global typedef declarations                                         *
\**************************************************************************/

/** @addtogroup CSL_PLLC_ENUM
 @{ */

/** \brief  PLLC Bypass Mode */
typedef enum {
    /** PLL Bypass Mode.                */
    CSL_PLLC_PLL_BYPASS_MODE   = CSL_PLLC_PLLCTL_PLLEN_BYPASS,
    /** PLL PLL Mode (not Bypass Mode). */
    CSL_PLLC_PLL_PLL_MODE      = CSL_PLLC_PLLCTL_PLLEN_PLL
} CSL_PllcPllBypassMode;


/** \brief  Enumeration for PLL divide enable/disable */
typedef enum {
    /** <b>: PLL Divider Disable</b> */
    CSL_PLLC_PLLDIV_DISABLE = 0,
    /** <b>: PLL Divider Enable</b> */
    CSL_PLLC_PLLDIV_ENABLE = 1
} CSL_PllcDivCtrl;

/** \brief Enumeration for control commands passed to \a CSL_pllcHwControl()
 *
 *  This is the set of commands that are passed to the \a CSL_pllcHwControl()
 *  with an optional argument type-casted to \a void* .
 *  The arguments to be passed with each enumeration (if any) are specified
 *  next to the enumeration
 */
typedef enum {
    /** \brief Control PLL based on the bits set in the input argument
     *  This is valid only for PLLC instance 1
     *  \param CSL_BitMask32
     */
    CSL_PLLC_CMD_PLLCONTROL,
    /** \brief Set PLL multiplier value
     *  This is valid only for PLLC instance 1
     *  \param Uint32
     */
    CSL_PLLC_CMD_SET_PLLM,
    /** \brief Sets the pre divider value
     *  \param Uint32
     */
    CSL_PLLC_CMD_SET_POSTDIV,

    /** \brief Set PLL divide ratio
     *  \param CSL_PllcDivRatio
     */
    CSL_PLLC_CMD_SET_PLLRATIO,
    /** \brief Enable/disable  PLL divider
     *  \param CSL_PllcDivideControl
     */
    CSL_PLLC_CMD_PLLDIV_CONTROL
} CSL_PllcHwControlCmd;


/** \brief Enumeration for queries passed to \a CSL_PllcGetHwStatus()
 *
 *  This is used to get the status of different operations.The arguments
 *  to be passed with each enumeration if any are specified next to
 *  the enumeration
 */
typedef enum {
    /** \brief Queries PLL Controller Status
     *  \param (CSL_BitMask32*)
     */
    CSL_PLLC_QUERY_STATUS,
    /** \brief Queries PLL SYSCLK Status
     *  \param (CSL_BitMask32*)
     */
    CSL_PLLC_QUERY_SYSCLKSTAT,
    /** \brief Queries Reset Type Status
     *  \param (CSL_BitMask32*)
     */
    CSL_PLLC_QUERY_RESETSTAT
} CSL_PllcHwStatusQuery;
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/**
 *  \brief Input parameters for setting up PLL Divide ratio
 */
typedef struct CSL_PllcDivRatio {
    /** \brief Divider number  */
    Uint32 divNum;
    /** \brief Divider Ratio  */
    Uint32 divRatio;
} CSL_PllcDivRatio;


/**
 *  \brief Input parameters for enabling\disabling PLL Divide ratio
 *  \sa CSL_PLLC_DIVSEL_DEFINE
 */
typedef struct CSL_PllcDivideControl {
    /** \brief Divider Number  */
    Uint32 divNum;
    /** \brief Divider Control (Enable/Disable) */
    CSL_PllcDivCtrl divCtrl;
} CSL_PllcDivideControl;

/**
 *  \brief Input parameters for setting up PLL Controller
 *
 *  Used to put PLLC known useful state
 */
typedef struct CSL_PllcHwSetup {
    /** \brief PLL Mode PLL/BYPASS 
      * This is valid only for PLLC instance 1
      */
    Uint32        pllMode;
    /** \brief Divider Enable/Disable */
    CSL_BitMask32 divEnable;
    /** \brief Pre-Divider 
      * This is valid only for PLLC instance 1
      */
    Uint32        postDiv;
    /** \brief PLL Multiplier 
      * This is valid only for PLLC instance 1
      */
    Uint32        pllM;
    /** \brief PLL Divider 3  
      * This is valid only for PLLC instance 1
      */
    Uint32        pllDiv3;
    /** \brief PLL Divider 6
      * This is valid only for PLLC instance 1
      */
    Uint32        pllDiv6;
    /** \brief PLL Divider 7 
      * This is valid only for PLLC instance 1
      */
    Uint32        pllDiv7;
	/** \brief PLL Divider 8 
      * This is valid only for PLLC instance 1
      */
    Uint32        pllDiv8;
    /** \brief Setup that can be used for future implementation */
    void*         extendSetup;
} CSL_PllcHwSetup;
/**
@} */

/**
@addtogroup CSL_PLLC_SYMBOL
@{
*/
/** \brief Default hardware setup parameters
 */
/** \brief Default hardware setup parameters  for PLL1 */
#define CSL_PLLC_HWSETUP_DEFAULTS_PLL1 {  \
    CSL_PLLC_PLL_BYPASS_MODE,        \
    (CSL_PLLC_DIVEN_POSTDIV | \
    CSL_PLLC_DIVEN_PLLDIV3 | \
	CSL_PLLC_DIVEN_PLLDIV6 | \
	CSL_PLLC_DIVEN_PLLDIV7 | \
    CSL_PLLC_DIVEN_PLLDIV8) ,\
    CSL_PLLC_PREDIV_RATIO_RESETVAL + 1,  \
    CSL_PLLC_PLLM_PLLM_RESETVAL  + 1,    \
	CSL_PLLC_PLLDIV3_RATIO_RESETVAL + 1, \
    CSL_PLLC_PLLDIV6_RATIO_RESETVAL + 1, \
    CSL_PLLC_PLLDIV7_RATIO_RESETVAL + 1, \
    CSL_PLLC_PLLDIV8_RATIO_RESETVAL + 1, \
    NULL                             \
}


/** \brief Default hardware setup parameters for output clock frequency of 750 MHz , 
  * CLKIN = 50MHz
 */
#define CSL_PLLC_HWSETUP_DEFAULTS_750MHZ {  \
    CSL_PLLC_PLL_PLL_MODE,           \
    0,                               \
    0,                               \
    15,                              \
    0,                               \
    0,                               \
	0,								 \
    0,                               \
    NULL                             \
}


/** \brief Default hardware setup parameters for output clock frequency of 1GHz , 
  * CLKIN = 50MHz
 */
#define CSL_PLLC_HWSETUP_DEFAULTS_1GHZ {  \
    CSL_PLLC_PLL_PLL_MODE,           \
    0,                               \
    0,                               \
    20,                              \
    0,                               \
    0,                               \
	0,								 \
    0,                               \
    NULL                             \
}
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */

/** \brief Config-structure
 *
 *  Used to configure the pllc using CSL_pllcHwSetupRaw()
 */
typedef struct CSL_PllcConfig {
	/** PLL Reset Control register  
	 */
	Uint32 RSTCTRL;

	/** PLL Reset Configuration register  
	 */
	Uint32 RSTCFG;

    /** PLL Control register 
      * This should be configured only for PLLC instance 1
      */
    Uint32 PLLCTL;

    /** PLL Multiplier Control register 
      * This should be configured only for PLLC instance 1
      */
    Uint32 PLLM;

    /** PLL Pre-Divider Control register 
      * This should be configured only for PLLC instance 1
      */
    Uint32 POSTDIV;

    /** PLL Controller Divider 3 register 
      * This should be configured only for PLLC instance 1
      */
    Uint32 PLLDIV3;

    /** PLL Controller Divider 6 register 
      * This should be configured only for PLLC instance 1
      */
    Uint32 PLLDIV6;

	/** PLL Controller Divider 7 register 
      * This should be configured only for PLLC instance 1
      */
    Uint32 PLLDIV7;

	/** PLL Controller Divider 8 register 
      * This should be configured only for PLLC instance 1
      */
    Uint32 PLLDIV8;
} CSL_PllcConfig;
/**
@} */

/**
@addtogroup CSL_PLLC_SYMBOL
@{
*/
/** \brief Default values for config structure for PLL1
 */
#define CSL_PLLC_CONFIG_DEFAULTS_PLL1 {     \
	CSL_PLLC_RSTCTRL_RESETVAL,				\
	CSL_PLLC_RSTCFG_RESETVAL,				\
    CSL_PLLC_PLLCTL_RESETVAL,               \
    CSL_PLLC_PLLM_RESETVAL,                 \
    CSL_PLLC_POSTDIV_RESETVAL,               \
    CSL_PLLC_PLLDIV3_RESETVAL,              \
    CSL_PLLC_PLLDIV6_RESETVAL,               \
    CSL_PLLC_PLLDIV7_RESETVAL,              \
    CSL_PLLC_PLLDIV8_RESETVAL               \
}

/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/** \brief Module specific context information. Present implementation of pllc
 *   CSL doesn't have any context information.
 */
typedef struct {
    /** Context information of Pllc CSL.
     *  The below declaration is just a place-holder for future implementation.
     */
    Uint16  contextInfo;
} CSL_PllcContext;
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/** \brief Module specific parameters. Present implementation of pllc CSL
 *         doesn't have any module specific parameters.
 */
typedef struct {
    /** Bit mask to be used for module specific parameters. The below
     *  declaration is just a place-holder for future implementation.
     */
    CSL_BitMask16   flags;
} CSL_PllcParam;
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/** \brief This structure contains the base-address information for the
 *         peripheral instance of the PLLC
 */
typedef struct {
    /** Base-address of the configuration registers of the peripheral
     */
    CSL_PllcRegsOvly  regs;
} CSL_PllcBaseAddress;
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */
/** \brief This object contains the reference to the instance of PLLC
 *         opened using the @a CSL_pllcOpen()
 *
 *  The pointer to this is passed to all PLLC CSL APIs
 *  This structure has the fields required to configure PLLC for any test
 *  case/application. It should be initialized as per requirements of a
 *  test case/application and passed on to the setup function
 */
typedef struct CSL_PllcObj {
    /** This is a pointer to the registers of the instance of PLLC
     *   referred to by this object
     */
    CSL_PllcRegsOvly regs;
    /** This is the instance of PLLC being referred to by this object */
    CSL_InstNum  pllcNum;
} CSL_PllcObj;
/**
@} */

/**
\addtogroup CSL_PLLC_DATASTRUCT
@{ */

/** \brief This data type is used to return the handle to the pllc functions */
typedef struct CSL_PllcObj *CSL_PllcHandle;
/**
@} */

/*******************************************************************************
 * PLLC function declarations
 ******************************************************************************/

/** @addtogroup CSL_PLLC_FUNCTION
 @{ */

/** ============================================================================
 *   @n@b CSL_pllcInit
 *
 *   @b Description
 *   @n This is the initialization function for the pllc CSL. The function
 *      must be called before calling any other API from this CSL. This
 *      function is idem-potent. Currently, the function just return status
 *      CSL_SOK, without doing anything.
 *
 *   @b Arguments
 *   @verbatim
        pContext    Pointer to module-context. As pllc doesn't have
                    any context based information user is expected to pass NULL.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                        CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
            CSL_pllcInit(NULL);
     @endverbatim
 *  ============================================================================
 */
CSL_Status CSL_pllcInit (
    CSL_PllcContext       *pContext
);


/** ============================================================================
 *   @n@b CSL_pllcOpen
 *
 *   @b Description
 *   @n This function returns the handle to the PLLC
 *      instance. This handle is passed to all other CSL APIs.
 *
 *   @b Arguments
 *   @verbatim
            pPllcObj    Pointer to pllc object.

            pllcNum     Instance of pllc CSL to be opened.

            pPllcParam  Module specific parameters.

            status      Status of the function call
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_PllcHandle
 *   @n                     Valid pllc handle will be returned if
 *                          status value is equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK            - Valid pllc handle is returned
 *   @li            CSL_ESYS_FAIL      - The pllc instance is invalid
 *   @li            CSL_ESYS_INVPARAMS - Invalid parameter
 *   @n   2.    PLLC object structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *   @n    2. PLLC object structure
 *
 *   @b Example
 *   @verbatim
            CSL_status           status;
            CSL_PllcObj          pllcObj;
            CSL_PllcHandle       hPllc;
            ...

            hPllc = CSL_pllcOpen(&pllcObj, CSL_PLLC_1, NULL, &status);
            ...

    @endverbatim
 *  ============================================================================
 */
 CSL_PllcHandle CSL_pllcOpen (
    CSL_PllcObj               *pllcObj,
    CSL_InstNum                pllcNum,
    CSL_PllcParam             *pPllcParam,
    CSL_Status                *status
);

/** ============================================================================
 *   @n@b CSL_pllcGetBaseAddress
 *
 *   @b Description
 *   @n  Function to get the base address of the peripheral instance.
 *       This function is used for getting the base address of the peripheral
 *       instance. This function will be called inside the CSL_pllcOpen()
 *       function call. This function is open for re-implementing if the user
 *       wants to modify the base address of the peripheral object to point to
 *       a different location and there by allow CSL initiated write/reads into
 *       peripheral. MMR's go to an alternate location.
 *
 *   @b Arguments
 *   @verbatim
            pllcNum         Specifies the instance of the pllc to be opened.

            pPllcParam      Module specific parameters.

            pBaseAddress    Pointer to base address structure containing base
                            address details.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_OK             - Successful on getting the base 
 *                                               address of pllc
 *   @li                    CSL_ESYS_FAIL      - The instance number is invalid.
 *   @li                    CSL_ESYS_INVPARAMS - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *   @n    2. Base address structure is modified.
 *
 *   @b Example
 *   @verbatim
         CSL_Status              status;
         CSL_PllcBaseAddress     baseAddress;

         ...

         status = CSL_pllcGetBaseAddress(CSL_PLLC_1, NULL, &baseAddress);
         ...
    @endverbatim
 *  ============================================================================
 */
CSL_Status CSL_pllcGetBaseAddress (
    CSL_InstNum                      pllcNum,
    CSL_PllcParam                   *pPllcParam,
    CSL_PllcBaseAddress             *pBaseAddress
);


/** ============================================================================
 *   @n@b CSL_pllcClose
 *
 *   @b Description
 *   @n This function closes the specified instance of PLLC.
 *
 *   @b Arguments
 *   @verbatim
            hPllc            Handle to the pllc
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Close successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n   1. PLLC instance is closed and its usage is
 *           illegal until next open
 *
 *
 *   @b Modifies
 *   @n    1. PLLC object structure
 *
 *   @b Example
 *   @verbatim
            CSL_PllcHandle   hPllc;
            CSL_status       status;
            ...

            status = CSL_pllcClose(hPllc);
     @endverbatim
 *  ============================================================================
 */
 CSL_Status CSL_pllcClose (
    CSL_PllcHandle         hPllc
);

/** ============================================================================
 *   @n@b CSL_pllcHwSetup
 *
 *   @b Description
 *   @n It configures the pllc registers as per the values passed
 *      in the hardware setup structure.
 *
 *   @b Arguments
 *   @verbatim
            hPllc            Handle to the pllc

            hwSetup          Pointer to hardware setup structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Hardware setup successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Hardware structure is not
                                                  properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  PLLC registers are configured according
 *       to the hardware setup parameters
 *
 *   @b Modifies
 *   @n PLLC registers
 *
 *   @b Example
 *   @verbatim
            CSL_PllcHandle   hPllc;
            CSL_PllcObj      pllcObj;
            CSL_PllcHwSetup  hwSetup;
            CSL_status       status;
            ...

            hPllc = CSL_pllcOpen(&pllcObj, CSL_PLLC_1, NULL, &status);
            ...

            status = CSL_pllcHwSetup(hPllc, &hwSetup);
     @endverbatim
 *  ============================================================================
 */
 CSL_Status CSL_pllcHwSetup (
    CSL_PllcHandle            hPllc,
    CSL_PllcHwSetup          *hwSetup
);

/** ============================================================================
 *   @n@b CSL_pllcHwControl
 *
 *   @b Description
 *   @n Takes a command of PLLC with an optional argument & implements it.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance

            cmd             The command to this API indicates the action to be
                            taken on PLLC.

            arg             An optional argument.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li            CSL_SOK               - Status info return successful.
 *   @li            CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li            CSL_ESYS_INVCMD       - Invalid command
 *   @li            CSL_ESYS_INVPARAMS    - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  To change PLLM, PREDIV & PLLDIVn, PLLCTL_PLLEN bit must be in BYPASS
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of PLLC.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle         hPllc;
        CSL_PllcHwControlCmd   cmd = CSL_PLLC_CMD_SET_PLLM;
        void                   arg = 0x7;
        ...

        status = CSL_pllcHwControl (hPllc, cmd, &arg);
     @endverbatim
 *  ============================================================================
 */
CSL_Status CSL_pllcHwControl (
    CSL_PllcHandle              hPllc,
    CSL_PllcHwControlCmd        cmd,
    void                       *cmdArg
);

/** ============================================================================
 *   @n@b CSL_pllcGetHwStatus
 *
 *   @b Description
 *   @n Gets the status of the different operations of PLLC.
 *
 *   @b Arguments
 *   @verbatim
            hPllc         Handle to the PLLC instance

            query         The query to this API of PLLC which indicates the
                          status to be returned.

            response      Placeholder to return the status.
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVQUERY   - Invalid query command
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle          hPllc;
        CSL_PllcHwStatusQuery   query = CSL_PLLC_QUERY_STATUS;
        CSL_BitMask32           reponse;
        ...

        status = CSL_pllcGetHwStatus (hPllc, query, &response);
     @endverbatim
 *  ============================================================================
 */
CSL_Status CSL_pllcGetHwStatus (
    CSL_PllcHandle                hPllc,
    CSL_PllcHwStatusQuery         query,
    void                         *response
);

/** ============================================================================
 *   @n@b CSL_pllcHwSetupRaw
 *
 *   @b Description
 *   @n This function initializes the device registers with the register-values
 *      provided through the Config Data structure.
 *
 *   @b Arguments
 *   @verbatim
            hpllc        Handle to the PLLC instance

            config       Pointer to config structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li            CSL_SOK               - Configuration successful
 *   @li            CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li            CSL_ESYS_INVPARAMS    - Configuration is not
                                            properly initialized
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The registers of the specified PLLC instance will be setup
 *       according to input configuration structure values.
 *
 *   @b Modifies
 *   @n Hardware registers of the specified PLLC instance.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle       hPllc;
        CSL_PllcConfig       config = CSL_PLLC_CONFIG_DEFAULTS;
        CSL_Status           status;
        ...

        status = CSL_pllcHwSetupRaw (hPllc, &config);
     @endverbatim
 * =============================================================================
 */
CSL_Status CSL_pllcHwSetupRaw (
    CSL_PllcHandle               hPllc,
    CSL_PllcConfig              *config
);

/** ============================================================================
 *   @n@b CSL_pllcGetHwSetup
 *
 *   @b Description
 *   @n It retrives the hardware setup parameters of the pllc
 *      specified by the given handle.
 *
 *   @b Arguments
 *   @verbatim
            hPllc        Handle to the pllc

            hwSetup      Pointer to the hardware setup structure
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Retrieving the hardware setup
 *                                                parameters is successful
 *   @li                    CSL_ESYS_BADHANDLE  - The handle is passed is
 *                                                invalid
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  The hardware setup structure is populated with the hardware setup
 *       parameters
 *
 *   @b Modifies
 *   @n hwSetup variable
 *
 *   @b Example
 *   @verbatim
            CSL_PllcHandle   hPllc;
            CSL_PllcHwSetup  hwSetup;
            ...

            status = CSL_pllcGetHwSetup(hPllc, &hwSetup);
     @endverbatim
 *  ============================================================================
 */
CSL_Status CSL_pllcGetHwSetup (
    CSL_PllcHandle               hPllc,
    CSL_PllcHwSetup             *hwSetup
);
/**
@} */

#ifdef __cplusplus
}
#endif

#endif /* _CSL_PLLC_H_ */

