/*  ============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2008
 *
 *   Use of this software is controlled by the terms and conditions found in the
 *   license agreement under which this software has been supplied.
 *  ============================================================================
 */

/** @file csl_pllcAux.h
 *
 *  @brief API Auxilary header file for PLLC CSL.
 *
 *  Path: \(CSLPATH)\inc
 */

/*  ============================================================================
 *  Revision History
 *  ================
 *  20-Mar-2008 Added PLLC for new soc
 *  23-Jul-2008 Updated for Doxygen
 *  ============================================================================
 */


#ifndef _CSL_PPLCAUX_H_
#define _CSL_PLLCAUX_H_

#include <csl_pllc.h>

#ifdef __cplusplus
extern "C" {
#endif

/**
@addtogroup CSL_PLLC_FUNCTION
*/

/*
 *  Status command functions of the pllc
 */

/** ============================================================================
 *   @n@b CSL_pllcGetStatus
 *
 *   @b Description
 *   @n Gets the Status of the pll controller.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  CSL_BitMask32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle    hPllc;
        CSL_BitMask32     response;
        ...

        response = CSL_pllcGetStatus (hPllc);
     @endverbatim
 *  ============================================================================
 */
CSL_IDEF_INLINE
CSL_BitMask32 CSL_pllcGetStatus (
    CSL_PllcHandle                hPllc
)
{
    CSL_BitMask32 response;
    response = (CSL_BitMask32) hPllc->regs->PLLSTAT;
    return response;
}

/*
 *  Status command functions of the pllc
 */
 
/** ============================================================================
 *   @n@b CSL_pllcGetSysClkStatus
 *
 *   @b Description
 *   @n Gets the System Clock ON/OFF Status of the pllc.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  CSL_BitMask32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle    hPllc;
        CSL_BitMask32     response;
        ...

        response = CSL_pllcGetSysClkStatus (hPllc);
     @endverbatim
 *  ============================================================================
 */
CSL_IDEF_INLINE
CSL_BitMask32 CSL_pllcGetSysClkStatus (
    CSL_PllcHandle                      hPllc
)
{
    CSL_BitMask32 response;
    response = (CSL_BitMask32) hPllc->regs->SYSTAT;
    return response;
}

/*
 *  Status command functions of the pllc
 */
 
/** ============================================================================
 *   @n@b CSL_pllcGetResetStatus
 *
 *   @b Description
 *   @n Gets the Reset Type Status of the pllc.
 *
 *   @b Arguments
 *   @verbatim
            hPllc           Handle to the PLLC instance
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  CSL_BitMask32
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n None
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle    hPllc;
        CSL_BitMask32     response;
        ...

        response = CSL_pllcGetResetStatus (hPllc);
     @endverbatim
 *  ============================================================================
 */
CSL_IDEF_INLINE
CSL_BitMask32 CSL_pllcGetResetStatus (
    CSL_PllcHandle         hPllc
)
{
    CSL_BitMask32 response;
   	response = (Uint32) hPllc->regs->RSTYPE;
    return response;
}

/*
 *  Control command functions of the pllc
 */

/** ============================================================================
 *   @n@b CSL_pllcCommandCtrl
 *
 *   @b Description
 *      Controls the pllc operation.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc PLLCMD register
		status 		 Status variable
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc operation is controlled according to value set.
 *
 *   @b Modifies
 *   @n pllc PLLCMD register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_BitMask32       loadVal;
	    CSL_Status          status
        ...

        CSL_pllcCommandCtrl (hPllc, loadVal, &status);
     @endverbatim
 *  ============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcCommandCtrl (
    CSL_PllcHandle         hPllc,
    CSL_BitMask32          loadVal,
    CSL_Status             *status
)
{
    if (hPllc->pllcNum == CSL_PLLC_1) 
	{
	    hPllc->regs->PLLCTL = (loadVal & 0xFFFF);
	    loadVal = (loadVal & 0xFFFF0000)>>16;
	    if (loadVal)
	        hPllc->regs->PLLCMD = loadVal;
	}
    else
        *status = CSL_ESYS_INVPARAMS;
}

/** ============================================================================
 *   @n@b CSL_pllcMultiplierCtrl
 *
 *   @b Description
 *      Controls the pllc Multiplier.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc PLLM register
		status 	     Status variable		
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc multiplier is controlled accordingly.
 *
 *   @b Modifies
 *   @n pllc PLLM register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        Uint32              loadVal;
	    CSL_Status          status
        ...

        CSL_pllcMultiplierCtrl (hPllc, loadVal, &status);
     @endverbatim
 *  ============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcMultiplierCtrl (
    CSL_PllcHandle            hPllc,
    Uint32                    loadVal,
    CSL_Status                *status
)
{
    if (hPllc->pllcNum == CSL_PLLC_1) 
	    CSL_FINS (hPllc->regs->PLLM, PLLC_PLLM_PLLM, loadVal - 1);
    else
        *status = CSL_ESYS_INVPARAMS;
}

/** ============================================================================
 *   @n@b CSL_pllcSetPLLDivRatio
 *
 *   @b Description
 *      Sets the pllc Dividers ratios.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc divider registers
        status       Status variable
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc dividers ratios are set.
 *
 *   @b Modifies
 *   @n pllc divider registers.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        CSL_PllcDivRatio    loadVal;
        CSL_Status          status;
        ...

        CSL_pllcSetPLLDivRatio (hPllc, loadVal, &status);
     @endverbatim
 *  ============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcSetPLLDivRatio (
    CSL_PllcHandle             hPllc,
    CSL_PllcDivRatio           loadVal,
    CSL_Status                *status
)
{
    switch (loadVal.divNum) 
	{
        case CSL_PLLC_DIVSEL_PLLDIV3:
			CSL_FINS (hPllc->regs->PLLDIV3, PLLC_PLLDIV3_RATIO,
                          (Uint32)loadVal.divRatio - 1);
            break;
        case CSL_PLLC_DIVSEL_PLLDIV6:
            CSL_FINS (hPllc->regs->PLLDIV6, PLLC_PLLDIV6_RATIO,
                          (Uint32)loadVal.divRatio - 1);
            break;
        case CSL_PLLC_DIVSEL_PLLDIV7:
			CSL_FINS (hPllc->regs->PLLDIV7, PLLC_PLLDIV7_RATIO,
                          (Uint32)loadVal.divRatio - 1);
            break;
		case CSL_PLLC_DIVSEL_PLLDIV8:
			CSL_FINS (hPllc->regs->PLLDIV8, PLLC_PLLDIV8_RATIO,
                          (Uint32)loadVal.divRatio - 1);
            break;
        
        default:
            *status = CSL_ESYS_INVPARAMS;
            break;
    }
}

/** ============================================================================
 *   @n@b CSL_pllcPLLDivCtrl
 *
 *   @b Description
 *      Controls the pllc dividers.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        loadVal      Value to be loaded to pllc dividers register.
        status       Status variable
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc dividers are controlled.
 *
 *   @b Modifies
 *   @n pllc dividers register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle              hPllc;
        CSL_PllcDivideControl       loadVal;
        CSL_Status                  status;
        ...

        CSL_pllcPLLDivCtrl (hPllc, loadVal, &status);
     @endverbatim
 *  ============================================================================
 */
CSL_IDEF_INLINE
void CSL_pllcPLLDivCtrl (
    CSL_PllcHandle         hPllc,
    CSL_PllcDivideControl  loadVal,
    CSL_Status            *status
)
{
    switch (loadVal.divNum) 
	{
		case CSL_PLLC_DIVSEL_PLLDIV3:
				CSL_FINS (hPllc->regs->PLLDIV3, PLLC_PLLDIV3_D3EN,
                          (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;
		case CSL_PLLC_DIVSEL_PLLDIV6:
				CSL_FINS (hPllc->regs->PLLDIV6, PLLC_PLLDIV6_D6EN,
                          (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;
		case CSL_PLLC_DIVSEL_PLLDIV7:
				CSL_FINS (hPllc->regs->PLLDIV7, PLLC_PLLDIV7_D7EN,
                          (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;
		case CSL_PLLC_DIVSEL_PLLDIV8:
				CSL_FINS (hPllc->regs->PLLDIV8, PLLC_PLLDIV8_D8EN,
                          (CSL_PllcDivCtrl)loadVal.divCtrl);
            break;
		default:
            *status = CSL_ESYS_INVPARAMS;
            break;
    }
}

/** ============================================================================
 *   @n@b CSL_pllcSetPreDivider
 *
 *   @b Description
 *      Enables the predivider and sets the divider value in predivider register.
 *
 *   @b Arguments
 *   @verbatim
        hPllc        Handle to the PLLC instance
        arg		     Value to be loaded
     @endverbatim
 *
 *   <b> Return Value </b>
 *   @n  None
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  pllc operation is controlled according to value set.
 *
 *   @b Modifies
 *   @n pllc control register.
 *
 *   @b Example
 *   @verbatim
        CSL_PllcHandle      hPllc;
        ...

        CSL_pllcSetPreDivider(hPllc, arg);
     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE            
void CSL_pllcSetPostDivider (
    CSL_PllcHandle       hPllc, 
    Uint8                arg 
)
{
    CSL_FINST(hPllc->regs->POSTDIV, PLLC_POSTDIV_POSTDEN, ENABLE);
    CSL_FINS(hPllc->regs->POSTDIV, PLLC_POSTDIV_RATIO, arg);
}

/**
@}
*/

#ifdef __cplusplus
}
#endif

#endif /* _CSL_PLLCAUX_H_ */

