/* =============================================================================
 *   Copyright (c) Texas Instruments Inc 2008
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 * =============================================================================
 */ 
/** ============================================================================
 *   @file  csl_tetb.h
 *
 *   @path  $(CSLPATH)\inc
 *
 *   @desc  Header file for functional layer of TETB CSL
 *  - The different symbolic constants, enumerations, structure definitions
 *    and function prototype declarations
 *
 */
/* =============================================================================
 * Revision History
 * ================
 * 24-Apr-2008 File created
 *
 * 23-Jul-2008 Updated for Doxygen
 * =============================================================================
 */

/**
 * @defgroup CSL_TETB_API TETB
 *
 * @section Introduction
 *
 * @subsection xxx Overview
 *
 * The TETB is an optional module that can be instantiated external to a TI-DSP subsystem 
 * that supports cTools trace. The trace output from the DSP system is sent out in 10-bit packets. 
 * The ADTF is then used to pack this 10-bit data into a 32-bit data, of which 2 bits are used 
 * for keeping track of how many valid packets are in the data word, and the left over 30-bits 
 * are used to store up to 3 10-bit wide packets.  
 * The TETB module is used to capture the 32-bit wide data coming out of the ADTF.
 *
 *
 * @subsection References
 *   -# Highlander TETB Requirements and Functional Specification Revision 0.0.4.0

 *
 */

#ifndef _CSL_TETB_H_
#define _CSL_TETB_H_

#ifdef __cplusplus
extern "C" {
#endif

#include <soc.h>
#include <csl_types.h>
#include <csl_error.h>
#include <cslr_tetb.h>

/**
@defgroup CSL_TETB_SYMBOL  TETB Symbols Defined
@ingroup CSL_TETB_API
*/
/**
@defgroup CSL_TETB_DATASTRUCT  TETB Data Structures
@ingroup CSL_TETB_API
*/
/**
@defgroup CSL_TETB_FUNCTION  TETB Functions
@ingroup CSL_TETB_API
*/
/**
@defgroup CSL_TETB_ENUM TETB Enumerated Data Types
@ingroup CSL_TETB_API
*/

/**************************************************************************\
* TETB global macro declarations
\**************************************************************************/
/**
@addtogroup CSL_TETB_SYMBOL
@{
*/
/** Acquisition complete */
#define CSL_TETB_STS_ACQ_COMPLETE      1
/** Acquisition NOT complete */
#define CSL_TETB_STS_ACQ_NOTCOMPLETE   0

/** Value for trace capture enable */
#define CSL_TETB_TRACEACPEN_ENABLE       1
/** Value for trace capture disable */
#define CSL_TETB_TRACEACPEN_DISABLE      0

/**
@} */

/**
@addtogroup CSL_TETB_ENUM
@{
*/
/**
 * This is the set of control commands that are passed to
 * @a CSL_tetbHwControl(), with an optional argument type-casted to @a void*
 *
 * The arguments, if any, to be passed with each command are specified
 * next to that command.
 */
typedef enum {
    /**
     * Setup RAM read pointer register
     *
     * @param arg - value to set Trace RAM
     * @return CSL_SOK
     */
    CSL_TETB_CMD_SET_RAM_RD_POINTER = 0,

    /**
     * Setup RAM write pointer register
     *
     * @param arg - value to set the Trace RAM
     * @return CSL_SOK
     */
    CSL_TETB_CMD_SET_RAM_WR_POINTER = 1,

    /**
     * Setup Trigger Counter Register
     *
     * @param arg - value to set the trigger counter
     * @return CSL_SOK
     */
    CSL_TETB_CMD_SET_TRIGG_COUNT = 2,

    /**
     * Enable Trace Capture  
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_ENA_TRACE_CAPTURE = 3,

    /**
     * Disable Trace Capture  
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_DIS_TRACE_CAPTURE = 4,

	/**
     * Set TI Mode
     *
     * @param arg - TI Mode
     * @return CSL_SOK
     */
	CSL_TETB_CMD_SET_TI_MODE = 5,

	/**
     * Set Buffer Mode
     *
     * @param arg - 0 : Circular Buffer Mode
	 *				1 : Stop Buffer Mode 
     * @return CSL_SOK
     */
	CSL_TETB_CMD_SET_BUFF_MODE = 6,

	/**
     * Set TI Mode
     *
     * @param None
     * @return CSL_SOK
     */
	CSL_TETB_CMD_RESET = 7,

	/**
     * Enable Half Full interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_ENABLE_HALF_FULL = 8,

	/**
     * Enable Full interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_ENABLE_FULL = 9,

	/**
     * Enable Overflow interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_ENABLE_OVERFLOW = 10,

	/**
     * Enable Underflow interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_ENABLE_UNDERFLOW = 11,

	/**
     * Enable ACQ_COMP interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_ENABLE_ACQ_COMP = 12,

	/**
     * Disable Half Full interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_DISABLE_HALF_FULL = 13,

	/**
     * Disable Full interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_DISABLE_FULL = 14,

	/**
     * Disable Overflow interrupt
     *
     * @param None
     * @return CSL_SOK
     */
     CSL_TETB_CMD_DISABLE_OVERFLOW = 15,

	/**
     * Disable Underflow interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_DISABLE_UNDERFLOW = 16,

	/**
     * Disable ACQ_COMP interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_DISABLE_ACQ_COMP = 17,

	/**
     * Set Half Full interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_SET_HALF_FULL = 18,

	/**
     * Set Full interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_SET_FULL = 19,

	/**
     * Set Overflow interrupt
     *
     * @param None
     * @return CSL_SOK
     */
     CSL_TETB_CMD_SET_OVERFLOW = 20,

	/**
     * Set Underflow interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_SET_UNDERFLOW = 21,

	/**
     * Set ACQ_COMP interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_SET_ACQ_COMP = 22,

	/**
     * Clear Half Full interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_CLEAR_HALF_FULL = 23,

	/**
     * Clear Full interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_CLEAR_FULL = 24,

	/**
     * Clear Overflow interrupt
     *
     * @param None
     * @return CSL_SOK
     */
     CSL_TETB_CMD_CLEAR_OVERFLOW = 25,

	/**
     * Clear Underflow interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_CLEAR_UNDERFLOW = 26,

	/**
     * Clear ACQ_COMP interrupt
     *
     * @param None
     * @return CSL_SOK
     */
    CSL_TETB_CMD_CLEAR_ACQ_COMP = 27
} CSL_TetbControlCmd;


/**
 * This is the set of query commands to get the status of various
 * operations in TETB
 *
 * The arguments, if any, to be passed with each command are specified
 * next to that command. */
typedef enum {
    /**
     * Queries the RAM Full (RAM write pointer has wrapped around)
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_RAM_FULL = 0,

    /**
     * Queries the Trigger bit set when a trigger has been observed
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_TRIG_STAUS = 1,

    /**
     * Queries the Acquisition complete
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_ACQUISITION_COMPLETE = 2,

   /**
     * Queries the DID - Device ID
     * @param (Uint32 *)
     * @return CSL_SOK
     */
	CSL_TETB_QUERY_DEVICE_ID = 3,

    /**
     * Queries the Device Type Identification Register
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_DEVICE_TYPE = 4,

    /**
     * Queries the value of RAM DEPTH
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_RAM_DEPTH = 5,

    /**
     * Queries the value of READ POINTER
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_READ_POINTER = 6,

    /**
     * Queries the value of WRITE POINTER
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_WRITE_POINTER = 7,

    /**
     * Queries the value of TRACE STATUS
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_TRACECAP_STATUS = 8, 

    /**
     * Queries the value of Trigger Counter
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_TRIGGERCOUNT = 9, 
            
    /**
     * Queries the value of Half Full Interrupt bit
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_HALF_FULL_ENABLE = 10,

	/**
     * Queries the value of Full Interrupt bit
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_FULL_ENABLE = 11,

	/**
     * Queries the value of Overflow Interrupt bit
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_OVERFLOW_ENABLE = 12,

	/**
     * Queries the value of Underflow Interrupt bit
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_UNDERFLOW_ENABLE = 13,

	/**
     * Queries the value of Half Full Interrupt bit
     * @param (Uint32 *)
     * @return CSL_SOK
     */
    CSL_TETB_QUERY_ACQ_COMP_ENABLE = 14
} CSL_TetbHwStatusQuery;
/**
@} */


/** @addtogroup CSL_TETB_DATASTRUCT
 @{ */
/**
 * @brief This structure will have the base-address information for the peripheral
 *  instance
 */
typedef struct {
    /** Base-address of the Configuration registers of TETB.
     */
    CSL_TetbRegsOvly    regs;
} CSL_TetbBaseAddress;


/**
 * @brief TETB specific parameters. Present implementation doesn't have
 *  any specific parameters.
 */
typedef struct {
    /** Bit mask to be used for module specific parameters.
     *  The below declaration is just a place-holder for future
     *  implementation.
     */
    CSL_BitMask16   flags;
} CSL_TetbParam;

/**
 * @brief TETB specific context information. Present implementation doesn't
 *  have any Context information.
 */

typedef struct {
    /** Context information of TETB.
     *  The below declaration is just a place-holder for future
     *  implementation.
     */
    Uint16  contextInfo;
} CSL_TetbContext;


/**
 * @brief This structure/object holds the context of the instance of TETB
 * opened using @a CSL_tetbOpen() function.
 *
 * Pointer to this object is passed as TETB Handle to all TETB CSL APIs.
 * @a CSL_tetbOpen() function intializes this structure based on the parameters
 * passed */
typedef struct CSL_TetbObj {
    /** Pointer to the register overlay structure of the TETB */
    CSL_TetbRegsOvly    regs;

    /** Instance of TETB being referred by this object  */
    CSL_InstNum        tetbNum;
} CSL_TetbObj;


/** @brief This is a pointer to @a CSL_TetbObj and is passed as the first
 * parameter to all TETB CSL APIs
 */
typedef struct CSL_TetbObj *CSL_TetbHandle;
/**
@} */


/* ************************************************************************\
 * TETB global function declarations
\* ************************************************************************/

/** @addtogroup CSL_TETB_FUNCTION
 @{ */
/** ============================================================================
 *   @n@b CSL_tetbInit
 *
 *   @b Description
 *   @n This is the initialization function for the TI ETB. This function must
 *      be called before calling any other API from this CSL. This function is
 *		idem-potent. Currently, the function just returns status CSL_SOK, 
 *      without doing anything.
 *
 *   @b Arguments
     @verbatim
            pContext   Context information for the instance. Should be NULL
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK - Always returns
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim
         CSL_Status    status;
         ...
         status = CSL_tetbInit(NULL);
         ...

     @endverbatim
 *  ===========================================================================
 */
CSL_Status  CSL_tetbInit (
    CSL_TetbContext    *pContext
);

/** ============================================================================
 *   @n@b CSL_tetbOpen
 *
 *   @b Description
 *   @n This function returns the handle to the TI ETB controller
 *      instance. The open call sets up the data structures for the particular
 *      instance TETB device. The device can be re-opened anytime after it has 
 *      been normally closed if so required. The handle returned by this call is
 *      input as an essential argument for rest of the APIs described for
 *      this module.
 *
 *   @b Arguments
 *   @verbatim
 *          pTetbObj     Pointer to tetb object.
 *
 *          tetbNum      Instance of  DSP TETB to be opened. 
                      
 *
 *          pTetbParam   Module specific parameters.
 *
 *          pStatus     Status of the function call
 *    @endverbatim
 *
 *   <b> Return Value </b>  CSL_TetbHandle
 *   @n                         Valid tetb handle will be returned if
 *                              status value is equal to CSL_SOK.
 *
 *   <b> Pre Condition </b>
 *   @n  CSL_tetbInit() must be called successfully in order before calling 
 *       CSL_tetbOpen().
 *
 *   <b> Post Condition </b>
 *   @n   1.    The status is returned in the status variable. If status
 *              returned is
 *   @li            CSL_SOK             Valid etb handle is returned
 *   @li            CSL_ESYS_FAIL       The tetb instance is invalid
 *   @li            CSL_ESYS_INVPARAMS  Invalid parameter
 *
 *        2.    TETB object structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. TETB object structure
 *
 *   @b Example
 *   @verbatim
 *          CSL_Status			status;
 *          CSL_TetbObj			tetbObj;
 *          CSL_TetbHandle      htEtb;
 *
 *          ...
 *               
 *          htetb = CSL_tetbOpen(&tetbObj, 
 *							   CSL_TETB_0, 
 *							   NULL, 
 *							   &status
 *							  );
 *           
 *          ...
 *  @endverbatim
 * =============================================================================
 */
CSL_TetbHandle CSL_tetbOpen (
    CSL_TetbObj      *hTetbObj,
    CSL_InstNum      tetbNum,
    CSL_TetbParam    *pTetbParam,
    CSL_Status       *pStatus
);

/** ===========================================================================
 *   @n@b CSL_tetbClose
 *
 *   @b Description
 *   @n This function closes the specified instance of TETB. 
 *
 *   @b Arguments
 *   @verbatim
            hTetb            Handle to the TETB

     @endverbatim
 *
 *   <b> Pre Condition </b>
 *   @n  Both @a CSL_tetbInit() and @a CSL_tetbOpen() must be called 
 *       successfully in order before calling @a CSL_tetbClose().
 *
 *   <b> Post Condition </b>
 *   @n The ETB CSL APIs can not be called until the ETB CSL is reopened again 
 *      using CSL_tetbOpen().
 *
 *   @b Modifies
 *   @n Obj structure values
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Close successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *
 *   @b Example
 *   @verbatim
            CSL_TetbHandle      hTetb;
            CSL_Status         status;

            ...
            
            status = CSL_tetbClose(hTetb);
     @endverbatim
 * ===========================================================================
 */
CSL_Status  CSL_tetbClose (
    CSL_TetbHandle    hTetb
);

/** ============================================================================
 *   @n@b CSL_tetbHwControl
 *
 *   @b Description
 *   @n This function takes an input control command with an optional argument
 *      and accordingly controls the operation/configuration of TETB.
 *
 *   @b Arguments
 *   @verbatim
            hTetb       TETB handle returned by successful 'open'

            cmd         The command to this API indicates the action to be
                        taken on TETB.
                        Control command, refer @ CSL_TetbControlCmd for the 
                        list of commands supported

            arg         An optional argument.
                        Optional argument as per the control command, 
                        @ void * casted

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li         CSL_SOK               - Status info return successful.
 *   @li         CSL_ESYS_BADHANDLE    - Invalid handle
 *   @li         CSL_ESYS_INVCMD       - Invalid command
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_tetbInit() and CSL_tetbOpen() must be called successfully 
 *       in order before calling CSL_tetbHwControl(). 
 *
 *       Refer to @a CSL_TetbHwControlCmd for the argument type ( void*) that 
 *       needs to be passed with the control command
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n The hardware registers of TETB.
 *
 *   @b Example
 *   @verbatim
      CSL_Status    status;
      Uint32        arg;
      CSL_TetbHandle hTetb;
      ...
      // Init successfully done
      ...
      // Open successfully done
      ...
       
      arg = CSL_TETB_TRACEACPEN_ENABLE;
      status = CSL_tetbHwControl(hTetb,
                                CSL_TETB_CMD_ENA_TRACE_CAPTURE,
                                &arg);
      ...

     @endverbatim
 * =============================================================================
 */
CSL_Status  CSL_tetbHwControl (
    CSL_TetbHandle        hTetb,
    CSL_TetbControlCmd    cmd,
    void*                 arg
);

/** ============================================================================
 *   @n@b CSL_tetbGetHwStatus
 *
 *   @b Description
 *   @n Gets the status of different operations or some setup-parameters of 
 *      TETB. The status is returned through the third parameter.
 *
 *   @b Arguments
 *   @verbatim
            hTetb     TETB handle returned by successful 'open'

            query     The query to this API of TETB which indicates the
                      status to be returned. 

                      Query command, refer @ CSL_TetbHwStatusQuery for the list 
                      of query commands supported 

            response  Response from the query. Pointer to appropriate object 
                      corresponding to the query command needs to be passed here

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Status info return successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVQUERY   - Invalid query command
 *
 *   <b> Pre Condition </b>
 *   @n  Both CSL_tetbInit() and CSL_tetbOpen() must be called successfully 
 *       in order before calling CSL_tetbGetHwStatus(). 
 *       Refer to @a CSL_TetbHwStatusQuery for the argument to be passed along 
 *       with the corresponding query command.
 *
 *   <b> Post Condition </b>
 *   @n  None
 *
 *   @b Modifies
 *   @n Third parameter response
 *
 *   @b Example
 *   @verbatim
        CSL_TetbHandle hTetb;
        CSL_Status    status;
        Uint32        response;
        ...
        status = CSL_tetbGetHwStatus(hTetb,
                                     CSL_TETB_QUERY_ACQUISITION_COMPLETE,
                                     &response);
        ...


     @endverbatim
 * =============================================================================
 */
CSL_Status  CSL_tetbGetHwStatus (
    CSL_TetbHandle           hTetb,
    CSL_TetbHwStatusQuery    myQuery,
    void*                    response
);

/** ============================================================================
 *   @n@b CSL_tetbWrite
 *
 *   @b Description
 *   @n This function writes the specified data into ETB data register.
 *
 *   @b Arguments
 *   @verbatim
            hTetb        Handle to ETB instance
            
            wrData      write data into the RWD

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Success (doesnot verify 
                                                           written data)
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   @li                    CSL_ESYS_INVPARAMS  - Invalid parameter
 *
 *   <b> Pre Condition </b>
 *   @n  Both @a CSL_tetbInit() and @a CSL_tetbOpen() must be called 
 *       successfully in order before calling @a CSL_tetbWrite().
 *
 *   <b> Post Condition </b>
 *    @n Data is written to ETB RAM Write register
 *
 *   @b Modifies
 *   @n  ETB register
 *
 * @b Example:
 * @verbatim

      Uint32           wrData;
      CSL_Status       status;
      CSL_TetbHandle    hTetb;
      ...

      status = CSL_tetbWrite(hTetb, &wrData);
      ...

   @endverbatim
 * ===========================================================================
 */
 CSL_Status CSL_tetbWrite ( 
    CSL_TetbHandle    hTetb,
    void             *wrData
);

/** ============================================================================
 *   @n@b CSL_tetbRead
 *
 *   @b Description
 *   @n This function reads TETB data.
 *
 *   @b Arguments
 *   @verbatim
            hTetb		Handle to TETB instance
            
            rdData      read data from the RRD

     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             - Read operation successful
 *   @li                    CSL_ESYS_BADHANDLE  - Invalid handle
 *   
 *   <b> Pre Condition </b>
 *   @n  Both @a CSL_tetbInit() and @a CSL_tetbOpen() must be called 
 *       successfully in order before calling @a CSL_tetbRead().
 *
 *   <b> Post Condition </b>
 *   @n None
 *
 *   @b Modifies
 *   @n None
 *
 * @b Example:
 * @verbatim

        Uint32           rdData;
        CSL_Status       status;
        CSL_TetbHandle    hTetb;
        ...
        status = CSL_tetbRead(hTetb, &rdData);
        ...

   @endverbatim
 * ===========================================================================
 */

CSL_Status CSL_tetbRead (
    CSL_TetbHandle    hTetb,
    void*            rdData
);

/** ===========================================================================
 *   @n@b CSL_tetbGetBaseAddress
 *
 *   @b Description
 *   @n  This function gets the base address of the given TETB
 *       instance.
 *
 *   @b Arguments
 *   @verbatim
            tetb			Peripheral instance numberd

            ptetbParam		Module specific parameters

            pBaseAddress    Pointer to base address structure containing base
                             address details
     @endverbatim
 *
 *   <b> Return Value </b>  CSL_Status
 *   @li                    CSL_SOK             Successful on getting the base 
 *                                              address of tetb
 *   @li                    CSL_ESYS_FAIL       pwrdwn instance is not
 *                                              available.
 *   @li                    CSL_ESYS_INVPARAMS  Invalid parameter 
 *
 *   <b> Pre Condition </b>
 *   @n  None
 *
 *   <b> Post Condition </b>
 *   @n  Base address structure is populated
 *
 *   @b Modifies
 *   @n    1. The status variable
 *
 *         2. Base address structure is modified.
 * 
 * @b Example:
   @verbatim

    CSL_TetbHandle       hTetb;
    CSL_TetbBaseAddress  baseAddress;
    CSL_TetbParam        params; 

    CSL_tetbGetBaseAddress(CSL_TETB, &params, &baseAddress) ;

   @endverbatim
 * ============================================================================
 */
CSL_Status   CSL_tetbGetBaseAddress (
        CSL_InstNum              tetbNum,
        CSL_TetbParam           *pTetbParam,
        CSL_TetbBaseAddress     *pBaseAddress
);
/**
@} */


#ifdef __cplusplus
}
#endif

#endif

