/* =============================================================================
 *   Copyright (c) Texas Instruments Inc 2002, 2003, 2004, 2005, 2006, 2008
 *
 *   Use of this software is controlled by the terms and conditions found
 *   in the license agreement under which this software has been supplied.
 * =============================================================================
 */ 
/** ============================================================================
 *   @file  csl_tetbAux.h
 *
 *   @path  $(CSLPATH)\inc
 *
 *   @desc  Header file for functional layer of CSL
 *    - This decribes the definitions of the inline functions
 *
 */
/* =============================================================================
 * Revision History
 * ================
 * 24-Apr-2008 File created
 * 
 * 25-Jul-2008 Updated for Doxygen
 * =============================================================================
 */
#ifndef _CSL_TETBAUX_H_
#define _CSL_TETBAUX_H_

#include <csl_tetb.h>

#ifdef __cplusplus
extern "C" {
#endif

/** @addtogroup CSL_TETB_FUNCTION
 @{ */

/* Status Query Functions of tetb. */
/** ============================================================================
 *   @n@b CSL_tetbGetRam 
 *
 *   @b Description
 *   @n This function returns the RAM Full status 
 *      (RAM write pointer has wrapped around)
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetRam(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetRam (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT( hTetb->regs->STS, TETB_STS_FULL));
}

/** ============================================================================
 *   @n@b CSL_tetbGetTrigSts 
 *
 *   @b Description
 *   @n This function returns the Trigger bit status (CSL_TETB_QUERY_TRIG_STAUS)
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetTrigSts(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetTrigSts (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT( hTetb->regs->STS, TETB_STS_TRIGGERED));
}

/** ============================================================================
 *   @n@b CSL_tetbGetAcq 
 *
 *   @b Description
 *   @n This function returns the Acquisition complete status 
 *      (CSL_TETB_QUERY_ACQUISITION_COMPLETE)
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetAcq(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetAcq (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT( hTetb->regs->STS, TETB_STS_ACQCOMP));
}

/** ============================================================================
 *   @n@b CSL_tetbGetDevType 
 *
 *   @b Description
 *   @n This function returns the Device Type Identification 
 *      (CSL_TETB_QUERY_DEVICE_TYPE).
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetDevType(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetDevType (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT( hTetb->regs->DTIR, TETB_DTIR_DTIR));
}

/** ============================================================================
 *   @n@b CSL_tetbGetRamDepth 
 *
 *   @b Description
 *   @n This function reads the value RAM size.
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetRamDepth(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetRamDepth (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT(hTetb->regs->RDP, TETB_RDP_RDP));  
}

/** ============================================================================
 *   @n@b CSL_tetbGetReadPointer 
 *
 *   @b Description
 *   @n This function reads RAM read pointer value.
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetReadPointer(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetReadPointer (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT(hTetb->regs->RRP, TETB_RRP_RRP));
}

/** ============================================================================
 *   @n@b CSL_tetbGetWritePointer 
 *
 *   @b Description
 *   @n This function reads RAM write pointer value.
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetWritePointer(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetWritePointer (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT(hTetb->regs->RWP, TETB_RWP_RWP));
}

/** ============================================================================
 *   @n@b CSL_tetbGetTraceCapture 
 *
 *   @b Description
 *   @n This function reads the trace capture value.
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetTraceCapture(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetTraceCapture (
    CSL_TetbHandle    hTetb, 
    void             *response
) 
{
    *(Uint32 *)response = (CSL_FEXT(hTetb->regs->CTL, TETB_CTL_TRACECAPTEN));
}

/** ============================================================================
 *   @n@b CSL_tetbGetTriggerCounter 
 *
 *   @b Description
 *   @n This function reads the value of Trigger Counter.
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetTriggerCounter(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetTriggerCounter (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT(hTetb->regs->TRG, TETB_TRG_TRG ));
}

/** ============================================================================
 *   @n@b CSL_tetbGetHalffull_Enable 
 *
 *   @b Description
 *   @n This function reads the Half Full interrupt bit.
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetHalffull_Enable(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetHalffull_Enable (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT(hTetb->regs->IEST, TETB_IEST_HALFFULL_ENABLED));
}

/** ============================================================================
 *   @n@b CSL_tetbGetFull_Enable 
 *
 *   @b Description
 *   @n This function reads the Full interrupt bit.
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetFull_Enable(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetFull_Enable (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT(hTetb->regs->IEST, TETB_IEST_FULL_ENABLED));
}

/** ============================================================================
 *   @n@b CSL_tetbGetOverflow_Enable 
 *
 *   @b Description
 *   @n This function reads the Overflow interrupt bit.
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetOverflow_Enable(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetOverflow_Enable (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT(hTetb->regs->IEST, TETB_IEST_OVERFLOW_ENABLED));
}

/** ============================================================================
 *   @n@b CSL_tetbGetUnderflow_Enable 
 *
 *   @b Description
 *   @n This function reads the Underflow interrupt bit.
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetUnderflow_Enable(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetUnderflow_Enable (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT(hTetb->regs->IEST, TETB_IEST_UNDERFLOW_ENABLED));
}

/** ============================================================================
 *   @n@b CSL_tetbGetAcqComp_Enable 
 *
 *   @b Description
 *   @n This function reads the ACQ COMP interrupt bit.
 *
 *   @b Arguments
 *   @verbatim

            hTetb          Handle to TETB instance

            response       Placeholder to return the status. @a void* casted

     @endverbatim
 *
 *   <b> Return Value </b>  
 *	 @n  None
 *
 *   <b> Pre Condition </b>
 *	 @n  None
 *
 *   <b> Post Condition </b>
 *    @n None
 *
 *   @b Modifies
 *   @n  None
 *
 *   @b Example
 *   @verbatim

       CSL_tetbGetAcqComp_Enable(hTetb, response);

     @endverbatim
 * =============================================================================
 */
CSL_IDEF_INLINE
void CSL_tetbGetAcqComp_Enable (
    CSL_TetbHandle    hTetb,
    void             *response
)
{
    *(Uint32 *)response = (CSL_FEXT(hTetb->regs->IEST, TETB_IEST_ACQ_COMP_ENABLED));
}
/**
@}
*/

#ifdef __cplusplus
}
#endif

#endif /* _CSL_TETBAUX_H_ */
