# -*- coding: utf-8 -*-
from __future__ import absolute_import

from django.views.generic import ListView, DetailView, CreateView, UpdateView, DeleteView, FormView, RedirectView
from django.core.urlresolvers import reverse_lazy
from django.shortcuts import redirect
from django.http import HttpResponse

from kindergarten.models import Kindergarten, SmartKindergarten
from kindergarten.forms import KindergartenCreateForm, SmartKindergartenSearchForm, SmartKindergartenCreateForm
from core.views import LoginRequiredMixin, page_calc, PaginationMixin, OwnerRequiredMixin


# smart_kindergarten 기반으로 유치원 생성했을때의 함수 (유치원 검색후 등록하기)
def through_smart_create(request, pk):
	smart_kindergarten = SmartKindergarten.objects.get(id=pk)
	kindergarten = Kindergarten()
	kindergarten.user_profile = request.user.userprofile
	kindergarten.smart_kindergarten = smart_kindergarten
	kindergarten.name = smart_kindergarten.name
	kindergarten.address = smart_kindergarten.address
	kindergarten.first_address = smart_kindergarten.address
	kindergarten.tel = smart_kindergarten.tel
	kindergarten.area = smart_kindergarten.area
	kindergarten.save()
	# return redirect(reverse_lazy('kindergarten:update')+'?kindergarten_id='+str(kindergarten.id))
	# return redirect('kindergarten:update', pk=kindergarten.id)
	return redirect('kindergarten:list')


# 내 영업 유치원 목록
class KindergartenLV(LoginRequiredMixin, PaginationMixin, ListView):
	template_name = 'kindergarten/kindergarten.html'
	model = Kindergarten
	paginate_by = 5

	def get_queryset(self):
		user = self.request.user
		return Kindergarten.objects.filter(user_profile__user=user).order_by('-id')


# 검색했는데 유치원이 없을경우 새로만드는 유치원 생성 CreateView
class KindergartenCV(LoginRequiredMixin, CreateView):
	form_class = KindergartenCreateForm
	template_name = 'kindergarten/kindergarten_create.html'
	# model = Kindergarten
	# fields = '__all__'
	success_url = reverse_lazy('kindergarten:list')

	def get_form_kwargs(self):
		kwargs = super(KindergartenCV, self).get_form_kwargs()
		kwargs.update({'user': self.request.user})
		return kwargs

	def form_valid(self, form):
		kindergarten = form.save(commit=False)
		kindergarten.address = kindergarten.first_address + ' ' + kindergarten.detail_address
		kindergarten.user_profile = self.request.user.userprofile
		kindergarten.save()
		return redirect(self.success_url)


# 유치원 검색 기능 담당 뷰, 검색한 결과를 페이징으로 보여준다
class SmartKindergartenSearchFormView(LoginRequiredMixin, PaginationMixin, FormView, ListView):
	form_class = SmartKindergartenSearchForm
	model = SmartKindergarten
	template_name = 'kindergarten/smart_kindergarten_search.html'
	paginate_by = 7
	success_url = reverse_lazy('kindergarten:search')

	def get_queryset(self):
		try:
			search_word = self.request.GET.get('search_word')
		except:
			search_word = ''
		if (search_word != '' and search_word != None):
			object_list = self.model.objects.filter(name__icontains=search_word)
		else:
			object_list = self.model.objects.all()
		return object_list

	def get_context_data(self, **kwargs):
		context = super(SmartKindergartenSearchFormView, self).get_context_data(**kwargs)
		search_word = self.request.GET.get('search_word')
		if (search_word != '' and search_word != None):
			context['search_word'] = self.request.GET.get('search_word')
		else:
			context['search_word'] = ''
		context['search_paging'] = True
		return context

	def post(self, request, *args, **kwargs):
		search_word = '%s' % self.request.POST['search_word']
		return redirect(self.success_url+'?search_word='+search_word)
	

# 유치원 검색후 등록한다음 보여지는 UpdateView
# html 과 form class 는 createview 와 동일
class KindergartenUV(LoginRequiredMixin, UpdateView):
	template_name = 'kindergarten/kindergarten_create.html'
	form_class = KindergartenCreateForm
	success_url = reverse_lazy('kindergarten:list')

	def get_object(self, queryset=None):
		return Kindergarten.objects.get(id=self.kwargs['pk'])

	def get_form_kwargs(self):
		kwargs = super(KindergartenUV, self).get_form_kwargs()
		kwargs.update({'user': self.request.user})
		return kwargs

	def form_valid(self, form):
		kindergarten = form.save(commit=False)
		kindergarten.address = kindergarten.first_address + ' ' + kindergarten.detail_address
		kindergarten.user_profile = self.request.user.userprofile
		kindergarten.save()
		return redirect(self.success_url)


class KindergartenUserDeleteView(LoginRequiredMixin, DeleteView):
	model = Kindergarten
	success_url = reverse_lazy('kindergarten:list')

	def get(self, request, *args, **kwargs):
		self.delete(request, *args, **kwargs)
		return redirect(self.get_success_url())
