# -*- coding: utf-8 -*-
from __future__ import unicode_literals

from django.db import models

from django.contrib.auth.models import User
from core.models import TimeStampedModel


SEASON_CHOICES = (
    ('winter', u'동복'),
    ('summer', u'하복')
)
STYLE_CHOICES = (
    ('formal', u'정복'),
    ('training', u'체육복')
)
GENDER_CHOICES = (
    ('boy', u'남아'),
    ('girl', u'여아')
)
TYPE_CHOICES = (
    ('jacket', u'자켓'),
    ('vest', u'조끼'),
    ('shirts', u'상의'),
    ('pants', u'하의'),
    ('skirt', u'치마'),
    ('one-piece', u'원피스'),
    ('overall', u'멜빵바지'),
    ('dress', u'드레스'),
    ('acc', u'기타')
)
SIDE_CHOICES = (
    ('front', 'front'),
    ('back', 'back')
)
COUPLE_TYPE_CHOICES = (
    ('popular', '인기제품'),
    ('catalog', 'e-book 카탈로그'),
    ('normal', '일반')
)


def product_upload_path(instance, filename):
    # file will be uploaded to MEDIA_ROOT/user_<id>/<filename>
    # return '{0}/{1}'.format(instance.id, filename)
    return '/'.join(['product', str(instance.id), filename])


def fabric_upload_path(instance, filename):
    return '/'.join(['product', str(instance.product.id), filename])


def grid_upload_path(instance, filename):
    return '/'.join(['product', str(instance.fabric.product.id), filename])


def fitting_upload_path(instance, filename):
    return '/'.join(['fitting', str(instance.id), filename])


def couple_upload_path(instance, filename):
    return '/'.join(['couple', str(instance.id), filename])


def option_upload_path(instance, filename):
    return '/'.join(['option', str(instance.id), filename])


def catalog_upload_path(instance, filename):
    return '/'.join(['ebook_catalog', str(instance.id), filename])


# Create your models here.
class ProductType(TimeStampedModel):
    name = models.CharField(max_length=30, choices=TYPE_CHOICES)
    index = models.IntegerField()

    class Meta:
        db_table = 'sim_product_type'

    def __unicode__(self):
        return unicode(self.name)


class Function(TimeStampedModel):
    name = models.CharField(max_length=30)
    image = models.ImageField()
    default_formal = models.BooleanField(default=False)
    default_training = models.BooleanField(default=False)

    class Meta:
        db_table = 'sim_function'
        ordering = ['id']

    def __unicode__(self):
        return unicode(self.name)


class Product(TimeStampedModel):
    user = models.ForeignKey(User, null=True)
    product_type = models.ForeignKey(ProductType, null=True, blank=True)
    name = models.CharField(max_length=30)
    season = models.CharField(max_length=10, choices=SEASON_CHOICES, null=True)
    style = models.CharField(max_length=10, choices=STYLE_CHOICES, null=True)
    code = models.CharField(max_length=30, null=True)
    txm = models.FileField(upload_to=product_upload_path, null=True, blank=True)
    xml = models.FileField(upload_to=product_upload_path, null=True, blank=True)
    main_fabric_group = models.CharField(max_length=4, null=True, blank=True)

    class Meta:
        db_table = 'sim_product'
        ordering = ['-id']

    def __unicode__(self):
        return unicode(self.name)


class MainFabric(TimeStampedModel):
    product = models.ForeignKey(Product, on_delete=models.CASCADE)
    name = models.CharField(max_length=20)
    index = models.CharField(max_length=3)
    stock90 = models.IntegerField(null=True, blank=True)
    stock100 = models.IntegerField(null=True, blank=True)
    stock110 = models.IntegerField(null=True, blank=True)
    stock120 = models.IntegerField(null=True, blank=True)
    stock130 = models.IntegerField(null=True, blank=True)
    stock140 = models.IntegerField(null=True, blank=True)
    stock150 = models.IntegerField(null=True, blank=True)
    stock = models.IntegerField(null=True, blank=True)
    price = models.IntegerField(null=True, default=0)
    image = models.ImageField(upload_to=fabric_upload_path, null=True, blank=True)
    front_mapping_image = models.ImageField(upload_to=fabric_upload_path, null=True, blank=True)
    back_mapping_image = models.ImageField(upload_to=fabric_upload_path, null=True, blank=True)
    mapping_thumb = models.ImageField(upload_to=fabric_upload_path, null=True, blank=True)
    discontinue = models.NullBooleanField(default=False, null=True, blank=True)

    class Meta:
        db_table = 'sim_main_fabric'
        ordering = ['index']

    def __unicode__(self):
        return unicode(self.name)


class Grid(TimeStampedModel):
    fabric = models.ForeignKey(MainFabric, on_delete=models.CASCADE)
    image = models.ImageField(upload_to=grid_upload_path)
    side = models.CharField(max_length=10, choices=SIDE_CHOICES)

    class Meta:
        db_table = 'sim_grid'


class FittingCategory(TimeStampedModel):
    season = models.CharField(max_length=10, choices=SEASON_CHOICES)
    style = models.CharField(max_length=10, choices=STYLE_CHOICES)
    gender = models.CharField(max_length=10, choices=GENDER_CHOICES)
    product_type_set = models.ManyToManyField(ProductType)

    class Meta:
        db_table = 'sim_fitting_category'


class Fitting(TimeStampedModel):
    code = models.CharField(max_length=30, null=True, blank=True)
    front_image = models.ImageField(upload_to=fitting_upload_path, null=True, blank=True)
    back_image = models.ImageField(upload_to=fitting_upload_path, null=True, blank=True)
    jacket_off_front_image = models.ImageField(upload_to=fitting_upload_path, null=True, blank=True)
    jacket_off_back_image = models.ImageField(upload_to=fitting_upload_path, null=True, blank=True)
    product_set = models.ManyToManyField(MainFabric)
    fitting_category = models.ForeignKey(FittingCategory)

    class Meta:
        db_table = 'sim_fitting'
        ordering = ['-id']

    def __unicode__(self):
        return self.fitting_category.season + self.fitting_category.style + self.fitting_category.gender


class Couple(TimeStampedModel):
    girl_fitting = models.ForeignKey(Fitting, on_delete=models.CASCADE, related_name='girl')
    boy_fitting = models.ForeignKey(Fitting, on_delete=models.CASCADE, related_name='boy')
    front_image = models.ImageField(upload_to=couple_upload_path, null=True, blank=True)
    back_image = models.ImageField(upload_to=couple_upload_path, null=True, blank=True)
    type = models.CharField(max_length=50, choices=COUPLE_TYPE_CHOICES, null=True, blank=True)
    user = models.ForeignKey(User, null=True, blank=True)
    name = models.CharField(max_length=80, null=True, blank=True)

    class Meta:
        db_table = 'sim_couple'
        ordering = ['-id']

    def __unicode__(self):
        # return self.front_image.url
        return 'id_'+str(self.id)+'couple_'+str(self.boy_fitting.fitting_category.id) + str(self.girl_fitting.fitting_category.id)


class OptionMainCategory(TimeStampedModel):
    name = models.CharField(max_length=30)
    visible = models.BooleanField(default=True)

    class Meta:
        db_table = 'sim_option_main_category'


class OptionSubCategory(TimeStampedModel):
    main_category = models.ForeignKey(OptionMainCategory, on_delete=models.CASCADE)
    name = models.CharField(max_length=30)
    maximum_discount_rate = models.IntegerField(default=0)
    visible = models.BooleanField(default=True)

    class Meta:
        db_table = 'sim_option_sub_category'


class Option(TimeStampedModel):
    sub_category = models.ForeignKey(OptionSubCategory, on_delete=models.CASCADE)
    name = models.CharField(max_length=30)
    price = models.IntegerField(null=True, default=0)
    thumb = models.ImageField(upload_to=option_upload_path, null=True, blank=True)
    front_image = models.ImageField(upload_to=option_upload_path, null=True, blank=True)
    back_image = models.ImageField(upload_to=option_upload_path, null=True, blank=True)
    visible = models.BooleanField(default=True)

    class Meta:
        db_table = 'sim_option'


class FinalFittingSet(TimeStampedModel):
    name = models.CharField(max_length=30)
    order_contract = models.OneToOneField('order_contract.OrderContract', null=True, blank=True, on_delete=models.CASCADE)

    class Meta:
        db_table = 'sim_final_fitting_set'

    def __unicode__(self):
        return self.name


class SelectedCoupleSet(TimeStampedModel):
    final_uniform_set = models.ForeignKey(FinalFittingSet, on_delete=models.CASCADE)
    couple = models.ForeignKey(Couple, on_delete=models.CASCADE)
    function_set = models.ManyToManyField(Function)
    discount_rate = models.IntegerField(default=0)
    acc_discount_rate = models.IntegerField(default=0)

    class Meta:
        db_table = 'sim_selected_couple_set'


class SelectedFavoriteCoupleSet(TimeStampedModel):
    final_uniform_set = models.ForeignKey(FinalFittingSet, on_delete=models.CASCADE)
    couple = models.ForeignKey(Couple, on_delete=models.CASCADE)

    class Meta:
        db_table = 'sim_selected_favorite_couple_set'


class Mannequin(TimeStampedModel):
    season = models.CharField(max_length=10, choices=SEASON_CHOICES)
    style = models.CharField(max_length=10, choices=STYLE_CHOICES)
    gender = models.CharField(max_length=10, choices=GENDER_CHOICES)
    front = models.ImageField()
    back = models.ImageField()

    class Meta:
        db_table = 'sim_mannequin'


class ElectronicBookCatalog(TimeStampedModel):
    catalog_image = models.ImageField(upload_to=catalog_upload_path, null=True, blank=True)
    couple = models.OneToOneField(Couple, on_delete=models.CASCADE)

    # def save(self, **kwargs):
    #     catalog = super(ElectronicBookCatalog, self)
    #     catalog.couple = Couple.objects.get(id=self.kwargs['couple_id'])
    #     catalog.save()